﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <map>
#include <sstream>
#include <string>

#include "testFixtures.h"
#include "utils.h"

TEST_F(CurlTest, PostFieldsTest)
{
    CURLcode cResult;

    const char *testUrl = "http://natf.com/postrequest.php";
    const char *postData = "number=27&string=This%20is%20a%20test";

    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_POST, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_POSTFIELDS, postData);
    ASSERT_EQ(CURLE_OK, cResult);

    std::stringstream responseStream;
    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &responseStream);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, StringWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    pugi::xml_document doc;
    std::string responseString = responseStream.str();
    doc.load_buffer(responseString.c_str(), responseString.length());
    pugi::xml_node postVars = doc.child("PostVariables");

    std::map<std::string, std::string> varMap;
    for (pugi::xml_node variable = postVars.child("Variable"); variable; variable = variable.next_sibling("Variable"))
    {
        const char *key = variable.child("Key").child_value();
        const char *val = variable.child("Value").child_value();
        varMap.insert(std::make_pair(key, val));
    }

    ASSERT_STREQ("This is a test", varMap["string"].c_str());
    ASSERT_STREQ("27", varMap["number"].c_str());
}

TEST_F(CurlTest, CopyPostFieldsTest)
{
    CURLcode cResult;

    const char *testUrl = "http://natf.com/postrequest.php";
    const char *postData = "number=27&string=This%20is%20a%20test";

    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_POST, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_COPYPOSTFIELDS, postData);
    ASSERT_EQ(CURLE_OK, cResult);

    std::stringstream responseStream;
    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &responseStream);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, StringWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    pugi::xml_document doc;
    std::string responseString = responseStream.str();
    doc.load_buffer(responseString.c_str(), responseString.length());
    pugi::xml_node postVars = doc.child("PostVariables");

    std::map<std::string, std::string> varMap;
    for (pugi::xml_node variable = postVars.child("Variable"); variable; variable = variable.next_sibling("Variable"))
    {
        const char *key = variable.child("Key").child_value();
        const char *val = variable.child("Value").child_value();
        varMap.insert(std::make_pair(key, val));
    }

    ASSERT_STREQ("This is a test", varMap["string"].c_str());
    ASSERT_STREQ("27", varMap["number"].c_str());
}

TEST_F(CurlTest, HttpPostTest)
{
    CURLcode cResult;

    const char *testUrl = "http://natf.com/postrequest.php";

    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    std::stringstream responseStream;
    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &responseStream);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, StringWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    // Build up the form submission
    curl_httppost *form = nullptr;
    curl_httppost *tail = nullptr;

    CURLFORMcode cfResult = curl_formadd(&form, &tail,
        CURLFORM_COPYNAME, "copy_name",
        CURLFORM_COPYCONTENTS, "copy_content",
        CURLFORM_CONTENTSLENGTH, 12,
        CURLFORM_END);
    ASSERT_EQ(CURL_FORMADD_OK, cfResult);

    const char *formName = "pointer_name";
    const char *formData = "pointer_content";
    cfResult = curl_formadd(&form, &tail,
        CURLFORM_PTRNAME, formName,
        CURLFORM_PTRCONTENTS, formData,
        CURLFORM_CONTENTSLENGTH, 12,
        CURLFORM_END);
    ASSERT_EQ(CURL_FORMADD_OK, cfResult);

    // Give the form to the handle
    curl_easy_setopt(cHandle, CURLOPT_HTTPPOST, form);
    ASSERT_EQ(CURLE_OK, cResult);

    // Do the thing!
    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    pugi::xml_document doc;
    std::string responseString = responseStream.str();
    doc.load_buffer(responseString.c_str(), responseString.length());
    pugi::xml_node postVars = doc.child("PostVariables");

    std::map<std::string, std::string> varMap;
    for (pugi::xml_node variable = postVars.child("Variable"); variable; variable = variable.next_sibling("Variable"))
    {
        const char *key = variable.child("Key").child_value();
        const char *val = variable.child("Value").child_value();
        varMap.insert(std::make_pair(key, val));
    }

    ASSERT_STREQ("copy_content", varMap["copy_name"].c_str());
    ASSERT_STREQ("pointer_cont", varMap["pointer_name"].c_str());
}
