﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/hid/hid_Vibration.h>
#include <nn/xcd/xcd.h>

namespace nnt { namespace xcd {

/*
 * @brief   振動子の状態を表す構造体です。
 */
struct VibrationState
{
    nn::hid::VibrationDeviceHandle deviceHandle;    //!< 振動子のハンドル
    nn::hid::VibrationDeviceInfo deviceInfo;        //!< 振動子のデバイス情報
    int vibrationPatternId;                         //!< 振動パターン番号
    nn::hid::VibrationValue currentVibration;       //!< 現在発生している振動
};

/**
 * @brief   右コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadRightJoy
{
    typedef nn::hid::NpadStyleJoyRight Style;
    typedef nn::hid::NpadJoyRightState State;
    static const int VibrationDeviceCountMax = 1;
};

/**
 * @brief   フルキー操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadFullKey
{
    typedef nn::hid::NpadStyleFullKey Style;
    typedef nn::hid::NpadFullKeyState State;
    static const int VibrationDeviceCountMax = 2;
};

/**
 * @brief   携帯機操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadHandheld
{
    typedef nn::hid::NpadStyleHandheld Style;
    typedef nn::hid::NpadHandheldState State;
    static const int VibrationDeviceCountMax = 2;
};

/**
 * @brief   コントローラーに対する処理を扱うインターフェースクラスです。
 */
class INpadController
{
    NN_DISALLOW_COPY(INpadController);
    NN_DISALLOW_MOVE(INpadController);

public:
    INpadController() NN_NOEXCEPT {}
    virtual ~INpadController() NN_NOEXCEPT {}

    virtual void Initialize() NN_NOEXCEPT = 0;              //!< メインループの前に呼び出す初期化処理です。
    virtual void Finalize() NN_NOEXCEPT = 0;                //!< プログラム終了前に呼び出す停止処理です。
    virtual void Update() NN_NOEXCEPT = 0;                  //!< 毎フレーム一度だけ呼び出す更新処理です。
    virtual int GetVibrationDeviceCount() NN_NOEXCEPT = 0;  //!< 搭載されている振動子の個数を取得します。
    virtual const VibrationState& GetVibrationState(int idx) NN_NOEXCEPT = 0;   //!< 振動子の状態を取得します。

    /**
     * @brief   指定されたボタンが押されているか
     */
    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンが新たに押されたか
     */
    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンが離されたか
     */
    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;
};


/**
 * @brief   操作形態ごとの処理を記述した、 INpadController の派生クラスです。
 */
template<typename Npad>
class NpadController : public INpadController
{
    NN_DISALLOW_COPY(NpadController);
    NN_DISALLOW_MOVE(NpadController);

public:
    explicit NpadController(nn::hid::NpadIdType npadId) NN_NOEXCEPT :
        m_NpadId(npadId),
        m_LoopCount(0),
        m_CurrentButtons(),
        m_PreviousButtons() {}
    virtual ~NpadController() NN_NOEXCEPT
    {
        Finalize();
    }

    virtual void Initialize() NN_NOEXCEPT
    {
        // 振動子のハンドルを取得します
        nn::hid::VibrationDeviceHandle handles[Npad::VibrationDeviceCountMax];
        m_VibrationDeviceCount = nn::hid::GetVibrationDeviceHandles(
            handles, Npad::VibrationDeviceCountMax, m_NpadId, Npad::Style::Mask);

        for (int i = 0; i < m_VibrationDeviceCount; i++)
        {
            VibrationState& v = m_VibrationStateArray[i];
            v.deviceHandle = handles[i];
            v.vibrationPatternId = 0;
            v.currentVibration = nn::hid::VibrationValue::Make();

            // 振動子を初期化します
            nn::hid::InitializeVibrationDevice(v.deviceHandle);

            // 振動子の情報を取得します
            nn::hid::GetVibrationDeviceInfo(&v.deviceInfo, v.deviceHandle);
        }
    }

    virtual void Finalize() NN_NOEXCEPT
    {
        // 振幅ゼロの振動値を送信することによって振動を停止します
        nn::hid::VibrationValue zeroVibration = nn::hid::VibrationValue::Make();

        for (int i = 0; i < m_VibrationDeviceCount; i++)
        {
            VibrationState& v = m_VibrationStateArray[i];
            nn::hid::SendVibrationValue(v.deviceHandle, zeroVibration);
        }
    }

    virtual void Update() NN_NOEXCEPT
    {
        UpdateButtons();
        UpdateVibrationPattern();
        UpdateVibrationValue();
        m_LoopCount++;
    }

    virtual int GetVibrationDeviceCount() NN_NOEXCEPT
    {
        return m_VibrationDeviceCount;
    }

    virtual const VibrationState& GetVibrationState(int idx) NN_NOEXCEPT
    {
        return m_VibrationStateArray[idx];
    }


    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
    {
        return (m_CurrentButtons & buttons) == buttons;
    }

    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
    {
        return (m_CurrentButtons & buttons) == buttons &&
            (m_PreviousButtons & buttons) != buttons;
    }

    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
    {
        return (m_CurrentButtons & buttons) != buttons &&
            (m_PreviousButtons & buttons) == buttons;
    }

private:
    void UpdateButtons() NN_NOEXCEPT
    {
        m_PreviousButtons = m_CurrentButtons;

        typename Npad::State state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_CurrentButtons = state.buttons;
    }

    void UpdateVibrationPattern() NN_NOEXCEPT
    {
        const nn::hid::NpadButtonSet LeftHandButtonMask =
            nn::hid::NpadButton::Up::Mask | nn::hid::NpadButton::Right::Mask | nn::hid::NpadButton::Down::Mask |
            nn::hid::NpadButton::Left::Mask | nn::hid::NpadButton::L::Mask | nn::hid::NpadButton::ZL::Mask;
        const nn::hid::NpadButtonSet RightHandButtonMask =
            nn::hid::NpadButton::A::Mask | nn::hid::NpadButton::B::Mask | nn::hid::NpadButton::X::Mask |
            nn::hid::NpadButton::Y::Mask | nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask;

        // 押されているボタンの個数に応じて振動パターンを更新します
        for (int i = 0; i < m_VibrationDeviceCount; i++)
        {
            VibrationState& v = m_VibrationStateArray[i];
            switch (v.deviceInfo.position)
            {
            case nn::hid::VibrationDevicePosition_Left:
                v.vibrationPatternId = (m_CurrentButtons & LeftHandButtonMask).CountPopulation();
                break;
            case nn::hid::VibrationDevicePosition_Right:
                v.vibrationPatternId = (m_CurrentButtons & RightHandButtonMask).CountPopulation();
                break;
            default:
                v.vibrationPatternId = 0;
                break;
            }
        }
    }

    void UpdateVibrationValue() NN_NOEXCEPT
    {
        for (int i = 0; i < m_VibrationDeviceCount; i++)
        {
            VibrationState& v = m_VibrationStateArray[i];

            // 振動パターンに応じた振動値を生成します
            nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make();
            switch (v.vibrationPatternId)
            {
            case 0:
                // 振動を発生させない
                break;
            case 1:
                // 周波数 160 Hz で振幅が 0.25 の振動を発生します
                vib.amplitudeLow = 0.25f;
                vib.frequencyLow = 160.0f;
                vib.amplitudeHigh = 0.0f;
                vib.frequencyHigh = 320.0f;
                break;
            case 2:
                // 周波数 160 Hz で振幅が 1.00 と 0.00 に交互に切り替わる振動を発生します
                vib.amplitudeLow = (m_LoopCount % 20 < 10) ? 1.0f : 0.0f;
                vib.frequencyLow = 160.0f;
                vib.amplitudeHigh = 0.0f;
                vib.frequencyHigh = 320.0f;
                break;
            case 3:
                // 振幅 0.50 で周波数が 160 Hz と 320 Hz に交互に切り替わる振動を発生します
                vib.amplitudeLow = (m_LoopCount % 66 < 33) ? 0.5f : 0.0f;
                vib.frequencyLow = 160.0f;
                vib.amplitudeHigh = (m_LoopCount % 66 < 33) ? 0.0f : 0.5f;
                vib.frequencyHigh = 320.0f;
                break;
            default:
                // 振幅が 0.50 で周波数が動的に変化する振動を発生します
                vib.amplitudeLow = 0.5f;
                vib.frequencyLow = 120.0f * (1.0f + (m_LoopCount % 10) / 10.0f);
                vib.amplitudeHigh = 0.0f;
                vib.frequencyHigh = 320.0f;
                break;
            }

            // 振動値を送信します
            nn::hid::SendVibrationValue(v.deviceHandle, vib);

            // 現在の振動を取得します
            nn::hid::GetActualVibrationValue(&v.currentVibration, v.deviceHandle);
        }
    }

private:
    const nn::hid::NpadIdType m_NpadId;             //!< Npad ID
    uint32_t m_LoopCount;                           //!< 振動波形を更新するためのカウンタ
    nn::hid::NpadButtonSet    m_CurrentButtons;     //!< 現在のボタン情報
    nn::hid::NpadButtonSet    m_PreviousButtons;    //!< 前回のボタン情報
    int m_VibrationDeviceCount;
    VibrationState m_VibrationStateArray[Npad::VibrationDeviceCountMax];
};

/**
 * @brief   指定された NpadIdType のスタイルに対応する NpadController を作成します。
 */
INpadController* CreateNpadController(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

}}  // nnt::xcd
