﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ProcessorTaskBase.h"
#include <test_ConsoleFrameworkOpenCv.h>
#include <test_CvUtil.h>
//#include <nn/os/os_Thread.h>
#include <nn/xcd/xcd.h>

namespace {
    const char* WINDOW_NAME_COPIED = "ProcessorTestTask Copied";
    const char* WINDOW_NAME_MENU   = "ProcessorTestTask Menu";
    const CvSize WINDOW_SIZE_MENU = cvSize(320, 740);
    const CvSize FONT_SIZE = cvSize(16, 16);

    enum
    {
        PERF_SECTION_CAPTURE,
        PERF_SECTION_USER_CALC,
        PERF_SECTION_NUM
    };

    enum
    {
        MENU_ITEM_MOMENT_SIZE_FORMAT,
    };

    enum
    {
        MENU_ITEM_CLUSTERING_SIZE_FORMAT,
    };


    struct ImageSize
    {
        int width;
        int height;
    };

    const float PIXCLK = 31.2f; // カメラのピクセルクロック

    const char* OUTPUT_IMAGE_PATH_FORMAT = "..\\..\\out\\%s_%04d%02d%02d_%02d%02d%02d_%04d.png";
    const wchar_t* OUTPUT_LOG_DIRECTORY = L"Log_%04d%02d%02d_%02d%02d%02d_%04d";
} //namespace

const uint8_t ProcessorTaskBase::DIGITAL_GAIN_SETTING_DATA[][2] =
{
    {0x10, 0x00}, // X1 16
    {0x20, 0x00}, // X2 32
    {0x40, 0x00}, // X4 64
    {0x80, 0x00}, // X8 128
    {0x00, 0x01}, // X16 256
};

const char* ProcessorTaskBase::MENU_ITEM_NAMES[ProcessorTaskBase::MENU_ITEM_NUM] =
{
    "ExposureTime",
    "DigitalGain",
    "PostProcess",
    "IR-LED",
    "IR-LED_Intensity",
    "ON-OFF_Subtraction",
    "Fps",
    "HDR",
    "LensShading",
    "6Axis",
    "LRA",
};

//============================================================
void ProcessorTaskBase::DoInitialize()
{
    TEST_ASSERT(!m_IsInitialized);

    InitializeProcessor();

    cvNamedWindow(WINDOW_NAME_COPIED);
    cvNamedWindow(WINDOW_NAME_MENU);
    for (auto i = 0; i< FORMAT_SIZE_COUNT_MAX; i++)
    {
        m_pImageCaptured[i] = cvCreateImageHeader(GetImageSizeCv(i), IPL_DEPTH_8U, 1);
        TEST_ASSERT(m_pImageCaptured[i] != NULL);
        m_pImageCopied[i] = cvCreateImage(GetImageSizeCv(i), IPL_DEPTH_8U, 3);
        TEST_ASSERT(m_pImageCopied[i] != NULL);
    }
    m_pImageMenu = cvCreateImage(WINDOW_SIZE_MENU, IPL_DEPTH_8U, 3);
    TEST_ASSERT(m_pImageMenu != NULL);
    m_Cursor = 0;
    m_ImageSizeIndex = 0;
    m_TextWriter.Initialize(FONT_SIZE);
    DoSetDefaultConfig();
    m_Perf.Initialize(PERF_SECTION_NUM);
    m_IsInitialized = true;

    DoInitializeCore();   //Initialize sub instance.
} //ProcessorTaskBase::DoInitialize()

//============================================================
void ProcessorTaskBase::DoFinalize()
{
    if (m_IsInitialized)
    {
        DoFinalizeCore();

        m_Perf.Finalize();
        m_TextWriter.Finalize();
        for (auto i = 0; i < FORMAT_SIZE_COUNT_MAX; i++)
        {
            cvReleaseImageHeader(&m_pImageCaptured[i]);
            m_pImageCaptured[i] = NULL;
            cvReleaseImage(&m_pImageCopied[i]);
            m_pImageCopied[i] = NULL;
        }
        cvReleaseImage(&m_pImageMenu);
        m_pImageMenu = NULL;
        cvDestroyWindow(WINDOW_NAME_COPIED);
        cvDestroyWindow(WINDOW_NAME_MENU);
        FinalizeProcessor();
        m_IsInitialized = false;
    }
} //ProcessorTaskBase::DoFinalize()

//============================================================
void ProcessorTaskBase::InitializeProcessor()
{
} //ProcessorTaskBase::InitializeProcessor()

//============================================================
void ProcessorTaskBase::FinalizeProcessor()
{
} //ProcessorTaskBase::FinalizeProcessor()

//============================================================
void ProcessorTaskBase::DoCalc()
{
    DoCalcCore(m_pImageCaptured[m_ImageSizeIndex], m_pImageCopied[m_ImageSizeIndex]);
    m_pDriver->GetPadData(&m_ActivePadState, &m_SensorCal);
    DoCalcMenu();
    DoSetLog();
} //ProcessorTaskBase::DoCalc()

//============================================================
void ProcessorTaskBase::DoDraw()
{
    DoDrawCore(m_pImageCopied[m_ImageSizeIndex]);
    DoDrawMenu();
    cvShowImage(WINDOW_NAME_COPIED, m_pImageCopied[m_ImageSizeIndex]);
    cvShowImage(WINDOW_NAME_MENU, m_pImageMenu);
} //ProcessorTaskBase::DoDraw()

//============================================================
void ProcessorTaskBase::DoDrawMenu()
{
    DrawMenu();
    DoDrawMenuCore(m_pImageMenu);
}

//============================================================
void ProcessorTaskBase::DoCalcMenu()
{
    CalcMenu();
    int menuCnt = 0;
    DoCalcMenuCore(&menuCnt);

    // menu 切替
    int totalMenuCnt = MENU_ITEM_NUM + menuCnt;
    s32 keycode = GetKeycode();
    if (keycode == KEY_CODE_UP)
    {
        if (--m_Cursor < 0)
        {
            m_Cursor = totalMenuCnt - 1;
        }
    }
    if (keycode == KEY_CODE_DOWN)
    {
        if (++m_Cursor >= totalMenuCnt)
        {
            m_Cursor = 0;
        }
    }
}

//============================================================
void ProcessorTaskBase::DoSetDefaultConfig()
{
    DoSetDefaultConfigCore(m_Config);
    DoSetDefaultConfigCore(m_CurrentConfig);
}

//============================================================
void ProcessorTaskBase::DoSetDefaultConfigCore(ProcessorConfig& config)
{
    config.digitalGain = GAIN_X8;
    config.exposureTimeUsec = 300;
    config.irLedMode = IR_LED_BOTH;
    config.irLedIntensity = 16;
    config.postProcess = POSTPROCESS_NONE;
    config.isOnOffSubtractionEnabled = true;
    config.isHdrEnabled = false;
    config.isLensShadingEnabled = true;
    config.fps = 133;
    config.is6AxisEnabled = false;
    config.isLraEnabled = false;
}

//============================================================
void ProcessorTaskBase::DoSetLog()
{

    s32 keycode = GetKeycode();
    if (keycode == 'v' || keycode == 'l')
    {
        if (keycode == 'v')
        {
            TEST_PRINTF("Start Save Visual Log\n");
            m_IsVisualLogEnabled = true;
            m_IsLogEnabled = true;
        }
        else if (keycode == 'l')
        {
            TEST_PRINTF("Start Save Log\n");
            m_IsVisualLogEnabled = false;
            m_IsLogEnabled = true;
        }

        wchar_t dir[256];
        SYSTEMTIME st;
        GetLocalTime(&st);
        wsprintf(dir, OUTPUT_LOG_DIRECTORY, st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
        CreateDirectory(dir, NULL);
        wcstombs(m_DirectoryPath, dir, 256);
        DoCreateLogFileCore();
        TEST_PRINTF("Create Log Directory: %s\n", dir);

    }
    else if (keycode == 's' || ((m_TotalFrame != 0) && (m_CaptureFrameCount >= m_TotalFrame)))
    {
        m_TotalFrameCount = 0;
        m_CaptureFrameCount = 0;
        m_IsLogEnabled = false;
        m_IsVisualLogEnabled = false;
        DoCloseLogFileCore();
        TEST_PRINTF("Finish Save Log\n");
    }
    else if (keycode == 'k')
    {
        m_LogMenuSelect = 0;
    }
    else if (keycode == 'j')
    {
        m_LogMenuSelect = 1;
    }
    else if (keycode == 'p')
    {
        if (m_LogMenuSelect == 0)
        {
            if (m_SkipFrame > 10000)
            {
                m_SkipFrame -= 10000;
            }
            else if (m_SkipFrame > 100 && m_SkipFrame <= 10000)
            {
                m_SkipFrame -= 100;
            }
            else
            {
                m_SkipFrame--;
            }
            m_SkipFrame = std::max(m_SkipFrame, 0);
        }
        else if(m_LogMenuSelect == 1)
        {
            if (m_TotalFrame > 10000)
            {
                m_TotalFrame -= 10000;
            }
            else if (m_TotalFrame > 100 && m_TotalFrame <= 10000)
            {
                m_TotalFrame -= 100;
            }
            else
            {
                m_TotalFrame--;
            }
            m_TotalFrame = std::max(m_TotalFrame, 0);
        }
    }
    else if (keycode == 'n')
    {
        if (m_LogMenuSelect == 0)
        {
            if (m_SkipFrame >= 10000)
            {
                m_SkipFrame += 10000;
            }
            else if (m_SkipFrame >= 100 && m_SkipFrame < 10000)
            {
                m_SkipFrame += 100;
            }
            else
            {
                m_SkipFrame++;
            }
        }
        else if(m_LogMenuSelect == 1)
        {
            if (m_TotalFrame >= 10000)
            {
                m_TotalFrame += 10000;
            }
            else if (m_TotalFrame >= 100 && m_TotalFrame < 10000)
            {
                m_TotalFrame += 100;
            }
            else
            {
                m_TotalFrame++;
            }
        }
    }
} //NOLINT(readability/fn_size)

//============================================================
void ProcessorTaskBase::DoWriteRegSetting()
{
    nn::xcd::IrWriteRegisterSetting setting;
    int index = 0;

    if (m_Config.exposureTimeUsec != m_CurrentConfig.exposureTimeUsec)
    {
        int regValue = static_cast<int>(m_Config.exposureTimeUsec * PIXCLK);
        setting.registerBlock[index].bankId = 1;
        setting.registerBlock[index].address = 0x30;
        setting.registerBlock[index].data = regValue & 0x000000FF;
        index++;
        setting.registerBlock[index].bankId = 1;
        setting.registerBlock[index].address = 0x31;
        setting.registerBlock[index].data = (regValue >> 8) & 0x000000FF;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.digitalGain != m_CurrentConfig.digitalGain)
    {
        setting.registerBlock[index].bankId = 1;
        setting.registerBlock[index].address = 0x2E;
        setting.registerBlock[index].data = DIGITAL_GAIN_SETTING_DATA[m_Config.digitalGain][0];
        index++;
        setting.registerBlock[index].bankId = 1;
        setting.registerBlock[index].address = 0x2F;
        setting.registerBlock[index].data = DIGITAL_GAIN_SETTING_DATA[m_Config.digitalGain][1];
        index++;
        setting.registerCount = index;
    }
    if (m_Config.postProcess != m_CurrentConfig.postProcess)
    {
        uint8_t data = 0x00;
        if (m_Config.postProcess == POSTPROCESS_INVERSION)
        {
            // ImageTransfer モードの時は、Inversionできない
            if (m_ImageHeader.irProcessorType != nn::xcd::IrProcessorType::ImageTransfer)
            {
                data = 0x04;
            }
        }
        else
        {
            data = 0x00;
        }
        if (m_CurrentConfig.isOnOffSubtractionEnabled)
        {
            data |= 0x03;
        }
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x0E;
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.irLedMode != m_CurrentConfig.irLedMode)
    {
        uint8_t data = 0x00;

        if (m_Config.irLedMode == IR_LED_BOTH_ALWAYS || m_Config.irLedMode == IR_LED_WIDE_ALWAYS || m_Config.irLedMode == IR_LED_NARROW_ALWAYS)
        {
            data |= 0x01;
        }
        if (m_Config.irLedMode == IR_LED_NONE || m_Config.irLedMode == IR_LED_WIDE || m_Config.irLedMode == IR_LED_WIDE_ALWAYS)
        {
            data |= 0x10;
        }
        if (m_Config.irLedMode == IR_LED_NONE || m_Config.irLedMode == IR_LED_NARROW || m_Config.irLedMode == IR_LED_NARROW_ALWAYS)
        {
            data |= 0x20;
        }
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x10;
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.irLedIntensity != m_CurrentConfig.irLedIntensity)
    {
        uint8_t data = 0x00;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x11;
        setting.registerBlock[index].data = static_cast<uint8_t>(m_Config.irLedIntensity);
        index++;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x12;
        setting.registerBlock[index].data = static_cast<uint8_t>(m_Config.irLedIntensity);
        index++;
        setting.registerCount = index;
    }
    if (m_Config.isOnOffSubtractionEnabled != m_CurrentConfig.isOnOffSubtractionEnabled)
    {
        uint8_t data = 0x00;
        if (m_Config.isOnOffSubtractionEnabled)
        {
            data |= 0x03;
        }

        if (m_Config.postProcess == POSTPROCESS_INVERSION)
        {
            // ImageTransfer モードの時は、Inversionできない
            if (m_ImageHeader.irProcessorType != nn::xcd::IrProcessorType::ImageTransfer)
            {
                data |= 0x04;
            }
        }
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x0E;
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.fps != m_CurrentConfig.fps)
    {
        uint16_t data = 0x00;
        data = 10000 / static_cast<uint16_t>(m_Config.fps);
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x04;
        setting.registerBlock[index].data = data & 0xFF;
        index++;
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x05;
        setting.registerBlock[index].data = (data >> 8) & 0xFF;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.isHdrEnabled != m_CurrentConfig.isHdrEnabled)
    {
        uint8_t data = 0x00;
        if (m_Config.isHdrEnabled)
        {
            data |= 0x01;
        }

        setting.registerBlock[index].bankId = 1;
        setting.registerBlock[index].address = 0x35;
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.isLensShadingEnabled != m_CurrentConfig.isLensShadingEnabled)
    {
        uint8_t data = 0x00;
        if (m_Config.isLensShadingEnabled)
        {
            data |= 0x01;
        }
        setting.registerBlock[index].bankId = 1;
        setting.registerBlock[index].address = 0x50;
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
    }

    // その他のレジスタ設定
    DoWriteRegSettingCore(setting, index);

    // 1度に書けるサイズ上限内かどうかをチェック(Updateレジスタ用に1つ余裕を見る)
    NN_SDK_ASSERT(index < nn::xcd::IrWriteRegisterCountMax);

    // 最後にUpdate レジスタを書き込む
    setting.registerBlock[index].bankId = 0;
    setting.registerBlock[index].address = 0x07;
    setting.registerBlock[index].data = 1;
    index++;
    setting.registerCount = index;

    if (index > 0)
    {
        //// Format設定が変更された場合は、一度readyモードに戻してから変更をする
        //if (m_IsImageSizeChanged)
        //{
        //    RequestChangeMode(nn::xcd::IrProcessorType::Ready, block);
        //    while (IsReadyMode()) {}
        //    RequestChangeMode(nn::xcd::IrProcessorType::ImageTransfer, static_cast<nn::xcd::IrImageTransferProcessorFormat>(m_ImageSizeIndex), block);
        //    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        //    m_IsImageSizeChanged = false;
        //}
        auto trialCounter = 0;
        const int TrialCountMax = 50;
        while (!m_pDriver->RequestWriteRegister(setting))
        {
            NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
            nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
            trialCounter++;
        }

        // 設定値の更新
        m_CurrentConfig = m_Config;
    }

} //NOLINT(readability/fn_size) //ProcessorTaskBase::DoWriteRegSetting()

//============================================================
void ProcessorTaskBase::CalcMenu()
{
    s32 keycode = GetKeycode();
    if (keycode == KEY_CODE_A)
    {
        // 更新
        if (m_Config.is6AxisEnabled != m_CurrentConfig.is6AxisEnabled)
        {
            m_pDriver->SetSixAxisSampling(m_Config.is6AxisEnabled);
        }
        if (m_Config.isLraEnabled != m_CurrentConfig.isLraEnabled)
        {
            m_pDriver->SetLraSending(m_Config.isLraEnabled);
        }
        DoWriteRegSetting();
    }
    if (m_ImageHeader.irProcessorType != nn::xcd::IrProcessorType::TeraPlugin)
    {
        if (m_Cursor == MENU_ITEM_EXPOSURE_TIME)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                if (--m_Config.exposureTimeUsec < EXPOSURE_TIME_USEC_MIN)
                {
                    m_Config.exposureTimeUsec = EXPOSURE_TIME_USEC_MAX;
                }
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                if (++m_Config.exposureTimeUsec > EXPOSURE_TIME_USEC_MAX)
                {
                    m_Config.exposureTimeUsec = EXPOSURE_TIME_USEC_MIN;
                }
            }
        }
        if (m_Cursor == MENU_ITEM_DIGITAL_GAIN)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.digitalGain = static_cast<DigitalGain>(m_Config.digitalGain - 1);
                if (m_Config.digitalGain < 0)
                {
                    m_Config.digitalGain = static_cast<DigitalGain>(GAIN_MAX_NUM - 1);
                }
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.digitalGain = static_cast<DigitalGain>(m_Config.digitalGain + 1);
                if (m_Config.digitalGain >= GAIN_MAX_NUM)
                {
                    m_Config.digitalGain = static_cast<DigitalGain>(0);
                }
            }
        }
        if (m_Cursor == MENU_ITEM_POSTPROCESS_MODE)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.postProcess = static_cast<PostProcess>(m_Config.postProcess - 1);
                if (m_Config.postProcess < 0)
                {
                    m_Config.postProcess = static_cast<PostProcess>(POSTPROCESS_MAX_NUM - 1);
                }
                // ImageTransfer モードの時は、Inversionできない
                if (m_ImageHeader.irProcessorType == nn::xcd::IrProcessorType::ImageTransfer)
                {
                    m_Config.postProcess = PostProcess::POSTPROCESS_NONE;
                }
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.postProcess = static_cast<PostProcess>(m_Config.postProcess + 1);
                if (m_Config.postProcess >= POSTPROCESS_MAX_NUM)
                {
                    m_Config.postProcess = static_cast<PostProcess>(0);
                }
                // ImageTransfer モードの時は、Inversionできない
                if (m_ImageHeader.irProcessorType == nn::xcd::IrProcessorType::ImageTransfer)
                {
                    m_Config.postProcess = PostProcess::POSTPROCESS_NONE;
                }
            }
        }
        if (m_Cursor == MENU_ITEM_IR_LED_MODE)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.irLedMode = static_cast<IrLedMode>(m_Config.irLedMode - 1);
                if (m_Config.irLedMode < 0)
                {
                    m_Config.irLedMode = static_cast<IrLedMode>(IR_LED_MAX_NUM - 1);
                }
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.irLedMode = static_cast<IrLedMode>(m_Config.irLedMode + 1);
                if (m_Config.irLedMode >= IR_LED_MAX_NUM)
                {
                    m_Config.irLedMode = static_cast<IrLedMode>(0);
                }
            }
        }
        if (m_Cursor == MENU_ITEM_IR_LED_INTENSITY)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.irLedIntensity = m_Config.irLedIntensity - 1;
                if (m_Config.irLedIntensity < 0)
                {
                    m_Config.irLedIntensity = IR_LED_INTENSITY_MAX;
                }
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.irLedIntensity = m_Config.irLedIntensity + 1;
                if (m_Config.irLedIntensity > IR_LED_INTENSITY_MAX)
                {
                    m_Config.irLedIntensity = 0;
                }
            }
        }
        if (m_Cursor == MENU_ITEM_ON_OFF_SUBTRACTION_MODE)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.isOnOffSubtractionEnabled = !m_Config.isOnOffSubtractionEnabled;
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.isOnOffSubtractionEnabled = !m_Config.isOnOffSubtractionEnabled;
            }
        }
        if (m_Cursor == MENU_ITEM_FPS_MODE)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                if (--m_Config.fps < FPS_MIN)
                {
                    m_Config.fps = FPS_MAX;
                }
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                if (++m_Config.fps > FPS_MAX)
                {
                    m_Config.fps = FPS_MIN;
                }
            }
        }
        if (m_Cursor == MENU_ITEM_HDR_MODE)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.isHdrEnabled = !m_Config.isHdrEnabled;
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.isHdrEnabled = !m_Config.isHdrEnabled;
            }
        }
        if (m_Cursor == MENU_ITEM_LENS_SHADING_MODE)
        {
            if (keycode == KEY_CODE_LEFT)
            {
                m_Config.isLensShadingEnabled = !m_Config.isLensShadingEnabled;
            }
            if (keycode == KEY_CODE_RIGHT)
            {
                m_Config.isLensShadingEnabled = !m_Config.isLensShadingEnabled;
            }
        }
    }
    if (m_Cursor == MENU_ITEM_6AXIS_MODE)
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_Config.is6AxisEnabled = !m_Config.is6AxisEnabled;
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.is6AxisEnabled = !m_Config.is6AxisEnabled;
        }
    }
    if (m_Cursor == MENU_ITEM_LRA_MODE)
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_Config.isLraEnabled = !m_Config.isLraEnabled;
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.isLraEnabled = !m_Config.isLraEnabled;
        }
    }
} //NOLINT(readability/fn_size) //ProcessorTaskBase::CalcMenu()

//============================================================
void ProcessorTaskBase::DrawMenu()
{
    const char* POSTPROCESS_MODE_NAMES[] = {"NONE", "INVERSION"};
    const char* IR_LED_MODE_NAMES[] = {"NONE", "WIDE", "NARROW", "BOTH", "WIDE_ALWAYS", "NARROW_ALWAYS", "BOTH_ALWAYS"};
    const char* ON_OFF_SUBTRACTION_MODE_NAMES[] = {"Disable", "Enable"};
    const char* HDR_MODE_NAMES[] = {"Disable", "Enable"};
    const char* LENS_SHADING_MODE_NAMES[] = {"Disable", "Enable"};
    const char* SIXAXIS_MODE_NAMES[] = {"Disable", "Enable"};
    const char* LRA_MODE_NAMES[] = {"Disable", "Enable"};

    cvSet(m_pImageMenu, test::CONST_COLOR_BLACK);
    for (s32 i = 0; i < MENU_ITEM_NUM; i++)
    {
        s32 tx = 2 * FONT_SIZE.width;
        s32 ty = i * FONT_SIZE.height;

        m_TextWriter.SetFontColor((m_Cursor == i) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);
        m_TextWriter.PutText(m_pImageMenu, 0, ty, "%s", (m_Cursor == i) ? ">" : " ");

        if (m_ImageHeader.irProcessorType != nn::xcd::IrProcessorType::TeraPlugin)
        {
            if (i == MENU_ITEM_EXPOSURE_TIME)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %d", MENU_ITEM_NAMES[i], m_Config.exposureTimeUsec);
            }
            if (i == MENU_ITEM_DIGITAL_GAIN)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %d", MENU_ITEM_NAMES[i], 1 << m_Config.digitalGain);
            }
            if (i == MENU_ITEM_POSTPROCESS_MODE)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], POSTPROCESS_MODE_NAMES[m_Config.postProcess]);
            }
            if (i == MENU_ITEM_IR_LED_MODE)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], IR_LED_MODE_NAMES[m_Config.irLedMode]);
            }
            if (i == MENU_ITEM_IR_LED_INTENSITY)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %d", MENU_ITEM_NAMES[i], m_Config.irLedIntensity);
            }
            if (i == MENU_ITEM_ON_OFF_SUBTRACTION_MODE)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], ON_OFF_SUBTRACTION_MODE_NAMES[m_Config.isOnOffSubtractionEnabled]);
            }
            if (i == MENU_ITEM_FPS_MODE)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %d", MENU_ITEM_NAMES[i], m_Config.fps);
            }
            if (i == MENU_ITEM_HDR_MODE)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], HDR_MODE_NAMES[m_Config.isHdrEnabled]);
            }
            if (i == MENU_ITEM_LENS_SHADING_MODE)
            {
                m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], LENS_SHADING_MODE_NAMES[m_Config.isLensShadingEnabled]);
            }
        }
        if (i == MENU_ITEM_6AXIS_MODE)
        {
            m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], SIXAXIS_MODE_NAMES[m_Config.is6AxisEnabled]);
        }
        if (i == MENU_ITEM_LRA_MODE)
        {
            m_TextWriter.PutText(m_pImageMenu, tx, ty, "%s : %s", MENU_ITEM_NAMES[i], LRA_MODE_NAMES[m_Config.isLraEnabled]);
        }
    }

    m_TextWriter.SetFontColor(test::CONST_COLOR_WHITE);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 29, "Output Log:");
    if (m_IsLogEnabled && m_IsVisualLogEnabled)
    {
        m_TextWriter.SetFontColor(test::CONST_COLOR_CYAN);
        m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 28, "    Visual");
    }
    else if (m_IsLogEnabled)
    {
        m_TextWriter.SetFontColor(test::CONST_COLOR_YELLOW);
        m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 28, "    Text");
    }
    else
    {
        m_TextWriter.SetFontColor(test::CONST_COLOR_RED);
        m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 28, "    Off");
    }

    m_TextWriter.SetFontColor((m_LogMenuSelect == 0) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);

    auto totalMilliSecs = 0;
    auto hours = 0;
    auto minutes = 0;
    auto seconds = 0;
    auto millisecs = 0;

    if (m_ImageHeader.irProcessorType == nn::xcd::IrProcessorType::ImageTransfer)
    {
        totalMilliSecs = m_SkipFrame * 15 * 256;
    }
    else
    {
        totalMilliSecs = m_SkipFrame * 15;
    }
    hours = totalMilliSecs / (1000 * 60 * 60);
    totalMilliSecs = totalMilliSecs % (1000 * 60 * 60);
    minutes = totalMilliSecs / (1000 * 60);
    totalMilliSecs = totalMilliSecs % (1000 * 60);
    seconds = totalMilliSecs / 1000;
    totalMilliSecs = totalMilliSecs % 1000;
    millisecs = totalMilliSecs;
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 27, "  SkipFrame : %d (%dh %dm %ds %dms)", m_SkipFrame, hours, minutes, seconds, millisecs);
    m_TextWriter.SetFontColor((m_LogMenuSelect == 1) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);

    if (m_TotalFrame  == 0)
    {
        m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 26, "  TotalFrame : %d (infinity)", m_TotalFrame);
    }
    else
    {
        if (m_ImageHeader.irProcessorType == nn::xcd::IrProcessorType::ImageTransfer)
        {
            totalMilliSecs = m_TotalFrame * (m_SkipFrame + 1) * 15 * 256;
        }
        else
        {
            totalMilliSecs = m_TotalFrame * (m_SkipFrame + 1) * 15;
        }
        hours = totalMilliSecs / (1000 * 60 * 60);
        totalMilliSecs = totalMilliSecs % (1000 * 60 * 60);
        minutes = totalMilliSecs / (1000 * 60);
        totalMilliSecs = totalMilliSecs % (1000 * 60);
        seconds = totalMilliSecs / 1000;
        totalMilliSecs = totalMilliSecs % 1000;
        millisecs = totalMilliSecs;
        m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 26, "  TotalFrame : %d (%dh %dm %ds %dms)", m_TotalFrame, hours, minutes, seconds, millisecs);
    }


    m_TextWriter.SetFontColor(test::CONST_COLOR_WHITE);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 24, "ControllerData:");
    char buttons[21];
    buttons[0]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::A>())         ? 'A' : '_';
    buttons[1]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::B>())         ? 'B' : '_';
    buttons[2]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Y>())         ? 'Y' : '_';
    buttons[3]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::X>())         ? 'X' : '_';
    buttons[4]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::R>())         ? 'R' : '_';
    buttons[5]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::ZR>())        ? 'r' : '_';
    buttons[6]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::StickR>())    ? 'J' : '_';
    buttons[7]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Start>())     ? '+' : '_';
    buttons[8]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Home>())      ? 'H' : '_';
    buttons[9]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::SL>())        ? 'S' : '_';
    buttons[10]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::SR>())       ? 's' : '_';
    buttons[11]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Shot>())     ? 'C' : '_';
    buttons[12]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Select>())   ? '-' : '_';
    buttons[13]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::StickL>())   ? 'j' : '_';
    buttons[14]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::L>())        ? 'L' : '_';
    buttons[15]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::ZL>())       ? 'l' : '_';
    buttons[16]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Right>())    ? '>' : '_';
    buttons[17]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Down>())     ? 'v' : '_';
    buttons[18]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Left>())     ? '<' : '_';
    buttons[19]  = (m_ActivePadState.buttons.Test<nn::xcd::PadButton::Up>())       ? '^' : '_';
    buttons[20] = '\0';
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 23, "    PadState: %s", buttons);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 22,
        "    RStick: [x] %04d [y] %04d",
        m_ActivePadState.analogStickR.x, m_ActivePadState.analogStickR.y);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 21,
        "    LStick: [x] %04d [y] %04d",
        m_ActivePadState.analogStickL.x, m_ActivePadState.analogStickL.y);

    // 加速度のキャリブレーション
    for (auto i = 0; i < 3; i++)
    {
        float accelX = (m_ActivePadState.accelerometer[i].x - m_SensorCal.accelerometerOrigin.x) / static_cast<float>(m_SensorCal.accelerometerSensitivity.x);
        float accelY = (m_ActivePadState.accelerometer[i].y - m_SensorCal.accelerometerOrigin.y) / static_cast<float>(m_SensorCal.accelerometerSensitivity.y);
        float accelZ = (m_ActivePadState.accelerometer[i].z - m_SensorCal.accelerometerOrigin.z) / static_cast<float>(m_SensorCal.accelerometerSensitivity.z);
        m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * (20 - i),
            "  Acc[%d]: % 0.4f/% 0.4f/% 0.4f", i, accelX, accelY, accelZ);
    }
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 17,
        "  Gyr[0]: %05d/%05d/%05d", m_ActivePadState.gyroscope[0].x, m_ActivePadState.gyroscope[0].y, m_ActivePadState.gyroscope[0].z);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 16,
        "  Gyr[1]: %05d/%05d/%05d", m_ActivePadState.gyroscope[1].x, m_ActivePadState.gyroscope[1].y, m_ActivePadState.gyroscope[1].z);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 15,
        "  Gyr[2]: %05d/%05d/%05d", m_ActivePadState.gyroscope[2].x, m_ActivePadState.gyroscope[2].y, m_ActivePadState.gyroscope[2].z);

    m_TextWriter.SetFontColor(test::CONST_COLOR_WHITE);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 13, "HeaderData:");
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 12, "    processorType=%d", m_ImageHeader.irProcessorType);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 11, "    frameId=%d", m_ImageHeader.frameId);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 10, "    avgIntensity(ON)=%d (OFF)=%d", m_ImageHeader.averageIntensityOn, m_ImageHeader.averageIntensityOff);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 9, "    lightPixel(ON)=%d (OFF)=%d", m_ImageHeader.lightPixelOn, m_ImageHeader.lightPixelOff);
    //m_TextWriter.SetFontColor(test::CONST_COLOR_WHITE);
    //m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 7, "Performance:");
    //m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 6, "    sensor=%lld calc=%lld", m_Perf.GetAverageSpanUsec(PERF_SECTION_CAPTURE), m_Perf.GetAverageSpanUsec(PERF_SECTION_USER_CALC));
    m_TextWriter.SetFontColor(test::CONST_COLOR_RED);
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 7, "Press J,K to choose logging menu");
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 6, "Press N,P to change logging setting");
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 5, "Press A to update");
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 4, "Press V to start visual logging");
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 3, "Press L to start text logging");
    m_TextWriter.PutText(m_pImageMenu, 0, WINDOW_SIZE_MENU.height - FONT_SIZE.height * 2, "Press S to stop logging");
} //NOLINT(readability/fn_size) //ProcessorTaskBase::DrawMenu()

//============================================================
void ProcessorTaskBase::SaveImage(const IplImage* image) const
{
    char path[1024];

    SYSTEMTIME st;
    GetLocalTime(&st);
    sprintf_s(path, OUTPUT_IMAGE_PATH_FORMAT, GetName(), st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
    s32 success = cvSaveImage(path, image);
    if (success == 0)
    {
        TEST_PRINTF("Failed to save an image: %s\n", path);
        return;
    }
    TEST_PRINTF("Save an image: %s\n", path);
} //ProcessorTaskBase::SaveImage()

//============================================================
s32 ProcessorTaskBase::GetKeycode() const
{
    const test::ConsoleFrameworkOpenCv* pFramework = dynamic_cast<const test::ConsoleFrameworkOpenCv*>(test::TaskBase::GetFrameworkConst());
    if (pFramework != NULL)
    {
        return pFramework->GetKeycode();
    }
    return 0;
} //ProcessorTaskBase::GetKeycode()

//============================================================
void ProcessorTaskBase::PutText(IplImage* pImage, const s32 x, const s32 y, const CvScalar& color, const char* fmt, ...)
{
    char tmp[1024];

    va_list list;
    va_start(list, fmt);
    vsnprintf_s(tmp, 1024, 1024 - 1, fmt, list);
    va_end(list);

    m_TextWriter.SetFontColor(color);
    m_TextWriter.PutText(pImage, x, y, tmp);
} //ProcessorTaskBase::PutText()

//============================================================
