﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HandAnalysisModeTestTask.h"
#include <test_ConsoleFrameworkOpenCv.h>
#include <test_CvUtil.h>
#include <nn/xcd/xcd.h>
#include <nn/os/os_Tick.h>

namespace
{

    const CvScalar ColorRed = {36, 28, 255};
    const CvScalar ColorWhite    = { 255,255,255, 255 };
    const CvScalar ColorGray    = { 128,128,128, 255 };
    const CvScalar ColorCyan     = { 239,174,  0, 255 };
    const CvScalar ColorMagenta     = { 255,0,  255, 255 };
    const CvScalar ColorYellow     = {0, 255, 255, 255 };
    const CvScalar ColorGreen   = {0, 255, 0, 255 };

    void DrawLine(CvArr* pImage, int x1, int y1, int x2, int y2, CvScalar color, int width)
    {
        CvPoint p1 = cvPoint(x1, y1);
        CvPoint p2 = cvPoint(x2, y2);
        cvLine(pImage, p1, p2, color, width);
    }

    void DrawCircle(CvArr* pImage, int ox, int oy, int radius, CvScalar color, int width)
    {
        CvPoint p = cvPoint(ox, oy);
        cvCircle(pImage, p, radius, color, width);
    }

    const char* OUTPUT_IMAGE_FILE_NAME = "%s\\%s_%04d%02d%02d_%02d%02d%02d_%04d.png";
    const char* OUTPUT_HANDANALYSIS_FILE_NAME = "%s\\%s_%s.csv";
    FILE* fp;
    const size_t FILE_NAME_SIZE = 256;
    const size_t LINE_SIZE = 4096;
}

//============================================================
void HandAnalysisModeTestTask::DoInitializeCore()
{
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::TeraPlugin))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::TeraPlugin;
} //HandAnalysisModeTestTask::DoInitializeCore()

//============================================================
void HandAnalysisModeTestTask::DoFinalizeCore()
{
    m_pDriver->SetSixAxisSampling(false);
    m_pDriver->SetLraSending(false);
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Ready))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Ready;
} //HandAnalysisModeTestTask::DoFinalizeCore()

//============================================================
void HandAnalysisModeTestTask::DoCalcCore(IplImage* pImage, IplImage* pDstImage)
{
    NN_UNUSED(pImage);
    NN_UNUSED(pDstImage);

    m_SamplingResult = m_pDriver->GetHandAnalysisData(&m_ImageHeader, &m_State);
} //HandAnalysisModeTestTask::DoCalc()

//============================================================
void HandAnalysisModeTestTask::DoDrawCore(IplImage* pImage)
{
    cvSetZero(pImage);
    DrawHandAnalysis(m_State, pImage);
    if (m_IsLogEnabled)
    {
        SaveData(pImage, m_State);
    }
} //HandAnalysisModeTestTask::DoDraw()


void HandAnalysisModeTestTask::DrawHandAnalysis( nn::xcd::IrTeraPluginProcessorState& handAnalysisProcessorState, IplImage* pResultImage)
{
    NN_UNUSED(pResultImage);
    NN_UNUSED(handAnalysisProcessorState);
}

//============================================================
void HandAnalysisModeTestTask::DoDrawMenuCore(IplImage* pImage)
{
    NN_UNUSED(pImage);
} //HandAnalysisModeTestTask::DoDraw()


//============================================================
void HandAnalysisModeTestTask::DoCalcMenuCore(int* pMenuCnt)
{
    static nn::os::Tick keyPressedTime = nn::os::Tick();
    static int counter = 0;
    *pMenuCnt = MENU_ITEM_CLUSTERING_NUM;
}//NOLINT(readability/fn_size)

void HandAnalysisModeTestTask::DoWriteRegSettingCore(nn::xcd::IrWriteRegisterSetting& setting, int& index)
{
    NN_UNUSED(setting);
    NN_UNUSED(index);
}

void HandAnalysisModeTestTask::DoSetDefaultConfigCore(ProcessorConfig& config)
{
    NN_UNUSED(config);
    // HandAnalysisモードでは自動設定
}

void HandAnalysisModeTestTask::SetDefaultHandAnalysisConfig()
{
}

void HandAnalysisModeTestTask::SaveData(IplImage* pImage, nn::xcd::IrTeraPluginProcessorState& state)
{
    NN_UNUSED(state);
    char path[1024];

    SYSTEMTIME st;
    GetLocalTime(&st);

    m_TotalFrameCount++;
    bool isCaptureEnabled = false;
    if (m_SkipFrame == 0 || (m_TotalFrameCount % m_SkipFrame == 0))
    {
        m_CaptureFrameCount++;
        isCaptureEnabled = true;
    }

    if (isCaptureEnabled)
    {
        if (m_IsVisualLogEnabled && m_SamplingResult.IsSuccess())
        {
            sprintf_s(path, OUTPUT_IMAGE_FILE_NAME, m_DirectoryPath, GetName(), st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            s32 success = cvSaveImage(path, pImage);
            if (success == 0)
            {
                TEST_PRINTF("Failed to save an image: %s\n", path);
                return;
            }
        }

        // データログ
        if (fp != NULL)
        {
            char data[LINE_SIZE];
            sprintf_s(data, "%04d%02d%02d_%02d%02d%02d_%04d", st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            if (m_SamplingResult.IsSuccess())
            {
                sprintf_s(data, "%s\n", data);
            }
            else
            {
                sprintf_s(data, "%s,SamplingFailed,%0X\n", data, m_SamplingResult.GetInnerValueForDebug());
            }
            fputs(data, fp);
        }
    }
}

void HandAnalysisModeTestTask::DoCreateLogFileCore()
{
    char path[FILE_NAME_SIZE];
    sprintf_s(path, OUTPUT_HANDANALYSIS_FILE_NAME, m_DirectoryPath, GetName(), m_DirectoryPath);
    if ((fp = fopen(path, "w")) == NULL)
    {
        TEST_PRINTF("Failed to create file\n");
    }
    char data[LINE_SIZE];
    sprintf_s(data, "time,samplingNumber,deltaTime,objectsCount");
    //for (int i = 0; i< nn::xcd::IrTeraPluginProcessorObjectCountMax; i++)
    //{
    //    sprintf_s(data, "%s,objects[%d].averageIntensity,objects[%d].centroidX,objects[%d].centoridY,objects[%d].pixelCount,objects[%d].boundX,objects[%d].boundY,objects[%d].boundWidth,objects[%d].boundHeight", data, i, i, i, i, i, i, i, i);
    //}
    sprintf_s(data, "%s\n", data);
    fputs(data, fp);
}

void HandAnalysisModeTestTask::DoCloseLogFileCore()
{
    fclose(fp);
}

