﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_Common.h"
#include "test_Time.h"

namespace test {

//============================================================
//! @brief 1 区間の CPU 時間の計測情報です。
class PerfUnit
{
public:
    PerfUnit()
    : m_BeginTick()
    , m_LatestSpan()
    , m_SumSpan()
    {}

    virtual ~PerfUnit()
    {}

private:
    PerfUnit(const PerfUnit& src);
    PerfUnit& operator=(const PerfUnit& rhs);

public:
    //! @brief 情報をクリアします。
    void Clear()
    {
        m_BeginTick = TickTime();
        m_LatestSpan = TimeSpan();
        m_SumSpan = TimeSpan();
        m_Count = 0;
    }

    //! @brief 計測を開始します。
    void Begin()
    {
        m_BeginTick.SetCurrentCpuTime();
    }

    //! @brief 計測を終了します。
    void End()
    {
        TickTime endTick;
        endTick.SetCurrentCpuTime();
        m_LatestSpan = (endTick - m_BeginTick).ToTimeSpan();
        m_SumSpan += m_LatestSpan;
        m_Count++;
    }

    //! @brief 計測値を直接セットします。
    void SetLatestSpan(const TimeSpan& span)
    {
        m_LatestSpan = span;
        m_SumSpan += m_LatestSpan;
        m_Count++;
    }

    //! @brief 最新の計測時間を取得します。
    const TimeSpan& GetLatestSpan() const
    {
        return m_LatestSpan;
    }

    //! @brief 合計時間を取得します。
    const TimeSpan& GetSumSpan() const
    {
        return m_SumSpan;
    }

    //! @brief 計測回数を取得します。
    const s32 GetCount() const
    {
        return m_Count;
    }

private:
    TickTime m_BeginTick;
    TimeSpan m_LatestSpan;
    TimeSpan m_SumSpan;
    s32 m_Count;
}; //class PerfUnit

//============================================================
//! @brief CPU 時間の計測器です。
class Perf
{
public:
    Perf()
    : m_IsInitialized(false)
    , m_SectionNum(0)
    , m_aPerfUnit(NULL)
    {}

    virtual ~Perf()
    {
        if (m_IsInitialized)
        {
            Finalize();
        }
    }

public:
    //! @brief 初期化します。内部で new[] を呼びます。
    void Initialize(const s32 sectionNum);

    //! @brief 終了します。内部で delete[] を呼びます。
    void Finalize();

    //! @brief 計測結果をクリアします。
    void Clear()
    {
        TEST_ASSERT(m_IsInitialized);
        for (s32 i = 0; i < m_SectionNum; i++)
        {
            m_aPerfUnit[i].Clear();
        }
    }

    //! @brief 計測を開始します。
    void BeginMeasure(const s32 sectionIndex)
    {
        TEST_ASSERT(m_IsInitialized);
        TEST_ASSERT(sectionIndex < m_SectionNum);
        m_aPerfUnit[sectionIndex].Begin();
    }

    //! @brief 計測を終了します。
    void EndMeasure(const s32 sectionIndex)
    {
        TEST_ASSERT(m_IsInitialized);
        TEST_ASSERT(sectionIndex < m_SectionNum);
        m_aPerfUnit[sectionIndex].End();
    }

    //! @brief 計測値を直接セットします。
    void SetMeasure(const s32 sectionIndex, const TimeSpan& span)
    {
        TEST_ASSERT(m_IsInitialized);
        TEST_ASSERT(sectionIndex < m_SectionNum);
        m_aPerfUnit[sectionIndex].SetLatestSpan(span);
    }

    //! @brief 最新の計測時間を取得します。
    TimeSpan GetLatestSpan(const s32 sectionIndex) const
    {
        TEST_ASSERT(m_IsInitialized);
        TEST_ASSERT(sectionIndex < m_SectionNum);
        return m_aPerfUnit[sectionIndex].GetLatestSpan();
    }

    //! @brief 平均の計測時間 (単位 usec) を取得します。
    s64 GetAverageSpanUsec(const s32 sectionIndex) const
    {
        TEST_ASSERT(m_IsInitialized);
        TEST_ASSERT(sectionIndex < m_SectionNum);
        const TimeSpan& sumSpan = m_aPerfUnit[sectionIndex].GetSumSpan();
        const s32 count = m_aPerfUnit[sectionIndex].GetCount();
        if (count > 0)
        {
            return (sumSpan.GetMicroSeconds() / static_cast<s64>(count));
        }
        return 0;
    }

    //! @brief 計測回数を取得します。
    s32 GetMeasureCount(const s32 sectionIndex) const
    {
        TEST_ASSERT(m_IsInitialized);
        TEST_ASSERT(sectionIndex < m_SectionNum);
        return m_aPerfUnit[sectionIndex].GetCount();
    }

    //! @brief 計測区間の数を取得します。
    s32 GetSectionNum() const
    {
        return m_SectionNum;
    }

    //! @brief 計測結果を出力します。
    void Print() const;

private:
    bool m_IsInitialized;
    s32 m_SectionNum;
    PerfUnit* m_aPerfUnit;
}; //class Perf

//============================================================

} //namespace test

