﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_TimeSpan.h>
#include <nn/xcd/xcd.h>

namespace nnt {

class XcdIrSensorDriver final
{
    NN_DISALLOW_COPY(XcdIrSensorDriver);
    NN_DISALLOW_MOVE(XcdIrSensorDriver);

private:
    enum class IrSensorDriverState
    {
        InitializeMcu,
        FinalizeMcu,
        ModeSetting,
        DataRead,
        ReadRegister,
        WriteRegister,
        Ready,
    };

    struct XcdDeviceBlock
    {
        nn::xcd::DeviceHandle handle;
        nn::os::SystemEventType event;
        nn::os::MultiWaitHolderType holder;
        nn::xcd::DeviceInfo info;
        nn::xcd::PadState pad;
        nn::xcd::DeviceStatus status;
        nn::xcd::SensorCalibrationValue cal;
        nn::xcd::PadButtonSet previousButton;
        nn::TimeSpan previousTime;
        bool connected;
    };

    nn::os::Mutex m_Mutex;
    bool m_IsDriverInitialized;
    IrSensorDriverState m_DriverState;
    XcdDeviceBlock m_XcdDevice;
    nn::xcd::IrProcessorType m_CurrentProcessorType;
    nn::xcd::IrProcessorType m_NextProcessorType;

    static const nn::TimeSpan SamplingInterval;
    static const int TrialRetryCountMax;

    nn::os::ThreadType m_Thread;
    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[4096];

    nn::xcd::IrCommonData m_CommonData;
    nn::xcd::IrCommonData m_CommonWorkBuffer;
    nn::xcd::IrMomentProcessorState m_MomentData;
    nn::xcd::IrMomentProcessorState m_MomentWorkBuffer;
    nn::xcd::IrClusteringProcessorState m_ClusteringData;
    nn::xcd::IrClusteringProcessorState m_ClusteringWorkBuffer;
    nn::xcd::IrImageTransferProcessorState m_ImageTransferData;
    nn::xcd::IrImageTransferProcessorState m_ImageTransferWorkBuffer;
    nn::xcd::IrDpdProcessorState m_DpdData[nn::xcd::IrDpdProcessorStateCountMax];
    nn::xcd::IrDpdProcessorState m_DpdWorkBuffer[nn::xcd::IrDpdProcessorStateCountMax];
    nn::xcd::IrTeraPluginProcessorState m_HandAnalysisData;
    nn::xcd::IrTeraPluginProcessorState m_HandAnalysisWorkBuffer;
    int m_OutPacketCnt;
    nn::Result m_Result;

    nn::xcd::IrWriteRegisterSetting m_WriteRegisterSetting;

    nn::os::SystemEventType m_IrSamplingEvent;
    nn::os::SystemEventType m_IrCommandCompletionEvent;
    nn::os::SystemEventType m_RequestCompletionEvent;
    nn::os::TimerEventType m_SamplingTimer;

#if defined (USE_OPENCV)
    bool m_IsLraEnabled;
#endif

public:
    XcdIrSensorDriver() NN_NOEXCEPT;
    ~XcdIrSensorDriver() NN_NOEXCEPT;

    void Initialize(nn::xcd::DeviceHandle handle) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    bool RequestWriteRegister(nn::xcd::IrWriteRegisterSetting writeConfig) NN_NOEXCEPT;
    bool RequestReadRegister() NN_NOEXCEPT;
    bool RequestDataRead() NN_NOEXCEPT;
    bool RequestMcuInitialize() NN_NOEXCEPT;
    bool RequestMcuFinalize() NN_NOEXCEPT;
    bool RequestModeSet(nn::xcd::IrProcessorType type) NN_NOEXCEPT;

    nn::Result GetMomentData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrMomentProcessorState* pOutData) NN_NOEXCEPT;
    nn::Result GetClusteringData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrClusteringProcessorState* pOutData) NN_NOEXCEPT;
    nn::Result GetImageTransferData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrImageTransferProcessorState* pOutData) NN_NOEXCEPT;
    nn::Result GetDpdData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrDpdProcessorState* pOutData, int* pOutPacketCnt) NN_NOEXCEPT;
    nn::Result GetHandAnalysisData(nn::xcd::IrCommonData* pOutCommonData, nn::xcd::IrTeraPluginProcessorState* pOutData) NN_NOEXCEPT;

#if defined (USE_OPENCV)
    void GetPadData(nn::xcd::PadState* pOutValue, nn::xcd::SensorCalibrationValue* pSensorCal) NN_NOEXCEPT;

    void SetSixAxisSampling(bool isSixAxisSamplingEnabled) NN_NOEXCEPT;
    void SetLraSending(bool isLraEnabled) NN_NOEXCEPT;
#endif
private:
    static void RequestHandlerThreadLauncher(void* arg) NN_NOEXCEPT;
    void RequestHandlerThreadFunc() NN_NOEXCEPT;

    void DoWriteRegisterSequence() NN_NOEXCEPT;
    void DoReadRegisterSequence() NN_NOEXCEPT;
    void DoModeSettingSequence() NN_NOEXCEPT;
    void DoDataReadSequence() NN_NOEXCEPT;
    void DoInitializeMcuSequence() NN_NOEXCEPT;
    void DoFinalizeMcuSequence() NN_NOEXCEPT;

    nn::Result SetMcuState(nn::xcd::McuState state) NN_NOEXCEPT;
    nn::Result SetMode(nn::xcd::IrProcessorType mode) NN_NOEXCEPT;
};

}
