﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/nn_TimeSpan.h>

#include <nn/xcd/xcd.h>

namespace {
    const size_t SamplingRateCount = 4;
    struct XcdDeviceBlock
    {
        nn::xcd::DeviceHandle handle;
        nn::os::SystemEventType event;
        nn::os::MultiWaitHolderType holder;
        nn::xcd::DeviceInfo info;
        nn::xcd::PadState pad;
        nn::xcd::DeviceStatus status;
        nn::xcd::SensorCalibrationValue cal;
        nn::xcd::AnalogStickValidRange lStick;
        nn::xcd::AnalogStickValidRange rStick;
        nn::xcd::PadButtonSet previousButton;
        uint16_t sampleCount[SamplingRateCount];
        nn::TimeSpan previousTime;
        bool connected;
    };
    XcdDeviceBlock g_Devices[nn::xcd::DeviceCountMax];

    nn::os::MultiWaitType g_MultiWait;
    nn::os::SystemEventType g_LinkEvent;
    nn::os::MultiWaitHolderType g_LinkHolder;
    nn::os::EventType g_TerminateEvent;
    nn::os::MultiWaitHolderType g_TerminateHolder;

    nn::os::ThreadType g_Thread;
    NN_ALIGNAS(4096) char g_ThreadStack[4096];

    enum TestDisplayMode
    {
        TestDisplayMode_Status,
        TestDisplayMode_Log,
    };
    TestDisplayMode g_DisplayMode;

    const uint8_t LedPattern[] = {0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0d, 0x06};
}

void PrintTestLog(const char *fmt, ...)
{
    if(g_DisplayMode == TestDisplayMode_Log)
    {
         NN_LOG(fmt);
    }
}

void SwitchDisplayMode()
{
    g_DisplayMode = (g_DisplayMode == TestDisplayMode_Log) ? TestDisplayMode_Status : TestDisplayMode_Log;
    system("cls");
    PrintTestLog("Logs:\n");
}

void PrintError(const char *fmt, ...)
{
    if(g_DisplayMode == TestDisplayMode_Status)
    {
        SwitchDisplayMode();
    }
    PrintTestLog(fmt);
}

void XcdMonitoringThread(void* arg)
{
    NN_UNUSED(arg);

    while (NN_STATIC_CONDITION(true))
    {
        // デバイス情報の更新を待機
        auto pHolder = nn::os::WaitAny(&g_MultiWait);

        if(nn::xcd::Proceed(pHolder) == true)
        {
            // 何もしない
        }
        else if(pHolder == &g_LinkHolder)
        {
            nn::os::ClearSystemEvent(&g_LinkEvent);

            // nn::hid::GetDebugPadStates による状態取得
            nn::xcd::DeviceList list;
            if(nn::xcd::ListDevices(&list).IsFailure())
            {
                PrintError("List Failure\n");
            }

            // 切断されたデバイスの探索
            for(auto& device : g_Devices)
            {
                // 切断されたかどうか
                bool detached = true;
                if(device.connected == false)
                {
                    continue;
                }

                for(int i = 0; i < list.deviceCount; i++)
                {
                    if(device.handle == list.handleList[i])
                    {
                        // 接続されている
                        detached = false;
                        break;
                    }
                }
                if(detached == true)
                {
                    device.connected = false;
                }
            }

            // 接続されたデバイスの探索
            for(int i = 0; i < list.deviceCount; i++)
            {
                // 接続されたかどうか
                bool connected = true;

                for(auto& device : g_Devices)
                {
                    if(device.connected == true)
                    {
                        if(device.handle == list.handleList[i])
                        {
                            // 接続済み
                            connected = false;
                            break;
                        }
                    }
                }

                if(connected == true)
                {
                    for(int j = 0; j < nn::xcd::DeviceCountMax; j++)
                    {
                        if(g_Devices[j].connected == false)
                        {
                            g_Devices[j].handle = list.handleList[i];
                            nn::xcd::SetSamplingEvent(&g_Devices[j].event, g_Devices[j].handle);
                            nn::xcd::SleepSensor(false, g_Devices[j].handle);
                            nn::xcd::SetPlayerIndicatorPattern(LedPattern[j], g_Devices[j].handle);
                            g_Devices[j].connected = true;
                            nn::xcd::GetDeviceInfo(&g_Devices[j].info, g_Devices[j].handle);
                            nn::xcd::GetSensorCalibrationValue(&g_Devices[j].cal, g_Devices[j].handle);
                            nn::xcd::GetLeftAnalogStickValidRange(&g_Devices[j].lStick, g_Devices[j].handle);
                            nn::xcd::GetRightAnalogStickValidRange(&g_Devices[j].rStick, g_Devices[j].handle);
                            break;
                        }
                    }
                }
            }
        }
        else
        {
            for(auto& device : g_Devices)
            {
                if(pHolder == &device.holder && device.connected == true)
                {
                    nn::os::ClearSystemEvent(&device.event);
                    nn::xcd::GetPadState(&device.pad, device.handle);
                    nn::xcd::GetDeviceStatus(&device.status, device.handle);
                    device.sampleCount[SamplingRateCount - 1]++;

                    auto currentTime = nn::os::GetSystemTick().ToTimeSpan();
                    if((currentTime - device.previousTime).GetMilliSeconds() > 985)
                    {
                        for(int i=0; i < SamplingRateCount - 1; i++)
                        {
                            device.sampleCount[i] = device.sampleCount[i + 1];
                        }
                        device.sampleCount[SamplingRateCount - 1] = 0;
                        device.previousTime = currentTime;
                    }
                }
            }
        }
    }
}

void Display()
{
    for(auto& device : g_Devices)
    {
        if(device.connected == true)
        {
            NN_LOG("#%d bd_addr:%02x %02x %02x %02x %02x %02x", device.handle,
                                                                  device.info.address.address[0], device.info.address.address[1], device.info.address.address[2],
                                                                  device.info.address.address[3], device.info.address.address[4], device.info.address.address[5]);
            switch (device.info.deviceType)
            {
            case nn::xcd::DeviceType_Left:
                NN_LOG("    :Left\n");
                break;
            case nn::xcd::DeviceType_Right:
                NN_LOG("    :Right\n");
                break;
            case nn::xcd::DeviceType_FullKey:
                NN_LOG("    :FullKey\n");
                break;
            default:
                NN_LOG("\n");
                break;
            }
            NN_LOG("  Acc: %05d/%05d/%05d   %05d/%05d/%05d\n", device.cal.accelerometerOrigin.x, device.cal.accelerometerOrigin.y, device.cal.accelerometerOrigin.z,
                                                               device.cal.accelerometerSensitivity.x, device.cal.accelerometerSensitivity.y, device.cal.accelerometerSensitivity.z);
            NN_LOG("  Gyr: %05d/%05d/%05d   %05d/%05d/%05d\n", device.cal.gyroscopeOrigin.x, device.cal.gyroscopeOrigin.y, device.cal.gyroscopeOrigin.z,
                                                               device.cal.gyroscopeSensitivity.x, device.cal.gyroscopeSensitivity.y, device.cal.gyroscopeSensitivity.z);
            NN_LOG("  LStick: Zero %05d/%05d  Min %05d/%05d  ZeroPlay %05d\n", device.lStick.origin.x, device.lStick.origin.y, device.lStick.circuitMin.x, device.lStick.circuitMin.y, device.lStick.originPlay);
            NN_LOG("  RStick: Zero %05d/%05d  Min %05d/%05d  ZeroPlay %05d\n", device.rStick.origin.x, device.rStick.origin.y, device.rStick.circuitMin.x, device.rStick.circuitMin.y, device.rStick.originPlay);
            char buttons[21];
            buttons[0]  = (device.pad.buttons.Test<nn::xcd::PadButton::A>())         ? 'A' : '_';
            buttons[1]  = (device.pad.buttons.Test<nn::xcd::PadButton::B>())         ? 'B' : '_';
            buttons[2]  = (device.pad.buttons.Test<nn::xcd::PadButton::Y>())         ? 'Y' : '_';
            buttons[3]  = (device.pad.buttons.Test<nn::xcd::PadButton::X>())         ? 'X' : '_';
            buttons[4]  = (device.pad.buttons.Test<nn::xcd::PadButton::R>())         ? 'R' : '_';
            buttons[5]  = (device.pad.buttons.Test<nn::xcd::PadButton::ZR>())        ? 'r' : '_';
            buttons[6]  = (device.pad.buttons.Test<nn::xcd::PadButton::StickR>())    ? 'J' : '_';
            buttons[7]  = (device.pad.buttons.Test<nn::xcd::PadButton::Start>())     ? '+' : '_';
            buttons[8]  = (device.pad.buttons.Test<nn::xcd::PadButton::Home>())      ? 'H' : '_';
            buttons[9]  = (device.pad.buttons.Test<nn::xcd::PadButton::SL>())        ? 'S' : '_';
            buttons[10]  = (device.pad.buttons.Test<nn::xcd::PadButton::SR>())       ? 's' : '_';
            buttons[11]  = (device.pad.buttons.Test<nn::xcd::PadButton::Shot>())     ? 'C' : '_';
            buttons[12]  = (device.pad.buttons.Test<nn::xcd::PadButton::Select>())   ? '-' : '_';
            buttons[13]  = (device.pad.buttons.Test<nn::xcd::PadButton::StickL>())   ? 'j' : '_';
            buttons[14]  = (device.pad.buttons.Test<nn::xcd::PadButton::L>())        ? 'L' : '_';
            buttons[15]  = (device.pad.buttons.Test<nn::xcd::PadButton::ZL>())       ? 'l' : '_';
            buttons[16]  = (device.pad.buttons.Test<nn::xcd::PadButton::Right>())    ? '>' : '_';
            buttons[17]  = (device.pad.buttons.Test<nn::xcd::PadButton::Down>())     ? 'v' : '_';
            buttons[18]  = (device.pad.buttons.Test<nn::xcd::PadButton::Left>())     ? '<' : '_';
            buttons[19]  = (device.pad.buttons.Test<nn::xcd::PadButton::Up>())       ? '^' : '_';
            buttons[20] = '\0';

            uint32_t samplingRate = 0;
            for(int i = 0; i < SamplingRateCount - 1; i++)
            {
                samplingRate += device.sampleCount[i];
            }
            samplingRate = samplingRate * 3 / (2 * (SamplingRateCount - 1));
            NN_LOG("[%03d-%3d% %2d %2d %2d] %s  Battery:%d Cable:%c Charge:%c \n", device.pad.sampleNumber, samplingRate, device.sampleCount[0], device.sampleCount[1], device.sampleCount[2],
                                                                      buttons, device.status.batteryLevel,
                                                                      (device.status.cablePlugged) ? 'o' : 'x',
                                                                      (device.status.charged ) ? 'o' : 'x'
                                                                      );
            NN_LOG("  RStick: [x] %04d  [y] %04d\n", device.pad.analogStickR.x, device.pad.analogStickR.y);
            NN_LOG("  LStick: [x] %04d  [y] %04d\n", device.pad.analogStickL.x, device.pad.analogStickL.y);
            NN_LOG("  Acc: %05d/%05d/%05d   %05d/%05d/%05d   %05d/%05d/%05d\n", device.pad.accelerometer[0].x, device.pad.accelerometer[0].y, device.pad.accelerometer[0].z,
                device.pad.accelerometer[1].x, device.pad.accelerometer[1].y, device.pad.accelerometer[1].z,
                device.pad.accelerometer[2].x, device.pad.accelerometer[2].y, device.pad.accelerometer[2].z);
            NN_LOG("  Gyr: %05d/%05d/%05d   %05d/%05d/%05d   %05d/%05d/%05d\n", device.pad.gyroscope[0].x, device.pad.gyroscope[0].y, device.pad.gyroscope[0].z,
                device.pad.gyroscope[1].x, device.pad.gyroscope[1].y, device.pad.gyroscope[1].z,
                device.pad.gyroscope[2].x, device.pad.gyroscope[2].y, device.pad.gyroscope[2].z);

            NN_LOG("\n");
        }
    }
}

void Proc()
{
    for(auto& device : g_Devices)
    {
        if(device.connected == true)
        {
            if(!device.previousButton.Test<nn::xcd::PadButton::SL>() && device.pad.buttons.Test<nn::xcd::PadButton::SL>())
            {
                SwitchDisplayMode();
            }
            device.previousButton = device.pad.buttons;
        }
    }
}

extern "C" void nnMain()
{
    g_DisplayMode = TestDisplayMode_Status;

    nn::os::InitializeMultiWait(&g_MultiWait);

    for(int i = 0; i < nn::xcd::DeviceCountMax; i++)
    {
        g_Devices[i].connected = false;
        for(int j = 0; j < SamplingRateCount; j++)
        {
            g_Devices[i].sampleCount[j] = 0;
        }
        nn::os::CreateSystemEvent(&g_Devices[i].event, nn::os::EventClearMode_ManualClear, false);
        nn::os::InitializeMultiWaitHolder(&g_Devices[i].holder, &g_Devices[i].event);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_Devices[i].holder);
    }

    nn::os::CreateSystemEvent(&g_LinkEvent, nn::os::EventClearMode_ManualClear, false);
    nn::os::InitializeMultiWaitHolder(&g_LinkHolder, &g_LinkEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_LinkHolder);

    nn::os::InitializeEvent(&g_TerminateEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_TerminateHolder, &g_TerminateEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_TerminateHolder);

    // XCDの初期化
    nn::xcd::Initialize(&g_MultiWait);
    nn::xcd::BindLinkUpdateEvent(&g_LinkEvent);

    nn::os::CreateThread(&g_Thread, XcdMonitoringThread, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    nn::os::StartThread(&g_Thread);

    while(NN_STATIC_CONDITION(true))
    {
        if(g_DisplayMode == TestDisplayMode_Status)
        {
            std::system("cls");
            Display();
        }

        Proc();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }
}
