﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/xcd/xcd.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

#include "testXcd_TeraMcuFixture.h"
#include "testXcd_TeraMcuTestParameters.h"

/**
 *  Xcd の TeraMcu ドライバの基本機能をテストします。
 */

namespace {

// テストケースのインスタンス化
INSTANTIATE_TEST_CASE_P(
    StateTransition,
    XcdTeraMcuStateTransitionImmediate,
    ::testing::Values(::nnt::xcd::StateTransitionTestCount));

INSTANTIATE_TEST_CASE_P(
    StateTransition,
    XcdTeraMcuStateTransition,
    ::testing::Values(::nnt::xcd::StateTransitionTestCount));

// MCU ステート遷移関数の型
typedef nn::Result(*SetMcuStateFunction)(nn::xcd::McuState, nn::xcd::DeviceHandle);

// ステート名を取得
const char* GetStateName(nn::xcd::McuState state) NN_NOEXCEPT
{
    switch (state)
    {
    case nn::xcd::McuState_Idle:         return "Idle";
    case nn::xcd::McuState_Standby:      return "Standby";
    case nn::xcd::McuState_Background:   return "Background";
    case nn::xcd::McuState_Nfc:          return "Nfc";
    case nn::xcd::McuState_Ir:           return "Ir";
    case nn::xcd::McuState_Initializing: return "Initializing";
    default:                             return "???";
    }
}

// DataFormat を変更し、完了を待つ
void SetDataFormatSync(nn::xcd::PeriodicDataFormat targetFormat, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    ASSERT_TRUE(nn::xcd::SetDataFormat(targetFormat, handle).IsSuccess());

    nn::xcd::PeriodicDataFormat currentFormat;
    auto startTick = nn::os::GetSystemTick();
    do
    {
        nn::os::SleepThread(::nnt::xcd::CommandInterval);
        ASSERT_TRUE(nn::xcd::GetDataFormat(&currentFormat, handle).IsSuccess());
        auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        ASSERT_LT(diffTime, ::nnt::xcd::ModeTransitionTimeout);
    } while (currentFormat != targetFormat);
}

// Busy 時にリトライしつつステート遷移要求を出す
void SetMcuStateWithRetry(
    SetMcuStateFunction setStateFunc,
    nn::xcd::McuState state,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(setStateFunc);

    auto setMcuFailureCount = 0;
    nn::Result result;

    while ((result = setStateFunc(state, handle)).IsFailure())
    {
        // McuBusy 以外のエラーの場合は即失敗
        ASSERT_TRUE(::nn::xcd::ResultMcuBusy::Includes(result));

        // McuBusy エラーの場合はリトライする
        ASSERT_LT(setMcuFailureCount, ::nnt::xcd::SetMcuStateRetryCountMax);
        setMcuFailureCount++;
        nn::os::SleepThread(::nnt::xcd::CommandInterval);
        NN_LOG("Retry SetMcuState\n");
    }
}

}  // anonymous

/**
 *  DataFormat 変更を必要としない、Hid コマンドによる Tera MCU ステート遷移をテストします。
 */
TEST_P(XcdTeraMcuStateTransitionImmediate, StateTransition)
{
    nn::hid::NpadIdType npadId;

    // デバイスの接続を待つ
    // Tera関連部分をテストするので、ペアリング部分は hid の機能を使用する。
    {
        auto startTick = nn::os::GetSystemTick();

        while (nn::hid::system::GetNpadsWithNfc(&npadId, 1) == 0)
        {
            auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime.GetMilliSeconds(), nnt::xcd::DeviceAttachTimeout.GetMilliSeconds());
            nn::os::SleepThread(::nnt::xcd::CommandInterval);
        }
    }

    ASSERT_TRUE(nn::hid::system::ActivateNfc(npadId).IsSuccess());

    while (!nn::hid::system::IsNfcActivated(npadId))
    {
        nn::os::SleepThread(::nnt::xcd::CommandInterval);
    }

    NN_UTIL_SCOPE_EXIT
    {
        ASSERT_TRUE(nn::hid::system::DeactivateNfc(npadId).IsSuccess());
    };

    nn::xcd::DeviceHandle handle;
    ASSERT_TRUE(nn::hid::system::GetXcdHandleForNpadWithNfc(
        &handle,
        npadId).IsSuccess());

    // DataFormat は Basic
    SetDataFormatSync(nn::xcd::PeriodicDataFormat_Basic, handle);

    // 最初に一旦 Standby に落とす
    {
        SetMcuStateWithRetry(nn::xcd::SetMcuStateImmediate, nn::xcd::McuState_Standby, handle);

        nn::xcd::McuState currentState;
        auto startTick = nn::os::GetSystemTick();
        do
        {
            nn::os::SleepThread(::nnt::xcd::CommandInterval);
            ASSERT_TRUE(nn::xcd::GetMcuState(&currentState, handle).IsSuccess());
            auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime, ::nnt::xcd::ModeTransitionTimeout);
        } while (currentState != nn::xcd::McuState_Standby);
    }

    auto testCount = GetParam();
    for (auto i = 0; i < testCount; i++)
    {
        // Background と Standby を行き来する
        nn::xcd::McuState statesForTest[] = { nn::xcd::McuState_Background, nn::xcd::McuState_Standby };
        for (auto destState : statesForTest)
        {
            NN_LOG("[Loop:%d] Transit to %-10s : ", i, GetStateName(destState));

            auto startTick = nn::os::GetSystemTick();
            SetMcuStateWithRetry(nn::xcd::SetMcuStateImmediate, destState, handle);

            nn::xcd::McuState currentState;
            nn::TimeSpan diffTime;
            do
            {
                nn::os::SleepThread(::nnt::xcd::CommandInterval);
                ASSERT_TRUE(nn::xcd::GetMcuState(&currentState, handle).IsSuccess());
                diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            } while (currentState != destState && diffTime < ::nnt::xcd::ModeTransitionTimeout);

            ASSERT_EQ(currentState, destState);
            NN_LOG("%d[ms]\n", diffTime.GetMilliSeconds());
        }
    }
}

/**
 *  Standby -> IR -> Standby -> NFC -> Standby というシーケンスを繰り返すテスト
 *  規定された遷移時間内で、遷移が完了することをチェックします。
 */
TEST_P(XcdTeraMcuStateTransition, StateTransition)
{
    nn::xcd::DeviceHandle handle;
    nn::hid::NpadIdType npadId;

    // デバイスの接続を待つ
    // Tera関連部分をテストするので、ペアリング部分は hid の機能を使用する。
    {
        auto startTick = nn::os::GetSystemTick();

        while (nn::hid::system::GetNpadsWithNfc(&npadId, 1) == 0)
        {
            auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime.GetMilliSeconds(), nnt::xcd::DeviceAttachTimeout.GetMilliSeconds());
            nn::os::SleepThread(::nnt::xcd::CommandInterval);
        }
    }

    ASSERT_TRUE(nn::hid::system::ActivateNfc(npadId).IsSuccess());

    while (!nn::hid::system::IsNfcActivated(npadId))
    {
        nn::os::SleepThread(::nnt::xcd::CommandInterval);
    }

    NN_UTIL_SCOPE_EXIT
    {
        ASSERT_TRUE(nn::hid::system::DeactivateNfc(npadId).IsSuccess());
    };

    ASSERT_TRUE(nn::hid::system::GetXcdHandleForNpadWithNfc(
        &handle,
        npadId).IsSuccess());

    // DataFormat を変更
    SetDataFormatSync(nn::xcd::PeriodicDataFormat_MCU, handle);

    NN_UTIL_SCOPE_EXIT
    {
        ASSERT_TRUE(nn::xcd::SetDataFormat(
            nn::xcd::PeriodicDataFormat_Basic,
            handle).IsSuccess());
    };

    // 最初に一旦 Standby に落とす
    {
        SetMcuStateWithRetry(nn::xcd::SetMcuState, nn::xcd::McuState_Standby, handle);

        nn::xcd::McuState currentState;
        auto startTick = nn::os::GetSystemTick();
        do
        {
            nn::os::SleepThread(::nnt::xcd::CommandInterval);
            ASSERT_TRUE(nn::xcd::GetMcuState(&currentState, handle).IsSuccess());
            auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime, ::nnt::xcd::ModeTransitionTimeout);
        } while (currentState != nn::xcd::McuState_Standby);
    }

    // 繰り替えし回数を指定
    auto testCount = GetParam();
    for (auto i = 0; i < testCount; i++)
    {
        nn::os::Tick startTick;
        nn::TimeSpan diffTime;

        //
        // Standby -> IR
        //
        startTick = nn::os::GetSystemTick();
        SetMcuStateWithRetry(nn::xcd::SetMcuState, nn::xcd::McuState_Ir, handle);

        // 遷移の完了をポーリングする
        ::nn::xcd::McuState mcuState = ::nn::xcd::McuState_Standby;
        while (mcuState != ::nn::xcd::McuState_Ir)
        {
            diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime.GetMilliSeconds(), nnt::xcd::ModeTransitionTimeout.GetMilliSeconds());
            nn::os::SleepThread(::nnt::xcd::CommandInterval);
            ASSERT_TRUE(
                nn::xcd::GetMcuState(&mcuState, handle).IsSuccess());
        }
        diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        // 遷移時間が期待値を超えているかどうかをチェック
        EXPECT_LE(diffTime.GetMilliSeconds(), ::nnt::xcd::StandbyToIrTransitionTimeMax.GetMilliSeconds());
        NN_LOG("[Loop:%d] Standby->IR : %d[ms]\n", i, diffTime.GetMilliSeconds());

        //
        // IR -> Standby
        //
        startTick = nn::os::GetSystemTick();
        SetMcuStateWithRetry(nn::xcd::SetMcuState, nn::xcd::McuState_Standby, handle);

        // 遷移の完了をポーリングする
        mcuState = ::nn::xcd::McuState_Ir;
        while (mcuState != ::nn::xcd::McuState_Standby)
        {
            diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime.GetMilliSeconds(), nnt::xcd::ModeTransitionTimeout.GetMilliSeconds());
            nn::os::SleepThread(::nnt::xcd::CommandInterval);

            ASSERT_TRUE(
                nn::xcd::GetMcuState(&mcuState, handle).IsSuccess());
        }
        diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        // 遷移時間が期待値を超えているかどうかをチェック
        EXPECT_LE(diffTime.GetMilliSeconds(), ::nnt::xcd::IrToStandbyTransitionTimeMax.GetMilliSeconds());
        NN_LOG("[Loop:%d] IR->Standby : %d[ms]\n", i, diffTime.GetMilliSeconds());
        nn::os::SleepThread(::nnt::xcd::IrToStandbyInterval);

        //
        // Standby -> NFC
        //
        startTick = nn::os::GetSystemTick();
        SetMcuStateWithRetry(nn::xcd::SetMcuState, nn::xcd::McuState_Nfc, handle);

        // 遷移の完了をポーリングする
        mcuState = ::nn::xcd::McuState_Standby;
        while (mcuState != ::nn::xcd::McuState_Nfc)
        {
            diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime.GetMilliSeconds(), nnt::xcd::ModeTransitionTimeout.GetMilliSeconds());
            nn::os::SleepThread(::nnt::xcd::CommandInterval);

            ASSERT_TRUE(
                nn::xcd::GetMcuState(&mcuState, handle).IsSuccess());
        }
        diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        // 遷移時間が期待値を超えているかどうかをチェック
        EXPECT_LE(diffTime.GetMilliSeconds(), ::nnt::xcd::StandbyToNfcTransitionTimeMax.GetMilliSeconds());
        NN_LOG("[Loop:%d] Standby->NFC : %d[ms]\n", i, diffTime.GetMilliSeconds());

        //
        // NFC -> Standby
        //
        startTick = nn::os::GetSystemTick();
        SetMcuStateWithRetry(nn::xcd::SetMcuState, nn::xcd::McuState_Standby, handle);

        // 遷移の完了をポーリングする
        mcuState = ::nn::xcd::McuState_Nfc;
        while (mcuState != ::nn::xcd::McuState_Standby)
        {
            diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime.GetMilliSeconds(), nnt::xcd::ModeTransitionTimeout.GetMilliSeconds());
            nn::os::SleepThread(::nnt::xcd::CommandInterval);

            ASSERT_TRUE(
                nn::xcd::GetMcuState(&mcuState, handle).IsSuccess());
        }
        diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        // 遷移時間が期待値を超えているかどうかをチェック
        EXPECT_LE(diffTime.GetMilliSeconds(), ::nnt::xcd::NfcToStandbyTransitionTimeMax.GetMilliSeconds());
        NN_LOG("[Loop:%d] NFC->Standby : %d[ms]\n", i, diffTime.GetMilliSeconds());
        nn::os::SleepThread(::nnt::xcd::NfcToStandbyInterval);
    }

}  // NOLINT(readability/fn_size)
