﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

#include "testXcd_NfcPassThruAutoFixture.h"

// リトライ処理
#define NNT_XCD_RETRY(format, ...) \
    NN_LOG("%s: " format, NN_CURRENT_FUNCTION_NAME, ##__VA_ARGS__); \
    RetryAndWait(); \
    continue

void XcdNfcBase::SetUp() NN_NOEXCEPT
{
    ::testing::Test::SetUp();

    m_IsNfcInitialized = false;
    nn::os::InitializeMultiWait(&m_MultiWait);
    ::srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));

    nnt::xcd::SetupNpad();

    nn::hid::system::BindNfcDeviceUpdateEvent(&m_DeviceUpdateEvent, nn::os::EventClearMode_ManualClear);
    std::memset(&m_LastNfcInfo, 0, sizeof(m_LastNfcInfo));
}

void XcdNfcBase::TearDown() NN_NOEXCEPT
{
    nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
    nn::os::FinalizeMultiWait(&m_MultiWait);

    if (m_IsNfcInitialized)
    {
        // NFC 関連の終了処理
        nn::os::FinalizeMultiWaitHolder(&m_GeneralEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_DetectEventHolder);
        nn::os::DestroySystemEvent(&m_GeneralEvent);
        nn::os::DestroySystemEvent(&m_DetectEvent);

        nn::xcd::DeviceHandle xcdHandle;
        EXPECT_TRUE(GetNfcDeviceHandle(&xcdHandle));

        EXPECT_TRUE(
            nn::xcd::SetMcuState(
                nn::xcd::McuState_Standby,
                xcdHandle).IsSuccess());

        // Standby が完了しそうな時間待つ
        // XXX: Standby -> NFC を短期間で連投すると動作しなくなる不具合のワークアラウンド
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));

        EXPECT_TRUE(
            nn::xcd::SetDataFormat(
                nn::xcd::PeriodicDataFormat_Basic,
                xcdHandle).IsSuccess());

        m_IsNfcInitialized = false;
    }

    ::testing::Test::TearDown();
}

bool XcdNfcBase::SetupTest() NN_NOEXCEPT
{
    // デバイスの接続検知
    if (!WaitDeviceConnection())
    {
        NN_LOG("%s: Device is not connected\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }
    NN_LOG("NFC Device is connected\n");

    nn::xcd::DeviceHandle xcdHandle;
    {
        nn::Result result;
        for (int i = 0; i < 10; i++)
        {
            if (!GetNfcDeviceHandle(&xcdHandle))
            {
                NN_LOG("%s: Failed to get xcd handle\n", NN_CURRENT_FUNCTION_NAME);
                return false;
            }

            result = nn::xcd::SetDataFormat(
                nn::xcd::PeriodicDataFormat_MCU,
                xcdHandle);
            if (result.IsSuccess())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }

        if (result.IsFailure())
        {
            NN_LOG("%s: Failed to set data format (0x%08X)\n",
                NN_CURRENT_FUNCTION_NAME,
                result.GetInnerValueForDebug());
            return false;
        }
    }

    if (nn::xcd::SetMcuState(
        nn::xcd::McuState_Nfc,
        xcdHandle).IsFailure())
    {
        NN_LOG("%s: Failed to set state\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    // NFC ステートへの遷移を待つ
    auto startTick = nn::os::GetSystemTick();
    nn::xcd::McuState state = nn::xcd::McuState_Initializing;
    while (state != nn::xcd::McuState_Nfc)
    {
        // 規定時間以内に NFC ステートにならない場合は失敗
        auto diffTick = nn::os::GetSystemTick() - startTick;
        if (diffTick.ToTimeSpan() >= nnt::xcd::StateTransitionTimeout)
        {
            NN_LOG("%s: Timed out\n", NN_CURRENT_FUNCTION_NAME);
            return false;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        if (nn::xcd::GetMcuState(
            &state,
            xcdHandle).IsFailure())
        {
            NN_LOG("%s: Failed to get state\n", NN_CURRENT_FUNCTION_NAME);
            return false;
        }
    }

    // NFC イベントの通知を受け取れるようにする
    if (nn::xcd::SetNfcEvent(
        &m_GeneralEvent,
        &m_DetectEvent,
        xcdHandle).IsFailure())
    {
        NN_LOG("%s: Failed to set events\n", NN_CURRENT_FUNCTION_NAME);
        return false;
    }

    nn::os::InitializeMultiWaitHolder(&m_GeneralEventHolder, &m_GeneralEvent);
    nn::os::InitializeMultiWaitHolder(&m_DetectEventHolder, &m_DetectEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_GeneralEventHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_DetectEventHolder);

    m_IsNfcInitialized = true;

    return true;
}

bool XcdNfcBase::DetectTag(const nn::xcd::NfcDiscoveryParameter& parameter) NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        NN_LOG("Failed to get xcd handle\n");
        return false;
    }

    for (int i = 0; i < nnt::xcd::RetryCount; i++)
    {
        {
            auto result = nn::xcd::StartNfcDiscovery(parameter, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-1\n", i + 1);
            }
        }

        // タグの検出待ち
        if (WaitEvent() != EventType::Detect)
        {
            NNT_XCD_RETRY("Retry #%d-2\n", i + 1);
        }

        // タグ検出イベントであることを確認
        {
            auto result = nn::xcd::GetNfcInfo(&m_LastNfcInfo, xcdHandle);
            if (result.IsFailure())
            {
                NN_LOG("Result: ");
                nnt::xcd::PrintResult(result);
                NNT_XCD_RETRY("Retry #%d-3\n", i + 1);
            }
        }

        if (m_LastNfcInfo.reason == nn::xcd::NfcEventReason_Detected)
        {
            return true;
        }

        NNT_XCD_RETRY("Retry #%d-4\n", i + 1);
    }

    return false;
}

bool XcdNfcBase::StopTagDetection() NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    if (!GetNfcDeviceHandle(&xcdHandle))
    {
        NN_LOG("Failed to get xcd handle\n");
        return false;
    }

    return nn::xcd::StopNfcDiscovery(xcdHandle).IsSuccess();
}

bool XcdNfcBase::WaitDeviceConnection() NN_NOEXCEPT
{
    auto startTick = nn::os::GetSystemTick();

    // NFC デバイスの接続待ち
    int deviceCount = 0;
    do
    {
        auto passTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        if (passTime >= nnt::xcd::DeviceAttachTimeout)
        {
            // タイムアウト
            break;
        }

        auto remainTime = nnt::xcd::DeviceAttachTimeout - passTime;
        if (!nn::os::TimedWaitSystemEvent(&m_DeviceUpdateEvent, remainTime))
        {
            // タイムアウト
            break;
        }

        nn::os::ClearSystemEvent(&m_DeviceUpdateEvent);
        deviceCount = nn::hid::system::GetNpadsWithNfc(&m_DeviceNpadId, 1);
    } while (deviceCount <= 0);

    if (deviceCount <= 0)
    {
        NN_LOG("No NFC device connected\n");
        return false;
    }

    NN_LOG("DeviceNpadId: 0x%02X\n", m_DeviceNpadId);

    nn::os::SystemEventType activateEvent;
    nn::hid::system::BindNfcActivateEvent(
        m_DeviceNpadId,
        &activateEvent,
        nn::os::EventClearMode_ManualClear);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::DestroySystemEvent(&activateEvent);
    };

    if (nn::hid::system::ActivateNfc(m_DeviceNpadId).IsFailure())
    {
        NN_LOG("Failed to activate NFC\n");
        return false;
    }

    // NFC 有効化待ち
    NN_ABORT_UNLESS(nn::os::TimedWaitSystemEvent(&activateEvent, nn::TimeSpan::FromSeconds(10)));

    NN_LOG("NFC is activated\n");

    return nn::hid::system::IsNfcActivated(m_DeviceNpadId);
}

bool XcdNfcBase::GetNfcDeviceHandle(nn::xcd::DeviceHandle* pOutHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);

    return nn::hid::system::GetXcdHandleForNpadWithNfc(pOutHandle, m_DeviceNpadId).IsSuccess();
}

XcdNfcBase::EventType XcdNfcBase::WaitEvent() NN_NOEXCEPT
{
    const auto Timeout = nn::TimeSpan::FromSeconds(5);
    auto* pHolder = nn::os::TimedWaitAny(&m_MultiWait, Timeout);
    if (pHolder == &m_GeneralEventHolder)
    {
        nn::os::ClearSystemEvent(&m_GeneralEvent);
        return EventType::General;
    }
    else if (pHolder == &m_DetectEventHolder)
    {
        nn::os::ClearSystemEvent(&m_DetectEvent);
        return EventType::Detect;
    }
    else
    {
        return EventType::None;
    }
}

void XcdNfcBase::RetryAndWait() NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    NN_ABORT_UNLESS(GetNfcDeviceHandle(&xcdHandle));

    nn::xcd::StopNfcDiscovery(xcdHandle);
    nn::os::SleepThread(nnt::xcd::RetryWait);
}

bool XcdNfcBase::DetectTagBasic() NN_NOEXCEPT
{
    nn::xcd::NfcDiscoveryParameter parameter = {};
    parameter.pollingMask       = nn::xcd::NfcPollingMask_All;
    parameter.activationTimeout = 3000;
    parameter.discoveryPeriod   = 300;

    return DetectTag(parameter);
}

bool XcdNfcBase::SendRawData(
    nn::xcd::NfcPassThruData* pOutReceivedData,
    const nn::xcd::NfcPassThruParameter& parameter) NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    NN_ABORT_UNLESS(GetNfcDeviceHandle(&xcdHandle));

    auto startTick = nn::os::GetSystemTick();
    {
        auto result = nn::xcd::SendNfcRawData(parameter, xcdHandle);
        if (result.IsFailure())
        {
            NN_LOG("Result: ");
            nnt::xcd::PrintResult(result);
            return false;
        }
    }

    // 応答受信待ち
    if (WaitEvent() != EventType::General)
    {
        NN_LOG("Unexpected event\n");
        return false;
    }

    // PassThru 応答イベントであることを確認
    nn::xcd::NfcInfo info;
    {
        auto result = nn::xcd::GetNfcInfo(&info, xcdHandle);
        if (result.IsFailure())
        {
            NN_LOG("Result: ");
            nnt::xcd::PrintResult(result);
            return false;
        }
    }

    if (info.reason == nn::xcd::NfcEventReason_PassThruResult)
    {
        auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        NN_LOG("Received pass-thru response. Elapsed time: %lld msec\n", diffTime.GetMilliSeconds());
        *pOutReceivedData = info.passThruData;
        return true;
    }

    NN_LOG("Unexpected reason: %d\n", info.reason);
    return false;
}
