﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

#include "testXcd_NfcPassThruAutoFixture.h"
#include "testXcd_NfcPassThruAutoHelper.h"
#include "testXcd_NfcPassThruAutoTestParameters.h"

//#define VERBOSE

#ifdef VERBOSE
    #define NNT_XCD_LOG_VERBOSE(...)    NN_LOG(__VA_ARGS__)
#else  // VERBOSE
    #define NNT_XCD_LOG_VERBOSE(...)
#endif  // VERBOSE

/**

 FeliCa コマンドの発行テスト

 [テスト内容]
   1. FeliCa タグを検出できる
   2. FeliCa ポーリングコマンドを発行し、結果を取得できる
   3. Read without encryption コマンドを発行し、結果を取得できる

 */
TEST_F(XcdNfcBase, FeliCaCommand)
{
    ASSERT_TRUE(SetupTest());

    // Type F タグのみを探す
    {
        nn::xcd::NfcDiscoveryParameter parameter = {};
        parameter.pollingMask       = nn::xcd::NfcPollingMask_TechnologyF;
        parameter.activationTimeout = nnt::xcd::DiscoveryTimeoutMsec;
        parameter.discoveryPeriod   = 300;

        NN_LOG("Detecting...\n");

        // UID を取得するためにタグを検出
        ASSERT_TRUE(DetectTag(parameter));
    }

    nn::xcd::NfcTagInfo tagInfo;
    {
        // タグ検出直後は TagInfo が入っている
        nn::xcd::NfcInfo nfcInfo;
        GetNfcInfo(&nfcInfo);
        NN_ASSERT_EQUAL(nfcInfo.reason, nn::xcd::NfcEventReason_Detected);
        tagInfo = nfcInfo.tagInfo;

        NN_LOG("Tag detected\n");
    }

    // ポーリングコマンドを送信し、結果を確認
    {
        nn::xcd::NfcPassThruData passThruData;
        {
            nn::xcd::NfcPassThruParameter parameter = {};
            parameter.tagId        = tagInfo.tagId;
            parameter.timeoutMsec  = nnt::xcd::FeliCaCommandTimeoutMsec;
            parameter.sendDataSize = sizeof(nnt::xcd::FeliCaPollingCommand);
            std::memcpy(parameter.sendData, nnt::xcd::FeliCaPollingCommand, parameter.sendDataSize);

            NN_LOG("Sending FeliCaPolling...\n");
            ASSERT_TRUE(SendRawData(&passThruData, parameter));
        }

        ASSERT_EQ(passThruData.responseSize, nnt::xcd::FelicaPollingResponseSize);

        // ヘッダの確認
        ASSERT_EQ(
            std::memcmp(
                passThruData.responseData,
                nnt::xcd::FelicaPollingResponseHeader,
                sizeof(nnt::xcd::FelicaPollingResponseHeader)),
            0);

        // IDm の確認
        ASSERT_EQ(
            std::memcmp(
                passThruData.responseData + sizeof(nnt::xcd::FelicaPollingResponseHeader),
                tagInfo.tagId.uid,
                tagInfo.tagId.length),
            0);
    }

    // Read without encryption コマンドを送信し、結果を確認
    {
        nn::xcd::NfcPassThruData passThruData;
        {
            nn::xcd::NfcPassThruParameter parameter = {};
            parameter.tagId        = tagInfo.tagId;
            parameter.timeoutMsec  = nnt::xcd::FeliCaCommandTimeoutMsec;
            parameter.sendDataSize = sizeof(nnt::xcd::ReadWithoutEncryptionCommand);
            std::memcpy(parameter.sendData, nnt::xcd::ReadWithoutEncryptionCommand, parameter.sendDataSize);

            // (2-10 バイト目には IDm = UID が入る)
            std::memcpy(
                parameter.sendData + 2,
                tagInfo.tagId.uid,
                tagInfo.tagId.length);

            NN_LOG("Sending ReadWithoutEncryption...\n");
            ASSERT_TRUE(SendRawData(&passThruData, parameter));
        }

        // 想定データを作成
        // (2-10 バイト目には IDm = UID が入る)
        const auto expectedSize = sizeof(nnt::xcd::ReadWithoutEncryptionResponse);
        nn::Bit8 expectedData[expectedSize];
        std::memcpy(
            expectedData,
            nnt::xcd::ReadWithoutEncryptionResponse,
            expectedSize);
        std::memcpy(
            expectedData + 2,
            tagInfo.tagId.uid,
            tagInfo.tagId.length);

        // 想定したデータが来ているか判定
        ASSERT_GE(passThruData.responseSize, expectedSize);
        ASSERT_EQ(std::memcmp(passThruData.responseData, expectedData, expectedSize), 0);
    }

    ASSERT_TRUE(StopTagDetection());
}
