﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/xcd/xcd.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include "testXcd_NfcAutoCommon.h"
#include "testXcd_NfcAutoHelper.h"

namespace
{

// 初期化する Npad の ID リスト
const nn::hid::NpadIdType NpadIds[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::Handheld
};

}  // anonymous

namespace nnt { namespace xcd {

void SetupNpad() NN_NOEXCEPT
{
    static bool s_IsConfigured = false;

    if (s_IsConfigured)
    {
        return;
    }

    nn::hid::InitializeNpad();

    // 認識可能なスタイルを指定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleJoyRight::Mask
        | nn::hid::NpadStyleFullKey::Mask
        | nn::hid::NpadStyleHandheld::Mask);

    // Npad の初期化
    nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));
    for (const auto& id : NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(id);
    }

    s_IsConfigured = true;
}

bool ContainsSystemArea(const nn::xcd::NtagWriteData& writeData) NN_NOEXCEPT
{
    const uint8_t SystemAreaHead = 0x03;
    const uint8_t SystemAreaTail = 0x82;

    for (int i = 0; i < writeData.blockCount; i++)
    {
        const auto& block = writeData.dataBlocks[i];

        // Serial number, static lock, CC の領域
        if (block.startPageAddress <= SystemAreaHead)
        {
            return true;
        }

        // Dynamic lock, CFG 等の領域
        int tailAddress = block.startPageAddress + block.dataSize / nn::xcd::Type2TagPageSize - 1;
        if (tailAddress >= SystemAreaTail)
        {
            return true;
        }
    }

    return false;
}

void CreateRandomWriteData(
    nn::xcd::NtagWriteParameter* pOutParameter,
    int writePages,
    uint8_t blockPageMax) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutParameter);

    pOutParameter->ntagWriteData.blockCount =
        (writePages + blockPageMax - 1) / blockPageMax;
    NN_ABORT_UNLESS_MINMAX(blockPageMax, 1, nn::xcd::NtagWriteBlockPageCountMax);

    nn::Bit8 writeData[nnt::xcd::NtagWritePageMaxForBlank * nn::xcd::Type2TagPageSize];
    int totalBytes = writePages * nn::xcd::Type2TagPageSize;
    for (int i = 0; i < totalBytes; i++)
    {
        writeData[i] = static_cast<nn::Bit8>(::rand() & 0xFF);
    }

    int doneBytes = 0;
    for (int i = 0; i < pOutParameter->ntagWriteData.blockCount; i++)
    {
        const auto BlockBytes = blockPageMax * nn::xcd::Type2TagPageSize;

        auto& block = pOutParameter->ntagWriteData.dataBlocks[i];
        block.startPageAddress = static_cast<uint8_t>(
            nnt::xcd::NtagWriteTopPageForBlank + i * blockPageMax);
        block.dataSize = static_cast<uint8_t>(
            std::min(
                static_cast<decltype(BlockBytes)>(totalBytes - doneBytes),
                BlockBytes));
        std::memcpy(block.data, writeData + doneBytes, block.dataSize);

        doneBytes += block.dataSize;
        NN_LOG("  Block %d: Start %d, Size %d (%d pages)\n",
            i,
            block.startPageAddress,
            block.dataSize,
            block.dataSize / nn::xcd::Type2TagPageSize);
    }
}

void PrintResult(nn::Result result) NN_NOEXCEPT
{
    if (result.IsSuccess())
    {
        NN_LOG("Success\n");
    }
    else if (nn::xcd::ResultNotConnected().Includes(result))
    {
        NN_LOG("NotConnected\n");
    }
    else if (nn::xcd::ResultInvalidMcuState().Includes(result))
    {
        NN_LOG("InvalidMcuState\n");
    }
    else if (nn::xcd::ResultMcuBusy().Includes(result))
    {
        NN_LOG("McuBusy\n");
    }
    else
    {
        NN_LOG("Unknown\n");
    }
}

}}  // nnt::xcd
