﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

#include "testXcd_NfcAutoFixture.h"
#include "testXcd_NfcAutoHelper.h"
#include "testXcd_NfcAutoTestParameters.h"

//#define VERBOSE

#ifdef VERBOSE
    #define NNT_XCD_LOG_VERBOSE(...)    NN_LOG(__VA_ARGS__)
#else  // VERBOSE
    #define NNT_XCD_LOG_VERBOSE(...)
#endif  // VERBOSE

// アクティベーション情報のクリア値
const nn::Bit8 nnt::xcd::NtagActivationClearData[] = { 0xFF, 0xFF, 0xFF, 0xFF };

// アクティベーション情報
const nn::Bit8 nnt::xcd::NtagActivationData[] = { 0xA5, 0x00, 0x00, 0x00 };

// テストケースのインスタンス化
INSTANTIATE_TEST_CASE_P(
    StateTransition,
    XcdNfcStateTransition,
    ::testing::Values(nnt::xcd::StateTransitionTestCount));

INSTANTIATE_TEST_CASE_P(
    TagDetection,
    XcdNfcDetect,
    ::testing::ValuesIn(nnt::xcd::DiscoveryParameterList));

INSTANTIATE_TEST_CASE_P(
    TagRead,
    XcdNfcRead,
    ::testing::ValuesIn(nnt::xcd::ReadParameterList));

INSTANTIATE_TEST_CASE_P(
    TagRead,
    XcdNfcReadRandom,
    ::testing::ValuesIn(nnt::xcd::ReadRandomParameterList));

INSTANTIATE_TEST_CASE_P(
    TagWrite,
    XcdNfcWrite,
    ::testing::ValuesIn(nnt::xcd::WriteParameterList));

INSTANTIATE_TEST_CASE_P(
    TagWrite,
    XcdNfcWriteRandom,
    ::testing::ValuesIn(nnt::xcd::WriteRandomParameterList));

/**

 Tera のステート遷移が繰り返し動作することのテスト

 [テスト内容]
   1. Tera を NFC ステートへ遷移させる
   2. Tera を Standby ステートへ遷移させる
   3. 上記を規定回数繰り返す

 */
TEST_P(XcdNfcStateTransition, StateTransition)
{
    nnt::xcd::SetupNpad();

    nn::hid::NpadIdType npadId;

    // デバイスの接続を待つ
    {
        auto startTick = nn::os::GetSystemTick();
        while (nn::hid::system::GetNpadsWithNfc(&npadId, 1) == 0)
        {
            auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
            ASSERT_LT(diffTime, nnt::xcd::DeviceAttachTimeout);
            nn::os::SleepThread(nnt::xcd::DevicePollInterval);
        }

        ASSERT_TRUE(nn::hid::system::ActivateNfc(npadId).IsSuccess());

        while (!nn::hid::system::IsNfcActivated(npadId))
        {
            nn::os::SleepThread(nnt::xcd::DevicePollInterval);
        }
    }

    nn::xcd::DeviceHandle handle;
    ASSERT_TRUE(nn::hid::system::GetXcdHandleForNpadWithNfc(
        &handle,
        npadId).IsSuccess());

    ASSERT_TRUE(nn::xcd::SetDataFormat(
        nn::xcd::PeriodicDataFormat_MCU,
        handle).IsSuccess());

    auto testCount = GetParam();
    for (int i = 0; i < testCount; i++)
    {
        NN_LOG("[%s] Test #%d\n", NN_CURRENT_FUNCTION_NAME, i + 1);

        // NFC ステートに遷移
        {
            ASSERT_TRUE(nn::xcd::SetMcuState(
                nn::xcd::McuState_Nfc,
                handle).IsSuccess());

            auto startTick = nn::os::GetSystemTick();
            nn::xcd::McuState state = nn::xcd::McuState_Initializing;
            while (state != nn::xcd::McuState_Nfc)
            {
                // 規定時間以内に NFC ステートにならない場合は失敗
                auto diffTick = nn::os::GetSystemTick() - startTick;
                ASSERT_LT(diffTick.ToTimeSpan(), nnt::xcd::StateTransitionTimeout);

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
                ASSERT_TRUE(nn::xcd::GetMcuState(&state, handle).IsSuccess());
            }
        }

        // Standby ステートに遷移
        {
            ASSERT_TRUE(nn::xcd::SetMcuState(
                nn::xcd::McuState_Standby,
                handle).IsSuccess());

            // 遷移完了するであろう時間待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));

            // ステート確認
            nn::xcd::McuState state;
            ASSERT_TRUE(nn::xcd::GetMcuState(&state, handle).IsSuccess());
            ASSERT_EQ(state, nn::xcd::McuState_Standby);
        }
    }

    ASSERT_TRUE(nn::xcd::SetDataFormat(
        nn::xcd::PeriodicDataFormat_Basic,
        handle).IsSuccess());

    ASSERT_TRUE(nn::hid::system::DeactivateNfc(npadId).IsSuccess());
}

/**

 NFC の初期化・終了が成功することのテスト

 このテストケースは、他のテストでも常に実行される。

 [テスト内容]
   1. NFC ステートへ遷移できる
   2. NFC イベントを登録できる
   3. Standby ステートへ遷移できる

 */
TEST_F(XcdNfcBase, Initialize)
{
    // 実際の動作は SetupTest 内で実行
    ASSERT_TRUE(SetupTest());
}

#if defined(NNT_XCD_ENABLE_DEATH_TEST)
/**

 異常パラメータが ASSERT にかかることのテスト

 */
TEST_F(XcdNfcBase, BasicDeath)
{
    // NFC API を呼ぶために NFC ステートへの遷移が必要
    ASSERT_TRUE(SetupTest());

    EXPECT_DEATH_IF_SUPPORTED(
        nn::xcd::SetNfcEvent(nullptr, nullptr, m_DeviceHandle), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::xcd::GetNfcInfo(nullptr, m_DeviceHandle), "");

    // Read パラメータ
    {
        nn::xcd::NtagReadParameter parameter = {};
        parameter.timeoutMsec        = 3000;
        parameter.isPasswordRequired = false;

        // TagId が異常
        parameter.tagId.length = -1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagRead(parameter, m_DeviceHandle), "");
        parameter.tagId.length = nn::xcd::NfcUidLengthMax + 1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagRead(parameter, m_DeviceHandle), "");

        parameter.tagId.length = 0;

        // blockCount が異常
        parameter.blockCount = 0;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagRead(parameter, m_DeviceHandle), "");
        parameter.blockCount = nn::xcd::NtagReadBlockCountMax + 1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagRead(parameter, m_DeviceHandle), "");

        // アドレスが異常
        for (int i = 0; i < nn::xcd::NtagReadBlockCountMax; i++)
        {
            parameter.blockCount = i + 1;

            auto&& address = parameter.addresses[i];

            address.startPage = 0x00;
            address.endPage   = nn::xcd::NtagReadBlockPageCountMax + 1;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::xcd::StartNtagRead(parameter, m_DeviceHandle), "");

            address.endPage = 0x00;
        }
    }

    // Write パラメータ
    {
        nn::xcd::NtagWriteParameter parameter = {};
        parameter.timeoutMsec        = 5000;
        parameter.isPasswordRequired = false;
        parameter.type2TagVersion    = 0;
        parameter.ntagWriteData.isActivationNeeded = true;

        // TagId が異常
        parameter.tagId.length = -1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagWrite(parameter, m_DeviceHandle), "");
        parameter.tagId.length = nn::xcd::NfcUidLengthMax + 1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagWrite(parameter, m_DeviceHandle), "");

        parameter.tagId.length = 0;

        // blockCount が異常
        parameter.ntagWriteData.blockCount = 0;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagWrite(parameter, m_DeviceHandle), "");
        parameter.ntagWriteData.blockCount = nn::xcd::NtagWriteBlockCountMax + 1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::StartNtagWrite(parameter, m_DeviceHandle), "");

        // ブロックサイズが異常
        for (int i = 0; i < nn::xcd::NtagWriteBlockCountMax; i++)
        {
            parameter.ntagWriteData.blockCount = i + 1;

            auto&& block = parameter.ntagWriteData.dataBlocks[i];
            block.startPageAddress = 0;

            block.dataSize = 3;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::xcd::StartNtagWrite(parameter, m_DeviceHandle), "");
            block.dataSize = nn::xcd::NtagWriteBlockSizeMax + 4;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::xcd::StartNtagWrite(parameter, m_DeviceHandle), "");

            block.dataSize = 4;
        }
    }

    // PassThru パラメータ
    {
        nn::xcd::NfcPassThruParameter parameter = {};
        parameter.timeoutMsec = 3000;

        // TagId が異常
        parameter.tagId.length = -1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::SendNfcRawData(parameter, m_DeviceHandle), "");
        parameter.tagId.length = nn::xcd::NfcUidLengthMax + 1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::SendNfcRawData(parameter, m_DeviceHandle), "");

        parameter.tagId.length = 0;

        // データサイズが異常
        parameter.sendDataSize = nn::xcd::NfcPassThruDataSizeMax + 1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::xcd::SendNfcRawData(parameter, m_DeviceHandle), "");
    }

}  // NOLINT(readability/fn_size)
#endif  // if defined(NNT_XCD_ENABLE_DEATH_TEST)

/**

 タグの検出ができることのテスト

 このテストケースは、タグの読み書きテストでも内部的に実行される。

 [テスト内容]
   1. タグの検出を開始できる
   2. タグ検出時にタグ検出イベント (nn::xcd::NfcEventReason_Detected) が発生する

 */
TEST_P(XcdNfcDetect, TagDetection)
{
    ASSERT_TRUE(SetupTest());

    auto parameter = GetParam();
    ASSERT_TRUE(DetectTag(parameter));

    ASSERT_TRUE(StopTagDetection());
}

/**

 タグの読み取りができることのテスト

 [テスト内容]
   1. タグを検出できる
   2. タグの読み取りを開始できる
   3. タグの読み取り完了時に完了イベント (nn::xcd::NfcEventReason_ReadFinish) が発生する
   4. nn::xcd::NfcInfo を取得し、要求パラメータ通りの結果が返ってくる
     - データの中身は書き込みテストのベリファイで確認するため、このテストでは無視する

 */
TEST_P(XcdNfcRead, TagRead)
{
    ASSERT_TRUE(SetupTest());

    auto testParameter = GetParam();
    nn::xcd::NtagReadParameter parameter = {};
    parameter.timeoutMsec        = 3000;    // タイムアウト: 3000 msec
    parameter.isPasswordRequired = false;   // パスワード認証なし
    parameter.tagId.length       = 0;       // UID 指定なし
    parameter.blockCount         = testParameter.blockCount;

    auto* addresses = parameter.addresses;
    for (decltype(testParameter.blockCount) i = 0; i < testParameter.blockCount; i++)
    {
        addresses[i].startPage = testParameter.addresses[i].startPage;
        addresses[i].endPage   = testParameter.addresses[i].endPage;
    }

    // タグの読み取り
    nn::xcd::NfcInfo info = {};
    ASSERT_TRUE(ReadNtagWithRetry(&info, parameter));
    NNT_XCD_LOG_VERBOSE("  Read   [OK]\n");

    // 指定した通りのパラメータで読めているか確認
    //  ※ データの中身は見ない
    ASSERT_EQ(info.ntagData.blockCount, parameter.blockCount);
    for (decltype(parameter.blockCount) i = 0; i < parameter.blockCount; i++)
    {
        ASSERT_EQ(
            info.ntagData.readDataBlocks[i].address.startPage,
            testParameter.addresses[i].startPage);
        ASSERT_EQ(
            info.ntagData.readDataBlocks[i].address.endPage,
            testParameter.addresses[i].endPage);
    }
    NNT_XCD_LOG_VERBOSE("  Verify [OK]\n");

    ASSERT_TRUE(StopTagDetection());
}

/**

 タグの読み取りができることのテスト (ランダムアドレス版)

 [テスト内容]
   読み込みアドレスをランダムに変更しながら、XcdNfcRead と同様のテストを規定回数行う

 */
TEST_P(XcdNfcReadRandom, TagRead)
{
    ASSERT_TRUE(SetupTest());

    auto testParameter = GetParam();
    for (int testCount = 0; testCount < testParameter.testCount; testCount++)
    {
        NN_LOG("[%s] Test #%d\n", NN_CURRENT_FUNCTION_NAME, testCount + 1);

        nn::xcd::NtagReadParameter parameter = {};
        parameter.timeoutMsec        = 3000;    // タイムアウト: 3000 msec
        parameter.isPasswordRequired = false;   // パスワード認証なし
        parameter.tagId.length       = 0;       // UID 指定なし
        parameter.blockCount         = 1 + ::rand() % testParameter.blockCountMax;

        // 読み取るアドレスはランダムに決める
        auto* addresses = parameter.addresses;
        for (decltype(parameter.blockCount) i = 0; i < parameter.blockCount; i++)
        {
            int range = ::rand() % nn::xcd::NtagReadBlockPageCountMax;
            auto startPage = static_cast<uint8_t>(::rand() % (nnt::xcd::NtagReadPageMax - range));
            auto endPage   = static_cast<uint8_t>(startPage + range);
            addresses[i].startPage = startPage;
            addresses[i].endPage   = endPage;
            NN_LOG("  Address %d: %02X-%02X\n", i + 1, startPage, endPage);
        }

        // タグの読み取り
        nn::xcd::NfcInfo info = {};
        ASSERT_TRUE(ReadNtagWithRetry(&info, parameter));
        NNT_XCD_LOG_VERBOSE("  Read   [OK]\n");

        // 指定した通りのパラメータ (ブロック数、アドレス) で読めているか確認
        //  ※ データの中身は見ない
        ASSERT_EQ(info.ntagData.blockCount, parameter.blockCount);
        for (decltype(parameter.blockCount) i = 0; i < parameter.blockCount; i++)
        {
            ASSERT_EQ(
                info.ntagData.readDataBlocks[i].address.startPage,
                parameter.addresses[i].startPage);
            ASSERT_EQ(
                info.ntagData.readDataBlocks[i].address.endPage,
                parameter.addresses[i].endPage);
        }
        NNT_XCD_LOG_VERBOSE("  Verify [OK]\n");

        ASSERT_TRUE(StopTagDetection());
    }
}

/**

 タグへの書き込みができることのテスト

 [テスト内容]
   1. タグを検出できる
   2. タグへの書き込みを開始できる
   3. タグへの書き込み完了時に完了イベント (nn::xcd::NfcEventReason_WriteFinish) が発生する
   4. タグからデータを読み取り、読み取ったデータが書き込んだデータと一致する

 */
TEST_P(XcdNfcWrite, TagWrite)
{
    ASSERT_TRUE(SetupTest());

    auto testParameter = GetParam();

    nn::xcd::NtagWriteParameter writeParameter = {};
    writeParameter.timeoutMsec        = 5000;    // タイムアウト: 5000 msec
    writeParameter.isPasswordRequired = false;   // パスワード認証なし
    writeParameter.tagId.length       = 0;       // UID 指定なし
    writeParameter.type2TagVersion    = 0;       // Type2 タグバージョン 0

    writeParameter.ntagWriteData.isActivationNeeded = true;  // アクティベーションあり

    // アクティベーション情報の設定
    std::memcpy(
        writeParameter.ntagWriteData.clearData,
        nnt::xcd::NtagActivationClearData,
        nn::xcd::NtagActivationAreaSize);
    std::memcpy(
        writeParameter.ntagWriteData.activationData,
        nnt::xcd::NtagActivationData,
        nn::xcd::NtagActivationAreaSize);

    // 書き込むデータを生成
    nnt::xcd::CreateRandomWriteData(
        &writeParameter,
        testParameter.writePages,
        testParameter.blockPageMax);
    NN_LOG("  Block count: %d\n", writeParameter.ntagWriteData.blockCount);

    // 書き込みとベリファイ
    ASSERT_TRUE(WriteNtagWithRetry(writeParameter));
    NNT_XCD_LOG_VERBOSE("  Write  [OK]\n");
    ASSERT_TRUE(VerifyWrittenData(writeParameter));
    NNT_XCD_LOG_VERBOSE("  Verify [OK]\n");

    ASSERT_TRUE(StopTagDetection());
}

/**

 タグへの書き込みができることのテスト (ランダムサイズ版)

 [テスト内容]
   書き込むデータ量をランダムに変更しながら、XcdNfcWrite と同様のテストを規定回数行う

 */
TEST_P(XcdNfcWriteRandom, TagWrite)
{
    ASSERT_TRUE(SetupTest());

    auto testParameter = GetParam();
    for (int testCount = 0; testCount < testParameter.testCount; testCount++)
    {
        NN_LOG("[%s] Test #%d\n", NN_CURRENT_FUNCTION_NAME, testCount + 1);

        nn::xcd::NtagWriteParameter writeParameter = {};
        writeParameter.timeoutMsec        = 5000;    // タイムアウト: 5000 msec
        writeParameter.isPasswordRequired = false;   // パスワード認証なし
        writeParameter.tagId.length       = 0;       // UID 指定なし
        writeParameter.type2TagVersion    = 0;       // Type2 タグバージョン 0

        writeParameter.ntagWriteData.isActivationNeeded = true;  // アクティベーションあり

        // アクティベーション情報の設定
        std::memcpy(
            writeParameter.ntagWriteData.clearData,
            nnt::xcd::NtagActivationClearData,
            nn::xcd::NtagActivationAreaSize);
        std::memcpy(
            writeParameter.ntagWriteData.activationData,
            nnt::xcd::NtagActivationData,
            nn::xcd::NtagActivationAreaSize);

        // 書き込むデータを生成
        int writePages = 1 + (::rand() % nnt::xcd::NtagWritePageMaxForBlank);
        nnt::xcd::CreateRandomWriteData(&writeParameter, writePages, 32);
        NN_LOG("  Block count: %d\n", writeParameter.ntagWriteData.blockCount);

        // 書き込みとベリファイ
        ASSERT_TRUE(WriteNtagWithRetry(writeParameter));
        NNT_XCD_LOG_VERBOSE("  Write  [OK]\n");
        ASSERT_TRUE(VerifyWrittenData(writeParameter));
        NNT_XCD_LOG_VERBOSE("  Verify [OK]\n");

        ASSERT_TRUE(StopTagDetection());
    }
}
