﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include "TA_pad.h"
#include "TA_common.h"

namespace TestAgent {

const nn::hid::NpadIdType NPad::m_NpadIds[NPAD_ID_TYPE_COUNT] = {
        nn::hid::NpadId::No1, nn::hid::NpadId::No2, nn::hid::NpadId::No3,
        nn::hid::NpadId::No4, nn::hid::NpadId::No5, nn::hid::NpadId::No6,
        nn::hid::NpadId::No7, nn::hid::NpadId::No8, nn::hid::NpadId::Handheld };

NPad::PadInfo NPad::m_PadInfo[REMOTE_COUNT_MAX];
bool NPad::m_IsIrConfigurationEnabled = false;

int ToDebugPadButton(Button button) {
    switch (button) {
    case A:
        return 0;
    case B:
        return 1;
    case X:
        return 2;
    case Y:
        return 3;
    case L:
        return 4;
    case R:
        return 5;
    case ZL:
        return 6;
    case ZR:
        return 7;
    case START:
        return 8;
    case SELECT:
        return 9;
    case LEFT:
        return 10;
    case UP:
        return 11;
    case RIGHT:
        return 12;
    case DOWN:
        return 13;
    default:
        return 31;
    }
}

int ToNPadButton(Button button) {
    switch (button) {
    case A:
        return 0;
    case B:
        return 1;
    case X:
        return 2;
    case Y:
        return 3;
    case SL:
        return 4;
    case SR:
        return 5;
    case L:
        return 6;
    case R:
        return 7;
    case ZL:
        return 8;
    case ZR:
        return 9;
    case START:
    case PLUS:
        return 10;
    case SELECT:
    case MINUS:
        return 11;
    case LEFT:
        return 12;
    case UP:
        return 13;
    case RIGHT:
        return 14;
    case DOWN:
        return 15;
    case SL_LEFT:
        return 16;
    case SL_UP:
        return 17;
    case SL_RIGHT:
        return 18;
    case SL_DOWN:
        return 19;
    case SR_LEFT:
        return 20;
    case SR_UP:
        return 21;
    case SR_RIGHT:
        return 22;
    case SR_DOWN:
        return 23;
    default:
        return 63;
    }
}

void Print(nn::hid::DebugPadButtonSet buttons) {
    for (int i = 0; i < 32; ++i) {
        NN_LOG("%d", buttons.Test(i));
        if ((i + 1) % 4 == 0)
            NN_LOG(" ");
    }
    NN_LOG("\n");
}

/*---------------------------------------------------------------------------
 DebugPad
 ---------------------------------------------------------------------------*/

bool DebugPad::m_IsEnabled = true;

DebugPad::DebugPad() {
    m_HoldDetectionDelay = nn::TimeSpan::FromMilliSeconds(300);
    m_HoldDetectionInterval = nn::TimeSpan::FromMilliSeconds(30);
    Clear();
}

DebugPad::~DebugPad() {
}

DebugPad& DebugPad::GetInstance() {
    static DebugPad pad;
    static bool first = true;

    if (first) {
        if (pad.m_IsEnabled) {
            nn::hid::InitializeDebugPad();
        }

        first = false;
    }

    return pad;
}
;

void DebugPad::UpdatePadState() {
    if (!m_IsEnabled) {
        return;
    }

    DebugPad& pad = GetInstance();

    static nn::hid::DebugPadState previousBuf;
    static nn::hid::DebugPadState currentBuf;

    nn::hid::GetDebugPadState(&currentBuf);
    nn::hid::DebugPadButtonSet currentButtons = currentBuf.buttons;
    nn::hid::DebugPadButtonSet previousButtons = previousBuf.buttons;

    pad.m_CurrentState.trigger = (~previousButtons) & currentButtons;
    pad.m_CurrentState.hold = currentButtons;
    pad.m_CurrentState.release = previousButtons & (~currentButtons);

    if (pad.m_CurrentState.trigger.IsAnyOn()) {
        pad.m_PressTime = nn::os::GetSystemTick().ToTimeSpan();
        pad.m_PreviousDetectionTime = 0;
    }

    // 一つ前の入力を保存
    previousBuf = currentBuf;
}

bool DebugPad::IsTrigger(Button button) {
    if (m_CurrentState.trigger.Test(ToDebugPadButton(button))) {
        return true;
    }

    return false;
}

bool DebugPad::IsHold(Button button) {
    if (m_CurrentState.hold.Test((ToDebugPadButton(button)))) {
        nn::TimeSpan now = nn::os::GetSystemTick().ToTimeSpan();

        if (((now - m_PressTime) > m_HoldDetectionDelay)
                && (m_PreviousDetectionTime == 0)) {
            m_PreviousDetectionTime = now;

            return true;
        } else if (((now - m_PreviousDetectionTime) > m_HoldDetectionInterval)
                && (m_PreviousDetectionTime != 0)) {
            m_PreviousDetectionTime = now;
            return true;
        }
    }

    return false;
}

bool DebugPad::IsRelease(Button button) {
    if (m_CurrentState.release.Test((ToDebugPadButton(button)))) {
        return true;
    }

    return false;
}

void DebugPad::Clear() {
    m_CurrentState.trigger.Reset();
    m_CurrentState.hold.Reset();
    m_CurrentState.release.Reset();

    m_PreviousDetectionTime -= m_PreviousDetectionTime;
}

void DebugPad::EnablePad() {
    m_IsEnabled = true;
}

void DebugPad::DisablePad() {
    m_IsEnabled = false;
}

/*---------------------------------------------------------------------------
 NPad
 ---------------------------------------------------------------------------*/

bool NPad::m_IsEnabled = true;

NPad::NPad() {
    m_HoldDetectionDelay = nn::TimeSpan::FromMilliSeconds(300);
    m_HoldDetectionInterval = nn::TimeSpan::FromMilliSeconds(30);
    Clear();
}

NPad::~NPad() {
}

NPad& NPad::GetInstance() {
    static NPad pad;
    static bool first = true;

    if (first) {
        if (pad.m_IsEnabled) {
            pad.Clear();

            nn::hid::InitializeNpad();

            // 使用する操作形態を設定
            nn::hid::SetSupportedNpadStyleSet(
                    nn::hid::NpadStyleFullKey::Mask
                            | nn::hid::NpadStyleJoyLeft::Mask
                            | nn::hid::NpadStyleJoyRight::Mask
                            | nn::hid::NpadStyleJoyDual::Mask
                            | nn::hid::NpadStyleHandheld::Mask);

            // 使用するNpadId の設定
            nn::hid::SetSupportedNpadIdType(pad.m_NpadIds, NPAD_ID_TYPE_COUNT);

            // 6 軸センサーとモーションIRカメラの設定
            for (int i = 0; i < REMOTE_COUNT_MAX; ++i) {
                // 1本ずつ割り当て
                nn::hid::SetNpadJoyAssignmentModeSingle(m_NpadIds[i]);

                m_PadInfo[i].npadId = m_NpadIds[i];
                nn::hid::GetSixAxisSensorHandles(&m_PadInfo[i].leftHandle,
                        nn::hid::NpadSixAxisSensorHandleCountMax, m_NpadIds[i],
                        nn::hid::NpadStyleJoyLeft::Mask);

                nn::hid::GetSixAxisSensorHandles(&m_PadInfo[i].rightHandle,
                        nn::hid::NpadSixAxisSensorHandleCountMax, m_NpadIds[i],
                        nn::hid::NpadStyleJoyRight::Mask);

                nn::hid::GetSixAxisSensorHandles(&m_PadInfo[i].fullKeyHandle,
                        nn::hid::NpadSixAxisSensorHandleCountMax, m_NpadIds[i],
                        nn::hid::NpadStyleFullKey::Mask);

                nn::hid::StartSixAxisSensor(m_PadInfo[i].leftHandle);
                nn::hid::StartSixAxisSensor(m_PadInfo[i].rightHandle);
                nn::hid::StartSixAxisSensor(m_PadInfo[i].fullKeyHandle);

                m_PadInfo[i].irHandle = nn::irsensor::GetIrCameraHandle(
                        m_NpadIds[i]);
                nn::irsensor::Initialize(m_PadInfo[i].irHandle);
                m_PadInfo[i].isIrEnabled = false;
            }
        }

        first = false;
    }

    return pad;
}
;

void NPad::Finalize() {
    {
        NPad& pad = NPad::GetInstance();
        if (pad.m_IsEnabled) {
            for (int i = 0; i < REMOTE_COUNT_MAX; ++i) {
                nn::irsensor::Finalize(pad.m_PadInfo[i].irHandle);
            }
        }
    }
}

void NPad::GetPadInfo(const PadInfo* info[REMOTE_COUNT_MAX], uint8_t* pCount) {
    NN_ASSERT(pCount != nullptr);
    *pCount = REMOTE_COUNT_MAX;
    for (int i = 0; i < *pCount; ++i) {
        info[i] = &m_PadInfo[i];
    }
}

void NPad::ClearStatistics() {
}

void NPad::UpdatePadState() {
    if (!m_IsEnabled) {
        return;
    }

    static nn::hid::NpadFullKeyState previousBuf;
    static nn::hid::NpadFullKeyState currentBuf;

    NPad& pad = GetInstance();
    nn::hid::NpadButtonSet buttons;

    buttons.Reset();

    // すべての入力の論理和を入力とする
    for (int i = 0; i < NPAD_ID_TYPE_COUNT; ++i) {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(
                pad.m_NpadIds[i]);

        // フルキー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleFullKey>() == true) {
            nn::hid::NpadFullKeyState currentFullKeyBuf;
            nn::hid::GetNpadState(&currentFullKeyBuf, pad.m_NpadIds[i]);
            buttons |= currentFullKeyBuf.buttons;

            // samplingNumber の保存
            nn::hid::SixAxisSensorState state = { };
            nn::hid::GetSixAxisSensorState(&state,
                    pad.m_PadInfo[i].fullKeyHandle);
        }

        // Joy-Con 操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true) {
            nn::hid::NpadJoyDualState currentJoyDualBuf;
            nn::hid::GetNpadState(&currentJoyDualBuf, pad.m_NpadIds[i]);
            buttons |= currentJoyDualBuf.buttons;
        }

        if (style.Test<nn::hid::NpadStyleJoyLeft>() == true) {
            nn::hid::NpadJoyLeftState currentJoyLeftBuf;
            nn::hid::GetNpadState(&currentJoyLeftBuf, pad.m_NpadIds[i]);
            buttons |= currentJoyLeftBuf.buttons;

            // samplingNumber の保存
            nn::hid::SixAxisSensorState state = { };
            nn::hid::GetSixAxisSensorState(&state, pad.m_PadInfo[i].leftHandle);
        }

        if (style.Test<nn::hid::NpadStyleJoyRight>() == true) {
            static nn::hid::NpadButtonSet prevButtons;

            nn::hid::NpadJoyRightState currentJoyRightBuf;
            nn::hid::GetNpadState(&currentJoyRightBuf, pad.m_NpadIds[i]);
            buttons |= currentJoyRightBuf.buttons;

            // samplingNumber の保存
            nn::hid::SixAxisSensorState state = { };
            nn::hid::GetSixAxisSensorState(&state,
                    pad.m_PadInfo[i].rightHandle);

            // Enable IrSensor
            nn::hid::NpadButtonSet trigger = (~prevButtons) & buttons;
            prevButtons = buttons;
            if (m_IsIrConfigurationEnabled
                    && trigger.Test<nn::hid::NpadJoyButton::X>()) {
                if (m_PadInfo[i].isIrEnabled) {
                    NN_LOG(" [#%d] stop IrSensor\n", i + 1);
                    nn::irsensor::StopImageProcessor(m_PadInfo[i].irHandle);
                    m_PadInfo[i].isIrEnabled = false;
                } else {
                    NN_LOG(" [#%d] run IrSensor\n", i + 1);
                    nn::irsensor::MomentProcessorConfig config;
                    nn::irsensor::GetMomentProcessorDefaultConfig(&config);
                    nn::irsensor::RunMomentProcessor(m_PadInfo[i].irHandle,
                            config);
                    m_PadInfo[i].isIrEnabled = true;
                }
            }
        }

        // 携帯機コントローラー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleHandheld>() == true) {
            nn::hid::NpadHandheldState currentHandheldBuf;
            nn::hid::GetNpadState(&currentHandheldBuf, pad.m_NpadIds[i]);
            buttons |= currentHandheldBuf.buttons;
        }
    }

    currentBuf.buttons = buttons;

    nn::hid::NpadButtonSet currentButtons = currentBuf.buttons;
    nn::hid::NpadButtonSet previousButtons = previousBuf.buttons;

    pad.m_CurrentState.trigger = (~previousButtons) & currentButtons;
    pad.m_CurrentState.hold = currentButtons;
    pad.m_CurrentState.release = previousButtons & (~currentButtons);

    if (pad.m_CurrentState.trigger.IsAnyOn()) {
        pad.m_PressTime = nn::os::GetSystemTick().ToTimeSpan();
        pad.m_PreviousDetectionTime = 0;
    }

    // 一つ前の入力を保存
    previousBuf = currentBuf;
}

bool NPad::IsTrigger(Button button) {
    if (m_CurrentState.trigger.Test(ToNPadButton(button))) {
        return true;
    }

    return false;
}

bool NPad::IsHold(Button button) {
    if (m_CurrentState.hold.Test((ToNPadButton(button)))) {
        nn::TimeSpan now = nn::os::GetSystemTick().ToTimeSpan();

        if (((now - m_PressTime) > m_HoldDetectionDelay)
                && (m_PreviousDetectionTime == 0)) {
            m_PreviousDetectionTime = now;

            return true;
        } else if (((now - m_PreviousDetectionTime) > m_HoldDetectionInterval)
                && (m_PreviousDetectionTime != 0)) {
            m_PreviousDetectionTime = now;
            return true;
        }
    }
    return false;
}

bool NPad::IsRelease(Button button) {
    if (m_CurrentState.release.Test((ToNPadButton(button)))) {
        return true;
    }

    return false;
}

void NPad::Clear() {
    m_CurrentState.trigger.Reset();
    m_CurrentState.hold.Reset();
    m_CurrentState.release.Reset();

    m_PreviousDetectionTime -= m_PreviousDetectionTime;
}

void NPad::EnableIrConfiguration(const bool& isEnabled) {
    m_IsIrConfigurationEnabled = isEnabled;
}

void NPad::DisconnectAllNpads() {
    NPad& pad = GetInstance();
    for (int i = 0; i < NPAD_ID_TYPE_COUNT; ++i) {
        nn::hid::DisconnectNpad(pad.m_NpadIds[i]);
    }
}

void NPad::EnablePad() {
    m_IsEnabled = true;
}

void NPad::DisablePad() {
    m_IsEnabled = false;
}

/*---------------------------------------------------------------------------
 Pad
 ---------------------------------------------------------------------------*/

DebugPad* Pad::m_pDebugPad = nullptr;
NPad* Pad::m_pNPad = nullptr;

Pad::Pad() {
}

Pad::~Pad() {
}

Pad& Pad::GetInstance() {
    static Pad pad;
    static bool first = true;

    if (first) {
        m_pDebugPad = &DebugPad::GetInstance();
        m_pNPad = &NPad::GetInstance();
        first = false;
    }

    return pad;
}
;

void Pad::UpdatePadState() {
    Pad& pad = GetInstance();
    pad.m_pDebugPad->UpdatePadState();
    pad.m_pNPad->UpdatePadState();
}

bool Pad::IsTrigger(Button button) {
    Pad& pad = GetInstance();
    return pad.m_pDebugPad->IsTrigger(button) || pad.m_pNPad->IsTrigger(button);
}

bool Pad::IsHold(Button button) {
    Pad& pad = GetInstance();
    return pad.m_pDebugPad->IsHold(button) || pad.m_pNPad->IsHold(button);
}

bool Pad::IsRelease(Button button) {
    Pad& pad = GetInstance();
    return pad.m_pDebugPad->IsRelease(button) || pad.m_pNPad->IsRelease(button);
}

void Pad::Disconnect() {
    GetInstance().m_pNPad->DisconnectAllNpads();
}

void Pad::Clear() {
    Pad& pad = GetInstance();
    pad.m_pDebugPad->Clear();
    pad.m_pNPad->Clear();
}

void Pad::EnablePad() {
    Pad& pad = GetInstance();
    pad.m_pDebugPad->EnablePad();
    pad.m_pNPad->EnablePad();
}

void Pad::DisablePad() {
    DebugPad::DisablePad();
    NPad::DisablePad();
}

}
