﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/wlan/wlan_InfraApi.h>

#include "TestAgent.h"
#include "TA_agentpage.h"

extern void StartAgentThread() NN_NOEXCEPT;
extern void StopAgentThread() NN_NOEXCEPT;

namespace TestAgent {

const char* DISPLAY_SEPARATOR =
        "----------------------------------------------------------------";

enum SettingPageId
{
    PageId_StandAlone,
    PageId_HostDriven,
    PageId_TestAgent,
    PageId_Max,
    PageId_AgentMode,
};

typedef struct Type_Setting_Page
{
    int PageId;
    char TitleInfo[64];
    char SettingPageHelpInfo[64];
    char ExecutePageHelpInfo[64];
} SettingPage_t;

SettingPage_t SettingPageTable[] =
{
    {PageId_StandAlone, ""
            " - detect stand alone mode",
            "<A>Start <L/R>Switch mode",
            "<B>Stop <Y>Clear <A>Set H <X>Add/Del <+>Fetch"
    },

    {PageId_HostDriven,
            " - detect host driven mode",
            "<A>Start <L/R>Switch mode",
            "<A>Send/StopSend <B>Stop <Y>Clear"
    },

    {PageId_TestAgent,
            " - agent for auto test",
            "<A>Start <L/R>Switch mode",
            "<B>Stop"
    },

    {PageId_Max, "", "", ""},

    {PageId_AgentMode,
            " - Running in agent mode",
            "No GUI operation for current mode",
            ""
    },
};

uint8_t hash_tmp[8] = {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

void AddSelectorElement(SelectableValue<NamedValue<nn::TimeSpan> >* sv,
                        const nn::TimeSpan& value,
                        const nn::TimeSpan& initialValue)
{
    sv->Add(NamedValue<nn::TimeSpan>(ToStringSimpleTimeSpan(value), value),
            value == initialValue);
}

void GenerateTxInterval(SelectableValue<NamedValue<uint32_t> >* sv,
                      const uint32_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 1; i < 500 + 1; i++) {
        oss << i << " ms";
        AddSelectorElement<uint32_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateTxCount(SelectableValue<NamedValue<uint32_t> >* sv,
                      const uint32_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 1; i < 1000 + 1; i++) {
        oss << i;
        AddSelectorElement<uint32_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateIdleCount(SelectableValue<NamedValue<uint32_t> >* sv,
                      const uint32_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 0; i < 1000 + 1; i++) {
        oss << i;
        AddSelectorElement<uint32_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateRxStart(SelectableValue<NamedValue<uint32_t> >* sv,
                      const uint32_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 0; i < 1000 + 1; i++) {
        oss << i;
        AddSelectorElement<uint32_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateRxCount(SelectableValue<NamedValue<uint32_t> >* sv,
                      const uint32_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 1; i < 1000 + 1; i++) {
        oss << i;
        AddSelectorElement<uint32_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateDataSize(SelectableValue<NamedValue<uint32_t> >* sv,
                      const uint32_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 64; i < 1300 + 1; i++) {
        oss << i << " bytes";
        AddSelectorElement<uint32_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateChannel(SelectableValue<NamedValue<uint8_t> >* sv,
                     const uint8_t& initialValue)
{
    ostringstream oss;
    int i;

    for (i = 0; i < 13; i++) {
        oss << "Ch " << i + 1;
        AddSelectorElement<uint8_t>(sv, oss.str(), i + 1, initialValue);
        oss.str("");
    }

#if 0
    for (i = 0; i < 4; i++) {
        oss << "Ch " << 36 + i * 4;
        AddSelectorElement<uint8_t>(sv, oss.str(), 36 + i * 4, initialValue);
        oss.str("");
    }

    for (i = 0; i < 11; i++)
    {
        oss << "Ch " << 100 + i * 4;
        AddSelectorElement<uint8_t>(sv, oss.str(), 100 + i * 4, initialValue);
        oss.str("");
    }

    for (i = 0; i < 5; i++)
    {
        oss << "Ch " << 149 + i * 4;
        AddSelectorElement<uint8_t>(sv, oss.str(), 149 + i * 4, initialValue);
        oss.str("");
    }
#endif
}

void GenerateDataHash(SelectableValue<NamedValue<uint8_t> >* sv,
                      const uint8_t& initialValue)
{
    ostringstream oss;
    int i;

    AddSelectorElement<uint8_t>(sv, "0x0123456789ABCDEF", 0, initialValue);
    for (i = 1; i < 256; i++) {
        oss << "All 0x" << hex << setw(2) << setfill('0') << uppercase << i;
        AddSelectorElement<uint8_t>(sv, oss.str(), i, initialValue);
        oss.str("");
    }
}

void GenerateDataCmd(SelectableValue<NamedValue<uint8_t> >* sv,
                     const uint8_t& initialValue)
{
    AddSelectorElement<uint8_t>(sv, "Periodic", 1, initialValue);
    AddSelectorElement<uint8_t>(sv, "OneShot", 2, initialValue);
    AddSelectorElement<uint8_t>(sv, "Periodic Cont.(same hash)", 3, initialValue);
    AddSelectorElement<uint8_t>(sv, "Periodic Cont.(varied hash)", 4, initialValue);
}

void GenerateIsSendPeriodical(SelectableValue<NamedValue<bool> >* sv,
                              const bool& initialValue)
{
    AddSelectorElement<bool>(sv, "Enable", true, initialValue);
    AddSelectorElement<bool>(sv, "Disable", false, initialValue);
}

void AgentPage::InitializeStandAlonePage()
{
    Color TextColor;

    m_StandAloneSetingPage.Add(selectorStandAlone);
    selectorStandAlone.X = SELECTOR_START_X;
    selectorStandAlone.Y = SELECTOR_START_Y;
    selectorStandAlone.Height = 450;

    selectorStandAlone.Register("Tx Interval", &m_SsTxInterval);
    GenerateTxInterval(&m_SsTxInterval, 127);

    selectorStandAlone.Register("Tx Count", &m_SsTxCount);
    GenerateTxCount(&m_SsTxCount, 30);

    selectorStandAlone.Register("Idle Count", &m_SsIdleCount);
    GenerateIdleCount(&m_SsIdleCount, 0);

    selectorStandAlone.Register("Rx Start", &m_SsRxStart);
    GenerateRxStart(&m_SsRxStart, 0);

    selectorStandAlone.Register("Rx Count", &m_SsRxCount);
    GenerateRxCount(&m_SsRxCount, 1);

    selectorStandAlone.Register("Data Length", &m_SsDataLength);
    GenerateDataSize(&m_SsDataLength, 500);

    selectorStandAlone.Register("Data Hash", &m_SsDataHash);
    GenerateDataHash(&m_SsDataHash, 0);

    selectorStandAlone.Register("Channel", &m_SsChannelNum);
    GenerateChannel(&m_SsChannelNum, 1);

    m_StandAloneSetingPage.Add(m_BlankLine);

    m_StandAloneSetingPage.Add(m_StandAloneInfo);
    m_StandAloneInfo.X = INFO_START_X;
    m_StandAloneInfo.Y = INFO_START_Y;
    m_StandAloneInfo.Text = "";
    m_StandAloneInfo.FitSize();

    m_StandAloneSetingPage.Add(m_StandAloneSettingStatus);
    m_StandAloneSettingStatus.X = STATUSBAR_START_X + STATUS_OFFSET_X;
    m_StandAloneSettingStatus.Y = STATUSBAR_START_Y;
    m_StandAloneSettingStatus.Text = "Ready";
    m_StandAloneSettingStatus.FitSize();
    TextColor = ToColor(GREEN);
    m_StandAloneSettingStatus.SetFgColor(TextColor);

    m_StandAloneRunningPage.Add(selectorStandAloneHashSet);
    selectorStandAloneHashSet.X = SELECTOR_START_X;
    selectorStandAloneHashSet.Y = SELECTOR_START_Y;
    selectorStandAloneHashSet.Height = 60;

    selectorStandAloneHashSet.Register("Hash Filter", &m_SssHashFilter);
    GenerateDataHash(&m_SssHashFilter, 0);

    m_StandAloneRunningPage.Add(m_StandAloneRunningInfo);
    m_StandAloneRunningInfo.X = INFO_START_X;
    m_StandAloneRunningInfo.Y = INFO_START_Y;
    m_StandAloneRunningInfo.Text = "";
    m_StandAloneRunningInfo.FitSize();

    m_StandAloneRunningPage.Add(m_StandAloneRunningDataInfo);
    m_StandAloneRunningDataInfo.X = DATA_START_X;
    m_StandAloneRunningDataInfo.Y = DATA_START_Y - 50;
    m_StandAloneRunningDataInfo.Text = "";
    m_StandAloneRunningDataInfo.FitSize();

    m_StandAloneRunningPage.Add(m_StandAloneRunningStatus);
    m_StandAloneRunningStatus.X = STATUSBAR_START_X + STATUS_OFFSET_X;
    m_StandAloneRunningStatus.Y = STATUSBAR_START_Y;
    m_StandAloneRunningStatus.Text = "Running";
    m_StandAloneRunningStatus.FitSize();
    TextColor = ToColor(ORANGE);
    m_StandAloneRunningStatus.SetFgColor(TextColor);
}

void AgentPage::InitializeHostDrivenPage()
{
    Color TextColor;

    m_HostDrivenSetingPage.Add(selectorHostDriven);
    selectorHostDriven.X = SELECTOR_START_X;
    selectorHostDriven.Y = SELECTOR_START_Y;
    selectorHostDriven.Height = 450;

    selectorHostDriven.Register("Tx Interval", &m_HdTxInterval);
    GenerateTxInterval(&m_HdTxInterval, 500);

    selectorHostDriven.Register("Tx Count", &m_HdTxCount);
    GenerateTxCount(&m_HdTxCount, 10);

    selectorHostDriven.Register("Idle Count", &m_HdIdleCount);
    GenerateIdleCount(&m_HdIdleCount, 2);

    selectorHostDriven.Register("Rx Start", &m_HdRxStart);
    GenerateRxStart(&m_HdRxStart, 7);

    selectorHostDriven.Register("Rx Count", &m_HdRxCount);
    GenerateRxCount(&m_HdRxCount, 2);

    selectorHostDriven.Register("Data Length(P)", &m_HdDataLength);
    GenerateDataSize(&m_HdDataLength, 500);

    selectorHostDriven.Register("Data Hash(P)", &m_HdDataHash);
    GenerateDataHash(&m_HdDataHash, 0);

    selectorHostDriven.Register("Channel", &m_HdChannelNum);
    GenerateChannel(&m_HdChannelNum, 1);

    selectorHostDriven.Register("Send Data(P)", &m_HdIsSendPeriodical);
    GenerateIsSendPeriodical(&m_HdIsSendPeriodical, true);

    m_HostDrivenSetingPage.Add(m_BlankLine);

    m_HostDrivenSetingPage.Add(m_HostDrivenInfo);
    m_HostDrivenInfo.X = INFO_START_X;
    m_HostDrivenInfo.Y = INFO_START_Y;
    m_HostDrivenInfo.Text = "";
    m_HostDrivenInfo.FitSize();

    m_HostDrivenSetingPage.Add(m_HostDrivenSettingStatus);
    m_HostDrivenSettingStatus.X = STATUSBAR_START_X + STATUS_OFFSET_X;
    m_HostDrivenSettingStatus.Y = STATUSBAR_START_Y;
    m_HostDrivenSettingStatus.Text = "Ready";
    m_HostDrivenSettingStatus.FitSize();
    TextColor = ToColor(GREEN);
    m_HostDrivenSettingStatus.SetFgColor(TextColor);

    m_HostDrivenRunningPage.Add(selectorHostDrivenSendManually);
    selectorHostDrivenSendManually.X = SELECTOR_START_X;
    selectorHostDrivenSendManually.Y = SELECTOR_START_Y;
    selectorHostDrivenSendManually.Height = 130;

    selectorHostDrivenSendManually.Register("Single Data Hash", &m_HdSingleDataHash);
    GenerateDataHash(&m_HdSingleDataHash, 0);

    selectorHostDrivenSendManually.Register("Single Data Length", &m_HdSingleDataLength);
    GenerateDataSize(&m_HdSingleDataLength, 600);

    selectorHostDrivenSendManually.Register("Single Data Cmd", &m_HdSingleDataCmd);
    GenerateDataCmd(&m_HdSingleDataCmd, 2);

    selectorHostDrivenSendManually.Register("Cont. Tx interval", &m_HdSingleTxInterval);
    AddSelectorElement<uint32_t>(&m_HdSingleTxInterval, "0", 0, 100);
    GenerateTxInterval(&m_HdSingleTxInterval, 100);

    m_HostDrivenRunningPage.Add(m_HostDrivenRunningDataInfo);
    m_HostDrivenRunningDataInfo.X = DATA_START_X;
    m_HostDrivenRunningDataInfo.Y = DATA_START_Y + 25;
    m_HostDrivenRunningDataInfo.Text = "";
    m_HostDrivenRunningDataInfo.FitSize();

    m_HostDrivenRunningPage.Add(m_HostDrivenRunningStatus);
    m_HostDrivenRunningStatus.X = STATUSBAR_START_X + STATUS_OFFSET_X;
    m_HostDrivenRunningStatus.Y = STATUSBAR_START_Y;
    m_HostDrivenRunningStatus.Text = "Running";
    m_HostDrivenRunningStatus.FitSize();
    TextColor = ToColor(ORANGE);
    m_HostDrivenRunningStatus.SetFgColor(TextColor);

    m_TestAgentSetingPage.Add(m_TestAgentInfo);
    m_TestAgentInfo.X = INFO_START_X;
    m_TestAgentInfo.Y = INFO_START_Y;
    m_TestAgentInfo.Text = "Need ncl_ucc in agent mode.";
    m_TestAgentInfo.FitSize();
    TextColor = ToColor(ORANGE);
    m_TestAgentInfo.SetFgColor(TextColor);

    m_TestAgentSetingPage.Add(m_TestAgentSettingStatus);
    m_TestAgentSettingStatus.X = STATUSBAR_START_X + STATUS_OFFSET_X;
    m_TestAgentSettingStatus.Y = STATUSBAR_START_Y;
    m_TestAgentSettingStatus.Text = "Ready";
    m_TestAgentSettingStatus.FitSize();
    TextColor = ToColor(GREEN);
    m_TestAgentSettingStatus.SetFgColor(TextColor);

    m_TestAgentRunningPage.Add(m_TestAgentRunningDataInfo);
    m_TestAgentRunningDataInfo.X = DATA_START_X;
    m_TestAgentRunningDataInfo.Y = 70;
    m_TestAgentRunningDataInfo.Text = "";
    m_TestAgentRunningDataInfo.FitSize();

    m_TestAgentRunningPage.Add(m_TestAgentRunningStatus);
    m_TestAgentRunningStatus.X = STATUSBAR_START_X + STATUS_OFFSET_X;
    m_TestAgentRunningStatus.Y = STATUSBAR_START_Y;
    m_TestAgentRunningStatus.Text = "Running";
    m_TestAgentRunningStatus.FitSize();
    TextColor = ToColor(GREEN);
    m_TestAgentRunningStatus.SetFgColor(TextColor);
}

AgentPage::AgentPage()
{
    m_Selected = false;
    m_AgentMode = false;
    m_SettingErrorStandAlone = false;
    m_SettingErrorHostDriven = false;
    m_RunningErrorStandAlone = false;
    m_RunningErrorHostDriven = false;
    m_RunningErrorTestAgent = false;
    m_IsRunningStandAlone = false;
    m_IsRunningHostDriven = false;
    m_IsContinueTx = false;
    m_IsRunningTestAgent = false;
    m_IsDetectFetching = false;
    m_PageId = PageId_StandAlone;
    m_CurrentDataHash = 0;
    m_CurrentDataLen = 0;
    m_CurrentDataCmd = 0;
    m_SingleSentCount = 0;
    memset(&m_HashListBitMask[0], 0x0, 32);
    WlanDetectGetMacAddress(&m_MacAddr[0]);

    InitializeInfoPage(m_PageId);
    InitializeStandAlonePage();
    InitializeHostDrivenPage();
}

AgentPage::~AgentPage()
{
}

void AgentPage::InitializeInfoPage(int PageId)
{
    uint32_t h = Display::GetInstance().GetFontHeight();

    m_InfoPage.Clear();
    m_InfoPage.Add(m_BlankLine);
    m_InfoPage.Add(m_BlankLine);
    m_BlankLine.Text = " ";
    m_BlankLine.X = 0;
    m_BlankLine.Y = 10;
    m_BlankLine.Alignment = MiddleLeft;
    m_BlankLine.FitSize();

    for (int i = 0; i < 2; ++i) {
        m_InfoPage.Add(m_Separator[i]);
        m_Separator[i].Text = DISPLAY_SEPARATOR;
        m_Separator[i].X = 0 + i * -15;
        m_Separator[i].Y = h;
        m_Separator[i].Alignment = MiddleLeft;
        m_Separator[i].FitSize();
    }

    m_InfoPage.Add(m_Title);
    m_Title.Text = APP_NAME;
    m_Title.X = TITLE_START_X;
    m_Title.Y = 0;
    m_Title.Alignment = MiddleLeft;
    m_Title.FitSize();

    m_InfoPage.Add(m_Help);
    m_Help.X = STATUSBAR_START_X;
    m_Help.Y = STATUSBAR_START_Y;
    m_Help.Text = "";
    m_Help.FitSize();

    m_InfoPage.Add(m_BlankLine);
}

void AgentPage::ShowCommonPageInfo(int PageId)
{
    Color TextColor;
    ostringstream oss;

    m_InfoPage.Show(Display::GetInstance());

    oss << APP_NAME << " " << APP_VERSION;
    oss << SettingPageTable[PageId].TitleInfo;
    m_Title.Text = oss.str();
    oss.str("");
    m_Title.FitSize();
    m_Title.ShowImpl(Display::GetInstance());

    if (m_Selected) {
        m_Help.Text = SettingPageTable[PageId].ExecutePageHelpInfo;
    } else {
        m_Help.Text = SettingPageTable[PageId].SettingPageHelpInfo;
    }
    m_Help.FitSize();
    m_Help.ShowImpl(Display::GetInstance());
}

void AgentPage::ShowSettingPageInfo(int PageId)
{
    switch (m_PageId) {
    case PageId_StandAlone:
        m_StandAloneSetingPage.Show(Display::GetInstance());
        break;
    case PageId_HostDriven:
        m_HostDrivenSetingPage.Show(Display::GetInstance());
        break;
    case PageId_TestAgent:
        m_TestAgentSetingPage.Show(Display::GetInstance());
        break;
    default:
        break;
    }
}

void AgentPage::ShowExecutePageInfo(int PageId)
{
    switch (m_PageId) {
    case PageId_StandAlone:
        m_StandAloneRunningPage.Show(Display::GetInstance());
        break;
    case PageId_HostDriven:
        m_HostDrivenRunningPage.Show(Display::GetInstance());
        break;
    case PageId_TestAgent:
        m_TestAgentRunningPage.Show(Display::GetInstance());
        break;
    default:
        break;
    }
}

void AgentPage::EnterExecutePageAction(int PageId)
{
    uint32_t tx_interval;
    uint32_t tx_count;
    uint32_t idle_count;
    uint32_t rx_start;
    uint32_t rx_count;
    uint8_t data_hash;
    uint32_t data_size;
    uint8_t channel;
    uint8_t hash[8];
    bool isSend;

    switch (m_PageId) {
    case PageId_StandAlone:
        if (m_SettingErrorStandAlone)
            break;
        if (m_IsRunningStandAlone)
            break;
        if (m_RunningErrorStandAlone)
            break;
        tx_interval = m_SsTxInterval.GetValue();
        tx_count    = m_SsTxCount.GetValue();
        idle_count  = m_SsIdleCount.GetValue();
        rx_start    = m_SsRxStart.GetValue();
        rx_count    = m_SsRxCount.GetValue();
        data_size   = m_SsDataLength.GetValue();
        data_hash   = m_SsDataHash.GetValue();
        channel     = m_SsChannelNum.GetValue();
        if (data_hash == 0) {
            memcpy(&hash[0], &hash_tmp[0], 8);
        } else {
            memset(&hash[0], data_hash, 8);
        }
        WlanDetectInitialize();
        if (DetectStandAloneStart(tx_interval, tx_count, idle_count, rx_start,
                                rx_count, hash, data_size, channel) < 0) {
            m_RunningErrorStandAlone = true;
        } else {
            m_IsRunningStandAlone = true;
            m_Selected = true;
        }
        break;
    case PageId_HostDriven:
        if (m_IsRunningHostDriven)
            break;
        if (m_RunningErrorHostDriven)
            break;
        tx_interval  = m_HdTxInterval.GetValue();
        tx_count     = m_HdTxCount.GetValue();
        idle_count   = m_HdIdleCount.GetValue();
        rx_start     = m_HdRxStart.GetValue();
        rx_count     = m_HdRxCount.GetValue();
        data_hash    = m_HdDataHash.GetValue();
        data_size    = m_HdDataLength.GetValue();
        channel      = m_HdChannelNum.GetValue();
        isSend       = m_HdIsSendPeriodical.GetValue();
        if (data_hash == 0) {
            memcpy(&hash[0], &hash_tmp[0], 8);
        } else {
            memset(&hash[0], data_hash, 8);
        }
        WlanDetectInitialize();
        if (DetectHostDrivenStart(tx_interval, tx_count, idle_count, rx_start,
                                rx_count, hash, data_size, isSend,
                                channel) < 0) {
            m_RunningErrorHostDriven = true;
        } else {
            m_IsRunningHostDriven = true;
            m_Selected = true;
        }
        break;
    case PageId_TestAgent:
        if (m_IsRunningTestAgent)
            break;
        if (m_RunningErrorTestAgent)
            break;
        StartAgentThread();
        m_IsRunningTestAgent = true;
        m_Selected = true;
        break;
    default:
        break;
    }
}

void AgentPage::LeaveExecutePageAction(int PageId)
{
    switch (m_PageId) {
    case PageId_StandAlone:
        if (m_IsRunningStandAlone) {
            if (m_IsDetectFetching) {
                DetectStandAloneStopFetchFrame();
                m_IsDetectFetching = false;
            }
            DetectStandAloneStop();
            WlanDetectUninitialize();
            ClearRecvDetectFrameBuff();
            memset(&m_HashListBitMask[0], 0x0, 32);
            m_IsRunningStandAlone = false;
        }
        m_Selected = false;
        break;
    case PageId_HostDriven:
        if (m_IsRunningHostDriven) {
            if (m_IsContinueTx) {
                DetectContTxThreadStop();
                m_IsContinueTx = false;
            }
            DetectHostDrivenStop();
            WlanDetectUninitialize();
            m_CurrentDataHash = 0;
            m_CurrentDataLen = 0;
            m_CurrentDataCmd = 0;
            m_SingleSentCount = 0;
            ClearRecvDetectFrameBuff();
            m_IsRunningHostDriven = false;
        }
        m_Selected = false;
        break;
    case PageId_TestAgent:
        if (TestAgent::GetAgentStatus())
            break;

        if (m_IsRunningTestAgent) {
            StopAgentThread();
            ClearCmdList();
            m_IsRunningTestAgent = false;
        }
        m_Selected = false;
        break;
    default:
        break;
    }
}

void AgentPage::ButtonAExecutePageAction(int PageId)
{
    uint8_t data_hash;
    uint32_t data_size;
    uint8_t data_cmd;
    uint32_t tx_interval;
    uint8_t hash[8];
    int byte_idx;
    int bit_idx;

    switch (m_PageId) {
    case PageId_StandAlone:
        ClearHashFilter();
        for (byte_idx = 0; byte_idx < 32; byte_idx++) {
            for (bit_idx = 0; bit_idx < 8; bit_idx++) {
                if (m_HashListBitMask[byte_idx] & (1 << bit_idx)) {
                    data_hash = 8 * byte_idx + bit_idx;
                    if (data_hash == 0) {
                        memcpy(&hash[0], &hash_tmp[0], 8);
                    } else {
                        memset(&hash[0], data_hash, 8);
                    }
                    SetHashFilter(&hash[0]);
                }
            }
        }
        if (m_IsDetectFetching) {
            DetectStandAloneStopFetchFrame();
            m_IsDetectFetching = false;
        }
        DetectStandAloneSetHashList();
        break;
    case PageId_HostDriven:
        if (m_IsContinueTx) {
            DetectContTxThreadStop();
            m_IsContinueTx = false;
            break;
        }
        data_hash    = m_HdSingleDataHash.GetValue();
        data_size    = m_HdSingleDataLength.GetValue();
        data_cmd     = m_HdSingleDataCmd.GetValue();
        tx_interval     = m_HdSingleTxInterval.GetValue();
        if ((data_hash != m_CurrentDataHash) ||
            (data_size != m_CurrentDataLen) ||
            (data_cmd != m_CurrentDataCmd)) {
            m_CurrentDataHash = data_hash;
            m_CurrentDataLen = data_size;
            m_CurrentDataCmd = data_cmd;
            m_SingleSentCount = 0;
        }
        if (data_hash == 0) {
            memcpy(&hash[0], &hash_tmp[0], 8);
        } else {
            memset(&hash[0], data_hash, 8);
        }
        switch (data_cmd) {
        case 1:
        case 2:
            DetectHostDrivenOneShot(hash, data_size, data_cmd, &m_MacAddr[0], 0);
            m_SingleSentCount++;
            break;
        case 3:
        case 4:
            DetectContTxThreadStart(hash, data_size, data_cmd - 3, &m_MacAddr[0], 0, tx_interval);
            m_IsContinueTx = true;
            break;
        default:
            break;
        }
        break;
    case PageId_TestAgent:
        break;
    default:
        break;
    }
}

void AgentPage::ButtonXExecutePageAction(int PageId)
{
    uint8_t data_hash;
    int byte_idx;
    int bit_idx;

    switch (m_PageId) {
    case PageId_StandAlone:
        data_hash    = m_SssHashFilter.GetValue();
        byte_idx = data_hash / 8;
        bit_idx = data_hash % 8;
        if (m_HashListBitMask[byte_idx] & (1 << bit_idx))
            m_HashListBitMask[byte_idx] &= ~(1 << bit_idx);
        else
            m_HashListBitMask[byte_idx] |= (1 << bit_idx);
        break;
    case PageId_HostDriven:

        break;
    case PageId_TestAgent:
        break;
    default:
        break;
    }
}

void AgentPage::ButtonYExecutePageAction(int PageId)
{
    switch (m_PageId) {
    case PageId_StandAlone:
        memset(&m_HashListBitMask[0], 0x0, 32);
        ClearRecvDetectFrameBuff();
        ClearRecvStandAloneFrameCount();
        break;
    case PageId_HostDriven:
        if (m_IsContinueTx) {
            DetectContTxThreadStop();
            m_IsContinueTx = false;
        }
        m_SingleSentCount = 0;
        m_CurrentDataHash = 0;
        m_CurrentDataLen = 0;
        m_CurrentDataCmd = 0;
        ClearRecvDetectFrameBuff();
        break;
    case PageId_TestAgent:
        break;
    default:
        break;
    }
}

void AgentPage::ButtonPlusExecutePageAction(int PageId)
{
    switch (m_PageId) {
    case PageId_StandAlone:
        if (m_IsDetectFetching) {
            DetectStandAloneStopFetchFrame();
            m_IsDetectFetching = false;
        } else {
            DetectStandAloneStartFetchFrame();
            m_IsDetectFetching = true;
        }
        break;
    case PageId_HostDriven:
        break;
    case PageId_TestAgent:
        break;
    default:
        break;
    }
}

void AgentPage::UpdateSettingPage(int PageId)
{
    Color TextColor;
    ostringstream oss;
    uint32_t tx_interval;
    uint32_t tx_count;
    uint32_t idle_count;
    uint32_t rx_start;
    uint32_t rx_count;

    switch (m_PageId) {
    case PageId_StandAlone:
        tx_interval = m_SsTxInterval.GetValue();
        tx_count    = m_SsTxCount.GetValue();
        idle_count  = m_SsIdleCount.GetValue();
        rx_start    = m_SsRxStart.GetValue();
        rx_count    = m_SsRxCount.GetValue();
        oss << "Detect parameters:\n";
        oss << "  Phase Len   : " << dec << setw(6) << setfill(' ') << tx_interval * tx_count << " ms\n";
        oss << "  InterPhase  : " << dec << setw(6) << setfill(' ') << tx_interval * idle_count << " ms\n";
        oss << "  Tx Interval : " << dec << setw(6) << setfill(' ') << tx_interval << " ms\n";
        oss << "  Rx Start    : " << dec << setw(6) << setfill(' ') << tx_interval * rx_start << " ms\n";
        oss << "  Rx Period   : " << dec << setw(6) << setfill(' ') << tx_interval * rx_count << " ms";
        m_StandAloneInfo.Text = oss.str();
        m_StandAloneInfo.FitSize();
        m_StandAloneInfo.ShowImpl(Display::GetInstance());
        oss.str("");

        if (m_RunningErrorStandAlone) {
            TextColor = ToColor(RED);
            m_StandAloneSettingStatus.SetFgColor(TextColor);
            m_StandAloneSettingStatus.Text = "Wlan error!!!";
            m_StandAloneSettingStatus.FitSize();
        } else if ((rx_start + rx_count) > (tx_count + idle_count)) {
            TextColor = ToColor(RED);
            m_StandAloneSettingStatus.SetFgColor(TextColor);
            m_StandAloneSettingStatus.Text = "Rx exceed idle phase";
            m_StandAloneSettingStatus.FitSize();
            m_SettingErrorStandAlone = true;
        } else {
            TextColor = ToColor(GREEN);
            m_StandAloneSettingStatus.SetFgColor(TextColor);
            m_StandAloneSettingStatus.Text = "Ready";
            m_StandAloneSettingStatus.FitSize();
            m_SettingErrorStandAlone = false;
        }
        break;
    case PageId_HostDriven:
        tx_interval = m_HdTxInterval.GetValue();
        tx_count    = m_HdTxCount.GetValue();
        idle_count  = m_HdIdleCount.GetValue();
        rx_start    = m_HdRxStart.GetValue();
        rx_count    = m_HdRxCount.GetValue();

        oss << "Detect parameters:\n";
        oss << "  Phase Len   : " << dec << setw(6) << setfill(' ') << tx_interval * tx_count << " ms\n";
        oss << "  InterPhase  : " << dec << setw(6) << setfill(' ') << tx_interval * idle_count << " ms\n";
        oss << "  Tx Interval : " << dec << setw(6) << setfill(' ') << tx_interval << " ms\n";
        oss << "  Rx Start    : " << dec << setw(6) << setfill(' ') << tx_interval * rx_start << " ms\n";
        oss << "  Rx Period   : " << dec << setw(6) << setfill(' ') << tx_interval * rx_count << " ms";
        m_HostDrivenInfo.Text = oss.str();
        m_HostDrivenInfo.FitSize();
        m_HostDrivenInfo.ShowImpl(Display::GetInstance());
        oss.str("");

        if (m_RunningErrorHostDriven) {
            TextColor = ToColor(RED);
            m_HostDrivenSettingStatus.SetFgColor(TextColor);
            m_HostDrivenSettingStatus.Text = "Wlan error!!!";
            m_HostDrivenSettingStatus.FitSize();
        } else if ((rx_start + rx_count) > (tx_count + idle_count)) {
            TextColor = ToColor(RED);
            m_HostDrivenSettingStatus.SetFgColor(TextColor);
            m_HostDrivenSettingStatus.Text = "Rx exceed idle phase";
            m_HostDrivenSettingStatus.FitSize();
            m_SettingErrorHostDriven = true;
        } else {
            TextColor = ToColor(GREEN);
            m_HostDrivenSettingStatus.SetFgColor(TextColor);
            m_HostDrivenSettingStatus.Text = "Ready";
            m_HostDrivenSettingStatus.FitSize();
            m_SettingErrorHostDriven = false;
        }
        break;
    case PageId_TestAgent:
        break;
    default:
        break;
    }

}

void AgentPage::UpdateExecutePageInfo(int PageId)
{
    Color TextColor;
    ostringstream oss;
    bool isSend;
    int RecvFrameCount;
    int TotalRecvFrameCountInSleep;
    int SendFrameCount;
    uint8_t *current_hash;
    int AgentCmdIdx;
    uint8_t channel;
    DetectFrameInfo_t* pRecvFrameBuff;
    TestAgentCmd_t* pAgentCmdList;
    nn::wlan::MacAddress bssid;
    char bssidStr[nn::wlan::MacAddress::MacStringSize];
    uint8_t data_hash;
    int byte_idx;
    int bit_idx;
    int filter_cnt;
    int filter_line;

    switch (m_PageId) {
    case PageId_StandAlone:
        if (m_Selected == false) {
            m_StandAloneRunningDataInfo.Text = "";
            m_StandAloneRunningDataInfo.FitSize();
        } else {
            TextColor = ToColor(WHITE);
            m_StandAloneRunningInfo.SetFgColor(TextColor);
            filter_cnt = 0;
            for (byte_idx = 0; byte_idx < 32; byte_idx++) {
                for (bit_idx = 0; bit_idx < 8; bit_idx++) {
                    if (m_HashListBitMask[byte_idx] & (1 << bit_idx)) {
                        filter_cnt++;
                    }
                }
            }
            oss << "Hash List Number:";
            oss << dec << setw(3) << setfill(' ') << filter_cnt;
            oss << "\n";
            m_StandAloneRunningInfo.Text = oss.str();
            m_StandAloneRunningInfo.FitSize();
            oss.str("");

            TextColor = ToColor(WHITE);
            m_StandAloneRunningDataInfo.SetFgColor(TextColor);
            filter_cnt = 0;
            filter_line = 1;
            oss << "Hash:";
            for (byte_idx = 0; byte_idx < 32 && filter_cnt < 33; byte_idx++) {
                for (bit_idx = 0; bit_idx < 8 && filter_cnt < 33; bit_idx++) {
                    if (m_HashListBitMask[byte_idx] & (1 << bit_idx)) {
                        data_hash = 8 * byte_idx + bit_idx;
                        if ((filter_cnt != 0) && (filter_cnt % 11 == 0)) {
                            oss << "\n     ";
                            filter_line++;
                        }
                        if (data_hash)
                            oss << " 0x" << hex << setw(2) << setfill('0') << uppercase << (int)data_hash;
                        else
                            oss << " x0-F";
                        filter_cnt++;
                    }
                }
            }
            oss << "\n";
            TotalRecvFrameCountInSleep = GetRecvStandAloneFrameCount();
            oss << dec << TotalRecvFrameCountInSleep << " frames received in sleep";
            nn::os::LockMutex(&m_RxBuffLock);
            RecvFrameCount = GetRecvDetectFrameCount();
            if (RecvFrameCount) {
                pRecvFrameBuff = (DetectFrameInfo_t*)GetRecvDetectFrameBuff();
                if (pRecvFrameBuff) {
                    oss << "(Fetched Frame: " << dec << RecvFrameCount << ")";
                    for(int i = RecvFrameCount - 1; i > RecvFrameCount - 15; i--)
                    {
                        if (i < 0)
                            break;
                        bssid.Set(pRecvFrameBuff[i].bssid);
                        bssid.GetString(bssidStr);
                        oss << "\n    " << bssidStr;
                        oss << ", " << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[0];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[1];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[2];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[3];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[4];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[5];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[6];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[7];
                        oss << ", " << dec << setw(4) << setfill(' ') << pRecvFrameBuff[i].recvLength << " bytes";
                        oss << ", " << dec << pRecvFrameBuff[i].recvTimes;
                    }
                }
            }
            nn::os::UnlockMutex(&m_RxBuffLock);

            m_StandAloneRunningDataInfo.Text = oss.str();
            m_StandAloneRunningDataInfo.FitSize();
            oss.str("");

            channel = m_SsChannelNum.GetValue();
            if (m_IsDetectFetching)
                TextColor = ToColor(RED);
            else
                TextColor = ToColor(ORANGE);
            m_StandAloneRunningStatus.SetFgColor(TextColor);
            oss << "Running(RX";
            if (m_IsDetectFetching) {
                oss << ",F";
            }
            oss << ",Ch" << dec << (int)channel << ")";
            m_StandAloneRunningStatus.Text = oss.str();
            m_StandAloneRunningStatus.FitSize();
            oss.str("");
        }
        break;
    case PageId_HostDriven:
        if (m_Selected == false) {
            m_HostDrivenRunningDataInfo.Text = "";
            m_HostDrivenRunningDataInfo.FitSize();
        } else {
            TextColor = ToColor(WHITE);
            m_HostDrivenRunningStatus.SetFgColor(TextColor);
            nn::os::LockMutex(&m_RxBuffLock);
            RecvFrameCount = GetRecvDetectFrameCount();
            if (RecvFrameCount) {
                pRecvFrameBuff = (DetectFrameInfo_t*)GetRecvDetectFrameBuff();
                if (pRecvFrameBuff) {
                    oss << "Received Detect Frame: " << dec << RecvFrameCount;
                    for(int i = RecvFrameCount - 1; i > RecvFrameCount - 15; i--)
                    {
                        if (i < 0)
                            break;
                        bssid.Set(pRecvFrameBuff[i].bssid);
                        bssid.GetString(bssidStr);
                        oss << "\n    " << bssidStr;
                        oss << ", " << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[0];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[1];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[2];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[3];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[4];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[5];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[6];
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)pRecvFrameBuff[i].dataHash[7];
                        oss << ", " << dec << setw(4) << setfill(' ') << pRecvFrameBuff[i].recvLength << " bytes";
                        oss << ", " << dec << pRecvFrameBuff[i].recvTimes;
                    }
                }
            }
            nn::os::UnlockMutex(&m_RxBuffLock);
            if (m_IsContinueTx) {
                nn::os::LockMutex(&m_ContTxLock);
                SendFrameCount = GetSendDetectFrameCount();
                current_hash = GetCurrentHash();
                if (SendFrameCount) {
                    oss << "\nSent Periodic Frame: ";
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[0];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[1];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[2];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[3];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[4];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[5];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[6];
                    oss << hex << setw(2) << setfill('0') << uppercase << (int)current_hash[7];
                    oss << ", " << dec << (int)m_CurrentDataLen;
                    oss << " bytes, " << dec << (int)SendFrameCount;
                }
                nn::os::UnlockMutex(&m_ContTxLock);
            } else {
                if (m_SingleSentCount) {
                    if (m_CurrentDataCmd == 1)
                        oss << "\nSent Periodic Frame: ";
                    else if (m_CurrentDataCmd == 2)
                        oss << "\nSent OneShot Frame: ";
                    if (m_CurrentDataHash) {
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                        oss << hex << setw(2) << setfill('0') << uppercase << (int)m_CurrentDataHash;
                    } else oss << "0123456789ABCDEF";
                    oss << ", " << dec << (int)m_CurrentDataLen;
                    oss << " bytes, " << dec << (int)m_SingleSentCount;
                }
            }

            m_HostDrivenRunningDataInfo.Text = oss.str();
            m_HostDrivenRunningDataInfo.FitSize();
            oss.str("");

            isSend = m_HdIsSendPeriodical.GetValue();
            channel = m_HdChannelNum.GetValue();
            if (m_IsContinueTx)
                TextColor = ToColor(RED);
            else
                TextColor = ToColor(ORANGE);
            m_HostDrivenRunningStatus.SetFgColor(TextColor);
            oss << "Running(RX";
            if (isSend) {
                oss << ",P";
                if (m_IsContinueTx)
                    oss << "C";
            } else if (m_IsContinueTx)
                oss << ",C";
            oss << ",Ch" << dec << (int)channel << ")";
            m_HostDrivenRunningStatus.Text = oss.str();
            m_HostDrivenRunningStatus.FitSize();
            oss.str("");
        }
        break;
    case PageId_TestAgent:
        if (m_Selected == false) {
            m_TestAgentRunningDataInfo.Text = "";
            m_TestAgentRunningDataInfo.FitSize();
        } else {
            nn::os::LockMutex(&m_CmdListLock);
            AgentCmdIdx = GetCmdListIdx();
            pAgentCmdList = (TestAgentCmd_t*)GetCmdListBuff();
            for (int i = 0; i < 20; i++)
            {
                if (pAgentCmdList[AgentCmdIdx].cmd[0] == '\0') {
                    break;
                }
                oss << pAgentCmdList[AgentCmdIdx].cmd << "\n";
                if (AgentCmdIdx == 0)
                    AgentCmdIdx = 99;
                else
                    AgentCmdIdx--;
            }
            m_TestAgentRunningDataInfo.Text = oss.str();
            m_TestAgentRunningDataInfo.FitSize();
            oss.str("");
            nn::os::UnlockMutex(&m_CmdListLock);
        }

        if (TestAgent::GetAgentStatus()) {
            TextColor = ToColor(RED);
            oss << "Running(Testing)";
        } else {
            TextColor = ToColor(ORANGE);
            oss << "Running";
        }
        m_TestAgentRunningStatus.Text = oss.str();
        m_TestAgentRunningStatus.FitSize();
        m_TestAgentRunningStatus.SetFgColor(TextColor);
        oss.str("");
        break;
    default:
        break;
    }
} // NOLINT(impl/function_size)

void AgentPage::ShowAgentModePage()
{
    Color TextColor;
    ostringstream oss;

    m_InfoPage.Show(Display::GetInstance());

    oss << APP_NAME << " " << APP_VERSION;
    oss << SettingPageTable[PageId_AgentMode].TitleInfo;
    m_Title.Text = oss.str();
    oss.str("");
    m_Title.FitSize();
    m_Title.ShowImpl(Display::GetInstance());

    TextColor = ToColor(ORANGE);
    m_Help.SetFgColor(TextColor);
    m_Help.Text = SettingPageTable[PageId_AgentMode].SettingPageHelpInfo;
    m_Help.FitSize();
    m_Help.ShowImpl(Display::GetInstance());
}

void AgentPage::ApplicationImpl()
{
    static nn::os::Tick tick;
    static bool isHoldStart;
    static bool isHoldSelect;
    static bool isHoldZr;
    static bool isHoldZl;

    if (m_AgentMode) {
        ShowAgentModePage();
        Display::GetInstance().SwapBuffer();
        return;
    }

    Pad &pad = Pad::GetInstance();
#if 0
    bool isTriggered = pad.IsTrigger(Button::A) || pad.IsTrigger(Button::B)
            || pad.IsTrigger(Button::X) || pad.IsTrigger(Button::Y)
            || pad.IsTrigger(Button::L) || pad.IsTrigger(Button::R)
            || pad.IsTrigger(Button::ZL) || pad.IsTrigger(Button::ZR)
            || pad.IsTrigger(Button::START) || pad.IsTrigger(Button::SELECT)
            || pad.IsTrigger(Button::LEFT) || pad.IsTrigger(Button::UP)
            || pad.IsTrigger(Button::RIGHT) || pad.IsTrigger(Button::DOWN);
#endif

    ShowCommonPageInfo(m_PageId);
    if (!m_Selected) {
        ShowSettingPageInfo(m_PageId);

        isHoldStart = isHoldStart || pad.IsHold(Button::START);
        isHoldSelect = isHoldSelect || pad.IsHold(Button::SELECT);
        isHoldZl = isHoldZl || pad.IsHold(Button::ZL);
        isHoldZr = isHoldZr || pad.IsHold(Button::ZR);

        if (Pad::GetInstance().IsTrigger(Button::A)) {
            EnterExecutePageAction(m_PageId);
        } else if (Pad::GetInstance().IsTrigger(Button::R)) {
            if (m_PageId + 1 == PageId_Max) {
                m_PageId = PageId_StandAlone;
            } else {
                m_PageId++;
            }
        } else if (Pad::GetInstance().IsTrigger(Button::L)) {
            if (m_PageId == PageId_StandAlone) {
                m_PageId = PageId_Max - 1;
            } else {
                m_PageId--;
            }
        } else if (Pad::GetInstance().IsTrigger(Button::B)
                || Pad::GetInstance().IsTrigger(Button::LEFT)
                || Pad::GetInstance().IsTrigger(Button::RIGHT)) {
        } else if (isHoldStart && isHoldSelect && isHoldZl && isHoldZr) {
            Exit();
        }

        UpdateSettingPage(m_PageId);
    } else {
        ShowExecutePageInfo(m_PageId);
        if (Pad::GetInstance().IsTrigger(Button::A)) {
            ButtonAExecutePageAction(m_PageId);
        }
        if (Pad::GetInstance().IsTrigger(Button::B)) {
            LeaveExecutePageAction(m_PageId);
        }
        if (Pad::GetInstance().IsTrigger(Button::X)) {
            ButtonXExecutePageAction(m_PageId);
        }
        if (Pad::GetInstance().IsTrigger(Button::Y)) {
            ButtonYExecutePageAction(m_PageId);
        }
        if (Pad::GetInstance().IsTrigger(Button::PLUS)) {
            ButtonPlusExecutePageAction(m_PageId);
        }
        UpdateExecutePageInfo(m_PageId);
    }

    if ((nn::os::GetSystemTick() - tick).ToTimeSpan()
            >= nn::TimeSpan::FromMilliSeconds(1000)) {
        isHoldStart = false;
        isHoldSelect = false;
        isHoldZr = false;
        isHoldZl = false;
        tick = nn::os::GetSystemTick();
    }
}
}
