﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>
#include <nn/wlan/wlan_Result.h>

namespace
{
    // スキャンバッファです。
    const size_t buffSize = 100 * 1024;
    NN_ALIGNAS(4096) char g_scanBuffer[ buffSize ];  // 100KB

    // ローカル通信用StaticAES鍵
    static const nn::Bit8 staticAesKey[16] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    class LocalLcsState : public ::testing::Test
    {
    protected:
        nn::wlan::ScanParameters scanParam;
        nn::wlan::MasterBssParameters masterParam;

        virtual void SetUp() NN_OVERRIDE
        {
            // 基本となるスキャンパラメータです。
            scanParam.scanType = nn::wlan::ScanType_Passive;
            memset(scanParam.channelList, 0, sizeof(scanParam.channelList));
            scanParam.channelCount = 0;
            scanParam.channelScanTime = 120;
            scanParam.homeChannelTime = 0;
            scanParam.ssidList = NULL;
            scanParam.ssidCount = 0;
            scanParam.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

            // スキャンバッファを初期化します。
            memset(g_scanBuffer, 0, sizeof(g_scanBuffer));

            // MASTER側のBSSの設定です。
            masterParam.channel = -1;
            masterParam.autoKeepAlive = false;
            masterParam.basicRates = nn::wlan::RateSetLegacy_11gMask;
            masterParam.supportedRates = nn::wlan::RateSetLegacy_11gMask;
            masterParam.beaconInterval = 100;
            masterParam.hiddenSsid = true;
            masterParam.inactivePeriod = 10;
            masterParam.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
            masterParam.security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
            std::memcpy(&masterParam.security.key[0], &staticAesKey[0], sizeof(staticAesKey));
            masterParam.ssid.Set("LocalLcsTestMaster.#$%");

            // ローカル通信モードを開始します。
            nn::wlan::InitializeLocalManager();
        }

        virtual void TearDown() NN_OVERRIDE
        {
            // ローカル通信モードを終了します。
            nn::wlan::FinalizeLocalManager();
        }
    public:
        void LocalScan()  NN_NOEXCEPT
        {
            // スキャンバッファを初期化します。
            memset(g_scanBuffer, 0, sizeof(g_scanBuffer));

            // スキャンを開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::StartScan(g_scanBuffer, buffSize, scanParam));

            // スキャン結果からRSSI値を取り出し、0でないか評価します。
            nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);

            // 見付かったAPが0個だと意味がないので、ここで見付かった個数の評価を行います。
            ASSERT_NE(0, resultReader.GetCount());

            for( int j = 0; j < resultReader.GetCount(); j++ )
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                ASSERT_NE(0, beacon.GetRssi());
            }
        }
    };
}

TEST_F(LocalLcsState, LcsMasterStateCheck)
{
    nn::Result result;

    // Lcs Masterの開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenLcsMasterMode());

    // stateの確認をします。
    nn::wlan::WlanState wlanState;
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
    ASSERT_EQ(nn::wlan::WlanState_LocalLcsMasterIdle, wlanState);

    // LcsMaster以外のモードのOpenCloseを行い、正しくResultFailureが返って来るか確認します。
    result = nn::wlan::Local::OpenMasterMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::CloseMasterMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::OpenClientMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::CloseClientMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::OpenLcsClientMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::CloseLcsClientMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);

    // BSS作成
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateBss(masterParam));
    // stateの確認をします。
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
    ASSERT_EQ(nn::wlan::WlanState_LocalLcsMasterBss, wlanState);

    LocalScan();

    // BSS破棄
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
    // stateの確認をします。
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
    ASSERT_EQ(nn::wlan::WlanState_LocalLcsMasterIdle, wlanState);

    // Lcs Masterモードを終了します。
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseLcsMasterMode());
}

TEST_F(LocalLcsState, LcsClientStateCheck)
{
    nn::Result result;

    // Lcs Clientの開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenLcsClientMode());

    // stateの確認をします。
    nn::wlan::WlanState wlanState;
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
    ASSERT_EQ(nn::wlan::WlanState_LocalLcsClientIdle, wlanState);

    // LcsClient以外のモードのOpenCloseを行い、正しくResultFailureが返って来るか確認します。
    result = nn::wlan::Local::OpenMasterMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::CloseMasterMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::OpenClientMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::CloseClientMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::OpenLcsMasterMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);
    result = nn::wlan::Local::CloseLcsMasterMode();
    NNT_ASSERT_RESULT_FAILURE(nn::wlan::ResultInvalidState, result);

    LocalScan();

    // Lcs Clientモードを終了します。
    NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseLcsClientMode());
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    // 無線スイッチをオフにします。
    {
        // デバッグ設定書き込み。
        const bool isEnabled = true;
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

        nn::nifm::Initialize();
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておきます。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    // テストを実行します。
    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
