﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace {

    const char* WlanTesStaticAesSSID                = LocalMasterTest2SSID;

    const char* WlanTesOpenSSID                     = LocalMasterTestSSID;

    const size_t WlanTestBufferSize                 = 512;

    const size_t WlanTestMsgBufferSize              = 128;

    const int64_t WlanTestClinetConnectTime         = 10 * 1000;

    const WlanTest::LocalClientConnectParameter NormalConnectParam = {
        nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
    };

    struct TestScanParams
    {
        uint32_t testNo;
        size_t size;
        nn::wlan::ScanParameters params;
    };

    nn::wlan::Ssid ssidList[] = { nn::wlan::Ssid(WlanTesOpenSSID) };

    static const uint8_t WlanTestMasterMac[] =
    {
        0x40, 0xD2, 0x8A, 0xB4, 0x93, 0xAA
    };

    static const uint8_t WlanTestMasterNotMac[] =
    {
        0x11, 0x22, 0x33, 0x44, 0x55, 0x66
    };

    const TestScanParams StartScanTestParams[] = {
        // 正常値テストNo30 アクティブスキャン、最小バッファサイズ
        { 30, sizeof(nn::wlan::ScanResultHeader) + 1,{ nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 30, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()} },
        // 正常値テストNo31 アクティブスキャン、サンプルバッファサイズ
        { 31, WlanTestBufferSize ,{ nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 30, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress() } },
        // 正常値テストNo32 アクティブスキャン、サンプルバッファサイズ、SSID指定なし
        { 32, WlanTestBufferSize ,{ nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 30, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() } },
        // 正常値テストNo33 パッシブスキャン、最小バッファサイズ
        { 33, sizeof(nn::wlan::ScanResultHeader) + 1,{ nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 30, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress() } },
        // 正常値テストNo34 パッシブスキャン、サンプルバッファサイズ
        { 34, WlanTestBufferSize ,{ nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 30, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress() } },
        // 正常値テストNo35 パッシブスキャン、サンプルバッファサイズ、SSID指定なし
        { 35, WlanTestBufferSize ,{ nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 30, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() } },
    };

    const TestScanParams StopScanTestParams[] = {
        // 正常値テストNo28 アクティブスキャン、最小バッファサイズ
        { 28, 1,{ nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 30, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress() }}
    };

    const uint8_t LocalVendorIe[] = {
        0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x78
    };

    const uint8_t TestEtherType[] = {
        0x80, 0xB7
    };

}

namespace WlanTest {

    class LocalClientNormalInitUnitTest : public ::testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }

        void MakeFrameBuffer(uint8_t buffer[], const size_t size, const nn::wlan::MacAddress putMacAddress, const nn::wlan::MacAddress getMacAddress) NN_NOEXCEPT
        {
            size_t offsetSize = 0;
            const uint32_t HeaderSize = nn::wlan::MacAddress::MacAddressSize * 2;

            if (size >= offsetSize + nn::wlan::MacAddress::MacAddressSize)
            {
                std::memcpy(&buffer[0], putMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
                offsetSize += nn::wlan::MacAddress::MacAddressSize;
            }

            if (size >= offsetSize + nn::wlan::MacAddress::MacAddressSize)
            {
                std::memcpy(&buffer[nn::wlan::MacAddress::MacAddressSize], getMacAddress.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
                offsetSize += nn::wlan::MacAddress::MacAddressSize;
            }

            if (size >= offsetSize + sizeof(TestEtherType))
            {
                // Payload埋め
                // EtherType
                std::memcpy(&buffer[HeaderSize], TestEtherType, sizeof(TestEtherType));
                offsetSize += sizeof(TestEtherType);
            }

        }

        // LocalClient
        LocalApiClass localClient;
        nn::wlan::DisconnectClient disClient;
        nn::wlan::ConnectionStatus connectStatus;
        nn::os::SystemEventType connectEvent;
        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress putMacAddress;
        uint32_t rxId;
        uint32_t actionRxId;
        int32_t rssi;
    };

    class LocalClientNormalUnitTest : public LocalClientNormalInitUnitTest
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            isEventStart = false;
            conParam = NormalConnectParam;
            LocalClientNormalInitUnitTest::SetUp();
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

            if (nn::wlan::Local::GetConnectionEvent(&connectionEvent).IsSuccess() == true)
            {
                isEventStart = true;
            }
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
            localClient.LocalRelease();
            if (isEventStart == true)
            {
                nn::os::DestroySystemEvent(&connectionEvent);
                isEventStart = false;
            }
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
            LocalClientNormalInitUnitTest::TearDown();
        }

        bool isEventStart;
        nn::os::SystemEventType connectionEvent;
        LocalClientConnectParameter conParam;
    };

    class LocalClientParamsTest : public LocalClientNormalUnitTest,
        public ::testing::WithParamInterface<TestScanParams>
    {
    protected:
        std::unique_ptr<char[]> failMessage;

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            std::unique_ptr<char[]> message(new char[WlanTestMsgBufferSize]);
            WLANTEST_ASSERT_TRUE(message.get() != nullptr);
            failMessage = std::move(message);

            LocalClientNormalUnitTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            LocalClientNormalUnitTest::TearDown();
        }
    };

    typedef LocalClientParamsTest   LocalClientStartScanUnitTest;
    typedef LocalClientParamsTest   LocalClientStopScanUnitTest;
    typedef LocalClientParamsTest   LocalClientSsidBssidCombiTest;

    TEST_F(LocalClientNormalInitUnitTest, OpenCloseClientMode_No7_No8)
    {
        // 正常値テストNo5 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

        // 正常値テストNo6 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
    }

    //No30~No35
    TEST_P(LocalClientStartScanUnitTest, ScanStart_No30_No35)
    {
        TestScanParams testScanParam = GetParam();

        NN_LOG("             LocalClient StartScan Normal Test No.%d START\n", testScanParam.testNo);
        nn::util::SNPrintf(failMessage.get(), WlanTestMsgBufferSize, "StartScan Test No.%d  size(%lu) ScanType(%d)",
            testScanParam.testNo, testScanParam.size, testScanParam.params.scanType);
        WLANTEST_SCOPED_TRACE(failMessage.get());

        nn::wlan::ScanIeMatchInfo info;

        std::memcpy(&info, &TestMatchInfo[0], sizeof(nn::wlan::ScanIeMatchInfo));

        std::unique_ptr<char[]> scanBuffer(new char[testScanParam.size]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::StartScan(scanBuffer.get(), testScanParam.size, testScanParam.params, &info));
        NN_LOG("             LocalClient StartScan Normal Test No.%d END\n", testScanParam.testNo);
    }

    TEST_P(LocalClientStartScanUnitTest, StopScan_No36)
    {
        TestScanParams testScanParam = GetParam();

        NN_LOG("             LocalClient StopScan Normal UnitTest Test No.36(Pattern:%d) START\n", testScanParam.testNo);
        nn::util::SNPrintf(failMessage.get(), WlanTestMsgBufferSize, "StopScan Test No.%d  size(%lu) ScanType(%d)",
            testScanParam.testNo, testScanParam.size, testScanParam.params.scanType);
        WLANTEST_SCOPED_TRACE(failMessage.get());

        nn::wlan::ScanIeMatchInfo info;
        std::memcpy(&info, &TestMatchInfo[0], sizeof(nn::wlan::ScanIeMatchInfo));

        // 正常値テストNo35 CLIENT(Target MASTER)
        std::unique_ptr<char[]> scanBuffer(new char[testScanParam.size]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::StartScan(scanBuffer.get(), testScanParam.size, testScanParam.params, &info));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::StopScan());
        NN_LOG("             LocalClient StopScan Normal UnitTest Test No.36(Pattern:%d) END\n", testScanParam.testNo);
    }

    TEST_F(LocalClientStopScanUnitTest, OpenSecurityConnect_No37_No38)
    {
        // 正常値テストNo37 - No38 CLIENT(Target MASTER)
        nn::wlan::Ssid ssid(WlanTesOpenSSID);
        nn::wlan::Ssid dummySsid;

        macAddress = nn::wlan::MacAddress(WlanTestMasterMac);

        const LocalClientConnectParameter ConnectParams[] = {
            // 正常値テストNo37 - 38 Open SSID/MacAddress
            { nn::wlan::Ssid(WlanTesOpenSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10 },
            { dummySsid, macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10 },
        };

        for (auto connectSet : ConnectParams)
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(connectSet, WlanTestClinetConnectTime));
            WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
        }

    }

    TEST_F(LocalClientNormalUnitTest, StaticAesSecurityConnect_No55_No56)
    {
        // 正常値テストNo55 - No56 CLIENT(Target MASTER)
        nn::wlan::Ssid ssid(WlanTesStaticAesSSID);
        nn::wlan::Ssid dummySsid;

        macAddress = nn::wlan::MacAddress(WlanTestMasterMac);

        const LocalClientConnectParameter ConnectParams[] = {
            // 正常値テストNo55 - 56 StaticAes SSID/MacAddress
            { ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,{ nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10 },
            { dummySsid, macAddress, -1,{ nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10 },
        };

        for (auto connectSet : ConnectParams)
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(connectSet, WlanTestClinetConnectTime));
            WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
        }

    }

    TEST_F(LocalClientNormalUnitTest, CancelConnect_No57)
    {
        // 正常値テストNo57
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(NormalConnectParam.ssid, NormalConnectParam.security/*, TestMatchInfo, 1*/));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientAsyncConnect());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CancelConnect());
    }

    TEST_F(LocalClientNormalUnitTest, Disconnect_No78)
    {
        // 正常値テストNo78
        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientNormalUnitTest, GetConnectionEvent_No81_No84)
    {
        nn::os::SystemEvent conEvent;

        // 正常値テストNo81
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(conEvent.GetBase()));

        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));

        // 正常値テストNo84
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(conEvent.GetBase(), nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));

        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientNormalUnitTest, GetConnectionStatus_No87)
    {
        // 正常値テストNo87 CLIENT(Target MASTER)
        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));
        WLANTEST_ASSERT_TRUE(connectStatus.state == nn::wlan::ConnectionState_Connected);
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientNormalUnitTest, GetState_No92)
    {
        // 正常値テストNo92 CLIENT(Target MASTER)
        nn::wlan::WlanState wlanState = nn::wlan::WlanState_Stop;

        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetState(&wlanState));
        WLANTEST_ASSERT_TRUE(wlanState >= nn::wlan::WlanState_LocalClientIdle && wlanState <= nn::wlan::WlanState_LocalClientScan);
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientNormalUnitTest, AddDeleteIe_No95_No98)
    {
        uint32_t ieId;

        // 正常値テストNo95 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddIe(&ieId, nn::wlan::ManagementFrameType_Beacon, LocalVendorIe, sizeof(LocalVendorIe)));

        // 正常値テストNo98 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteIe(ieId));
    }

    TEST_F(LocalClientNormalUnitTest, GetMacAddress_No100)
    {
        // 正常値テストNo100 MASTER
        nn::wlan::MacAddress macAddress;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
    }

    TEST_F(LocalClientNormalUnitTest, Rssi_No101_No105)
    {
        // 正常値テストNo101 CLIENT
        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetRssi(&rssi));
        WLANTEST_EXPECT_TRUE(rssi != -128);

        // 正常値テストNo105 CLIENT
        nn::wlan::LinkLevel link;
        link = nn::wlan::Local::ConvertRssiToLinkLevel(rssi);
        WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 <= link && nn::wlan::LinkLevel_3 >= link);

        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientNormalUnitTest, GetLinkLevel_No103)
    {
        // 正常値テストNo103 CLIENT
        nn::wlan::LinkLevel link;

        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetLinkLevel(&link));
        WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 <= link && nn::wlan::LinkLevel_3 >= link);

        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientNormalUnitTest, PutGetFrame_No107_No109)
    {
        size_t outSize = 0;

        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(NormalConnectParam.ssid, NormalConnectParam.security, TestMatchInfo, 1, true, false));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        uint16_t ethertypes[] = { 0x80b7 };
        LocalApiClass::LocalMakeFrame(putBuffer.get(), WlanTestBufferSize, putMacAddress, macAddress, ethertypes[0], TestMatchInfo[0]);

        // 正常値テストNo107 CLIENT(Target MASTER)
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestBufferSize));

        // 正常値テストNo109 CLIENT(Target MASTER)
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(reinterpret_cast<uint8_t*>(getBuffer.get()), WlanTestBufferSize, &outSize, localClient.m_entRxId));

        LocalFrameHeader* pFrameHeader = reinterpret_cast<LocalFrameHeader*>(putBuffer.get());
        LocalPayloadData* pFrameData = reinterpret_cast<LocalPayloadData*>(pFrameHeader->data);
        WLANTEST_EXPECT_TRUE(std::memcmp(pFrameData->matchData, TestMatchInfo[0].matchData, TestMatchInfo[0].matchLength) == 0);

        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));

        localClient.LocalRelease();
    }

    TEST_F(LocalClientNormalUnitTest, CancelGetFrame_No112)
    {
        // 正常値テストNo112 CLIENT(Target MASTER)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();


        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(NormalConnectParam.ssid, NormalConnectParam.security, TestMatchInfo, 1, true, false));
        localClient.SetCommandTime(WlanTestCancelWaitTime);

        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_EXPECT_TRUE(localClient.LocalGetFrame(WlanTestBufferSize));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CancelGetFrame(localClient.m_entRxId));
    }

    TEST_F(LocalClientNormalUnitTest, CreateDeleteRxEntry_No115_No118)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };

        // 正常値テストNo115 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));

        // 正常値テストNo118 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientNormalUnitTest, AddDeleteEtherType_No121_No124)
    {
        const uint16_t Ethertypes[] = { 0x88b7, 0x88b8 };
        /*uint32_t outRxId;*/

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));

        // 正常値テストNo121 CLIENT
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(rxId, Ethertypes[1]));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, Ethertypes[1]));

        // 正常値テストNo124 CLIENT
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientNormalUnitTest, AddRemoveMatchData_No127_No130)
    {
        const uint16_t Ethertypes[] = { 0x88b7 };

        // 正常値テストNo127 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, Ethertypes, 1, 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));

        // 正常値テストNo130 CLIENT
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, TestMatchInfo[0]));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientNormalUnitTest, GetActionFrame_No138)
    {
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        uint32_t actionRxId;
        size_t outSize;

        // 正常値テストNo138 CLIENT(Target MASTER)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, &outSize, actionRxId));

        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    TEST_F(LocalClientNormalUnitTest, CancelGetActionFrame_No141)
    {
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;

        // 正常値テストNo141
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(NormalConnectParam.ssid, NormalConnectParam.security, TestNotMatchInfo, 1, false, true));
        localClient.SetCommandTime(WlanTestCancelWaitTime);
        WLANTEST_EXPECT_TRUE(localClient.LocalClientConnect(conParam, WlanTestClinetConnectTime));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));

        WLANTEST_EXPECT_TRUE(localClient.LocalCancelActionFrameStart(500));
        nn::Result result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, &outSize, localClient.m_actionRxId);
        WLANTEST_EXPECT_TRUE(nn::wlan::ResultGetFrameCancelled::Includes(result));
        localClient.LocalCancelActionFrameStop();

        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientNormalUnitTest, CreateDeleteRxEntryForActionFrame_No144_No147)
    {
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };

        // 正常値テストNo144 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        // 正常値テストNo147 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    TEST_F(LocalClientNormalUnitTest, AddDeleteSubtypeToRxEntryForActionFrame_No150_No153)
    {
        uint32_t actionRxId/*, outActionRxId*/;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, 1, 10));

        // 正常値テストNo150 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(actionRxId, nn::wlan::ActionFrameType_Local));

        // 正常値テストNo153 CLIENT
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&actionRxId, nn::wlan::ActionFrameType_Local));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No156)
    {
        // 正常値テストNo156 CLIENT(Target MASTER)
        nn::Result result;
        macAddress = nn::wlan::MacAddress::CreateZeroMacAddress();

        const LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:0 MacAddress：0
            nn::wlan::Ssid(""), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        result = nn::wlan::Local::Connect(ConnectParams.ssid,
                                          ConnectParams.bssid,
                                          ConnectParams.channel,
                                          ConnectParams.security,
                                          ConnectParams.autoKeepAlive,
                                          ConnectParams.indication,
                                          ConnectParams.beaconLostTimeout);
        WLANTEST_ASSERT_TRUE(nn::wlan::ResultInvalidArgument::Includes(result));
        NN_LOG("Connect: bssid must not be ZeroMacAddress\n");
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No157)
    {
        // 正常値テストNo157 CLIENT(Target MASTER)
        macAddress = nn::wlan::MacAddress(WlanTestMasterMac);

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:0 MacAddress：一致
            nn::wlan::Ssid(""), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No158)
    {
        // 正常値テストNo158 CLIENT(Target MASTER)
        nn::Result result;

        const LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:0 MacAddress：broadcast address
            nn::wlan::Ssid(""), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        result = nn::wlan::Local::Connect(ConnectParams.ssid,
                                          ConnectParams.bssid,
                                          ConnectParams.channel,
                                          ConnectParams.security,
                                          ConnectParams.autoKeepAlive,
                                          ConnectParams.indication,
                                          ConnectParams.beaconLostTimeout);
        WLANTEST_ASSERT_TRUE(nn::wlan::ResultInvalidArgument::Includes(result));
        NN_LOG("Connect: SSID must not be empty when BSSID is broadcast address\n");
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No159)
    {
        // 正常値テストNo159 CLIENT(Target MASTER)
        macAddress = nn::wlan::MacAddress(WlanTestMasterNotMac);

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:0 MacAddress：不一致
            nn::wlan::Ssid(""), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No160)
    {
        // 正常値テストNo160 CLIENT(Target MASTER)
        nn::Result result;
        macAddress = nn::wlan::MacAddress::CreateZeroMacAddress();

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:一致 MacAddress：0
            nn::wlan::Ssid(WlanTesOpenSSID), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        result = nn::wlan::Local::Connect(ConnectParams.ssid,
                                          ConnectParams.bssid,
                                          ConnectParams.channel,
                                          ConnectParams.security,
                                          ConnectParams.autoKeepAlive,
                                          ConnectParams.indication,
                                          ConnectParams.beaconLostTimeout);
        WLANTEST_ASSERT_TRUE(nn::wlan::ResultInvalidArgument::Includes(result));
        NN_LOG("Connect: bssid must not be ZeroMacAddress\n");
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No161)
    {
        // 正常値テストNo161 CLIENT(Target MASTER)
        macAddress = nn::wlan::MacAddress(WlanTestMasterMac);

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:一致 MacAddress：一致
            nn::wlan::Ssid(WlanTesOpenSSID), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No162)
    {
        // 正常値テストNo162 CLIENT(Target MASTER)

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:一致 MacAddress：broadcast address
            nn::wlan::Ssid(WlanTesOpenSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No163)
    {
        // 正常値テストNo163 CLIENT(Target MASTER)
        macAddress = nn::wlan::MacAddress(WlanTestMasterNotMac);

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:一致 MacAddress：不一致
            nn::wlan::Ssid(WlanTesOpenSSID), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No164)
    {
        // 正常値テストNo164 CLIENT(Target MASTER)
        nn::Result result;
        macAddress = nn::wlan::MacAddress::CreateZeroMacAddress();

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:不一致 MacAddress：0
            nn::wlan::Ssid("WlanTestNotSSID"), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        result = nn::wlan::Local::Connect(ConnectParams.ssid,
                                          ConnectParams.bssid,
                                          ConnectParams.channel,
                                          ConnectParams.security,
                                          ConnectParams.autoKeepAlive,
                                          ConnectParams.indication,
                                          ConnectParams.beaconLostTimeout);
        WLANTEST_ASSERT_TRUE(nn::wlan::ResultInvalidArgument::Includes(result));
        NN_LOG("Connect: bssid must not be ZeroMacAddress\n");
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No165)
    {
        // 正常値テストNo165 CLIENT(Target MASTER)
        macAddress = nn::wlan::MacAddress(WlanTestMasterMac);

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:不一致 MacAddress：一致
            nn::wlan::Ssid("WlanTestNotSSID"), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No166)
    {
        // 正常値テストNo166 CLIENT(Target MASTER)

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:不一致 MacAddress：broadcast address
            nn::wlan::Ssid("WlanTestNotSSID"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    TEST_F(LocalClientSsidBssidCombiTest, Ssid_Bssid_combi_No167)
    {
        // 正常値テストNo167 CLIENT(Target MASTER)
        macAddress = nn::wlan::MacAddress(WlanTestMasterNotMac);

        LocalClientConnectParameter ConnectParams =
        {
            // Open SSID:不一致 MacAddress：不一致
            nn::wlan::Ssid("WlanTestNotSSID"), macAddress, 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParams, WlanTestClinetConnectTime));
        WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestClientWaitTimeOut * 1000));
    }

    INSTANTIATE_TEST_CASE_P(WithScanPattern,
        LocalClientStartScanUnitTest,
        ::testing::ValuesIn(StartScanTestParams));

    INSTANTIATE_TEST_CASE_P(WithScanPattern,
        LocalClientStopScanUnitTest,
        ::testing::ValuesIn(StopScanTestParams));

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest
