﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace {
    struct TestUnitConnectParams
    {
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress bssid;
        int16_t channel;
        nn::wlan::Security security;
        bool autoKeepAlive;
        nn::wlan::BeaconIndication indication;
        int beaconLostTimeout;
    };

    const int64_t WlanTestClinetConnectTime = 30 * 1000;

    const char* WlanTesOpenSSID = LocalMasterTestSSID;

    const WlanTest::LocalClientConnectParameter ExceptionConnectParam = {
        nn::wlan::Ssid(WlanTesOpenSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
        { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
    };

} // namespace

namespace WlanTest {

    /********************************************/
    // Class 定義
    /********************************************/
    class LocalClientExceptionStateUnitTest : public ::testing::Test
    {
    public:
        uint32_t m_framerxId;
        nn::os::ThreadType m_getFrameThread;
        nn::Result m_threadResult;
        nn::Result Result;
        LocalApiClass localClient;

        nn::wlan::Ssid ssid;
        size_t     size = BufferSize100k;
        std::unique_ptr<char[]>  pTestMemchar;

        nn::os::SystemEventType SystemEvent;
        nn::wlan::ConnectionStatus OutStatus;
        nn::wlan::WlanState state;

        uint8_t ieBody[7] = { 0x00, 0x22, 0xAA, 0x02, 0xBB, 0x55, 0x16 };
        size_t ie_size = 0;
        uint32_t ieIndex;

        uint16_t ethertypes[1] = { 0x88B7 };
        size_t     out_size = 0;
        uint32_t rxId;
        std::unique_ptr<uint8_t[]> pTestMemUnit8;

        nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Active,
            { 1, 6, 11 },
            0,
            -1,
            -1,
            &ssid,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };

    protected:
        LocalClientExceptionStateUnitTest()  NN_NOEXCEPT
        {
            std::unique_ptr<char[]>pTestMemchar_tmp(new char[size]);
            pTestMemchar = std::move(pTestMemchar_tmp);

            std::unique_ptr<uint8_t[]>pTestMemUnit8_tmp(new uint8_t[size]);
            pTestMemUnit8 = std::move(pTestMemUnit8_tmp);
        }
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_TRUE(pTestMemchar.get() != nullptr);
            WLANTEST_ASSERT_TRUE(pTestMemUnit8.get() != nullptr);
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        void GetFrameThreadStart() NN_NOEXCEPT
        {
            static NN_ALIGNAS(4096) uint8_t stack[StacSize];

            nn::os::CreateThread(&m_getFrameThread, GetFrameThread, this, stack, StacSize, nn::os::DefaultThreadPriority);
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        static void GetFrameThread(void* arg) NN_NOEXCEPT
        {
            size_t     size = BufferSize100k;
            size_t     outSize = 0;
            std::unique_ptr<uint8_t[]>pTestMemGetFrame(new uint8_t[size]);
            WLANTEST_ASSERT_TRUE(pTestMemGetFrame.get() != nullptr);
            LocalClientExceptionStateUnitTest* pthis = static_cast<LocalClientExceptionStateUnitTest*>(arg);

            pthis->m_threadResult = nn::wlan::Local::GetFrameRaw(pTestMemGetFrame.get(), size, &outSize, pthis->m_framerxId);

        }
    };

    class LocalClientExceptionArgmentUnitTest : public ::testing::Test
    {
    protected:
        LocalApiClass localClient;
        LocalClientConnectParameter connectParam;
        char strMacAddress[nn::wlan::MacAddress::MacStringSize];
        nn::Result Result;

        nn::wlan::Ssid ssid;
        uint16_t ethertypes_delete = 0x00;
        uint32_t rxId;
        uint16_t ethertypes[1] = { 0x88B7 };

        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };
        nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Active,
            { 1, 6, 11 },
            0,
            -1,
            -1,
            &ssid,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        uint8_t ieBody[7] = { 0x00, 0x22, 0xAA, 0x02, 0xBB, 0x55, 0x16 };
        size_t ie_size = 1;
        uint32_t ieIndex;

    protected:
        LocalClientExceptionArgmentUnitTest()  NN_NOEXCEPT
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            ssid.Set(LocalMasterTestSSID);

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
            if (nn::wlan::Local::GetConnectionEvent(&connectionEvent).IsSuccess() == true)
            {
                isEventStart = true;
            }

            connectParam = ExceptionConnectParam;
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localClient.LocalRelease();
            if (isEventStart == true)
            {
                nn::os::DestroySystemEvent(&connectionEvent);
                isEventStart = false;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }

        nn::Result ClientConnect(TestUnitConnectParams ConnectParams, uint32_t waitTime) NN_NOEXCEPT
        {
            char strMacAddress[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::ConnectionStatus connectionStatus;
            nn::os::Tick conTimeout;
            nn::Result result;

            // 指定時間接続リトライ
            conTimeout = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - conTimeout).ToTimeSpan().GetMilliSeconds() <= waitTime)
            {
                result = nn::wlan::Local::Connect(ConnectParams.ssid,
                    ConnectParams.bssid,
                    ConnectParams.channel,
                    ConnectParams.security,
                    ConnectParams.autoKeepAlive,
                    ConnectParams.indication,
                    ConnectParams.beaconLostTimeout);

                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(WlanTestConnectTimeOut)) != true)
                {
                    continue;
                }

                if (nn::wlan::Local::GetConnectionStatus(&connectionStatus).IsSuccess() == true)
                {
                    if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }
                    NN_LOG("             LocalClient UintTest Connect Status Change Wait\n", WlanTestConnectTimeOut);
                }
            }

            ConnectParams.bssid.GetString(strMacAddress);
            NN_LOG("             *** LocalClient Connect Parameters ***\n");
            NN_LOG("               SSID       : %s(LEN:%d)\n", ConnectParams.ssid.GetSsidData(), ConnectParams.ssid.GetLength());
            NN_LOG("               BSSID      : %s\n", strMacAddress);
            NN_LOG("               CHANNEL    : %d\n", ConnectParams.channel);
            NN_LOG("               SECURITY G : %s\n", SecurityTypeString[ConnectParams.security.groupPrivacyMode]);
            NN_LOG("               SECURITY P : %s\n", SecurityTypeString[ConnectParams.security.privacyMode]);
            NN_LOG("               KEY INDEX  : %d\n", ConnectParams.security.keyIdx);
            NN_LOG("               PS         : %s\n", ConnectParams.security.key);
            NN_LOG("               AUTOKEEP   : %s\n", ConnectParams.autoKeepAlive == true ? "Yes" : "No");
            NN_LOG("               INDICATION : %d\n", ConnectParams.indication);
            NN_LOG("               Result     : 0x%x(Module:%d Desc:%d)\n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
            NN_LOG("             *** LocalClient Connect Parameters ***\n");

            return result;
        }

        bool isEventStart;
        nn::os::SystemEventType connectionEvent;
    };

    TEST_F(LocalClientExceptionArgmentUnitTest, OpenCloseMasterMode_No1_No3)
    {
        Result = nn::wlan::Local::OpenMasterMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::CloseMasterMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, OpenCloseSpectatorMode_No10_No12)
    {
        Result = nn::wlan::Local::OpenSpectatorMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::CloseSpectatorMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateDestroyBss_No13_No15)
    {
        nn::wlan::MasterBssParameters bssParameters = {
            -1,
            true,
            30,
            false,
            nn::wlan::RateSetLegacy_11gMask,
            nn::wlan::RateSetLegacy_11gMask,
            security,
            ssid,
            0
        };
        Result = nn::wlan::Local::CreateBss(bssParameters);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::DestroyBss();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, StartScan_No17)
    {
        nn::wlan::ScanParameters param = { nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 30, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() };

        nn::wlan::ScanIeMatchInfo info;

        std::memcpy(&info, &TestMatchInfo[0], sizeof(nn::wlan::ScanIeMatchInfo));

        size_t size = 50 * 1024;
        std::unique_ptr<char[]> scanBuffer(new char[size]);

        // 不正ステートのためCloseしてReadyに
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::StartScan(scanBuffer.get(), size, param, &info);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, StartScan_No18)
    {
        nn::wlan::ScanParameters param = { nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 30, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress() };

        nn::wlan::ScanIeMatchInfo info;

        std::memcpy(&info, &TestMatchInfo[0], sizeof(nn::wlan::ScanIeMatchInfo));

        size_t size = 50 * 1024;

        // developビルドの場合、アサートします
        Result = nn::wlan::Local::StartScan(nullptr, size, param, &info);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionStateUnitTest, StopScan_No19)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

        Result = nn::wlan::Local::StopScan();
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, Connect_No23)
    {
        Result = nn::wlan::Local::Connect(ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1, security, 2, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CancelConnect_No26)
    {
        // 要Master
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Connect(ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1, security, true, nn::wlan::BeaconIndication_Enable, 10));

        // Sleep
        Result = nn::wlan::Local::CancelConnect();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CancelJoin_No31)
    {
        Result = nn::wlan::Local::CancelJoin();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, Disconnect_No33)
    {
        nn::wlan::DisconnectClient disClient;

        // 不正ステートのためCloseしてReadyに
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        disClient.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        disClient.reasonCode = nn::wlan::Dot11ReasonCode::Dot11ReasonCode_Reserved;
        Result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disClient);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, Disconnect_No34)
    {
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));
        Result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, 0);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetConnectionEvent_No36)
    {
        Result = nn::wlan::Local::GetConnectionEvent(NULL);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetConnectionStatus_No38)
    {
        Result = nn::wlan::Local::GetConnectionStatus(nullptr);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionStateUnitTest, GetConnectionStatus_No39)
    {
        NN_LOG("         GetConnectionStatus_No39->InitializeLocalManager START\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        NN_LOG("         GetConnectionStatus_No39->InitializeLocalManager END\n\n");

        // InitializeInfraManager
        NN_LOG("         GetConnectionStatus_No39->InitializeInfraManager START\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
        NN_LOG("         GetConnectionStatus_No39->InitializeInfraManager END\n\n");
        // OpenMode
        NN_LOG("         GetConnectionStatus_No39->nn::wlan::Infra::OpenMode START\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
        NN_LOG("         GetConnectionStatus_No39->nn::wlan::Infra::OpenMode END\n\n");

        nn::wlan::ConnectionStatus OutStatus;
        Result = nn::wlan::Local::GetConnectionStatus(&OutStatus);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // CloseMode
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
        // FinalizeInfraManager
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());

        NN_LOG("         GetConnectionStatus_No39->FinalizeLocalManager START\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        NN_LOG("         GetConnectionStatus_No39->FinalizeLocalManager END\n\n");
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetClientStatus_No40)
    {
        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        nn::Bit32 clientStatusBitMap = 0;
        Result = nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetState_No46)
    {
        Result = nn::wlan::Local::GetState(nullptr);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddIe_No48)
    {
        Result = nn::wlan::Local::AddIe(nullptr, nn::wlan::ManagementFrameType_Beacon, ieBody, ie_size);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddIe_No50)
    {
        // 不正ステートのためCloseしてReadyに
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::AddIe(&ieIndex, nn::wlan::ManagementFrameType_Beacon, ieBody, ie_size);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteIe_No51)
    {
        ieIndex = 0;
        Result = nn::wlan::Local::DeleteIe(ieIndex);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteIe_No52)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddIe(&ieIndex, nn::wlan::ManagementFrameType_Beacon, ieBody, ie_size));

        // 不正ステートのためCloseしてReadyに
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::DeleteIe(ieIndex);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetRssi_No54)
    {
        Result = nn::wlan::Local::GetRssi(NULL);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetLinkLevel_No57)
    {
        Result = nn::wlan::Local::GetLinkLevel(NULL);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    // No61~No64
    TEST_F(LocalClientExceptionArgmentUnitTest, PutFrame_No61_No64)
    {
        const size_t WlanTestBufferSize = 1024;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
        std::unique_ptr<uint8_t[]> emptyBuffer(new uint8_t[100]);
        nn::wlan::MacAddress putMacAddress;
        nn::os::SystemEventType connectEvent;
        LocalApiClass localClient;

        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&connectEvent));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ExceptionConnectParam.ssid, ExceptionConnectParam.security, TestMatchInfo, 1, true, false));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));

        // No61
        NN_LOG("             PutFrame_No61 START\n");
        Result = nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(putBuffer.get()), -1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
        NN_LOG("             PutFrame_No61 END\n");

        // No62 は連続でフレームを送受信しないと再現できず対向試験アプリでは再現できないためNT。FUNCTION試験で再現できている負荷試験で確認
#if 0
        NN_LOG("             PutFrame_No62 START\n");
        Result = nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(putBuffer.get()), WlanTestBufferSize);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultTxQueueIsFull::Includes(Result));
        NN_LOG("             PutFrame_No62 END\n");
#endif

        // No64
        NN_LOG("             PutFrame_No64 START\n");
        WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(WlanTestClinetConnectTime));
        Result = nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(putBuffer.get()), WlanTestBufferSize);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultCommandFailure::Includes(Result));
        NN_LOG("             PutFrame_No64 END\n");

        // No63
        NN_LOG("             PutFrame_No63 START\n");
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));
        Result = nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(putBuffer.get()), WlanTestBufferSize);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultNotAuthorized::Includes(Result));
        NN_LOG("             PutFrame_No63 END\n");

        WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(WlanTestClinetConnectTime));

        nn::os::DestroySystemEvent(&connectEvent);
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetFrameRaw_No65)
    {
        size_t outSize = 0;
        const size_t WlanTestBufferSize = 100;
        std::unique_ptr<uint8_t[]> pTestMemUnit8(new uint8_t[WlanTestBufferSize]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30));

        Result = nn::wlan::Local::GetFrameRaw(pTestMemUnit8.get(), WlanTestBufferSize, &outSize, rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    // No66, No68, No69, No71
    TEST_F(LocalClientExceptionArgmentUnitTest, GetFrameRaw_No66_No71)
    {
        const size_t WlanTestBufferSize = 1024;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        uint8_t emptyBuffer[10];
        nn::os::SystemEvent conEvent;
        LocalApiClass localClient;
        size_t outSize = 0;


        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(conEvent.GetBase()));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ExceptionConnectParam.ssid, ExceptionConnectParam.security, TestMatchInfo, 1, true, false));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));

        // No66
        NN_LOG("             GetFrameRaw_No66 START\n");
        Result = nn::wlan::Local::GetFrameRaw(reinterpret_cast<uint8_t*>(getBuffer.get()), WlanTestBufferSize, nullptr, localClient.m_entRxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
        NN_LOG("             GetFrameRaw_No66 END\n");

        // No71
        NN_LOG("             GetFrameRaw_No71 START\n");
        Result = nn::wlan::Local::GetFrameRaw(emptyBuffer, 1, &outSize, localClient.m_entRxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultBufferTooShort::Includes(Result));
        NN_LOG("             GetFrameRaw_No71 END\n");

        WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(WlanTestClinetConnectTime));
        localClient.LocalRelease();

        // No69
        NN_LOG("             GetFrameRaw_No69 START\n");
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ExceptionConnectParam.ssid, ExceptionConnectParam.security, TestNotMatchInfo, 1, true, false));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));

        localClient.LocalCancelFrameStart(500);
        Result = nn::wlan::Local::GetFrameRaw(reinterpret_cast<uint8_t*>(getBuffer.get()), WlanTestBufferSize, &outSize, localClient.m_entRxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultGetFrameCancelled::Includes(Result));
        localClient.LocalCancelFrameStop();
        WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(WlanTestClinetConnectTime));
        NN_LOG("             GetFrameRaw_No69 END\n");

        // No68
        NN_LOG("             GetFrameRaw_No68 START\n");
        Result = nn::wlan::Local::GetFrameRaw(reinterpret_cast<uint8_t*>(getBuffer.get()), WlanTestBufferSize, &outSize, localClient.m_entRxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultNotAuthorized::Includes(Result));
        NN_LOG("             GetFrameRaw_No68 END\n");

        localClient.LocalRelease();
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetCancelFrame_No70_No72)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        size_t outSize = 0;
        std::unique_ptr<uint8_t[]> pTestMemUnit8;

        // No70
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No70 START\n");
        Result = nn::wlan::Local::GetFrameRaw(pTestMemUnit8.get(), 100, &outSize, rxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryIsMuted::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No70 END\n\n");

        // No72
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No72 START\n");
        Result = nn::wlan::Local::CancelGetFrame(rxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No72 END\n\n");

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntry_No74)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntry_No75)
    {
        Result = nn::wlan::Local::CreateRxEntry(NULL, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntry_No76)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30));

        Result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultDuplicateRxEntry::Includes(Result));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntry_No77)
    {
        // nn::wlan::EthertypeCountMax配列数 + 1
        uint16_t etherTypes[] = {
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
            0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10
        };
        uint32_t etherTypeCount = sizeof(etherTypes[0]) / sizeof(etherTypes);

        NN_LOG("             CreateRxEntry_No77 ---> START\n");
        Result = nn::wlan::Local::CreateRxEntry(&rxId, etherTypes, etherTypeCount, 30);
        NN_LOG("             CreateRxEntry_No77 ---> END\n\n");
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryCountIsFull::Includes(Result));
        nn::wlan::Local::DeleteRxEntry(rxId);
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteRxEntry_No78)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::DeleteRxEntry(rxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddEthertypeToRxEntry_No80)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::AddEthertypeToRxEntry(rxId, ethertypes[0]);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddEthertypeToRxEntry_No82)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(ethertypes[0]), 30));

        Result = nn::wlan::Local::AddEthertypeToRxEntry(rxId, ethertypes[0]);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultDuplicateRxEntry::Includes(Result));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteEthertypeFromRxEntry_No83)
    {
        uint16_t testEthertype[1] = { 0x88b7 };

        Result = nn::wlan::Local::DeleteEthertypeFromRxEntry(nullptr, testEthertype[0]);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteEthertypeFromRxEntry_No85)
    {
        uint16_t testEthertype[2] = { 0x88b7, 0x88b9 };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(ethertypes[0]), 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(rxId, testEthertype[1]));

        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, testEthertype[1]);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddMatchDataToRxEntry_No86)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddMatchDataToRxEntry_No87_No89)
    {
        uint16_t ethertypes[] = { 0x88b7 };
        const nn::wlan::ReceivedDataMatchInfo ExceptionTestMatchInfo[] = {
            { { 0x00, 0x22,0xAA, 0xBB, 0xCC }, nn::wlan::ReceivedDataMatchLengthMax + 1 },
            { { 0x00 }, 0 },
        };

        // No87
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No87 START\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, 1, 30));
        Result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, ExceptionTestMatchInfo[0]);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No87 END\n");

        // No89
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No89 START\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));
        Result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, ExceptionTestMatchInfo[1]);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultDataMatchInfoError::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No89 END\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, RemoveMatchingDataFromRxEntry_No90)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, TestMatchInfo[0]);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, RemoveMatchingDataFromRxEntry_No91)
    {
        uint16_t ethertypes[] = { 0x88b7 };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, 1, 30));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, TestMatchInfo[0]));

        Result = nn::wlan::Local::RemoveMatchingDataFromRxEntry(-1, TestMatchInfo[0]);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(Result));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, SetActionFrameWithBeacon_No96)
    {
        Result = nn::wlan::Local::SetActionFrameWithBeacon(NULL, Setactionsizenormal);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, SetActionFrameWithBeacon_No97)
    {
        uint8_t  txBuffer[Setactionsizenormal] = { 0x7F, 0xD2, 0x8A, 0x01, nn::wlan::ActionFrameType_Beacon };
        Result = nn::wlan::Local::SetActionFrameWithBeacon(&txBuffer[0], Setactionsizenormal);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CancelActionFrameWithBeacon_No100)
    {
        Result = nn::wlan::Local::CancelActionFrameWithBeacon();
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalClientExceptionStateUnitTest, GetActionFrame_No101)
    {
        const size_t WlanTestBufferSize = 100 * 1024;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        uint32_t actionRxId;
        size_t outSize;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, &outSize, actionRxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetActionFrame_No102_No103_No107)
    {
        nn::wlan::DisconnectClient disClient;
        nn::wlan::ConnectionStatus connectStatus;
        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress putMacAddress;
        const size_t WlanTestBufferSize = 1024;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;

        // CLIENT(Target MASTER)
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&macAddress));
        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(connectParam.ssid, connectParam.security, TestMatchInfo, 1, true, true));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestConnectTimeOut));

        // No102
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No102 START\n");
        Result = nn::wlan::Local::GetActionFrame(nullptr, getBuffer.get(), WlanTestBufferSize, &outSize, localClient.m_actionRxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No102 END\n");

        // No103
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No103 START\n");
        Result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, nullptr, localClient.m_actionRxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No103 END\n");

        // No107
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No107 START\n");
        uint8_t emptyBuffer[10];
        Result = nn::wlan::Local::GetActionFrame(&outMacAddress, emptyBuffer, 1, &outSize, localClient.m_actionRxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultBufferTooShort::Includes(Result));
        NN_LOG("             LocalClientExceptionArgmentUnitTest ---> Test No107 END\n");

        WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(WlanTestConnectTimeOut));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetActionFrame_No105)
    {
        const size_t WlanTestBufferSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        uint32_t outRxId;
        size_t outSize;

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(nn::wlan::Ssid(LocalMasterTestSSID), security, TestNotMatchInfo, 1, false, true));
        outRxId = localClient.m_actionRxId;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outRxId, nn::wlan::ActionFrameType_Beacon));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestConnectTimeOut));

        localClient.LocalCancelActionFrameStart(WlanTestCancelWaitTime);
        Result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, &outSize, localClient.m_actionRxId);
        localClient.LocalCancelActionFrameStop();

        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultGetFrameCancelled::Includes(Result));

        WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(WlanTestConnectTimeOut));
        localClient.LocalRelease();
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, GetActionFrame_No106)
    {
        const size_t WlanTestBufferSize = ActionFrameBufferSize;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        uint32_t actionRxId;
        size_t outSize;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        //wlanプロセス内部でデータパスを一時的に無効化状態とするためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), WlanTestBufferSize, &outSize, actionRxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryIsMuted::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CancelGetActionFrame_No108_No109)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::CancelGetActionFrame(rxId);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

        Result = nn::wlan::Local::CancelGetActionFrame(rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntryForActionFrame_No110)
    {
        uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };

        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntryForActionFrame_No111)
    {
        uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal));
        Result = nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultDuplicateRxEntry::Includes(Result));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntryForActionFrame_No112)
    {
        // nn::wlan::ActionFrameRxCapacityMax配列数 + 1
        uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved3),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_End),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved3),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_End),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved3),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_End),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect)
        };
        uint32_t afTypeCount = sizeof(afTypes[0]) / sizeof(afTypes);

        Result = nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, afTypeCount, Createrxcapacitynormal);

        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryCountIsFull::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, CreateRxEntryForActionFrame_No113)
    {
        uint16_t afTypes = static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon);

        Result = nn::wlan::Local::CreateRxEntryForActionFrame(nullptr, &afTypes, Createrxcountnormal, Createrxcapacitynormal);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteRxEntryForActionFrame_No114)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddSubtypeToRxEntryForActionFrame_No116_No117)
    {
        uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, 1, Createrxcapacitynormal));

        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        NN_LOG("         AddSubtypeToRxEntryForActionFrame_No116->START\n");
        Result = nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(rxId, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        NN_LOG("         AddSubtypeToRxEntryForActionFrame_No116->END\n\n");

        NN_LOG("         AddSubtypeToRxEntryForActionFrame_No117->START\n");
        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));

        Result = nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(-1, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(Result));
        NN_LOG("         AddSubtypeToRxEntryForActionFrame_No117->END\n\n");
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, AddSubtypeToRxEntryForActionFrame_No119)
    {
        uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon)
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, 1, Createrxcapacitynormal));

        Result = nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(rxId, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultDuplicateRxEntry::Includes(Result));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientExceptionArgmentUnitTest, DeleteSubtypeFromRxEntryForActionFrame_No120_No122)
    {
        // 不正ステートのためClose
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

        Result = nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&rxId, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        // 再Open
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

        Result = nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(nullptr, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalClientExceptionStateUnitTest, DeleteSubtypeFromRxEntryForActionFrame_No75   )
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
        Result = nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&rxId, nn::wlan::ActionFrameType_Local);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(Result));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    TEST_F(LocalClientExceptionStateUnitTest, SetMaxAssociationNumber_No124)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

        Result = nn::wlan::Local::SetMaxAssociationNumber(1);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest

