﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "TcpClass.h"

namespace {
    enum ParamType
    {
        ParamType_Int16,
        ParamType_Uint16,
        ParamType_Int32,
        ParamType_Uint32,
        ParamType_Dword,
        ParamType_Char
    };

    struct Param
    {
        ParamType type;
        int16_t* pint16;
        uint16_t* puint16;
        int32_t* pint32;
        uint32_t* puint32;
        DWORD* pdw;
        char* pstr;
        size_t strSize;
    };

    struct TcpInputParam
    {
        std::string strSearch;
        std::unique_ptr<Param[]> pItem;
        std::string strDescription;
    };

    std::unique_ptr<Param[]> SetParam(int16_t* pValue) NN_NOEXCEPT;
    std::unique_ptr<Param[]> SetParam(uint16_t* pValue) NN_NOEXCEPT;
    std::unique_ptr<Param[]> SetParam(int32_t* pValue) NN_NOEXCEPT;
    std::unique_ptr<Param[]> SetParam(uint32_t* pValue) NN_NOEXCEPT;
    std::unique_ptr<Param[]> SetParam(DWORD* pValue) NN_NOEXCEPT;
    std::unique_ptr<Param[]> SetParam(char* str, size_t size) NN_NOEXCEPT;

    const size_t TraceLogBufferSize = 1512;
    const DWORD PacketWindMaxSize   = 256;

    char g_serverAddress[INET_ADDRSTRLEN];
    char g_ipAddress[INET_ADDRSTRLEN];
    char g_logFilePath[MAX_PATH + 1];
    USHORT g_serverPort;
    USHORT g_clientPort;
    DWORD g_sendSize;
    DWORD g_executTime;
    DWORD g_sendInterval;
    DWORD g_rateInfoInterval;
    DWORD g_packetWindSize;

    TcpInputParam TcpPrameterItem[] = {
        { "-sip=",      SetParam(g_serverAddress, sizeof(g_serverAddress)),
                        "Specify IP address of server(Ex：192.168.0.2)" },
        { "-ip=",       SetParam(g_ipAddress, sizeof(g_ipAddress)),
                        "Specify IP address of client to be connected(Ex：192.168.0.2)" },
        { "-sport=",    SetParam(&g_serverPort),
                        "Specify listen port of server" },
        { "-cport=",    SetParam(&g_clientPort),
                        "Specify port of client to be connected" },
        { "-csize=",    SetParam(&g_sendSize),
                        "Specify size to send by client in bytes" },
        { "-time=",     SetParam(&g_executTime),
                        "Specify time to execute in milliseconds" },
        { "-int=",      SetParam(&g_sendInterval),
                        "Specify interval in milliseconds to send on client" },
        { "-w=",        SetParam(&g_packetWindSize),
                        "Specify Window size of TCP send / receive in KByte units" },
        { "-srint=",    SetParam(&g_rateInfoInterval),
                        "Specify interval in milliseconds to display rate information of transmission / reception" },
        { "-log=",      SetParam(g_logFilePath, sizeof(g_logFilePath)),
                        "Specify Output Destination of Log File Path" },
        { "",           nullptr,
                        "" },
    };

    bool GetParam(int argc, char** argv) NN_NOEXCEPT NN_IMPLICIT
    {
        int itemCount   = sizeof(TcpPrameterItem) / sizeof(TcpPrameterItem[0]);
        char *search    = nullptr;
        int matchParam  = 0;
        size_t size     = 0;
        int length      = 0;
        int offset      = 0;

        for (int i = 1; i < argc; i++)
        {
            if (strstr(argv[i], "/help") != nullptr ||
                strstr(argv[i], "--help") != nullptr ||
                strstr(argv[i], "/?") != nullptr)
            {
                return false;
            }

            for (int item = 0; item < itemCount; item++)
            {
                if (TcpPrameterItem[item].pItem == nullptr)
                {
                    continue;
                }

                search = strstr(argv[i], TcpPrameterItem[item].strSearch.c_str());
                if (search == nullptr)
                {
                    continue;
                }

                matchParam++;

                switch (TcpPrameterItem[item].pItem.get()->type)
                {
                case ParamType::ParamType_Char:
                    size = TcpPrameterItem[item].pItem.get()->strSize;
                    std::memset(TcpPrameterItem[item].pItem.get()->pstr, 0x00, size);
                    length = nn::util::Strnlen(argv[i], static_cast<int>(size)) + 1;
                    offset = nn::util::Strnlen(TcpPrameterItem[item].strSearch.c_str(),
                                static_cast<int>(TcpPrameterItem[item].strSearch.length()));
                    if (static_cast<int>(size) > length - offset)
                    {
                        nn::util::Strlcpy(TcpPrameterItem[item].pItem.get()->pstr,
                                &search[offset], static_cast<int>(length - offset));
                    }
                    break;
                case ParamType::ParamType_Int16:
                    length = static_cast<int>(TcpPrameterItem[item].strSearch.length());
                    offset = nn::util::Strnlen(TcpPrameterItem[item].strSearch.c_str(), length);
                    *TcpPrameterItem[item].pItem.get()->pint16 =
                                static_cast<int16_t>(strtol(&search[offset], 0, 10));
                    break;
                case ParamType::ParamType_Uint16:
                    length = static_cast<int>(TcpPrameterItem[item].strSearch.length());
                    offset = nn::util::Strnlen(TcpPrameterItem[item].strSearch.c_str(), length);
                    *TcpPrameterItem[item].pItem.get()->puint16 =
                                static_cast<uint16_t>(strtoul(&search[offset], 0, 10));
                    break;
                case ParamType::ParamType_Int32:
                    length = static_cast<int>(TcpPrameterItem[item].strSearch.length());
                    offset = nn::util::Strnlen(TcpPrameterItem[item].strSearch.c_str(), length);
                    *TcpPrameterItem[item].pItem.get()->pint32 = strtol(&search[offset], 0, 10);
                    break;
                case ParamType::ParamType_Uint32:
                    length = static_cast<int>(TcpPrameterItem[item].strSearch.length());
                    offset = nn::util::Strnlen(TcpPrameterItem[item].strSearch.c_str(), length);
                    *TcpPrameterItem[item].pItem.get()->puint32 =
                                        static_cast<uint32_t>(strtoul(&search[offset], 0, 10));
                    break;
                case ParamType::ParamType_Dword:
                    length = static_cast<int>(TcpPrameterItem[item].strSearch.length());
                    offset = nn::util::Strnlen(TcpPrameterItem[item].strSearch.c_str(), length);
                    *TcpPrameterItem[item].pItem.get()->pdw =
                                        static_cast<DWORD>(strtoul(&search[offset], 0, 10));
                    break;
                default:
                    NN_LOG("There is a mistake in argument specification.\n");
                    NN_LOG("Please define type type for each case.\n\n");
                    return false;
                }
            }
        }
        if (argc > 1 && matchParam == 0)
        {
            NN_LOG("\nThere is a mistake in argument specification.\n\n");
            return false;
        }
        return true;
    }

    std::unique_ptr<Param[]> SetParam(int16_t* pValue) NN_NOEXCEPT
    {
        std::unique_ptr<Param[]> param(new Param());

        param.get()->type = ParamType::ParamType_Int16;
        param.get()->pint16 = pValue;
        return std::move(param);
    }

    std::unique_ptr<Param[]> SetParam(uint16_t* pValue) NN_NOEXCEPT
    {
        std::unique_ptr<Param[]> param(new Param());

        param.get()->type = ParamType::ParamType_Uint16;
        param.get()->puint16 = pValue;
        return std::move(param);
    }

    std::unique_ptr<Param[]> SetParam(int32_t* pValue) NN_NOEXCEPT
    {
        std::unique_ptr<Param[]> param(new Param());

        param.get()->type = ParamType::ParamType_Int32;
        param.get()->pint32 = pValue;
        return std::move(param);
    }

    std::unique_ptr<Param[]> SetParam(uint32_t* pValue) NN_NOEXCEPT
    {
        std::unique_ptr<Param[]> param(new Param());

        param.get()->type = ParamType::ParamType_Uint32;
        param.get()->puint32 = pValue;
        return std::move(param);
    }

    std::unique_ptr<Param[]> SetParam(DWORD* pValue) NN_NOEXCEPT
    {
        std::unique_ptr<Param[]> param(new Param());

        param.get()->type = ParamType::ParamType_Dword;
        param.get()->pdw = pValue;
        return std::move(param);
    }

    std::unique_ptr<Param[]> SetParam(char* str, size_t size) NN_NOEXCEPT
    {
        std::unique_ptr<Param[]> param(new Param());

        param.get()->type = ParamType::ParamType_Char;
        param.get()->pstr = str;
        param.get()->strSize = size;
        return std::move(param);
    }
}

class CTcpTool
{
    NN_DISALLOW_COPY( CTcpTool );
    NN_DISALLOW_MOVE( CTcpTool );

public:
    CTcpTool() NN_NOEXCEPT
    {
        m_isStop        = false;
    }

    void StartTcp() NN_NOEXCEPT
    {
        std::string strSizeUnit = CTcpClass::GetUnitByteCalc(g_sendSize);
        m_server.LogOutIpAddress(g_serverPort);
        TcpTraceLog("**** Client Connect Info ****");
        TcpTraceLog("    IP Address    : %s(%hd)", g_ipAddress, g_clientPort);
        TcpTraceLog("    Send Size     : %.1lf %s", CTcpClass::GetByteCalc(g_sendSize), strSizeUnit.c_str());
        TcpTraceLog("    Send Interval : %lu msec", g_sendInterval);
        TcpTraceLog("**** Client Connect Info ****\n");

        Sleep(1000);

        m_tcpThreads[0] = CreateThread(nullptr, 0,
                reinterpret_cast<LPTHREAD_START_ROUTINE>(TcpServerThread), this, 0, nullptr);
        m_tcpThreads[1] = CreateThread(nullptr, 0,
                reinterpret_cast<LPTHREAD_START_ROUTINE>(TcpClientThread), this, 0, nullptr);

        WaitForMultipleObjects(2, m_tcpThreads, TRUE, g_executTime);
        m_isStop = true;

        m_server.TearDownTcpServer();
        m_client.TearDownTcpClient();

        CloseHandle(m_tcpThreads[0]);
        CloseHandle(m_tcpThreads[1]);
    }

    static void TcpServerThread(LPVOID arg) NN_NOEXCEPT
    {
        CTcpTool* pThis = static_cast<CTcpTool*>(arg);
        DWORD dwResult;

        g_packetWindSize = std::min(PacketWindMaxSize, g_packetWindSize);
        if (g_packetWindSize > 0)
        {
            pThis->m_server.SetWindowsSize(g_packetWindSize * 1024);
            TcpTraceLog("TCP Server Packet Window Size : %lu Byte", g_packetWindSize * 1024);
        }
        pThis->m_server.SetLogoutTime(g_rateInfoInterval);
        TcpTraceLog("TCP Server Recv Log Interval Time : %lu msec", g_rateInfoInterval);

        TcpTraceLog("TCP Server Execution Time : %lu msec", g_executTime);
        dwResult = pThis->m_server.SetupTcpServer(g_serverAddress, g_serverPort,
                                                        g_sendSize, g_sendInterval);
        if (dwResult != 0)
        {
            TcpTraceLog("TCP Server Setup Faild : 0x%08x", dwResult);
            return;
        }
    }

    static void TcpClientThread(LPVOID arg) NN_NOEXCEPT
    {
        CTcpTool* pThis = static_cast<CTcpTool*>(arg);
        DWORD dwResult;

        g_packetWindSize = std::min(PacketWindMaxSize, g_packetWindSize);
        if (g_packetWindSize > 0)
        {
            pThis->m_client.SetWindowsSize(g_packetWindSize * 1024);
            TcpTraceLog("TCP Client Packet Window Size : %lu Byte", g_packetWindSize * 1024);
        }
        pThis->m_client.SetLogoutTime(g_rateInfoInterval);
        TcpTraceLog("TCP Client Send Log Interval Time : %lu msec", g_rateInfoInterval);

        TcpTraceLog("TCP Client Execution Time : %lu msec", g_executTime);

        Sleep(1000);

        while (pThis->m_isStop != true)
        {
            dwResult = pThis->m_client.SetupTcpClient(g_ipAddress, g_clientPort,
                                g_sendSize, g_sendInterval);
            if (dwResult != 0)
            {
                TcpTraceLog("TCP Client Setup Faild : 0x%08x\n", dwResult);
                return;
            }
            Sleep(1000);
        }
    }

protected:
    bool m_isStop;

    HANDLE m_tcpThreads[2];
    CTcpClass m_server;
    CTcpClass m_client;
    std::string m_strlogFile;
};

extern "C" void nnMain()
{
    CTcpTool tcpTool;
    int     argc    = nn::os::GetHostArgc();
    char**  argv    = nn::os::GetHostArgv();

    g_serverPort        = 5000;
    g_executTime        = 30000;
    g_clientPort        = 5001;
    g_sendSize          = 1024;
    g_sendInterval      = 10;
    g_rateInfoInterval  = 10000;
    g_packetWindSize    = 0;

    nn::util::Strlcpy(g_ipAddress, "192.168.1.2\0", sizeof(g_ipAddress));
    nn::util::Strlcpy(g_serverAddress, "000.000.000.000\0", sizeof(g_serverAddress));
    std::memset(g_logFilePath, 0x00, sizeof(g_logFilePath));

    TcpLogInitialize(TraceLogBufferSize);
    if (GetParam(argc, argv) != true)
    {
        NN_LOG("\n■Parameters that can be specified as arguments are as follows.\n\n");
        int itemCount = sizeof(TcpPrameterItem) / sizeof(TcpPrameterItem[0]);
        for (int item = 0; item < itemCount; item++)
        {
            if (TcpPrameterItem[item].strSearch != std::string(""))
            {
                NN_LOG("  %-10.8s : %s\n", TcpPrameterItem[item].strSearch.c_str(),
                                            TcpPrameterItem[item].strDescription.c_str());
            }
        }
        NN_LOG("\n\n");
        TcpLogFinalize();
        return;
    }

    TcpTraceLogFileOpen(g_logFilePath);
    tcpTool.StartTcp();
    TcpTraceLogFileClose();
    TcpLogFinalize();
}

