﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Include/testWlan_LocalInfraStressTest.h"

#include <curl/curl.h>
#include <nn/ldn.h>

#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>

#include <nn/settings/fwdbg/settings_SettingsCommon.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/socket.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>

#include "../include/Definition.h"
#include "../include/Util.h"

namespace {

    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    int SendThreadPriority = nn::os::DefaultThreadPriority + 4;
    int ReceiveThreadPriority = nn::os::DefaultThreadPriority + 3;
    int LdnThreadPriority = nn::os::DefaultThreadPriority + 2;
    int MonitorThreadPriority = nn::os::DefaultThreadPriority + 1;

    const int ModeSettingServerWaitTime = 30;
    const int RandamModeCount = 80;

    const int ModeScanWaitTime = 3;

    const int ModeChangeMinTime = 15;
    const int ModeChangeMaxTime = 30;
    const int ClientConRetry = 100;
    const int ConWaitTime = 3;
    const int ModeWaitTime = 180;
    const int ModeLocalConnectRetryCount = 10;

    // スレッドスタックサイズ定義
    const size_t StackSize = nn::os::ThreadStackAlignment * 4;

    const int16_t Channels_2_4Ghz[] = {
        1,  6,  11
    };

    const int16_t Channels_5Ghz[] = {
        36,  40,  44,   48
    };

    enum ApplicationState
    {
        ApplicationState_Initialized,
        ApplicationState_NetworkCreating,
        ApplicationState_NetworkCreated,
        ApplicationState_NetworkDestroying,
        ApplicationState_NetworkDestroyed,
        ApplicationState_Connecting,
        ApplicationState_Connected,
        ApplicationState_Disconnecting,
        ApplicationState_Disconnected,
        ApplicationState_ChangeMode
    };

    enum ChangeMode
    {
        Mode_Init = 0,
        Mode_InfraEtherServer,
        Mode_InfraEtherClient,
        Mode_InfraWiFiServer,
        Mode_InfraWiFiClient,
        Mode_LdnLocalMaster,
        Mode_LdnLocalClient,

        // 以降はテスト起動モードで使用
        Mode_TestMaster,
        Mode_TestClient
    };

    enum TestDataType
    {
        Type_Non = 0,
        Type_TestPattern,
    };

    enum TestNetPattern
    {
        NetPttern_2_4Ghz,
        NetPttern_5Ghz,
    };

    const char* ChangeModeString[] =
    {
        "Mode Init",
        "Mode Infra Ether Server Test",
        "Mode Infra Ether Client Test",
        "Mode Infra WiFi Server Test",
        "Mode Infra WiFi Client Test",
        "Mode Local Master Test",
        "Mode Local Client Test",

        // 以降はテスト起動モードで使用
        "Mode Test Master Execute",
        "Mode Test Client Execute"
    };

    // ローカル通信／インフラ通信周波数帯試験パターン
    // | 2.4GHz → 5GHz | 5GHz → 2.4GHz | 2.4GHz → 2.4 GHz | 5GHz → 5GHz |
    const TestNetPattern NetProfilePattern[] =
    {
        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_2_4Ghz,

        TestNetPattern::NetPttern_2_4Ghz,
        TestNetPattern::NetPttern_5Ghz,

        TestNetPattern::NetPttern_5Ghz,
        TestNetPattern::NetPttern_5Ghz,
    };

    struct TestDataPayload
    {
        uint32_t dataType;
        uint8_t data[1];
    };

    struct ChangeModePayload
    {
        uint32_t executeTime;
        uint32_t length;
        ChangeMode mode[RandamModeCount];
        uint32_t changeTimes[RandamModeCount];
    };

    struct ChangeModeInfo
    {
        uint32_t length;
        ChangeMode mode[RandamModeCount];
        uint32_t changeTimes[RandamModeCount];
    };

    struct ApplicationResource
    {
        nn::os::ThreadType          monitorThread;
        nn::os::ThreadType          sendThread;
        nn::os::ThreadType          receiveThread;
        nn::os::SystemEventType     stateChangeEvent;
        nn::os::EventType           cancelEvent;
        nn::os::EventType           initEvent;
        nn::os::MutexType           mutex;
        nn::os::MutexType           apiMutex;
        nn::ldn::NetworkInfo        networkInfo;
        nn::ldn::Ipv4Address        ipv4Address;
        nn::ldn::SubnetMask         subnetMask;
        uint16_t                    port;
        uint32_t                    myCounter;
        uint32_t                    counter[nn::ldn::NodeCountMax];
        int                         socket;
        int                         clientSocket;
        ApplicationState            state;
        uint32_t                    executeTime;
        ChangeMode                  mode;
        ChangeMode                  currentMode;
        ChangeModeInfo              TestPattern;
        bool                        isAgingEnable;
        nn::nifm::NetworkProfileData* pNetProfile;
        int16_t                     setChannel;
        void*                       pThis;
    };

    nn::ldn::NetworkInfo g_ScanResult[nn::ldn::ScanResultCountMax];
    nn::nifm::NetworkProfileData g_netProfile;

}

namespace WlanTest {

    class LocalInfraStressTest : public ::testing::Test
    {
    protected:

        const bool IsEnabled = true;

        ApplicationResource app;
        char localIp[INET_ADDRSTRLEN];
        int64_t agingTime;
    protected:
        LocalInfraStressTest() NN_NOEXCEPT
        {
            std::memset(&app, 0, sizeof(app));
            app.pThis = this;
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            agingTime = GetAgintTime();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm",
                "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(true);

            // スレッド毎にOS APIのInitialize／Finalizeを呼んでいた箇所をSetup/TearDownへ(スレッドタイミング調整不要に対応)
            nn::os::InitializeEvent(&app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
            nn::os::InitializeEvent(&app.initEvent, false, nn::os::EventClearMode_ManualClear);
            nn::os::InitializeMutex(&app.mutex, false, 0);
            nn::os::InitializeMutex(&app.apiMutex, false, 0);

            // Socket初期値設定
            app.socket = -1;
            app.clientSocket = -1;

            // ローカル通信のデフォルトチャンネル設定(自動取得)
            app.setChannel = nn::ldn::AutoChannel;
            // ローカル通信／インフラ通信切り替え試験のデフォルトネットワークプロファイル設定(2.4GHz設定)
            app.pNetProfile = nullptr;

        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            if (app.mode != ChangeMode::Mode_Init)
            {
                app.isAgingEnable = false;
                app.state = ApplicationState_NetworkDestroying;
                nn::os::SignalEvent(&app.cancelEvent);
                app.mode = ChangeMode::Mode_Init;
                app.state = ApplicationState_Initialized;
            }

            nn::os::FinalizeMutex(&app.mutex);
            nn::os::FinalizeMutex(&app.apiMutex);
            nn::os::FinalizeEvent(&app.cancelEvent);
            nn::os::FinalizeEvent(&app.initEvent);
        }

        void SettingProfileData(nn::nifm::NetworkProfileData& profile, nn::wlan::Ssid ssid,
                    const char* pIpAddress, const char* pSubnet, const char* pGateWay,
                    nn::nifm::Security* pSequrity) NN_NOEXCEPT
        {
            const char strSubnet[INET_ADDRSTRLEN] = "255.255.255.0";
            profile.id = nn::util::InvalidUuid;
            profile.networkProfileType = nn::nifm::NetworkProfileType_User;
            std::strncpy(profile.name, "", sizeof(profile.name));
            profile.networkInterfaceType = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Invalid;
            profile.isAutoConnect = true;
            profile.isLargeCapacity = true;

            profile.wirelessSetting.security.authEncryption.authentication  = pSequrity->authEncryption.authentication;
            profile.wirelessSetting.security.authEncryption.encryption      = pSequrity->authEncryption.encryption;

            nn::util::Strlcpy(profile.wirelessSetting.security.sharedKey.keyMaterial,
                pSequrity->sharedKey.keyMaterial, pSequrity->sharedKey.KeyMaterialSize);

            profile.wirelessSetting.ssidConfig.ssid.length = ssid.GetLength();
            ssid.GetHexString(reinterpret_cast<char*>(profile.wirelessSetting.ssidConfig.ssid.hex));

            if (pSequrity != nullptr)
            {
                memcpy(&profile.wirelessSetting.security, pSequrity, sizeof(profile.wirelessSetting.security));
            }

            nn::nifm::IpV4Address ipArray;
            nn::nifm::IpV4Address subnatArray;
            nn::nifm::IpV4Address gatewayArray;
            size_t strIpLength = nn::util::Strnlen(pIpAddress, INET_ADDRSTRLEN) + 1;

            WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(ipArray, pIpAddress, strIpLength));

            // サブネットマスク設定
            if (pSubnet != nullptr)
            {
                strIpLength = nn::util::Strnlen(pSubnet, INET_ADDRSTRLEN) + 1;
                WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(subnatArray, pSubnet, strIpLength));
            }
            else
            {
                WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(subnatArray, strSubnet, sizeof(strSubnet)));
            }

            // デフォルトゲートウェイ設定
            if (pGateWay != nullptr)
            {
                strIpLength = nn::util::Strnlen(pGateWay, INET_ADDRSTRLEN) + 1;
                WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(gatewayArray, pGateWay, strIpLength));
            }
            else
            {
                strIpLength = nn::util::Strnlen(pIpAddress, INET_ADDRSTRLEN) + 1;
                WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(gatewayArray, pIpAddress, strIpLength));
            }

            // 固定IP設定
            profile.ipSetting.ip.isAuto = false;
            // IP設定
            memcpy(&profile.ipSetting.ip.ipAddress.data, &ipArray, sizeof(profile.ipSetting.ip.ipAddress.data));
            // SubNetMask設定
            memcpy(&profile.ipSetting.ip.subnetMask.data, &subnatArray, sizeof(profile.ipSetting.ip.ipAddress.data));
            // SubNetMask設定
            memcpy(&profile.ipSetting.ip.defaultGateway.data, &gatewayArray, sizeof(profile.ipSetting.ip.defaultGateway.data));

            // DNS
            profile.ipSetting.dns.isAuto = false;
            memcpy(&profile.ipSetting.dns.preferredDns.data, &gatewayArray, sizeof(profile.ipSetting.dns.preferredDns.data));
            memcpy(&profile.ipSetting.dns.alternateDns.data, &gatewayArray, sizeof(profile.ipSetting.dns.preferredDns.data));

            profile.ipSetting.mtu = 1400;
        }

        void WiFiSettingProfileData(nn::nifm::NetworkProfileData& profile, nn::wlan::Ssid ssid,
                    const char* pIpAddress, nn::nifm::Security* pSequrity) NN_NOEXCEPT
        {
            std::string strGateWay = MakeIpAddress(pIpAddress, 1, pIpAddress);
            SettingProfileData(profile, ssid, pIpAddress, GetSubnetMaskString(), strGateWay.c_str(), pSequrity);
            profile.networkInterfaceType = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211;
        }

        void WiFiSettingProfileData(nn::nifm::NetworkProfileData& profile, nn::wlan::Ssid ssid,
                    const char* pIpAddress, const char* pSubnet, const char* pGateWay,
                    nn::nifm::Security* pSequrity) NN_NOEXCEPT
        {
            SettingProfileData(profile, ssid, pIpAddress, pSubnet, pGateWay, pSequrity);
            profile.networkInterfaceType = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211;
        }

        void EtherSettingProfileData(nn::nifm::NetworkProfileData& profile, nn::wlan::Ssid ssid,
                    const char* pIpAddress, nn::nifm::Security* pSequrity) NN_NOEXCEPT
        {
            std::string strGateWay = MakeIpAddress(pIpAddress, 1, pIpAddress);
            SettingProfileData(profile, ssid, pIpAddress, GetSubnetMaskString(), strGateWay.c_str(), pSequrity);
            profile.networkInterfaceType = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet;
        }

        void EtherSettingProfileData(nn::nifm::NetworkProfileData& profile, nn::wlan::Ssid ssid,
                    const char* pIpAddress, const char* pSubnet, const char* pGateWay,
                    nn::nifm::Security* pSequrity) NN_NOEXCEPT
        {
            SettingProfileData(profile, ssid, pIpAddress, pSubnet, pGateWay, pSequrity);
            profile.networkInterfaceType = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet;
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            return rand_ret;
        }

        std::string MakeIpAddress(const char* ipAddress, const int oct4, const char* defaultAddress) NN_NOEXCEPT
        {
            std::string srcIp = ipAddress;
            char* pPos = const_cast<char*>(srcIp.c_str());
            std::string resultString = "";

            for (int octPos = 0; octPos < 3; octPos++)
            {
                // IPの第3オクテッドまで分割
                char* pSearch = strstr(pPos, ".");
                if (pSearch != nullptr)
                {
                    *pSearch = '\0';
                    resultString.append(pPos).append(".");
                }

                // 0～255の範囲か判定
                if (pSearch != nullptr && strtoul(pPos, 0, 10) <= 255)
                {
                    pPos = ++pSearch;
                }
                else
                {
                    // 0～255の範囲外ならdefaultAddress引数を返す
                    return std::string(defaultAddress);
                }
            }

            // 第4オクテッドIPを追加してIPアドレス文字列として返す
            return resultString.append(std::to_string(oct4));
        }

        bool ConvertIpStringToArray(nn::nifm::IpV4Address& ipAddress, const char* strIpAddress, const size_t size) NN_NOEXCEPT
        {
            char strIp[INET_ADDRSTRLEN];

            if (strIpAddress == nullptr)
            {
                return false;
            }

            if (MakeIpAddress(strIpAddress, 2, "") == "")
            {
                return false;
            }

            sockaddr_in addr;
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(UdpPort);
            addr.sin_addr.s_addr = INADDR_ANY;

            nn::util::Strlcpy(strIp, strIpAddress, INET_ADDRSTRLEN);
            nn::socket::InetPton(AF_INET, strIp, &addr.sin_addr);

            memcpy(&ipAddress.data, &addr.sin_addr, sizeof(ipAddress.data));
            return true;
        }

        nn::Bit32 ConvertIpArrayToBit32(const nn::nifm::IpV4Address& ipAddress) NN_NOEXCEPT
        {
            nn::Bit32* pIpAddress;
            nn::Bit8 ipArray[nn::nifm::IpV4Address::Size];
            int offSet = nn::nifm::IpV4Address::Size;

            for (int i = 0; i < nn::nifm::IpV4Address::Size; i++)
            {
                offSet--;
                ipArray[i] = ipAddress.data[offSet];
            }

            pIpAddress = reinterpret_cast<nn::Bit32*>(&ipArray[0]);

            return *pIpAddress;
        }

        void TraceCurrentnetwork() NN_NOEXCEPT
        {
            nn::Result result;
            in_addr ipAddr, subAddr, gateAddr, dns1Addr, dns2Addr;
            char strAddress[INET_ADDRSTRLEN];

            result = nn::nifm::GetCurrentIpConfigInfo(&ipAddr, &subAddr, &gateAddr, &dns1Addr, &dns2Addr);
            if (result.IsSuccess())
            {
                nn::socket::InetNtop(AF_INET, &ipAddr.s_addr, strAddress, sizeof(strAddress));

                NN_LOG("現在のWlan接続ネットワーク設定情報です。\n");
                NN_LOG("IP Address     : %s\n",
                    nn::socket::InetNtop(AF_INET, &ipAddr.s_addr, strAddress, sizeof(strAddress)));
                NN_LOG("Subnet Mask    : %s\n",
                    nn::socket::InetNtop(AF_INET, &subAddr.s_addr, strAddress, sizeof(strAddress)));
                NN_LOG("Default GateWay: %s\n",
                    nn::socket::InetNtop(AF_INET, &gateAddr.s_addr, strAddress, sizeof(strAddress)));
                NN_LOG("Dns1 Address   : %s\n",
                    nn::socket::InetNtop(AF_INET, &dns1Addr.s_addr, strAddress, sizeof(strAddress)));
                NN_LOG("Dns2 Address   : %s\n\n",
                    nn::socket::InetNtop(AF_INET, &dns2Addr.s_addr, strAddress, sizeof(strAddress)));
            }
            else
            {
                NN_LOG("nn::nifm::GetCurrentIpConfigInfo Error Module(%d) Description(%d)\n\n", result.GetModule(), result.GetDescription());
            }
        }

        void CreateSocket() NN_NOEXCEPT
        {
            // データの送受信に使用する socket を生成します。
            app.socket = nn::socket::Socket(AF_INET, SOCK_DGRAM, 0);
            NN_ABORT_UNLESS_NOT_EQUAL(app.socket, -1);

            sockaddr_in addr;
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(UdpPort);
            addr.sin_addr.s_addr = INADDR_ANY;
            NN_LOG("%s ---> nn::socket::Bind Start\n", __FUNCTION__);
            NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
                app.socket, reinterpret_cast<sockaddr*>(&addr), sizeof(addr)), 0);
            NN_LOG("%s ---> nn::socket::Bind End\n\n", __FUNCTION__);

            // ブロードキャスト通信を有効化します。
            int isEnabled = 1;
            NN_LOG("%s ---> nn::socket::SetSockOpt SO_BROADCAST Start\n", __FUNCTION__);
            int nResult = nn::socket::SetSockOpt(
                app.socket, SOL_SOCKET, SO_BROADCAST, &isEnabled, sizeof(isEnabled));
            if (nResult != 0)
            {
                NN_LOG("nn::socket::SetSockOpt SO_BROADCAST Error : %d\n\n", nResult);
            }
            NN_LOG("%s ---> nn::socket::SetSockOpt SO_BROADCAST End\n\n", __FUNCTION__);

            // データ送信用のスレッドを生成します。
            static NN_ALIGNAS(4096) uint8_t SendThreadStack[StackSize];
            static NN_ALIGNAS(4096) uint8_t ReceiveThreadStack[StackSize];

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &app.sendThread, SendThread, &app,
                SendThreadStack, sizeof(SendThreadStack),
                SendThreadPriority));
            nn::os::StartThread(&app.sendThread);

            // データ受信用のスレッドを生成します。
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &app.receiveThread, ReceiveThread, &app,
                ReceiveThreadStack, sizeof(ReceiveThreadStack),
                ReceiveThreadPriority));
            nn::os::StartThread(&app.receiveThread);
        }

        void CreateTcpServerSocket() NN_NOEXCEPT
        {
            // データの送受信に使用する socket を生成します。
            app.socket = nn::socket::Socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
            NN_ABORT_UNLESS_NOT_EQUAL(app.socket, -1);

            int isEnabled = 1;
            int nResult = nn::socket::SetSockOpt(
                app.socket, SOL_SOCKET, SO_REUSEADDR, &isEnabled, sizeof(isEnabled));
            if (nResult != 0)
            {
                NN_LOG("nn::socket::SetSockOpt SO_REUSEADDR Error : %d\n\n", nResult);
            }

            nn::os::ClearEvent(&app.initEvent);

            static NN_ALIGNAS(4096) uint8_t SendThreadStack[StackSize];
            static NN_ALIGNAS(4096) uint8_t ReceiveThreadStack[StackSize];

            // データ送信用のスレッドを生成します。
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &app.sendThread, TcpSendThread, &app,
                SendThreadStack, sizeof(SendThreadStack),
                SendThreadPriority));
            nn::os::StartThread(&app.sendThread);

            // データ受信用のスレッドを生成します。
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &app.receiveThread, TcpServerReceiveThread, &app,
                ReceiveThreadStack, sizeof(ReceiveThreadStack),
                ReceiveThreadPriority));
            nn::os::StartThread(&app.receiveThread);
        }

        void CreateTcpClientSocket() NN_NOEXCEPT
        {
            sockaddr_in addr;
            char strIpAddress[INET_ADDRSTRLEN];
            int nResult;

            nn::os::ClearEvent(&app.initEvent);

            std::memset(&addr, 0x00, sizeof(struct sockaddr_in));
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(TcpServerPort);
            addr.sin_addr.s_addr = app.ipv4Address.raw;

            sprintf(strIpAddress, "%s", ConvertToString(app.ipv4Address));

            NN_LOG("TCP Client 接続先 : %s:%d\n\n", strIpAddress, TcpServerPort);
            nn::socket::InetPton(AF_INET, strIpAddress, &addr.sin_addr);

            for (int i = 0; i < ClientConRetry; i++)
            {
                // データの送受信に使用する socket を生成します。
                app.socket = nn::socket::Socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
                NN_ABORT_UNLESS_NOT_EQUAL(app.socket, -1);

                NN_LOG("%s ---> Connect Start %d/%d\n", __FUNCTION__, i + 1, ClientConRetry);
                nResult = nn::socket::Connect(app.socket, (struct sockaddr *)&addr, sizeof(struct sockaddr));
                NN_LOG("%s ---> Connect End %d/%d Result(%d)\n\n", __FUNCTION__, i + 1, ClientConRetry, nResult);
                if (nResult == 0)
                {
                    app.clientSocket = app.socket;
                    break;
                }
                else
                {
                    NN_LOG("nn::socket::Connect Error Retry Count : %d\n\n", i + 1);
                    TestSocketClose(app.socket, &app.mutex, false);
                    app.clientSocket = -1;
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(ConWaitTime));
                }
            }

            if (nResult == 0)
            {
                static NN_ALIGNAS(4096) uint8_t SendThreadStack[StackSize];
                static NN_ALIGNAS(4096) uint8_t ReceiveThreadStack[StackSize];

                // データ送信用のスレッドを生成します。
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                    &app.sendThread, TcpSendThread, &app,
                    SendThreadStack, sizeof(SendThreadStack),
                    SendThreadPriority));
                nn::os::StartThread(&app.sendThread);

                // データ受信用のスレッドを生成します。
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                    &app.receiveThread, TcpClientReceiveThread, &app,
                    ReceiveThreadStack, sizeof(ReceiveThreadStack),
                    ReceiveThreadPriority));
                nn::os::StartThread(&app.receiveThread);
            }
            else
            {
                nn::os::SignalEvent(&app.cancelEvent);
                NN_LOG("nn::socket::Connect Error : %d\n\n", nResult);
            }
        }

        static void TcpSendThread(void* arg) NN_NOEXCEPT
        {
            auto& app = *static_cast<ApplicationResource*>(arg);

            // ブロードキャスト送信用のアドレスを生成します。

            if (app.currentMode == ChangeMode::Mode_TestMaster)
            {
                nn::os::WaitEvent(&app.initEvent);
                NN_LOG("Tcp Server Accept Event Sygnal\n\n");
            }
            else
            {
                NN_LOG("Tcp Mode Check %s\n\n", ChangeModeString[app.currentMode]);
            }

            // キャンセルされるまで一定時間毎にパケットを送信します。
            const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
            while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
            {
                // 自身のカウンタをインクリメントします。
                nn::os::LockMutex(&app.mutex);
                app.counter[0] += 1;

                // 送信データを生成します。
                AccesspointData data;
                for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                {
                    data.counter[i] = nn::socket::InetHtonl(app.counter[i]);
                }
                nn::os::UnlockMutex(&app.mutex);

                NN_LOG("%s ---> Send Start\n", __FUNCTION__);
                auto sendSize = nn::socket::Send(
                    app.clientSocket,
                    &data,
                    sizeof(data),
                    0);
                NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, sendSize);

                if (sendSize < 0)
                {
                    NN_LOG("Tcp Send Error(%d)\n\n", nn::socket::GetLastErrno());
                    nn::os::SignalEvent(&app.cancelEvent);
                    break;
                }
                else
                {
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        if (app.counter[i] != 0)
                        {
                            NN_LOG("Tcp Send Data %d : %lu\n\n", i, app.counter[i]);
                        }
                    }
                }
            }
        }

        static void TcpServerReceiveThread(void* arg) NN_NOEXCEPT
        {
            auto& app = *static_cast<ApplicationResource*>(arg);
            // 受信用のバッファです。
            AccesspointData data;

            // パケットを受信するためのアドレスを生成します。
            sockaddr_in addr;

            // クライアント接続時に取得する情報です。
            sockaddr_in clientAddr;
            socklen_t nsaddrLen;
            char strIpAddress[INET_ADDRSTRLEN];

            std::memset(&addr, 0x00, sizeof(struct sockaddr_in));
            std::memset(&clientAddr, 0x00, sizeof(struct sockaddr_in));
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(TcpServerPort);
            addr.sin_addr.s_addr = INADDR_ANY;

            NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
                app.socket, reinterpret_cast<sockaddr*>(&addr), sizeof(addr)), 0);

            nn::socket::Listen(app.socket, 0);

            nsaddrLen = sizeof(struct sockaddr_in);
            NN_LOG("%s ---> Accept Start\n", __FUNCTION__);
            app.clientSocket = nn::socket::Accept(app.socket, (struct sockaddr *)&clientAddr, &nsaddrLen);
            if (app.clientSocket == -1)
            {
                NN_LOG("TCP Server nn::socket::Accept Error(%d)\n\n", nn::socket::GetLastErrno());
                return;
            }
            NN_LOG("%s ---> Accept End Socket(%d) \n\n", __FUNCTION__, app.clientSocket);

            nn::socket::InetNtop(AF_INET, &clientAddr.sin_addr,
                strIpAddress, INET_ADDRSTRLEN);
            NN_LOG("TCP Server Client Connect : %s(%d)\n", strIpAddress, nn::socket::InetNtohs(clientAddr.sin_port));

            nn::os::SignalEvent(&app.initEvent);

            // キャンセルされるまでパケットを受信し続けます。
            while (!nn::os::TryWaitEvent(&app.cancelEvent))
            {
                NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
                auto size = nn::socket::Recv(
                    app.clientSocket,
                    &data,
                    sizeof(data),
                    0);
                NN_LOG("%s ---> Recv End Result : %dByte\n\n", __FUNCTION__, size);
                if (size == 0 || size == -1)
                {
                    NN_LOG("Tcp Server Receive Error(%d)\n\n", nn::socket::GetLastErrno());
                    nn::os::SignalEvent(&app.cancelEvent);
                    break;
                }
                else
                {
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        if (nn::socket::InetHtonl(data.counter[i]) != 0)
                        {
                            NN_LOG("Tcp Server Receive Data %d : %lu\n\n", i, nn::socket::InetHtonl(data.counter[i]));
                        }
                    }
                }
            }
        }

        static void TcpClientReceiveThread(void* arg) NN_NOEXCEPT
        {
            auto& app = *static_cast<ApplicationResource*>(arg);
            // 受信用のバッファです。
            AccesspointData data;

            // キャンセルされるまでパケットを受信し続けます。
            while (!nn::os::TryWaitEvent(&app.cancelEvent))
            {
                NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
                auto size = nn::socket::Recv(
                    app.socket,
                    &data,
                    sizeof(data),
                    0);
                NN_LOG("%s ---> Recv End Result : %dByte\n\n", __FUNCTION__, size);
                if (size == 0 || size == -1)
                {
                    NN_LOG("Tcp Client Receive Error(%d)\n\n", nn::socket::GetLastErrno());
                    nn::os::SignalEvent(&app.cancelEvent);
                    break;
                }
                else
                {
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        if (nn::socket::InetHtonl(data.counter[i]) != 0)
                        {
                            NN_LOG("Tcp Client Receive Data %d : %lu\n\n", i, nn::socket::InetHtonl(data.counter[i]));
                        }
                    }
                }
            }
        }

        static void ReceiveThread(void* arg) NN_NOEXCEPT
        {
            auto& app = *static_cast<ApplicationResource*>(arg);
            char strIpAddress[INET_ADDRSTRLEN];
            bool isBroadConnect = false;

            if (app.state == ApplicationState_NetworkCreated)
            {
                // 受信用のバッファです。
                StationData data;

                // 任意のステーションからパケットを受信するためのアドレスを生成します。
                sockaddr_in addr;
                addr.sin_family = AF_INET;
                addr.sin_port = nn::socket::InetHtons(UdpPort);
                addr.sin_addr.s_addr = INADDR_ANY;

                nn::socket::InetNtop(AF_INET, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
                NN_LOG("Server Receive Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

                // キャンセルされるまでパケットを受信し続けます。
                while (!nn::os::TryWaitEvent(&app.cancelEvent))
                {
                    // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
                    socklen_t length = sizeof(addr);
                    NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
                    auto size = nn::socket::RecvFrom(
                        app.socket,
                        &data,
                        sizeof(data),
                        0,
                        reinterpret_cast<sockaddr*>(&addr),
                        &length);

                    nn::socket::InetNtop(AF_INET, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
                    NN_LOG("%s ---> Server Recv End From Address : %s Result : %dByte\n\n", __FUNCTION__, strIpAddress, size);
                    if (size == 0 || size == -1)
                    {
                        break;
                    }

                    nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.s_addr) };

                    // 初回且つ、自身のIPではない場合ブロードキャスト通信開始イベント
                    if (isBroadConnect != true && app.ipv4Address != from)
                    {
                        NN_LOG("%s ---> Server BroadCast Event Sygnal\n\n", __FUNCTION__);
                        nn::os::SignalEvent(&app.initEvent);
                        isBroadConnect = true;
                    }

                    // 受信したカウンタを保存します。
                    nn::os::LockMutex(&app.mutex);
                    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
                    {
                        const auto& node = app.networkInfo.ldn.nodes[i];
                        if (node.isConnected && node.ipv4Address == from)
                        {
                            app.counter[i] = nn::socket::InetNtohl(data.counter);
                        }
                    }
                    nn::os::UnlockMutex(&app.mutex);
                }
            }
            else if (app.state == ApplicationState_Connected)
            {
                // 受信用のバッファです。
                AccesspointData data;

                // アクセスポイントからパケットを受信するためのアドレスを生成します。
                const auto& ap = app.networkInfo.ldn.nodes[0];
                sockaddr_in addr;
                addr.sin_family = AF_INET;
                addr.sin_port = nn::socket::InetHtons(UdpPort);
                addr.sin_addr.s_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);;

                nn::socket::InetNtop(AF_INET, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
                NN_LOG("Client Receive Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

                // キャンセルされるまでパケットを受信し続けます。
                while (!nn::os::TryWaitEvent(&app.cancelEvent))
                {
                    // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
                    socklen_t length = sizeof(addr);
                    NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
                    auto size = nn::socket::RecvFrom(
                        app.socket,
                        &data,
                        sizeof(data),
                        0,
                        reinterpret_cast<sockaddr*>(&addr),
                        &length);

                    nn::socket::InetNtop(AF_INET, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
                    NN_LOG("%s ---> Client Recv End From Address : %s Result : %dByte\n\n", __FUNCTION__, strIpAddress, size);
                    if (size == 0 || size == -1)
                    {
                        break;
                    }

                    nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.s_addr) };

                    // 初回且つ、自身のIPではない場合ブロードキャスト通信開始イベント
                    if (isBroadConnect != true && app.ipv4Address != from)
                    {
                        NN_LOG("%s ---> Client BroadCast Event Sygnal\n\n", __FUNCTION__);
                        nn::os::SignalEvent(&app.initEvent);
                        isBroadConnect = true;
                    }

                    // 受信したカウンタを保存します。
                    nn::os::LockMutex(&app.mutex);
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        app.counter[i] = nn::socket::InetNtohl(data.counter[i]);
                    }

                    // 現在の状態をログ出力します。
                    NN_LOG("[Counter]\n");
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        if (app.networkInfo.ldn.nodes[i].isConnected)
                        {
                            NN_LOG("Node %d : %u\n", i, app.counter[i]);
                        }
                    }
                    NN_LOG("\n");
                    nn::os::UnlockMutex(&app.mutex);
                }
            }
        }

        static void SendThread(void* arg) NN_NOEXCEPT
        {
            auto& app = *static_cast<ApplicationResource*>(arg);
            char strIpAddress[INET_ADDRSTRLEN];

            if (app.state == ApplicationState_NetworkCreated)
            {
                // ブロードキャスト送信用のアドレスを生成します。
                sockaddr_in addr;
                addr.sin_family = AF_INET;
                addr.sin_port = nn::socket::InetHtons(UdpPort);
                addr.sin_addr.s_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);
                nn::socket::InetNtop(AF_INET, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
                NN_LOG("Server Send Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

                // キャンセルされるまで一定時間毎にパケットを送信します。
                const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
                while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
                {
                    // 自身のカウンタをインクリメントします。
                    nn::os::LockMutex(&app.mutex);
                    app.counter[0] += 1;

                    // 送信データを生成します。
                    AccesspointData data;
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        data.counter[i] = nn::socket::InetHtonl(app.counter[i]);
                    }

                    // 現在の状態をログ出力します。
                    NN_LOG("[Counter]\n");
                    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
                    {
                        if (app.networkInfo.ldn.nodes[i].isConnected)
                        {
                            NN_LOG("Node %d : %u\n", i, app.counter[i]);
                        }
                    }
                    NN_LOG("\n");
                    nn::os::UnlockMutex(&app.mutex);

                    NN_LOG("%s ---> Send Start\n", __FUNCTION__);
                    // パケットを送信します。
                    auto size = nn::socket::SendTo(
                        app.socket,
                        &data,
                        sizeof(data),
                        0,
                        reinterpret_cast<sockaddr*>(&addr),
                        sizeof(addr));
                    NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, size);
                }
            }
            else if (app.state == ApplicationState_Connected)
            {
                // アクセスポイントへの送信用のアドレスを生成します。
                const auto& ap = app.networkInfo.ldn.nodes[0];
                sockaddr_in addr;
                addr.sin_family = AF_INET;
                addr.sin_port = nn::socket::InetHtons(UdpPort);
                addr.sin_addr.s_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);

                nn::socket::InetNtop(AF_INET, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
                NN_LOG("Client Send Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

                // キャンセルされるまで一定時間毎にパケットを送信します。
                const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
                while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
                {
                    // 自身のカウンタをインクリメントします。
                    nn::os::LockMutex(&app.mutex);
                    ++app.myCounter;

                    // 送信データを生成します。
                    StationData data;
                    data.counter = nn::socket::InetHtonl(app.myCounter);
                    nn::os::UnlockMutex(&app.mutex);

                    NN_LOG("%s ---> Send Start\n", __FUNCTION__);
                    // パケットを送信します。
                    auto size = nn::socket::SendTo(
                        app.socket,
                        &data,
                        sizeof(data),
                        0,
                        reinterpret_cast<sockaddr*>(&addr),
                        sizeof(addr));
                    NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, size);
                }
            }
        }

        static void MonitorThread(void* arg) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(arg);
            auto& app = *static_cast<ApplicationResource*>(arg);
            nn::Result result;
            nn::ldn::NetworkInfo networkInfo;
            uint32_t connectCnt, disConnectCnt;

            NN_LOG("Monitorスレッドが開始されました。\n");

            nn::os::MultiWaitType       multiWait;
            nn::os::MultiWaitHolderType stateChangeHolder;
            nn::os::MultiWaitHolderType cancelHolder;
            nn::os::InitializeMultiWait(&multiWait);
            nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
            nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
            nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
            nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

            if (app.state == ApplicationState_NetworkCreating)
            {
                nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];

                // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
                while (nn::os::WaitAny(&multiWait) != &cancelHolder)
                {
                    // WaitAny では必ず手動でシグナルをクリアしなければなりません。
                    nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

                    // ネットワークが破棄された場合には監視を終了します。
                    nn::ldn::State ldnState = nn::ldn::GetState();
                    if (ldnState != nn::ldn::State_AccessPointCreated)
                    {
                        nn::os::SignalEvent(&app.cancelEvent);
                        break;
                    }

                    // 接続中のステーションを取得します。
                    result = nn::ldn::GetNetworkInfo(&networkInfo, updates, nn::ldn::NodeCountMax);
                    if (nn::ldn::ResultInvalidState::Includes(result))
                    {
                        nn::os::SignalEvent(&app.cancelEvent);
                        break;
                    }
                    else if (nn::ldn::ResultDeviceNotAvailable::Includes(result))
                    {
                        nn::os::SignalEvent(&app.cancelEvent);
                        break;
                    }
                    NN_ABORT_UNLESS(result.IsSuccess());

                    connectCnt = disConnectCnt = 0;
                    // 接続状態が変化したステーションを検索します。ノード番号 0 は AP なので無視します。
                    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
                    {
                        const auto& oldNode = app.networkInfo.ldn.nodes[i];
                        const auto& newNode = networkInfo.ldn.nodes[i];
                        const auto& update = updates[i];
                        if (update.stateChange == nn::ldn::NodeStateChange_Disconnect ||
                            update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
                        {
                            NN_LOG("ステーションが切断しました。\n");
                            NN_LOG("Node ID    : %d\n", oldNode.nodeId);
                            NN_LOG("IP Address : %s\n", ConvertToString(oldNode.ipv4Address));
                            NN_LOG("User Name  : %s\n\n", oldNode.userName);
                            disConnectCnt++;
                        }
                        if (update.stateChange == nn::ldn::NodeStateChange_Connect ||
                            update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
                        {
                            NN_LOG("新しいステーションの接続です。\n");
                            NN_LOG("Node ID    : %d\n", newNode.nodeId);
                            NN_LOG("IP Address : %s\n", ConvertToString(newNode.ipv4Address));
                            NN_LOG("User Name  : %s\n\n", newNode.userName);
                            connectCnt++;
                        }
                    }

                    // 接続状態が１台もなく切断が発生していたら監視終了追加
                    if (connectCnt == 0 && disConnectCnt > 0)
                    {
                        nn::os::SignalEvent(&app.cancelEvent);
                        break;
                    }
                    nn::os::LockMutex(&app.mutex);
                    std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
                    nn::os::UnlockMutex(&app.mutex);
                }
            }
            else if (app.state == ApplicationState_Connecting)
            {
                // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
                while (nn::os::WaitAny(&multiWait) != &cancelHolder)
                {
                    // WaitAny では必ず手動でシグナルをクリアしなければなりません。
                    nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

                    // ネットワークが破棄された場合には監視を終了します。
                    nn::ldn::State ldnState = nn::ldn::GetState();
                    if (ldnState != nn::ldn::State_StationConnected)
                    {
                        nn::os::SignalEvent(&app.cancelEvent);
                        break;
                    }

                    // 現在のネットワークの状態を取得します。
                    result = nn::ldn::GetNetworkInfo(&networkInfo);
                    if (result <= nn::ldn::ResultInvalidState())
                    {
                        nn::os::SignalEvent(&app.cancelEvent);
                        break;
                    }
                    nn::os::LockMutex(&app.mutex);
                    std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
                    nn::os::UnlockMutex(&app.mutex);
                }
            }
            nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
            nn::os::UnlinkMultiWaitHolder(&cancelHolder);
            nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
            nn::os::FinalizeMultiWaitHolder(&cancelHolder);
            nn::os::FinalizeMultiWait(&multiWait);

            NN_LOG("Monitorスレッドが終了されました。\n");
        } // NOLINT(impl/function_size)

        nn::Result Scan(int* pOutScanResultCount) NN_NOEXCEPT
        {
            // IntentId が一致する LDN ネットワークのみを探索対象とします。
            int scanChannel[] = { nn::ldn::AutoChannel, 36, 40, 44, 48 };
            nn::ldn::ScanFilter filter = {};
            filter.networkType = nn::ldn::NetworkType_Ldn;
            filter.networkId.intentId = BasicSampleIntentId;
            filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

            // LDN ネットワークを探索します。
            for (auto channel : scanChannel)
            {
                NNS_LDN_RETURN_IF_FAILED(nn::ldn::Scan(
                    g_ScanResult, pOutScanResultCount, nn::ldn::ScanResultCountMax,
                    filter, channel));
                if (*pOutScanResultCount > 0)
                {
                    break;
                }
            }

            return nn::ResultSuccess();
        }

        nn::Result Connect(const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
        {
            // 接続に必要なセキュリティパラメータを設定します。
            nn::ldn::SecurityConfig security = {};
            nn::ldn::UserConfig     user = {};
            security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Any);
            security.passphraseSize = sizeof(Passphrase);
            std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
            std::strncpy(user.userName, "Station", nn::ldn::UserNameBytesMax);
            nn::Result result;


            // LDN ネットワークに接続します。
            for (int i = 0; i < ModeLocalConnectRetryCount; i++)
            {
                result = nn::ldn::Connect(network, security, user, 0, nn::ldn::ConnectOption_None);
                if (result.IsSuccess() == true)
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(ConWaitTime));
                NN_LOG("LDN ネットワーク接続をリトライしています。Result(%d:%d) : 0x%08x\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }

            NNS_LDN_RETURN_IF_FAILED(result);

            // ネットワークの情報を取得します。
            NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&app.networkInfo));

            // 自身の IPv4 アドレスとサブネットマスクを取得します。
            NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&app.ipv4Address, &app.subnetMask));

            // ネットワークの状態を監視するスレッドを生成します。
            static NN_ALIGNAS(4096) uint8_t MonitorThreadStack[StackSize];
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                &app.monitorThread, MonitorThread, &app,
                MonitorThreadStack, sizeof(MonitorThreadStack),
                MonitorThreadPriority));
            nn::os::StartThread(&app.monitorThread);

            return nn::ResultSuccess();
        }

        void Disconnect() NN_NOEXCEPT
        {
            // ネットワークの状態を監視するスレッドの終了まで待機します。
            nn::os::WaitThread(&app.monitorThread);
            nn::os::DestroyThread(&app.monitorThread);

            // ネットワークから切断します。
            nn::ldn::Disconnect();
        }

        nn::Result CreateAccessPoint() NN_NOEXCEPT
        {
            static NN_ALIGNAS(4096) uint8_t MonitorThreadStack[StackSize];

            nn::Result result;

            NN_LOG("%s ---> nn::ldn::OpenAccessPoint Start\n", __FUNCTION__);
            result = nn::ldn::OpenAccessPoint();
            if (result.IsSuccess() != true)
            {
                NN_LOG("%s ---> nn::ldn::OpenAccessPoint Failler Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
                return result;
            }

            // Advertise で配信するデータを設定します。
            const char message[] = "Welcome to LDN network!";
            result = nn::ldn::SetAdvertiseData(message, sizeof(message));
            if (result.IsSuccess() != true)
            {
                NN_LOG("%s ---> nn::ldn::SetAdvertiseData Failler Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
                return result;
            }

            // 新規に構築する LDN ネットワークのパラメータを設定します。
            nn::ldn::NetworkConfig  network = {};
            nn::ldn::SecurityConfig security = {};
            nn::ldn::UserConfig     user = {};
            network.intentId = BasicSampleIntentId;
            network.channel = app.setChannel;
            network.nodeCountMax = nn::ldn::NodeCountMax;
            security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
            security.passphraseSize = sizeof(Passphrase);
            std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
            std::strncpy(user.userName, "AP", nn::ldn::UserNameBytesMax);

            // アクセスポイントとして新規にネットワークを構築します。
            NN_LOG("%s ---> nn::ldn::CreateNetwork Start\n", __FUNCTION__);
            result = nn::ldn::CreateNetwork(network, security, user);
            if (result.IsSuccess() != true)
            {
                NN_LOG("%s ---> nn::ldn::CreateNetwork Failler Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
                return result;
            }

            // ネットワークの情報を取得します。
            NN_LOG("%s ---> nn::ldn::GetNetworkInfo Start\n", __FUNCTION__);
            result = nn::ldn::GetNetworkInfo(&app.networkInfo);
            if (result.IsSuccess() != true)
            {
                NN_LOG("%s ---> nn::ldn::GetNetworkInfo Failler Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
                return result;
            }

            // 自身の IPv4 アドレスとサブネットマスクを取得します。
            NN_LOG("%s ---> nn::ldn::GetIpv4Address Start\n", __FUNCTION__);
            result = nn::ldn::GetIpv4Address(&app.ipv4Address, &app.subnetMask);
            NN_LOG("%s ---> nn::ldn::GetIpv4Address End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            NNS_LDN_RETURN_IF_FAILED(result);

            // ネットワークの状態を監視するスレッドを生成します。
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                &app.monitorThread, MonitorThread, &app,
                MonitorThreadStack, sizeof(MonitorThreadStack),
                MonitorThreadPriority));
            nn::os::StartThread(&app.monitorThread);

            return nn::ResultSuccess();
        }

        void DestroyAccessPoint() NN_NOEXCEPT
        {
            // ネットワークの状態を監視するスレッドの終了まで待機します。
            nn::os::WaitThread(&app.monitorThread);
            nn::os::DestroyThread(&app.monitorThread);

            // ネットワークを破棄し、 LDN ライブラリの使用を終了します。
            nn::ldn::DestroyNetwork();
            nn::ldn::CloseAccessPoint();
            NN_LOG("LDN MoniterThreadを解放しました\n\n");
        }

        static void LdnAccessPoint(void* arg) NN_NOEXCEPT
        {
            nn::Result result;

            // LDN ライブラリを初期化します。
            NN_LOG("%s ---> nn::ldn::Initialize Start\n", __FUNCTION__);
            result = nn::ldn::Initialize();
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
            }

            // LDN ネットワークを構築します。
            auto& app = *static_cast<ApplicationResource*>(arg);
            LocalInfraStressTest* pThis = static_cast<LocalInfraStressTest*>(app.pThis);

            // LDN ネットワーク構築前にイベント初期化
            nn::os::ClearEvent(&app.initEvent);

            app.currentMode = Mode_TestMaster;

            // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
            NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent Start\n", __FUNCTION__);
            nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);
            NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent End\n\n", __FUNCTION__);

            result = nn::ldn::OpenAccessPoint();
            WLANTEST_EXPECT_RESULT_SUCCESS(result);

            NN_LOG("%s ---> CreateAccessPoint Start\n", __FUNCTION__);
            result = pThis->CreateAccessPoint();
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                nn::ldn::CloseAccessPoint();
                NN_LOG("CreateAccessPoint Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
                WLANTEST_ASSERT_RESULT_SUCCESS(result);
            }
            NN_LOG("%s ---> CreateAccessPoint End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_ASSERT_RESULT_SUCCESS(result);
            NN_LOG("LDN ネットワークの構築に成功しました。\n");
            NN_LOG("Channel    : %d\n", app.networkInfo.common.channel);
            NN_LOG("IP Address : %s/%s\n",
                ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
            NN_LOG("User Name  : %s\n\n", app.networkInfo.ldn.nodes[0].userName);
            app.state = ApplicationState_NetworkCreated;

            // socket ライブラリを初期化します。
            NN_LOG("%s ---> nn::socket::Initialize Start\n", __FUNCTION__);
            result = nn::socket::Initialize(
                g_SocketMemoryPoolBuffer,
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit);
            NN_LOG("%s ---> nn::socket::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            // データの送受信を開始します.
            NN_LOG("%s ---> CreateSocket Start\n", __FUNCTION__);
            pThis->CreateSocket();
            NN_LOG("%s ---> CreateSocket End\n\n", __FUNCTION__);

            // ブロードキャスト通信は自身のパケット送信を受けてしまうため別IPの初回受信するまで待機
            NN_LOG("%s ---> WaitEvent BroadCast Connect Event Start\n", __FUNCTION__);
            nn::os::WaitEvent(&app.initEvent);
            NN_LOG("%s ---> WaitEvent BroadCast Connect Event End\n\n", __FUNCTION__);

            // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
            NN_LOG("%s ---> WaitEvent CancelEvent Start\n", __FUNCTION__);
            nn::os::WaitEvent(&app.cancelEvent);
            NN_LOG("%s ---> WaitEvent CancelEvent End\n\n", __FUNCTION__);

            // LDN ネットワークを破棄し、データの送受信を終了します。
            NN_LOG("%s ---> DestroySocket Start\n", __FUNCTION__);
            pThis->DestroySocket(true);
            NN_LOG("%s ---> DestroySocket End\n\n", __FUNCTION__);

            // socket ライブラリの使用を終了します。
            NN_LOG("%s ---> nn::socket::Finalize Start\n", __FUNCTION__);
            result = nn::socket::Finalize();
            NN_LOG("%s ---> nn::socket::Finalize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            NN_LOG("%s ---> DestroyAccessPoint Start\n", __FUNCTION__);
            pThis->DestroyAccessPoint();
            NN_LOG("%s ---> DestroyAccessPoint End\n\n", __FUNCTION__);
            app.state = ApplicationState_NetworkDestroyed;
            NN_LOG("LDN ネットワークが破棄されました。\n\n");

            nn::os::DestroySystemEvent(&app.stateChangeEvent);
        }


        void TestSocketClose(int &sock, nn::os::MutexType* pMutex, bool isShutdown = false)
        {
            if (pMutex != nullptr)
            {
                nn::os::LockMutex(pMutex);
            }

            int nResult;
            if (sock != -1)
            {
                if (isShutdown == true)
                {
                    nResult = nn::socket::Shutdown(sock, SHUT_RDWR);
                    NN_LOG("Socket Shutdown Result : %d\n", nResult);
                    if (nResult != 0)
                    {
                        NN_LOG("Socket Shutdown Last Error : %d\n", nn::socket::GetLastErrno());
                    }
                }

                // ソケットを破棄します。
                nResult = nn::socket::Close(sock);
                NN_LOG("Socket Close Result : %d\n", nResult);
                sock = -1;
            }

            if (pMutex != nullptr)
            {
                nn::os::UnlockMutex(pMutex);
            }
        }

        void DestroySocket(bool isClientClose) NN_NOEXCEPT
        {
            nn::os::SignalEvent(&app.cancelEvent);

            if (isClientClose)
            {
                TestSocketClose(app.clientSocket, &app.mutex, true);
            }

            TestSocketClose(app.socket, &app.mutex, true);

            // スレッドの終了まで待機します。
            NN_LOG("%s ---> Send Thread Wait Start\n", __FUNCTION__);
            nn::os::WaitThread(&app.sendThread);
            NN_LOG("%s ---> Send Thread Wait End\n\n", __FUNCTION__);
            NN_LOG("%s ---> Send Thread Destroy Start\n", __FUNCTION__);
            nn::os::DestroyThread(&app.sendThread);
            NN_LOG("%s ---> Send Thread Destroy End\n\n", __FUNCTION__);

            NN_LOG("%s ---> Recv Thread Wait Start\n", __FUNCTION__);
            nn::os::WaitThread(&app.receiveThread);
            NN_LOG("%s ---> Recv Thread Wait End\n\n", __FUNCTION__);
            NN_LOG("%s ---> Recv Thread Destroy Start\n", __FUNCTION__);
            nn::os::DestroyThread(&app.receiveThread);
            NN_LOG("%s ---> Recv Thread Destroy End\n\n", __FUNCTION__);
        }

        static void LdnStation(void* arg) NN_NOEXCEPT
        {
            auto& app = *static_cast<ApplicationResource*>(arg);
            LocalInfraStressTest* pThis = static_cast<LocalInfraStressTest*>(app.pThis);
            nn::Result result;

            app.currentMode = Mode_TestClient;

            // LDN ライブラリを初期化します。
            NN_LOG("%s ---> nn::ldn::Initialize Start\n", __FUNCTION__);
            result = nn::ldn::Initialize();
            NN_LOG("%s ---> nn::ldn::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
                return;
            }

            // LDN ステーション起動前にイベント初期化
            nn::os::ClearEvent(&app.initEvent);

            if (result <= nn::ldn::ResultDeviceOccupied())
            {
                // 他の通信機能を使用中のためローカル通信機能を利用できません。
                // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
            }

            // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
            NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent Start\n", __FUNCTION__);
            nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);
            NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent End\n\n", __FUNCTION__);

            // ステーションとして起動します。
            NN_LOG("%s ---> nn::ldn::OpenStation Start\n", __FUNCTION__);
            result = nn::ldn::OpenStation();
            NN_LOG("%s ---> nn::ldn::OpenStation End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            if (result <= nn::ldn::ResultInvalidState())
            {
                // 無線スイッチオフなど通信機能を利用できない状態です。
                // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
            }
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            // LDN ネットワークを探索します。
            NN_LOG("LDN ネットワークを探索中です・・・。\n\n");
            int scanResultCount;

            // LDN ネットワークを探索します。
            NN_LOG("%s ---> nn::ldn::Scan Start\n", __FUNCTION__);
            for (int i = 0; i < (ModeWaitTime / ModeScanWaitTime); i++)
            {
                result = pThis->Scan(&scanResultCount);
                if (result.IsSuccess() != true)
                {
                    // 無線スイッチオフなど通信機能を利用できない状態です。
                    // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
                    NN_LOG("Scan Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
                    nn::ldn::CloseStation();
                    app.state = ApplicationState_Disconnected;
                    return;
                }
                else if (scanResultCount == 0)
                {
                    NN_LOG("接続対象のネットワークが見つかりませんでした。\n\n");
                }
                else
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(ModeScanWaitTime));
            }
            NN_LOG("%s ---> nn::ldn::Scan End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            // LDN ネットワークへの接続を試行します。
            NN_LOG("%s ---> nn::ldn::Connect Start\n", __FUNCTION__);
            result = pThis->Connect(g_ScanResult[0]);
            NN_LOG("%s ---> nn::ldn::Connect End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            if (result <= nn::ldn::ResultInvalidState() ||
                result <= nn::ldn::ResultConnectionFailed())
            {
                NN_LOG("LDN ネットワークへの接続に失敗しました。\n\n");
                nn::ldn::CloseStation();
                app.state = ApplicationState_Disconnected;
                return;
            }
            WLANTEST_ASSERT_RESULT_SUCCESS(result);
            NN_LOG("LDN ネットワークへの接続に成功しました。\n");
            NN_LOG("Name (AP)        : %s\n", app.networkInfo.ldn.nodes[0].userName);
            NN_LOG("IP Address (AP)  : %s/%s\n",
                ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address),
                ConvertToString(app.subnetMask));
            NN_LOG("IP Address (STA) : %s/%s\n\n",
                ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
            app.state = ApplicationState_Connected;

            // socket ライブラリを初期化します。
            NN_LOG("%s ---> nn::socket::Initialize Start\n", __FUNCTION__);
            result = nn::socket::Initialize(
                g_SocketMemoryPoolBuffer,
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit);
            NN_LOG("%s ---> nn::socket::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            // データの送受信を開始します.
            NN_LOG("%s ---> CreateSocket Start\n", __FUNCTION__);
            pThis->CreateSocket();
            NN_LOG("%s ---> CreateSocket End\n\n", __FUNCTION__);

            // ブロードキャスト通信は自身のパケット送信を受けてしまうため別IPの初回受信するまで待機
            NN_LOG("%s ---> WaitEvent BroadCast Connect Event Start\n", __FUNCTION__);
            nn::os::WaitEvent(&app.initEvent);
            NN_LOG("%s ---> WaitEvent BroadCast Connect Event End\n\n", __FUNCTION__);

            // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
            NN_LOG("%s ---> WaitEvent CancelEvent Start\n", __FUNCTION__);
            nn::os::WaitEvent(&app.cancelEvent);
            NN_LOG("%s ---> WaitEvent CancelEvent End\n\n", __FUNCTION__);

            // LDN ネットワークから切断し、データの送受信を終了します。
            NN_LOG("%s ---> DestroySocket Start\n", __FUNCTION__);
            pThis->DestroySocket(false);
            NN_LOG("%s ---> DestroySocket End\n\n", __FUNCTION__);

            // socket ライブラリの使用を終了します。
            NN_LOG("%s ---> nn::socket::Finalize Start\n", __FUNCTION__);
            result = nn::socket::Finalize();
            NN_LOG("%s ---> nn::socket::Finalize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            NN_LOG("%s ---> Disconnect Start\n", __FUNCTION__);
            pThis->Disconnect();
            NN_LOG("%s ---> Disconnect End\n\n", __FUNCTION__);

            NN_LOG("%s ---> nn::ldn::CloseStation Start\n", __FUNCTION__);
            result = nn::ldn::CloseStation();
            NN_LOG("%s ---> nn::ldn::CloseStation End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
            app.state = ApplicationState_Disconnected;
            NN_LOG("LDN ネットワークから切断されました。\n\n");

            nn::os::DestroySystemEvent(&app.stateChangeEvent);
        } // NOLINT(impl/function_size)

        void LdnNetWorkCreaate(nn::os::ThreadType* pldnThread) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(pldnThread);
            static NN_ALIGNAS(4096) uint8_t LdnThreadStack[StackSize];

            app.state = ApplicationState_NetworkCreating;
            NN_LOG("LDN ネットワークの構築を開始します。\n\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                pldnThread, LdnAccessPoint, &app,
                LdnThreadStack, sizeof(LdnThreadStack), LdnThreadPriority));
            nn::os::StartThread(pldnThread);
        }

        void NetWorkDestroy(nn::os::ThreadType* pldnThread) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(pldnThread);

            nn::os::WaitThread(pldnThread);
            nn::os::DestroyThread(pldnThread);
        }


        bool TestPatternSettingUdpServer() NN_NOEXCEPT
        {
            sockaddr_in addr;
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(UdpPort);
            addr.sin_addr.s_addr = INADDR_ANY;
            size_t bufSize = 1024;
            std::unique_ptr<uint8_t[]> randomBuffer(new uint8_t[bufSize]);
            nn::Result result;
            bool isSetting = false;

            // LDN ライブラリを初期化します。
            result = nn::ldn::Initialize();
            if (result.IsSuccess() != true)
            {
                NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
                return false;
            }

            // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
            nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

            result = CreateAccessPoint();
            if (result.IsSuccess() != true)
            {
                // 無線スイッチオフなど通信機能を利用できない状態です。
                // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
                nn::ldn::Finalize();
                nn::os::DestroySystemEvent(&app.stateChangeEvent);
                return false;
            }
            NN_LOG("LDN ネットワークの構築に成功しました。\n");
            NN_LOG("Channel    : %d\n", app.networkInfo.common.channel);
            NN_LOG("IP Address : %s/%s\n",
                ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
            NN_LOG("User Name  : %s\n\n", app.networkInfo.ldn.nodes[0].userName);
            app.state = ApplicationState_NetworkCreated;

            // socket ライブラリの初期処理。
            CreateTestSocket(addr);
            if (app.socket == -1)
            {
                return false;
            }

            // 任意のステーションからパケットを受信するためのアドレスを生成します。
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(UdpPort);
            addr.sin_addr.s_addr = INADDR_ANY;

            // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
            socklen_t length = sizeof(addr);
            NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
            auto size = nn::socket::RecvFrom(
                app.socket,
                randomBuffer.get(),
                bufSize,
                0,
                reinterpret_cast<sockaddr*>(&addr),
                &length);
            NN_LOG("%s ---> Recv End Result : %dByte\n\n", __FUNCTION__, size);

            nn::os::ClearSystemEvent(&app.stateChangeEvent);
            nn::os::TimedWaitSystemEvent(&app.stateChangeEvent, nn::TimeSpan::FromSeconds(ModeSettingServerWaitTime));

            nn::os::LockMutex(&app.mutex);
            if (size <= static_cast<size_t>(sizeof(ChangeModePayload)))
            {
                // ランダムな試験パターンデータ解析
                ChangeModePayload* pPayload = reinterpret_cast<ChangeModePayload*>(randomBuffer.get());

                app.executeTime = pPayload->executeTime;
                app.TestPattern.length = pPayload->length;
                for (int i = 0; i < pPayload->length; i++)
                {
                    app.TestPattern.mode[i] = pPayload->mode[i];
                    app.TestPattern.changeTimes[i] = pPayload->changeTimes[i];
                }
                isSetting = true;
            }
            else
            {
                isSetting = false;
            }
            nn::os::UnlockMutex(&app.mutex);

            nn::os::WaitThread(&app.monitorThread);
            nn::os::DestroyThread(&app.monitorThread);

            // socket ライブラリの終了処理。
            DestroyTestSocket();

            nn::ldn::Finalize();

            nn::os::DestroySystemEvent(&app.stateChangeEvent);
            return isSetting;
        }

        void CreateRandomTestData(ChangeMode testPatternMin, ChangeMode testPatternMax,
            int changeTimeMin, int changeTimeMax, int agingTimeMin, int agingTimeMax) NN_NOEXCEPT
        {
            int minMode, maxMode, pos;

            minMode = testPatternMin;
            maxMode = testPatternMax;

            app.TestPattern.length = 0;
            pos = 0;
            while (pos < RandamModeCount)
            {
                app.TestPattern.mode[pos] = static_cast<ChangeMode>(GetRandom(minMode, maxMode));
                app.TestPattern.changeTimes[pos] = static_cast<ChangeMode>(GetRandom(changeTimeMin, changeTimeMax));
                app.TestPattern.length++;

                // 2.4GHz／5GHzのランダムの2セットにするため同じモードを２個１のランダムに変更
                if (pos + 1 < RandamModeCount)
                {
                    app.TestPattern.mode[pos + 1] = app.TestPattern.mode[pos];
                    app.TestPattern.changeTimes[pos + 1] = app.TestPattern.changeTimes[pos];
                    app.TestPattern.length++;
                    pos++;
                }
                pos++;
            }

            app.executeTime = GetRandom(agingTimeMin, agingTimeMax);
        }

        void CreateTestSocket(sockaddr_in& addr) NN_NOEXCEPT
        {
            nn::Result result;
            int socketResult;

            // socket ライブラリを初期化します。
            result = nn::socket::Initialize(
                g_SocketMemoryPoolBuffer,
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit);

            app.socket = nn::socket::Socket(AF_INET, SOCK_DGRAM, 0);
            WLANTEST_ASSERT_TRUE(app.socket != -1);

            socketResult = nn::socket::Bind(
                app.socket, reinterpret_cast<sockaddr*>(&addr), sizeof(addr));
            WLANTEST_ASSERT_TRUE(socketResult != -1);

            // ブロードキャスト通信を有効化します。
            int isEnabled = 1;
            nn::socket::SetSockOpt(
                app.socket, SOL_SOCKET, SO_BROADCAST, &isEnabled, sizeof(isEnabled));
        }

        void DestroyTestSocket() NN_NOEXCEPT
        {
            TestSocketClose(app.socket, &app.mutex);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        void TestPatternSettingUdpClient(ChangeMode testPatternMin, ChangeMode testPatternMax,
                int changeTimeMin, int changeTimeMax, int agingTimeMin, int agingTimeMax) NN_NOEXCEPT
        {
            std::unique_ptr<ChangeModePayload[]> sendBuffer(new ChangeModePayload);
            nn::Result result;
            WLANTEST_ASSERT_TRUE(sendBuffer != nullptr);

            // ランダムな試験パターンデータ作成
            CreateRandomTestData(testPatternMin, testPatternMax, changeTimeMin, changeTimeMax, agingTimeMin, agingTimeMax);

            sendBuffer.get()->length = app.TestPattern.length;
            sendBuffer.get()->executeTime = app.executeTime;
            std::memcpy(sendBuffer.get()->mode, app.TestPattern.mode, sizeof(app.TestPattern.mode));
            std::memcpy(sendBuffer.get()->changeTimes, app.TestPattern.changeTimes, sizeof(app.TestPattern.changeTimes));

            // ここからLDN・UDP通信処理
            // LDN ライブラリを初期化します。
            result = nn::ldn::Initialize();
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
                return;
            }

            // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
            nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

            // ステーションとして起動します。
            result = nn::ldn::OpenStation();
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("nn::ldn::OpenStation Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
                nn::ldn::Finalize();
                nn::os::DestroySystemEvent(&app.stateChangeEvent);
                return;
            }

            // LDN ネットワークを探索します。
            for (int i = 0; i < (ModeWaitTime / ModeScanWaitTime); i++)
            {
                NN_LOG("LDN ネットワークを探索中です・・・。\n\n");
                int scanResultCount;
                result = Scan(&scanResultCount);
                if (result.IsSuccess() != true)
                {
                    // 無線スイッチオフなど通信機能を利用できない状態です。
                    // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
                    nn::ldn::CloseStation();
                    nn::ldn::Finalize();
                    nn::os::DestroySystemEvent(&app.stateChangeEvent);
                    return;
                }
                else if (scanResultCount == 0)
                {
                    NN_LOG("接続対象のネットワークが見つかりませんでした。\n\n");
                }
                else
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(ModeScanWaitTime));
            }
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            // LDN ネットワークへの接続を試行します。
            result = Connect(g_ScanResult[0]);
            if (result <= nn::ldn::ResultInvalidState() ||
                result <= nn::ldn::ResultConnectionFailed())
            {
                NN_LOG("LDN ネットワークへの接続に失敗しました。\n\n");
                nn::ldn::CloseStation();
                app.state = ApplicationState_Disconnected;
                nn::ldn::Finalize();
                nn::os::DestroySystemEvent(&app.stateChangeEvent);
                return;
            }
            WLANTEST_ASSERT_RESULT_SUCCESS(result);
            app.state = ApplicationState_Connected;

            // データの送受信に使用する socket を生成します。
            sockaddr_in addr;
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(UdpPort);
            addr.sin_addr.s_addr = INADDR_ANY;

            // socket ライブラリの初期処理。
            CreateTestSocket(addr);
            WLANTEST_ASSERT_TRUE(app.socket != -1);

            const auto& ap = app.networkInfo.ldn.nodes[0];
            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(UdpPort);
            addr.sin_addr.s_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);

            // ランダムな試験パターンデータ送信
            NN_LOG("%s ---> Send Start\n", __FUNCTION__);
            auto size = nn::socket::SendTo(
                app.socket,
                sendBuffer.get(),
                sizeof(ChangeModePayload),
                0,
                reinterpret_cast<sockaddr*>(&addr),
                sizeof(addr));
            NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, size);

            // LDN ネットワークから切断し、データの送受信を終了します。
            // socket ライブラリの使用を終了します。
            DestroyTestSocket();

            Disconnect();
            nn::ldn::CloseStation();
            app.state = ApplicationState_Disconnected;
            NN_LOG("LDN ネットワークから切断されました。\n\n");

            nn::ldn::Finalize();
            nn::os::DestroySystemEvent(&app.stateChangeEvent);
        }

        void LdnNifmAlternately() NN_NOEXCEPT
        {
            nn::os::ThreadType ldnThread;
            nn::os::Tick sysTick;
            uint32_t count = 0;
            int pattern = 0;
            int channelPattern = 0;
            int netLdnPattern = 0;
            int netNifmEtherPattern = 0;
            int netNifmWifiPattern = 0;
            nn::nifm::IpV4Address ipV4;

            std::string strClientIp, strGateWay;
            std::string strSsid2g   = GetSsid2g();
            std::string strSsid5g   = GetSsid5g();
            std::string strIp2g     = GetIpAddress2g();
            std::string strIp5g     = GetIpAddress5g();
            std::string strSubnet   = GetSubnetMaskString();
            int oct4 = GetOct4IpAddress();

            bool isStart = true;

            uint32_t secTime = app.executeTime % 60;
            uint32_t minuteTime = app.executeTime % (60 * 60) / 60;
            uint32_t hourTime = app.executeTime / (60 * 60);

            NN_LOG("LDN AND NIFM試験を開始します。実行予定時間 : %lu:%02lu:%02lu\n\n", hourTime, minuteTime, secTime);

            sysTick = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetSeconds() <= app.executeTime && app.isAgingEnable == true)
            {
                nn::os::ClearEvent(&app.cancelEvent);

                // ローカル通信のデフォルトチャンネル設定(自動取得)
                app.setChannel = nn::ldn::AutoChannel;
                // ローカル通信／インフラ通信切り替え試験のデフォルトネットワークプロファイル設定(2.4GHz設定)
                app.pNetProfile = nullptr;

                NN_LOG("ローカル通信とインフラ通信が交互切り替えを開始します。%s[%d回目]パターン 時間：%d 秒\n\n", ChangeModeString[app.TestPattern.mode[pattern]], ++count, app.TestPattern.changeTimes[pattern]);
                if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalClient) ||
                    (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalMaster))
                {
                    if (NetProfilePattern[netLdnPattern] == TestNetPattern::NetPttern_2_4Ghz)
                    {
                        channelPattern = GetRandom(0, (sizeof(Channels_2_4Ghz) / sizeof(Channels_2_4Ghz[0]) - 1));
                        app.setChannel = Channels_2_4Ghz[channelPattern];
                        NN_LOG("ローカル通信 AccessPoint(2.4GHz) Channel(%d) 生成を開始します。\n\n", app.setChannel);
                    }
                    else
                    {
                        channelPattern = GetRandom(0, (sizeof(Channels_5Ghz) / sizeof(Channels_5Ghz[0]) - 1));
                        app.setChannel = Channels_5Ghz[channelPattern];
                        NN_LOG("ローカル通信 AccessPoint(5GHz) Channel(%d) 生成を開始します。\n\n", app.setChannel);
                    }
                    app.currentMode = Mode_TestMaster;
                    ++netLdnPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));

                    LdnNetWorkCreaate(&ldnThread);

                    nn::os::WaitEvent(&app.initEvent);
                    nn::os::ClearEvent(&app.initEvent);

                    if (isStart == true)
                    {
                        isStart = false;
                        sysTick = nn::os::GetSystemTick();
                    }

                    nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern]));

                    app.state = ApplicationState_NetworkDestroying;
                    NN_LOG("LDN ネットワークを破棄します。\n\n");
                    nn::os::SignalEvent(&app.cancelEvent);

                    NetWorkDestroy(&ldnThread);
                    nn::ldn::Finalize();
                }
                else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalMaster) ||
                    (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalClient))
                {
                    static NN_ALIGNAS(4096) uint8_t LdnThreadStack[StackSize];
                    app.state = ApplicationState_Connecting;
                    NN_LOG("LDN ネットワークをスキャンして接続します。\n\n");

                    app.currentMode = Mode_TestClient;
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnStation, &app,
                        LdnThreadStack, sizeof(LdnThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);

                    nn::os::WaitEvent(&app.initEvent);
                    nn::os::ClearEvent(&app.initEvent);

                    if (isStart == true)
                    {
                        isStart = false;
                        sysTick = nn::os::GetSystemTick();
                    }

                    nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern]));
                    nn::os::SignalEvent(&app.cancelEvent);
                    NetWorkDestroy(&ldnThread);
                    nn::ldn::Finalize();
                }
                else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherClient) ||
                    (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherServer))
                {
                    nn::os::Tick loopTick;
                    loopTick = nn::os::GetSystemTick();

                    if (NetProfilePattern[netNifmEtherPattern] == TestNetPattern::NetPttern_2_4Ghz)
                    {
                        EtherSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid2g.c_str()), strIp2g.c_str(), GetEtherSecurity2g());
                        app.pNetProfile = &g_netProfile;
                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetServerProfileData2_4Ghz);
                        NN_LOG("インフラ通信 Ethernetサーバー(2.4GHz)を開始します。\n\n");
                    }
                    else
                    {
                        EtherSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid5g.c_str()), strIp5g.c_str(), GetEtherSecurity5g());
                        app.pNetProfile = &g_netProfile;
                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetServerProfileData5Ghz);
                        NN_LOG("インフラ通信 Ethernetサーバー(5GHz)を開始します。\n\n");
                    }
                    app.currentMode = Mode_TestMaster;
                    ++netNifmEtherPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));
                    EthernetTcpServer(app.TestPattern.changeTimes[pattern]);
                }
                else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherServer) ||
                    (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherClient))
                {
                    nn::os::Tick loopTick;
                    loopTick = nn::os::GetSystemTick();

                    if (NetProfilePattern[netNifmEtherPattern] == TestNetPattern::NetPttern_2_4Ghz)
                    {
                        WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(ipV4, strIp2g.c_str(), strIp2g.length()));
                        app.ipv4Address.raw = ConvertIpArrayToBit32(ipV4);

                        strClientIp = MakeIpAddress(strIp2g.c_str(), oct4, "");
                        strGateWay = MakeIpAddress(strIp2g.c_str(), 1, "");
                        EtherSettingProfileData(g_netProfile, nn::wlan::Ssid(GetSsid2g().c_str()),
                            strClientIp.c_str(), strSubnet.c_str(), strGateWay.c_str(), GetEtherSecurity2g());
                        app.pNetProfile = &g_netProfile;

                        //app.ipv4Address.raw = ConvertIpArrayToBit32(EthernetServerProfileData2_4Ghz.ipSetting.ip.ipAddress);
                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetClientProfileData2_4Ghz);
                        NN_LOG("インフラ通信 Ethernetクライアント(2.4GHz)を開始します。\n\n");
                    }
                    else
                    {
                        WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(ipV4, strIp5g.c_str(), strIp5g.length()));
                        app.ipv4Address.raw = ConvertIpArrayToBit32(ipV4);

                        strClientIp = MakeIpAddress(strIp5g.c_str(), oct4, "");
                        strGateWay = MakeIpAddress(strIp5g.c_str(), 1, "");
                        EtherSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid5g.c_str()),
                            strClientIp.c_str(), strSubnet.c_str(), strGateWay.c_str(), GetEtherSecurity5g());
                        app.pNetProfile = &g_netProfile;

                        //app.ipv4Address.raw = ConvertIpArrayToBit32(EthernetServerProfileData5Ghz.ipSetting.ip.ipAddress);
                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetClientProfileData5Ghz);
                        NN_LOG("インフラ通信 Ethernetクライアント(5GHz)を開始します。\n\n");
                    }
                    app.currentMode = Mode_TestClient;
                    ++netNifmEtherPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));
                    EthernetTcpClient(app.TestPattern.changeTimes[pattern]);
                }
                else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiClient) ||
                    (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiServer))
                {
                    nn::os::Tick loopTick;
                    loopTick = nn::os::GetSystemTick();

                    if (NetProfilePattern[netNifmWifiPattern] == TestNetPattern::NetPttern_2_4Ghz)
                    {
                        WiFiSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid2g.c_str()), strIp2g.c_str(), GetWifiSecurity2g());
                        app.pNetProfile = &g_netProfile;

                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiServerProfileData2_4Ghz);
                        NN_LOG("インフラ通信 WiFiサーバー(2.4GHz)を開始します。\n\n");
                    }
                    else
                    {
                        WiFiSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid5g.c_str()), strIp5g.c_str(), GetWifiSecurity5g());
                        app.pNetProfile = &g_netProfile;

                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiServerProfileData5Ghz);
                        NN_LOG("インフラ通信 WiFiサーバー(5GHz)を開始します。\n\n");
                    }
                    app.currentMode = Mode_TestMaster;
                    ++netNifmWifiPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));

                    WifiServer(app.TestPattern.changeTimes[pattern]);
                }
                else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiServer) ||
                    (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiClient))
                {
                    nn::os::Tick loopTick;
                    loopTick = nn::os::GetSystemTick();

                    if (NetProfilePattern[netNifmWifiPattern] == TestNetPattern::NetPttern_2_4Ghz)
                    {
                        WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(ipV4, strIp2g.c_str(), strIp2g.length()));
                        app.ipv4Address.raw = ConvertIpArrayToBit32(ipV4);

                        strClientIp = MakeIpAddress(strIp2g.c_str(), oct4, "");
                        WiFiSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid2g.c_str()),
                            strClientIp.c_str(), strSubnet.c_str(), strIp2g.c_str(), GetWifiSecurity2g());

                        app.pNetProfile = &g_netProfile;

                        //app.ipv4Address.raw = ConvertIpArrayToBit32(WifiServerProfileData2_4Ghz.ipSetting.ip.ipAddress);
                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiClientProfileData2_4Ghz);
                        NN_LOG("インフラ通信 WiFiクライアント(2.4GHz)を開始します。\n\n");
                    }
                    else
                    {
                        WLANTEST_EXPECT_TRUE(ConvertIpStringToArray(ipV4, MakeIpAddress(strIp5g.c_str(), oct4, "").c_str(),
                            MakeIpAddress(strIp5g.c_str(), oct4, "").length()));
                        app.ipv4Address.raw = ConvertIpArrayToBit32(ipV4);

                        strClientIp = MakeIpAddress(strIp5g.c_str(), oct4, "");
                        WiFiSettingProfileData(g_netProfile, nn::wlan::Ssid(strSsid5g.c_str()),
                            MakeIpAddress(strIp5g.c_str(), oct4, "").c_str(), GetWifiSecurity5g());
                        app.pNetProfile = &g_netProfile;

                        //app.ipv4Address.raw = ConvertIpArrayToBit32(WifiServerProfileData5Ghz.ipSetting.ip.ipAddress);
                        //app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiClientProfileData5Ghz);
                        NN_LOG("インフラ通信 WiFiクライアント(5GHz)を開始します。\n\n");
                    }
                    app.currentMode = Mode_TestClient;
                    ++netNifmWifiPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));
                    WifiClient(app.TestPattern.changeTimes[pattern]);
                }
                ++pattern %= app.TestPattern.length;
            }

            NN_LOG("LDN AND NIFM試験を終了します。\n\n", hourTime, minuteTime, secTime);
        } // NOLINT(impl/function_size)

        void EthernetTcpServer(uint32_t waitTime) NN_NOEXCEPT
        {
            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_netProfile);
            nn::nifm::NetworkConnection networkConnection;
            nn::nifm::RequestHandle requestHandle;
            nn::util::Uuid profileId;
            nn::Result result;

            if (app.pNetProfile == nullptr)
            {
                result = temporaryNetworkProfile.Update(g_netProfile);
            }
            else
            {
                result = temporaryNetworkProfile.Update(*app.pNetProfile);
            }
            if (result.IsSuccess() != true)
            {
                app.isAgingEnable = false;
                NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                return;
            }

            requestHandle = networkConnection.GetRequestHandle();
            profileId = temporaryNetworkProfile.GetId();
            result = nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);
            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            while (!nn::os::TryWaitEvent(&app.cancelEvent))
            {
                NN_LOG("%s ---> SubmitRequestAndWait\n", __FUNCTION__);
                networkConnection.SubmitRequestAndWait();
                if (networkConnection.IsAvailable() == true)
                {
                    break;
                }
            }

            // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
            WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable());
            if (networkConnection.IsAvailable() == true)
            {
                TraceCurrentnetwork();

                nn::os::ClearEvent(&app.initEvent);
                nn::os::ClearEvent(&app.cancelEvent);

                // socket ライブラリを初期化します。
                result = nn::socket::Initialize(
                    g_SocketMemoryPoolBuffer,
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::DefaultSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);
                WLANTEST_ASSERT_RESULT_SUCCESS(result);

                app.state = ApplicationState_Connected;

                // TCP ソケット生成、サーバー送受信を別スレッドで開始します
                CreateTcpServerSocket();

                // サーバーにクライアントが接続されたか待機します。
                if (nn::os::TimedWaitEvent(&app.initEvent, nn::TimeSpan::FromSeconds(ModeWaitTime)) == true)
                {
                    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
                    nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(waitTime));
                }

                // ネットワークを破棄し、データの送受信を終了します。
                DestroySocket(true);

                networkConnection.CancelRequest();
                WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable() != true);
                if (networkConnection.IsAvailable() != true)
                {
                    NN_LOG("NIFM ネットワークが切断されました。\n\n");
                }
                else
                {
                    result = networkConnection.GetResult();
                    NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                        result.GetDescription(), result.GetInnerValueForDebug());
                }

                // socket ライブラリの使用を終了します。
                result = nn::socket::Finalize();
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
            }
            else
            {
                app.isAgingEnable = false;
                result = networkConnection.GetResult();
                NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                    result.GetDescription(), result.GetInnerValueForDebug());
            }
        }

        void EthernetTcpClient(uint32_t waitTime) NN_NOEXCEPT
        {
            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_netProfile);
            nn::nifm::NetworkConnection networkConnection;
            nn::nifm::RequestHandle requestHandle;
            nn::util::Uuid profileId;
            nn::Result result;

            if (app.pNetProfile == nullptr)
            {
                result = temporaryNetworkProfile.Update(g_netProfile);
            }
            else
            {
                result = temporaryNetworkProfile.Update(*app.pNetProfile);
            }
            if (result.IsSuccess() != true)
            {
                app.isAgingEnable = false;
                NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                return;
            }

            requestHandle = networkConnection.GetRequestHandle();
            profileId = temporaryNetworkProfile.GetId();
            result = nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);

            WLANTEST_ASSERT_RESULT_SUCCESS(result);

            while (!nn::os::TryWaitEvent(&app.cancelEvent))
            {
                NN_LOG("%s ---> SubmitRequestAndWait\n", __FUNCTION__);
                networkConnection.SubmitRequestAndWait();
                if (networkConnection.IsAvailable() == true)
                {
                    break;
                }
            }

            // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
            WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable());
            if (networkConnection.IsAvailable() == true)
            {
                TraceCurrentnetwork();

                nn::os::ClearEvent(&app.initEvent);
                nn::os::ClearEvent(&app.cancelEvent);


                // socket ライブラリを初期化します。
                result = nn::socket::Initialize(
                    g_SocketMemoryPoolBuffer,
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::DefaultSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);
                WLANTEST_ASSERT_RESULT_SUCCESS(result);

                app.state = ApplicationState_Connected;

                // TCP ソケット生成、クライアント送受信を別スレッドで開始します
                CreateTcpClientSocket();

                // メインスレッドから停止を指示されるか、試験実行時間が経過するか、ネットワークが破棄されるまで待機します。
                nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(waitTime));

                // ネットワークを破棄し、データの送受信を終了します。
                DestroySocket(false);
                app.clientSocket = -1;

                networkConnection.CancelRequest();
                WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable() != true);
                if (networkConnection.IsAvailable() != true)
                {
                    NN_LOG("NIFM ネットワークが切断されました。\n\n");
                }
                else
                {
                    result = networkConnection.GetResult();
                    NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                        result.GetDescription(), result.GetInnerValueForDebug());
                }

                // socket ライブラリの使用を終了します。
                result = nn::socket::Finalize();
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
            }
            else
            {
                app.isAgingEnable = false;
                result = networkConnection.GetResult();
                NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                    result.GetDescription(), result.GetInnerValueForDebug());
            }
        }


        void WifiServer(uint32_t waitTime) NN_NOEXCEPT
        {
            nn::Result result;

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_netProfile);
            if (app.pNetProfile == nullptr)
            {
                result = temporaryNetworkProfile.Update(g_netProfile);
            }
            else
            {
                result = temporaryNetworkProfile.Update(*app.pNetProfile);
            }
            if (result.IsSuccess() != true)
            {
                app.isAgingEnable = false;
                NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                return;
            }

            nn::nifm::NetworkConnection networkConnection;
            nn::nifm::RequestHandle requestHandle = networkConnection.GetRequestHandle();
            nn::util::Uuid profileId = temporaryNetworkProfile.GetId();
            nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);
            networkConnection.SubmitRequestAndWait();

            // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
            WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable());
            if (networkConnection.IsAvailable() == true)
            {
                TraceCurrentnetwork();

                nn::os::ClearEvent(&app.initEvent);
                nn::os::ClearEvent(&app.cancelEvent);

                // socket ライブラリを初期化します。
                result = nn::socket::Initialize(
                    g_SocketMemoryPoolBuffer,
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::DefaultSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);
                WLANTEST_ASSERT_RESULT_SUCCESS(result);

                app.state = ApplicationState_Connected;

                // TCP ソケット生成、サーバー送受信を別スレッドで開始します
                CreateTcpServerSocket();

                // Clientが接続されるか、試験時間が経過するまで待機します
                if (nn::os::TimedWaitEvent(&app.initEvent, nn::TimeSpan::FromSeconds(ModeWaitTime)) == true)
                {
                    // メインスレッドから停止を指示されるか、試験実行時間が経過するか、ネットワークが破棄されるまで待機します。
                    nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(waitTime));
                }

                // ネットワークを破棄し、データの送受信を終了します。
                DestroySocket(true);

                networkConnection.CancelRequest();
                WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable() != true);
                if (networkConnection.IsAvailable() != true)
                {
                    NN_LOG("NIFM ネットワークが切断されました。\n\n");
                }
                else
                {
                    result = networkConnection.GetResult();
                    NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                        result.GetDescription(), result.GetInnerValueForDebug());
                }

                // socket ライブラリの使用を終了します。
                result = nn::socket::Finalize();
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
            }
            else {
                app.isAgingEnable = false;
                result = networkConnection.GetResult();
                NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                    result.GetDescription(), result.GetInnerValueForDebug());
            }

        }

        void WifiClient(uint32_t waitTime) NN_NOEXCEPT
        {
            nn::Result result;

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_netProfile);
            if (app.pNetProfile == nullptr)
            {
                result = temporaryNetworkProfile.Update(g_netProfile);
            }
            else
            {
                result = temporaryNetworkProfile.Update(*app.pNetProfile);
            }
            if (result.IsSuccess() != true)
            {
                app.isAgingEnable = false;
                NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                return;
            }

            nn::nifm::NetworkConnection networkConnection;
            nn::nifm::RequestHandle requestHandle = networkConnection.GetRequestHandle();
            nn::util::Uuid profileId = temporaryNetworkProfile.GetId();
            nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);
            NN_LOG("%s ---> SubmitRequestAndWait\n", __FUNCTION__);
            networkConnection.SubmitRequestAndWait();

            // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
            WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable());
            if (networkConnection.IsAvailable() == true)
            {
                TraceCurrentnetwork();

                nn::os::ClearEvent(&app.initEvent);
                nn::os::ClearEvent(&app.cancelEvent);

                // socket ライブラリを初期化します。
                result = nn::socket::Initialize(
                    g_SocketMemoryPoolBuffer,
                    nn::socket::DefaultSocketMemoryPoolSize,
                    nn::socket::DefaultSocketAllocatorSize,
                    nn::socket::DefaultConcurrencyLimit);
                WLANTEST_ASSERT_RESULT_SUCCESS(result);

                app.state = ApplicationState_Connected;

                // TCP ソケット生成、クライアント送受信を別スレッドで開始します
                CreateTcpClientSocket();

                // メインスレッドから停止を指示されるか、試験実行時間が経過するか、ネットワークが破棄されるまで待機します。
                nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(waitTime));

                // ネットワークを破棄し、データの送受信を終了します。
                DestroySocket(false);
                app.clientSocket = -1;

                networkConnection.CancelRequest();
                WLANTEST_EXPECT_TRUE(networkConnection.IsAvailable() != true);
                if (networkConnection.IsAvailable() != true)
                {
                    NN_LOG("NIFM ネットワークが切断されました。\n\n");
                }
                else
                {
                    result = networkConnection.GetResult();
                    NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                        result.GetDescription(), result.GetInnerValueForDebug());
                }

                // socket ライブラリの使用を終了します。
                result = nn::socket::Finalize();
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
            }
            else {
                app.isAgingEnable = false;
                result = networkConnection.GetResult();
                NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                    result.GetDescription(), result.GetInnerValueForDebug());
            }
        }
    };

    TEST_F(LocalInfraStressTest, LdnMaster) NN_NOEXCEPT
    {
        app.isAgingEnable = true;
        app.mode = ChangeMode::Mode_TestMaster;

        WLANTEST_ASSERT_TRUE(TestPatternSettingUdpServer());

        LdnNifmAlternately();
    }

    TEST_F(LocalInfraStressTest, LdnClient) NN_NOEXCEPT
    {
        app.isAgingEnable = true;
        app.mode = ChangeMode::Mode_TestClient;

        int agingSecTime = static_cast<int>(agingTime / 1000);

        TestPatternSettingUdpClient(
            Mode_InfraEtherServer, Mode_LdnLocalClient,
            ModeChangeMinTime, ModeChangeMaxTime,
            agingSecTime, agingSecTime);

        LdnNifmAlternately();

    }

}; // namespace WlanTest
