﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "../Include/testWlan_LocalInfraStressTest.h"

#define STATIC_IP

namespace {
    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    const uint16_t EtherTypes[1] = { 0x88b7 };

    const size_t threadStackSize = nn::os::ThreadStackAlignment * 4;
    NN_OS_ALIGNAS_THREAD_STACK char  threadStack[threadStackSize];

    int32_t g_ScanTime = WlanTestLdnScanTime;

};

namespace WlanTest {
    class LocalClientStressTest : public ::testing::Test
    {
    protected:
        nn::os::ThreadType actionframeThread;
        nn::os::EventType actionframeEvent;

        LocalApiClass localClient;

        nn::os::ThreadType scanThread;
        nn::wlan::ScanParameters unsyncScanParam;
        uint64_t intervalScanTime;
        bool isStartScan;

        nn::os::SystemEventType connectionEvent;
        nn::wlan::ConnectionStatus connectionStatus;

        nn::wlan::DisconnectClient disClient;

        nn::os::Tick sysTick;
        nn::os::Tick logOutTime;
        LocalConnectsInfo connectsInfo;

        const bool isEnabled = true;

        nn::wlan::MacAddress testPutMac;
        int16_t testChannel;

        int64_t agingTime;

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        nn::os::Tick connectLogApiStart;
        nn::os::Tick linkApiStart;
        nn::os::Tick connectStsApiStart;
        bool isConnectLogCheck;
        bool isLinkApiCheck;
        bool isConnectStsCheck;

        nn::os::ThreadType testCheckThread;
        bool isCheck;
#endif

    protected:
        LocalClientStressTest() NN_NOEXCEPT
        {
            isStartScan = false;
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Local);

            agingTime = GetAgintTime();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm",
                    "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                kLocalConcurrencyLimit));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

            localClient.InitializeBtm();

            // デフォルトのジョイコン設定は０台
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNodeNone));
#ifdef WLAN_TEST_STATE_ASSER_STOP
            localClient.StateThreadStart(Time100msec, TraceState);
#endif

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&connectionEvent));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localClient.LocalClientDisconnect(Time60s);
            localClient.LocalRelease();

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

#ifdef WLAN_TEST_STATE_ASSER_STOP
            localClient.StateThreadStop();
#endif

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        void AllScan() NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            const nn::wlan::ScanParameters StartScanTestParams = {
                nn::wlan::ScanType_Active,{ 1 }, 0, 40, -1, nullptr,
                0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
            };

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(),
                    TestScanMaxSize, StartScanTestParams, &info));
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            NN_LOG("             rand : %d\n", rand_ret);
            return rand_ret;
        }

        bool CompareScanSsid(uint8_t pScanBuffer[], size_t bufferSize, nn::wlan::Ssid compareSsid) NN_NOEXCEPT
        {
            nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
            uint32_t bssCount = resultReader.GetCount();
            NN_UNUSED(bufferSize);

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                if (compareSsid == beacon.GetSsid())
                {
                    return true;
                }
            }

            return false;
        }

        bool ScanGetMacAddress(nn::wlan::MacAddress& resultMacAddress,
                uint8_t pScanBuffer[], size_t bufferSize, nn::wlan::Ssid compareSsid) NN_NOEXCEPT
        {
            nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
            uint32_t bssCount = resultReader.GetCount();
            NN_UNUSED(bufferSize);

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                if (compareSsid == beacon.GetSsid())
                {
                    resultMacAddress = beacon.GetBssid();
                    return true;
                }
            }
            return false;
        }

        static void ScanActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            TestScanConnectParam* pInfo = static_cast<TestScanConnectParam*>(arg);
            char macStrArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress outMacAddress;
            uint32_t* pPos = &(pInfo->writePos = 0);
            TestScanActionFrame* pBufInfo = &pInfo->actInfo[*pPos];
            int64_t recvInterval = pInfo->recvInterval;
            int64_t recvWait;
            nn::os::Tick recvTick;
            size_t outSize = 0;
            nn::Result result;

            NN_LOG("             ScanActionFrameThread -> ActionRxID : %d START\n", pInfo->rxId);
            while (pInfo->isStop != true)
            {
                recvTick = nn::os::GetSystemTick();
                result = nn::wlan::Local::GetActionFrame(&outMacAddress, pBufInfo->buffer,
                    pBufInfo->bufferSize, &outSize, pInfo->rxId);
                if (result.IsSuccess() == true)
                {
                    ++*pPos %= pInfo->maxPool;
                    pBufInfo = &pInfo->actInfo[*pPos];
                    pInfo->recvEvent.SetSygnal();
                    NN_LOG("             ScanActionFrameThread -> GetAction Frame From(%s) %lu Byte\n\n",
                            outMacAddress.GetString(macStrArray), outSize);

                    recvWait = (nn::os::GetSystemTick() - recvTick).ToTimeSpan().GetMilliSeconds();
                    if (recvInterval > 0 && recvInterval > recvWait)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(recvInterval - recvWait));
                    }
                }
                else if (nn::wlan::ResultGetFrameCancelled().Includes(result) != true)
                {
                    NN_LOG("             ScanActionFrameThread -> GetActionFrame Error : 0x%08x\n", result.GetInnerValueForDebug());
                }
            }
            NN_LOG("             ScanActionFrameThread -> END\n\n");
        }

        bool ScanConnectRate(int32_t resultArray[Type_End], LocalClientConnectParameter conParameter, int32_t retryCount) NN_NOEXCEPT
        {
            nn::wlan::ScanIeMatchInfo info;
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            bool isScanMatch = false;

            const nn::wlan::ScanParameters Scan2GHz = {
                nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, 0, nullptr,
                0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
            };
            const int16_t Scan5GHzChannel[] = {
                36, 40, 44, 48
            };

            if (pTestBuffer.get() == nullptr)
            {
                NN_LOG("             ScanConnectRate Heap Memory Allocate Error\n\n");
                return false;
            }

            std::memset(resultArray, 0x00, sizeof(int32_t) * Type_End);
            for (int i = 0; i < retryCount; i++)
            {
                std::memset(pTestBuffer.get(), 0x00, TestScanMaxSize);
                // 2.4GHzスキャン
                nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, Scan2GHz, &info);
                isScanMatch = localClient.GetScanResult(conParameter.ssid, pTestBuffer.get(), TestScanMaxSize);
                if (isScanMatch != true)
                {
                    // 5GHzスキャン（LDN仕様）
                    for (auto nChannel : Scan5GHzChannel)
                    {
                        nn::wlan::ScanParameters Scan5GHz = {
                            nn::wlan::ScanType_Passive,{ nChannel }, 1, 110, 0, nullptr,
                            0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
                        };

                        nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, Scan2GHz, &info);
                        isScanMatch = localClient.GetScanResult(conParameter.ssid, pTestBuffer.get(), TestScanMaxSize);
                        if (isScanMatch == true)
                        {
                            break;
                        }
                    }
                }

                if (isScanMatch != true)
                {
                    resultArray[Type_ScanApiFail]++;
                    continue;
                }

                resultArray[Type_ScanPass]++;
                if (localClient.LocalClientConnect(conParameter, static_cast<int64_t>(Time10s)) == true)
                {
                    resultArray[Type_ConnectPass]++;
                    localClient.LocalClientDisconnect(Time60s);
                }
                else
                {
                    resultArray[Type_ConnectFail]++;
                }
            }

            return true;
        }

        bool ScanActionFrameStart(TestScanConnectParam& actInfo, uint32_t actionRxId) NN_NOEXCEPT
        {
            return ScanActionFrameStart(actInfo, actionRxId, 0);
        }

        bool ScanActionFrameStart(TestScanConnectParam& actInfo, uint32_t actionRxId, int32_t recvInterval) NN_NOEXCEPT
        {
            std::unique_ptr<TestScanActionFrame[]> pActInfo(new TestScanActionFrame[100]);

            if (pActInfo == nullptr)
            {
                return false;
            }

            actInfo.isStop = false;
            actInfo.maxPool = 100;
            actInfo.actInfo = std::move(pActInfo);
            actInfo.recvEvent.SetEvent(false, nn::os::EventClearMode_ManualClear);
            actInfo.rxId = actionRxId;
            actInfo.recvInterval = recvInterval;

            AllScan();
            if (nn::os::CreateThread(&actionframeThread, ScanActionFrameThread,
                &actInfo, threadStack, threadStackSize, nn::os::DefaultThreadPriority).IsSuccess() != true)
            {
                return false;
            }
            nn::os::StartThread(&actionframeThread);

            // 一旦ActionFrameが残っていないか待機して破棄する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            actInfo.readPos = actInfo.writePos;

            return true;
        }

        bool ScanActionConnectRate(int32_t resultArray[Type_End],
                TestScanConnectParam& actInfo, LocalClientConnectParameter conParameter, int32_t retryCount) NN_NOEXCEPT
        {
            bool isScanMatch = false;

            std::memset(resultArray, 0x00, sizeof(int32_t) * Type_End);
            for (int i = 0; i < retryCount; i++)
            {
                // スキャン開始前に一旦受信済みシグナルリセットと読み込み済み設定
                actInfo.recvEvent.ClearSygnal();
                actInfo.readPos = actInfo.writePos;

                isScanMatch = localClient.LocalPassiveScanResult(conParameter.ssid, conParameter.channel, g_ScanTime);
                if (isScanMatch == true)
                {
                    // Scan中のみActionFrame受信待機
                    isScanMatch = actInfo.recvEvent.TrySygnal();
                    // Scan後も110msecのActionFrame受信待機
                    //isScanMatch = actInfo.recvEvent.WaitSygnal(Time100msec + Time10msec);
                    if (isScanMatch != true)
                    {
                        resultArray[Type_ScanActionFrameFail]++;
                        continue;
                    }
                }
                else
                {
                    resultArray[Type_ScanApiFail]++;
                    continue;
                }

                ++actInfo.readPos %= actInfo.maxPool;
                if (localClient.LocalClientConnect(conParameter, static_cast<int64_t>(Time10s)) == true)
                {
                    resultArray[Type_ConnectPass]++;
                    localClient.LocalClientDisconnect(Time60s);
                }
                else
                {
                    resultArray[Type_ConnectFail]++;
                }
            }

            return true;
        }

        void ScanActionFrameStop(TestScanConnectParam& actInfo, LocalClientConnectParameter conParameter) NN_NOEXCEPT
        {
            if (actionframeThread._state > nn::os::ThreadType::State_NotInitialized)
            {
                actInfo.isStop = true;
                if (nn::wlan::Local::CancelGetActionFrame(localClient.m_actionRxId).IsSuccess() == true)
                {
                    if (localClient.LocalClientConnect(conParameter, static_cast<int64_t>(Time5s)) == true)
                    {
                        // MasterのBeacon停止条件の接続待機（10秒間接続状態でMasterのループを終了させる）
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time10s));
                        localClient.LocalClientDisconnect(Time10s);
                    }

                    nn::os::WaitThread(&actionframeThread);
                    nn::os::DestroyThread(&actionframeThread);
                }
            }
        }

        static void TraceState(nn::wlan::WlanState oldState, nn::wlan::WlanState newState) NN_NOEXCEPT
        {
            NN_LOG("\n             **** WLAN STATE CHANGE EVENT *****\n");
            NN_LOG("             Previous State : %d\n", oldState);
            NN_LOG("             Current State  : %d\n", newState);
            NN_LOG("             **** WLAN STATE CHANGE EVENT *****\n\n");
        }

        static void TraceState(LocalApiClass* plocal) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(plocal);

            nn::wlan::WlanState state;
            nn::Result result;

#ifdef WLAN_TEST_STATE_ASSER_STOP
            result = plocal->GetState(&state);
#else
            NN_UNUSED(plocal);
            result = nn::wlan::Local::GetState(&state);
#endif
            if (result.IsSuccess() == true)
            {
                NN_LOG("\n             **** WLAN STATE *****\n");
                NN_LOG("             Current State  : %d\n", state);
                NN_LOG("             **** WLAN STATE *****\n\n");
            }
            else
            {
                NN_LOG("             GetState Result failed : 0x%08x\n", result.GetInnerValueForDebug());
            }
        }

        void StartUnsyncScanThread(nn::wlan::ScanParameters TestScanParam, uint64_t intervalTime) NN_NOEXCEPT
        {
            if (isStartScan == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t scanStack[threadStackSize];
            unsyncScanParam = TestScanParam;
            intervalScanTime = intervalTime;

            isStartScan = true;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanThread, UnSyncScanThread,
                    this, scanStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&scanThread);
        }

        void StopUnsyncScanThread() NN_NOEXCEPT
        {
            if (isStartScan != true)
            {
                return;
            }

            isStartScan = false;

            nn::os::WaitThread(&scanThread);
            nn::os::DestroyThread(&scanThread);
        }

        static void SendActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientStressTest* pThis = static_cast<LocalClientStressTest*>(arg);

            size_t putSize = ActionFrameBufferSize;
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
            nn::wlan::MacAddress fromMacAddress;

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

            for (int i = 0; i < 1000; i++)
            {
                WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(),
                        putSize, pThis->testPutMac, fromMacAddress, EtherTypes[0],
                        TestMatchInfo[0], nn::wlan::ActionFrameType_Local, i));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(pThis->testPutMac,
                        putBuffer.get(), putSize, pThis->testChannel, 30));
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
            }
        }

        static void ReceiveActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientStressTest* pThis = static_cast<LocalClientStressTest*>(arg);
            const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
            nn::wlan::MacAddress outMacAddress;
            std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[GetActionFrameBufferSize]);
            nn::wlan::ConnectionStatus conStatus;
            nn::Result result;
            size_t outSize;

            WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

            // 3回受信する
            for (int i = 0; i < 3; i++)
            {
                WLANTEST_ASSERT_TRUE(pThis->localClient.LocalCancelActionFrameStart(Time5s));
                result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(),
                        GetActionFrameBufferSize, &outSize, pThis->localClient.m_actionRxId);
                pThis->localClient.LocalCancelActionFrameStop();
                if (result.IsSuccess() != true)
                {
                    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&conStatus));
                    // MASTERからの接続が切られていたら評価せずに終了
                    if (conStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        NN_LOG("             Master Disconnected\n");
                        break;
                    }
                    WLANTEST_EXPECT_RESULT_SUCCESS(result);
                }

                // 設定したフレームタイプ比較
                bool isMatchFramType = false;
                LocalActionFrameData* pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());
                for (auto frameType : FrameTypes)
                {
                    if (frameType == pActionFrame->frameType)
                    {
                        isMatchFramType = true;
                        break;
                    }
                }

                if (outSize > ActionFrameBufferSize)
                {
                    nn::wlan::MacAddress fromMac(pActionFrame->fromMacAddress);
                    char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                    NN_LOG("             ReceiveActionFrameThread UnMatch ActionFrame Size : %lu\n", outSize);
                    NN_LOG("                                              From MacAddress  : %s\n", fromMac.GetString(clientMacArray));
                    i--;
                    continue;
                }
                WLANTEST_EXPECT_TRUE(isMatchFramType);
            }
        }

        static void UnSyncScanThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientStressTest* pThis = static_cast<LocalClientStressTest*>(arg);

            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            NN_LOG("             Client UnSyncScanThread -> StartScan Start\n");
            while (pThis->isStartScan == true)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pThis->intervalScanTime));
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(),
                        TestScanMaxSize, pThis->unsyncScanParam, &info));
            }
            NN_LOG("             Client UnSyncScanThread -> StartScan End\n\n");
        }

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        static void TestApiCallTimeThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientStressTest* pThis = static_cast<LocalClientStressTest*>(arg);
            nn::os::Tick logTick;
            bool isLogOut = false;

            logTick = nn::os::GetSystemTick();
            while (pThis->isCheck)
            {
                if ((nn::os::GetSystemTick() - pThis->connectStsApiStart).ToTimeSpan().GetMilliSeconds() >= Time120s)
                {
                    if (pThis->isConnectStsCheck && (nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= Time60s)
                    {
                        NN_LOG("             GetConnectionStatus Api Call Time Over(%lld)\n",
                                (nn::os::GetSystemTick() - pThis->connectStsApiStart).ToTimeSpan().GetMilliSeconds());
                        isLogOut = true;
                    }
                }

                if ((nn::os::GetSystemTick() - pThis->linkApiStart).ToTimeSpan().GetMilliSeconds() >= Time120s)
                {
                    if (pThis->isLinkApiCheck && (nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= Time60s)
                    {
                        NN_LOG("             GetLinkLevel Api Call Time Over(%lld)\n",
                                (nn::os::GetSystemTick() - pThis->linkApiStart).ToTimeSpan().GetMilliSeconds());
                        isLogOut = true;
                    }
                }

                if ((nn::os::GetSystemTick() - pThis->connectLogApiStart).ToTimeSpan().GetMilliSeconds() >= Time600s * 2)
                {
                    if (pThis->isConnectLogCheck && (nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= Time60s)
                    {
                        NN_LOG("             connection Log Api Call Time Over(%lld)\n",
                                (nn::os::GetSystemTick() - pThis->connectLogApiStart).ToTimeSpan().GetMilliSeconds());
                        isLogOut = true;
                    }
                }

                if (isLogOut)
                {
                    logTick = nn::os::GetSystemTick();
                    isLogOut = false;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            }

        }
#endif

        void Connect(TestLocalConnectParams ConnectParams) NN_NOEXCEPT
        {
            TraceState(&localClient);
            while (NN_STATIC_CONDITION(1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParams.ssid,
                    ConnectParams.bssid,
                    ConnectParams.channel,
                    ConnectParams.security,
                    ConnectParams.autoKeepAlive,
                    ConnectParams.indication,
                    ConnectParams.beaconLostTimeout));

                NN_LOG("             WiFi Connect Start\n");
                nn::os::WaitSystemEvent(&connectionEvent);
                NN_LOG("             WiFi WaitSystemEvent Signal\n");

                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             WlanTest: Connected\n");
                    break;
                }

                NN_LOG("             WiFi ConnectionStatus(%d)\n", connectionStatus.state);
            }
            TraceState(&localClient);
        }

        void Disconnect() NN_NOEXCEPT
        {
            TraceState(&localClient);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            NN_LOG("             GetConnectionStatus(%d)\n", connectionStatus.state);
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("             Disconnect Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
                NN_LOG("             Disconnect End\n");
            }

            TraceState(&localClient);
            nn::os::Tick disTimeout = nn::os::GetSystemTick();
            while (NN_STATIC_CONDITION(1))
            {
                WLANTEST_ASSERT_TRUE((nn::os::GetSystemTick() - disTimeout).ToTimeSpan().GetMilliSeconds() < WlanTestDisconnectTimeOut);

                nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             DIS CONNECTED!!!\n");
                    break;
                }
            }

            TraceState(&localClient);
        }
    }; // Class LocalClientStressTest


    // 同一Masterに1時間連続で接続と切断を繰り返し
    TEST_F(LocalClientStressTest, L103)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" },
            true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // 1時間連続で接続/切断を行う
            Connect(ConnectParam);
            Disconnect();
        }
    }

    // Masterがn台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンをn時間連続通信 CLIENT
    // Masterがn台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンをn時間連続通信とClientのリンク維持 CLIENT
    TEST_F(LocalClientStressTest, L109_L110)
    {
        int64_t agingTime = Time8h;
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::LinkLevel level;
        nn::Result result;

        // 通信評価定義
        const uint64_t frameByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t frameByteMin = frameByteNormal - (frameByteNormal * 0.1);
        const uint64_t frameByteMax = frameByteNormal + (frameByteNormal * 0.1);

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, nullptr,
            0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest3SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key },
            true, nn::wlan::BeaconIndication_Enable, 10
        };

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        static NN_ALIGNAS(4096) uint8_t testApiCallStack[threadStackSize];
        connectLogApiStart = connectStsApiStart = linkApiStart = nn::os::GetSystemTick();
        isConnectStsCheck = isConnectLogCheck = isLinkApiCheck = false;

        isCheck = true;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&testCheckThread,
                TestApiCallTimeThread, this, testApiCallStack, threadStackSize,
                nn::os::DefaultThreadPriority));
        nn::os::StartThread(&testCheckThread);
#endif

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security,
                        TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize,
                        1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize,
                        agingTime, Interval_16m, true));
        StartUnsyncScanThread(StartScanTestParams, 1000);

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // 一定周期送信。通信が切断されたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time600s)
            {
#ifdef WLAN_TEST_LINKLEVEL_ERROR
                isConnectStsCheck = isLinkApiCheck = false;
                isConnectLogCheck = true;
                connectLogApiStart = nn::os::GetSystemTick();
#endif
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }

#ifdef WLAN_TEST_LINKLEVEL_ERROR
            isConnectLogCheck = isConnectStsCheck = false;
            isLinkApiCheck = true;
            linkApiStart = nn::os::GetSystemTick();
#endif
#ifdef WLAN_TEST_LINKLEVEL_WAIT
            result = nn::wlan::Local::GetLinkLevel(&level);
            if (result.IsSuccess() != true)
            {

#ifdef WLAN_TEST_LINKLEVEL_ERROR
                isConnectLogCheck = isLinkApiCheck = false;
                isConnectStsCheck = true;
                connectStsApiStart = nn::os::GetSystemTick();
#endif
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 != level);
                }
            }
#endif
        }
#ifdef WLAN_TEST_LINKLEVEL_ERROR
        isCheck = false;
        nn::os::WaitThread(&testCheckThread);
        nn::os::DestroyThread(&testCheckThread);
#endif
        localClient.LocalTraceConnectsInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        StopUnsyncScanThread();
        // 1秒待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

        Disconnect();

        // フレーム送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L109_L110_2 Frame Send Total Size:%lld(Range:%lld - %lld)\n",
                    connectsInfo.totalSendSize, frameByteMin, frameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(frameByteMin, connectsInfo.totalSendSize),
                    frameByteMax) == connectsInfo.totalSendSize);
    }

}; // namespace WlanTest
