﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Include/testWlan_LocalInfraStressTest.h"

extern "C"
{
    int testMain(int argc, char **argv);
    void cleanup(void);
}

namespace {
    #define WLANTEST_ACTIVESCAN         nn::wlan::ScanType::ScanType_Active
    #define WLANTEST_PASSIVESCAN        nn::wlan::ScanType::ScanType_Passive
    #define WLANTEST_BROADCASTADDRESS   nn::wlan::MacAddress::CreateBroadcastMacAddress()
    #define WLAN_INTERFACE_NAME "wl0"
    #define STATIC_IP           false
    #define STATIC_SUBNET_MASK  "255.255.255.0"
    #define STATIC_DNS_1        "8.8.8.8"
    #define STATIC_DNS_2        "8.8.8.9"

    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    const uint16_t  PortNo2 = 5000;
    const int Echoserverbacklog = 5;

    WlanTest::ScanTestPattern scanTestParam[] = {
        // N1
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N2
        { BufferSize100k,
        { WLANTEST_PASSIVESCAN,{ 1 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N3
        { BufferSize100k,
        { WLANTEST_PASSIVESCAN,{ 36 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N6
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 13 }, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N12以降
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 1, 6, 11 }, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N117
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 1 }, 1, 100, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N118,8,9,10,11
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 1 }, 0, 100, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N15
#ifdef SDEVEDEV
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 48 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
#else
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 140 }, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
#endif
        // N49
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13 }, 13, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        { BufferSize100k,
        { WLANTEST_ACTIVESCAN,{ 36, 40, 44, 48 }, 4, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
        // N129
        { BufferSize100k,
        { WLANTEST_PASSIVESCAN,{ 1 }, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS } },
    };

    enum SCANTESTKIND
    {
        SCAN_N1_TEST,
        SCAN_N2_TEST,
        SCAN_N3_TEST,
        SCAN_N6_TEST,
        SCAN_N12_TEST,
        SCAN_N117_TEST,
        SCAN_N118_TEST,
        SCAN_N15_TEST,
        SCAN_N49_1_TEST,
        SCAN_N49_2_TEST,
        SCAN_N129_TEST,
    };

    bool config_status = false;
};

namespace WlanTest{
    class TCPConnecter
    {
    protected:
        int sockOpen;
        int sockTcp;
        const int Bufsize = 128 * 1024;
        std::unique_ptr<char[]> sendBuf;
        std::unique_ptr<char[]> recvBuf;

    public:
        TCPConnecter() NN_NOEXCEPT
        {
            std::unique_ptr<char[]> sendmessage(new char[Bufsize]);
            sendBuf = std::move(sendmessage);

            std::unique_ptr<char[]> recvmessage(new char[Bufsize]);
            recvBuf = std::move(recvmessage);
        }

        void ConfigureInterface(const char* interfaceName, bool enable) NN_NOEXCEPT
        {
            if (true == config_status) return;
            nn::Result result;
            struct ifreq ifr = { WLAN_INTERFACE_NAME };
            ifr.ifr_flags = IFF_UP;
            int socket = nn::socket::Socket(PF_INET, SOCK_DGRAM, 0);
            NN_ABORT_UNLESS(socket >= 0);
            nn::socket::Ioctl(socket, SIOCSIFFLAGS, &ifr, sizeof(struct ifreq));

            if (enable)
            {
                nn::bsdsocket::cfg::IfSettings ifcfg;
                memset(&ifcfg, 0, sizeof(ifcfg));

#if STATIC_IP
                // static ip
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                ifcfg.mtu = 1500;
                nn::socket::InetAton(IpAdder, &ifcfg.u.modeStatic.addr);
                nn::socket::InetAton(GWIpAdder, &ifcfg.u.modeStatic.gatewayAddr);
                nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
                ifcfg.u.modeStatic.broadcastAddr.s_addr =
                    (ifcfg.u.modeStatic.addr.s_addr & ifcfg.u.modeStatic.subnetMask.s_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.s_addr;
                nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
                nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
#else
                // dhcp
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                ifcfg.mtu = 1500;
#endif

                result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
                if (result.IsFailure())
                {
                    NN_LOG("             failed to configure interface %s - %d:%d\n",
                        interfaceName,
                        result.GetModule(),
                        result.GetDescription());
                }
                else {
                    config_status = true;
                }
            }
            else
            {
                nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
            }
        }

        static unsigned long inet_addr(const char* cp) NN_NOEXCEPT
        {
            struct in_addr      val;

            if (nn::socket::InetAton(cp, &val))
            {
                return val.s_addr;
            }
            else
            {
                NN_LOG("             IP Addr err\n");
                return INADDR_NONE;
            }
        }

        void Connect() NN_NOEXCEPT
        {
            int ret;

            ConfigureInterface(WLAN_INTERFACE_NAME, true);
            struct timeval tv;
            memset(&tv, 0, sizeof(tv));
            tv.tv_sec = 20000;

            int bufsize = 128 * 1024;
            int time_wait = 1;

            /* ソケットを作成します */
            sockOpen = nn::socket::Socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
            if (0 > sockOpen)
            {
                NN_LOG("             Socket error : %d\n", sockOpen);
                return;
            }

            // TIME_WAIT状態対策
            ret = nn::socket::SetSockOpt(sockOpen, SOL_SOCKET, SO_REUSEADDR,
                (const char *)&time_wait, sizeof(time_wait));
            if (0 > ret)
            {
                NN_LOG("             SetSockOpt ReuseAddr error : %d\n", nn::socket::GetLastErrno());
                Disconnect();
                return;
            }

            // 送信タイムアウトを設定
            ret = nn::socket::SetSockOpt(sockOpen, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
            if (0 > ret)
            {
                NN_LOG("             SetSockOpt time error : %d\n", nn::socket::GetLastErrno());
                Disconnect();
                return;
            }

            // 受信タイムアウトを設定
            ret = nn::socket::SetSockOpt(sockOpen, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
            if (0 > ret)
            {
                NN_LOG("             SetSockOpt time error : %d\n", nn::socket::GetLastErrno());
                Disconnect();
                return;
            }

            // 送信サイズを設定
            ret = nn::socket::SetSockOpt(sockOpen, SOL_SOCKET, SO_SNDBUF,
                (char*)&bufsize, sizeof(bufsize));
            if (0 > ret)
            {
                NN_LOG("             SetSockOpt bufsize error : %d\n", nn::socket::GetLastErrno());
                Disconnect();
                return;
            }

            // 受信サイズを設定
            ret = nn::socket::SetSockOpt(sockOpen, SOL_SOCKET, SO_RCVBUF,
                (char*)&bufsize, sizeof(bufsize));
            if (0 > ret)
            {
                NN_LOG("             SetSockOpt bufsize error : %d\n", nn::socket::GetLastErrno());
                Disconnect();
                return;
            }
        }

        void Disconnect() NN_NOEXCEPT
        {
            // ソケットを解放.
            if (0 <= sockTcp)
            {
                nn::socket::Shutdown(sockTcp, SHUT_RDWR);
                sockTcp = nn::socket::Close(sockTcp);
                if (0 > sockTcp)
                {
                    NN_LOG("             Close error : %d\n", nn::socket::GetLastErrno());
                }
                sockTcp = 0;
            }

            // ソケットを解放.
            if (0 <= sockOpen)
            {
                nn::socket::Shutdown(sockOpen, SHUT_RDWR);
                sockOpen = nn::socket::Close(sockOpen);
                if (0 > sockOpen)
                {
                    NN_LOG("             Close error : %d\n", nn::socket::GetLastErrno());
                }
                sockOpen = 0;
            }
            config_status = false;
        }

        void Send(int wait_time) NN_NOEXCEPT
        {
            int ret;
            int err_cnt = 0;
            sockaddr_in server = { 0 };

            server.sin_family = AF_INET;
            server.sin_port = nn::socket::InetHtons(PortNo);
            in_addr outAddr = { 0 };
            nn::socket::InetAton(IpAdder, &outAddr);
            server.sin_addr = outAddr;

            ret = nn::socket::Connect(sockOpen, reinterpret_cast<sockaddr *>(&server), sizeof(server));
            if (0 > ret)
            {
                NN_LOG("             Connect error : %d\n", nn::socket::GetLastErrno());
                return;
            }

            while (NN_STATIC_CONDITION(1))
            {
                if (0 == sockOpen) break;
                int len = nn::socket::Send(sockOpen, (const char*)sendBuf.get(), Bufsize, 0);
                if (0 > len)
                {
                    NN_LOG("             SendTo error : %d\n", nn::socket::GetLastErrno());
                    err_cnt++;
                    if (100 <= err_cnt) break;
                }
                if (0 < wait_time)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wait_time));
                }
            }
        }

        void Receive() NN_NOEXCEPT
        {
            int ret;
            sockaddr_in client = { 0 };
            sockaddr_in addr = { 0 };
            socklen_t len;

            addr.sin_family = AF_INET;
            addr.sin_port = nn::socket::InetHtons(PortNo2);
            addr.sin_addr.s_addr = nn::socket::InetHtonl(INADDR_ANY);

            // サーバーソケットに名前をつける
            ret = nn::socket::Bind(sockOpen, reinterpret_cast<sockaddr *>(&addr), sizeof(addr));
            if (0 > ret)
            {
                NN_LOG("             Bind error : %d\n", nn::socket::GetLastErrno());
                return;
            }

            // ソケットを受信待機モードにし，保留接続キューのサイズを確保
            ret = nn::socket::Listen(sockOpen, Echoserverbacklog);
            if (0 > ret)
            {
                NN_LOG("             Listen error : %d\n", nn::socket::GetLastErrno());
                return;
            }
            len = sizeof(client);

            // サーバーソケットを接続要求待ち状態
            sockTcp = nn::socket::Accept(sockOpen, reinterpret_cast<sockaddr *>(&client), &len);
            if (0 > sockTcp)
            {
                NN_LOG("             Server Socket error : %d\n", nn::socket::GetLastErrno());
                return;
            }

            // 受信処理.
            int status;
            int err_cnt = 0;
            while (NN_STATIC_CONDITION(1))
            {
                if (0 == sockTcp) break;
                status = nn::socket::Recv(sockTcp, recvBuf.get(), Bufsize, 0);
                if (0 == status)
                {
                    NN_LOG("             Disconnect\n");
                    break;
                }
                else if (0 > status)
                {
                    NN_LOG("             Recv failed (error %d)\n", nn::socket::GetLastErrno());
                    err_cnt++;
                    if (100 <= err_cnt) {
                        break;
                    }
                }
            }
        }
    };

    TCPConnecter tcp_Send;
    TCPConnecter tcp_Recv;
    class InfraStressTest : public ::testing::Test
    {
    protected:
        nn::os::SystemEventType connectionEvent;
        nn::wlan::ConnectionStatus connectionStatus;
        const int KLocalConcurrencyLimit = 4;
        nn::os::Tick sysTick;

        nn::os::ThreadType scanStopThread;
        nn::os::EventType scanStopEvent;

        nn::os::ThreadType connectCancelThread;
        nn::os::EventType connectCancelEvent;

        nn::os::ThreadType agingn117Thread;
        nn::os::EventType agingn117Event;

        nn::os::ThreadType agingn118Thread;
        nn::os::EventType agingn118Event;

        nn::os::ThreadType tcpThread;
        nn::os::EventType tcpEvent;

        nn::os::ThreadType tcpSendThread;

        nn::os::ThreadType tcpRecvThread;

        nn::os::ThreadType iperfServerThread;
        nn::os::EventType iperfServerEvent;

        nn::os::ThreadType iperfClientThread;
        nn::os::EventType iperfClientEvent;

        nn::os::ThreadType wlanStateThread;

        nn::os::ThreadType wlanConnectStateThread;

        int connectionTime = 0;
        const int IPaddresstime = 3000;

        nn::wlan::WlanState State;

        static const int Time5m = 5;
        static const int Time16m = 16;
        static const int Time20m = 20;
        static const int Time100m = 100;
        static const int Time300m = 300;
        static const int Time1s = 1000;
        static const int Time3s = Time1s * 3;
        static const int Time5s = Time1s * 5;
        static const int Time10s = Time1s * 10;
        static const int Time15s = Time1s * 15;
        static const int Time20s = Time1s * 20;
        static const int Time60s = Time1s * 60;
        static const int Time1h = 3600000;
        static const int Time4h = 3600000 * 4;
        static const int Time12h = 3600000 * 12;

        unsigned oldRandtime = 0;
        bool confSet = false;

        bool loopTime = false;
        const bool IsEnabled = true;

        int64_t agingTime = 0;
    protected:
        InfraStressTest() NN_NOEXCEPT
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Infra);

            agingTime = GetAgintTime();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm",
                    "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(
                reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                KLocalConcurrencyLimit));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            if ((nn::wlan::WlanState_InfraIdle != State) && (nn::wlan::WlanState_InfraIdleScan != State))
            {
                NN_LOG("              Wlan State:%d\n", State);
                WLANTEST_ASSERT_TRUE(false);
            }

            confSet = false;
            WlanState();
            loopTime = true;
#ifdef WLAN_TEST_STATE_STOP
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&wlanStateThread, WlanStateThread,
                    &loopTime, ThreadStack3, ThreadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&wlanStateThread);
#endif
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
#ifdef WLAN_TEST_STATE_STOP
            loopTime = false;
            nn::os::WaitThread(&wlanStateThread);
            nn::os::DestroyThread(&wlanStateThread);
#endif
            nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(WLAN_INTERFACE_NAME));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Finalize());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
        }

        static void DumpScanResultWithReader(void* pBuf) NN_NOEXCEPT
        {
            if (pBuf == nullptr)
            {
                return;
            }

            nn::wlan::BeaconScanResultReader resultReader(pBuf);
            uint32_t bssCount = resultReader.GetCount();
            char macStr[nn::wlan::MacAddress::MacStringSize];
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
            const nn::wlan::VendorInfoElementReader* vie;
            nn::Bit8 matchingOui1[nn::wlan::MacAddress::OuiSize] = { 0x00, 0x50, 0xf2 };
            nn::Bit8 matchingOui2[nn::wlan::MacAddress::OuiSize] = { 0x00, 0x07, 0x40 };
            uint8_t matchType = 1;

            NN_LOG("\n             ***** SCAN RESULT ***** \n");
            NN_LOG("             BSS count : %d\n", bssCount);
            NN_LOG("             Scan Result Total Length : %d\n", resultReader.GetTotalResultLength());

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();

                NN_LOG("             -----------------------%d BSS------------------------\n", i + 1);
                NN_LOG("             BSS size      : %d\n", beacon.GetLength());
                NN_LOG("             IE count      : %d\n", beacon.GetIeCount());
                NN_LOG("             Channel       : %d\n", beacon.GetChannel());
                NN_LOG("             RSSI          : %d[dBm]\n", beacon.GetRssi());
                NN_LOG("             LINKLEVEL     : %d\n", beacon.GetLinkLevel());
                NN_LOG("             Beacon Period : %d[ms]\n", beacon.GetInterval());
                NN_LOG("             BSSID : %s\n", beacon.GetBssid().GetString(macStr));
                NN_LOG("             SSID : %s\n", beacon.GetSsid().GetHexString(ssidStr));

                // WPA/WPA2
                if (beacon.IsWpaSupported() == true)
                {
                    const nn::wlan::WpaInfoElementReader<nn::wlan::WifiInfoElementReader>* pWpa
                        = beacon.GetWpaIe();

                    NN_LOG("                 == WPA: version[%d]\n", pWpa->GetVersion());
                    for (uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++)
                    {
                        NN_LOG("                    PairCipher(%d) %s\n", k, CipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
                    }
                    NN_LOG("                    GroupCipher %s\n", CipherTypeString[pWpa->GetGroupCipherSuite()]);
                    NN_LOG("                    AKM %s\n", AkmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
                }
                if (beacon.IsWpa2Supported())
                {
                    const nn::wlan::WpaInfoElementReader<nn::wlan::InfoElementReader>* pWpa = beacon.GetWpa2Ie();

                    NN_LOG("                 == WPA2: version[%d]\n", pWpa->GetVersion());
                    for (uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++)
                    {
                        NN_LOG("                    PairCipher(%d) %s\n", k, CipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
                    }
                    NN_LOG("                    GroupCipher %s\n", CipherTypeString[pWpa->GetGroupCipherSuite()]);
                    NN_LOG("                    AKM %s\n", AkmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
                }

                // VIE matching
                vie = beacon.GetVendorIeWithOui(matchingOui1);
                if (vie != nullptr)
                {
                    NN_LOG("             --- OUI matched!\n");
                    NN_LOG("             IE element id : %d\n", vie->GetElementId());
                    NN_LOG("             IE length     : %d\n", vie->GetLength());
                    NN_LOG("             IE OUI        : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
                }
                else
                {
                    NN_LOG("             --- OUI not matched...\n");
                }
                vie = beacon.GetVendorIeWithOuiAndType(matchingOui2, matchType);
                if (vie != nullptr)
                {
                    NN_LOG("             --- OUI and type matched!\n");
                    NN_LOG("             IE element id : %d\n", vie->GetElementId());
                    NN_LOG("             IE length     : %d\n", vie->GetLength());
                    NN_LOG("             IE OUI        : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
                    NN_LOG("             IE type       : %d\n", vie->GetVendorData()[0]);
                }
                else
                {
                    NN_LOG("             --- OUI and type not matched...\n");
                }
            }
        }

        void WlanState()
        {
            nn::wlan::WlanState State;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            NN_LOG("             %s\n", WlanStateStr[State]);
        }

        void ElapsedTimeChk(int eltim) NN_NOEXCEPT
        {
            if ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > eltim)
            {
                // 指定秒以上経過はNGとする
                NN_LOG("             Connect Time : %d msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
                NN_LOG("             Connect Time : %d msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
            }
        }

        void Connect(APKIND Apkind, bool timechk = false, int time = Time5s) NN_NOEXCEPT
        {
            nn::os::Tick sysTick2;
            int count = 0;
            memset(IpAdder, NULL, sizeof(IpAdder));
            memset(GWIpAdder, NULL, sizeof(GWIpAdder));
            std::strcpy(IpAdder, &IPAddr[Apkind][0]);
            std::strcpy(GWIpAdder, &IPAddr[Apkind][0]);
            char *GWstr = &GWIpAdder[0];
            connectionTime = time;
            for (int i = 0; i < 3; i++)
            {
                GWstr = std::strchr(GWstr, '.');
                GWstr++;
                if (i == 2)
                {
                    memset(GWstr, 0, 3);
                    *GWstr = '1';
                }
            }

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
            while (NN_STATIC_CONDITION(1))
            {
                NN_LOG("             WiFi Connect Start\n");
                if (true == timechk)
                {
                    sysTick = nn::os::GetSystemTick();
                }
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[Apkind].ssid,
                    ConnectParams[Apkind].bssid,
                    ConnectParams[Apkind].channel,
                    ConnectParams[Apkind].security,
                    ConnectParams[Apkind].autoKeepAlive,
                    ConnectParams[Apkind].beaconLostTimeout));

                NN_LOG("             WiFi Connected\n");

                nn::os::WaitSystemEvent(&connectionEvent);
                NN_LOG("             WiFi WaitSystemEvent Signal\n");

                WlanState();
                sysTick2 = nn::os::GetSystemTick();
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                if ((nn::os::GetSystemTick() - sysTick2).ToTimeSpan().GetMilliSeconds() > Time100m)
                {
                    NN_LOG("             Connection Status 100msec over!!\n");
                    WLANTEST_ASSERT_TRUE(false);
                }
                if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    if (true == timechk)
                    {
                        ElapsedTimeChk(connectionTime);  // 接続までの時間チェック
                    }
                    NN_LOG("             WlanTest: CONNECTED!!!\n");
                    break;
                }
                else
                {
                    NN_LOG("             WlanTest: NOT CONNECTED!!! STATE %d\n", connectionStatus.state);
                    PrintConnectionStatus(&connectionStatus);
                    count++;
                    if (100 <= count) {
                        nn::os::DestroySystemEvent(&connectionEvent);
                        NN_LOG("             WlanTest: 100 failures\n");
                        WLANTEST_ASSERT_TRUE(false);
                        return;
                    }
                }

                // 3秒待って再試行
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
            }
            nn::os::DestroySystemEvent(&connectionEvent);
        }

        void Disconnect() NN_NOEXCEPT
        {
            nn::os::Tick sysTick2;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
            while (NN_STATIC_CONDITION(1))
            {
                // イベントシグナル状態の確認
                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true)
                {
                    WlanState();
                    sysTick2 = nn::os::GetSystemTick();
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                    if ((nn::os::GetSystemTick() - sysTick2).ToTimeSpan().GetMilliSeconds() > Time100m)
                    {
                        NN_LOG("             Connection Status 100msec over!!\n");
                        WLANTEST_ASSERT_TRUE(false);
                    }
                    if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        NN_LOG("             WlanTest: DIS CONNECTED!!!\n");
                        break;
                    }
                }
            }
            nn::os::DestroySystemEvent(&connectionEvent);
        }

        void ConfigureInterface(const char* interfaceName, bool enable) NN_NOEXCEPT
        {
            if (true == confSet)
            {
                return;
            }

            nn::Result result;
            struct ifreq ifr = { WLAN_INTERFACE_NAME };
            ifr.ifr_flags = IFF_UP;
            int socket = nn::socket::Socket(PF_INET, SOCK_DGRAM, 0);
            NN_ABORT_UNLESS(socket >= 0);
            nn::socket::Ioctl(socket, SIOCSIFFLAGS, &ifr, sizeof(struct ifreq));

            if (enable)
            {
                nn::bsdsocket::cfg::IfSettings ifcfg;
                memset(&ifcfg, 0, sizeof(ifcfg));

#if STATIC_IP
                // static ip
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                ifcfg.mtu = 1500;
                nn::socket::InetAton(IpAdder, &ifcfg.u.modeStatic.addr);
                nn::socket::InetAton(GWIpAdder, &ifcfg.u.modeStatic.gatewayAddr);
                nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
                ifcfg.u.modeStatic.broadcastAddr.s_addr =
                    (ifcfg.u.modeStatic.addr.s_addr & ifcfg.u.modeStatic.subnetMask.s_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.s_addr;
                nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
                nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
#else
                // dhcp
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                ifcfg.mtu = 1500;
#endif

                result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
                if (result.IsFailure())
                {
                    NN_LOG("             failed to configure interface %s - %d:%d\n",
                        interfaceName,
                        result.GetModule(),
                        result.GetDescription());
                }
            }
            else
            {
                nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
            }
            confSet = true;
        }

        void PrintConnectionStatus(nn::wlan::ConnectionStatus* pStatus) NN_NOEXCEPT
        {
            char macStr[nn::wlan::MacAddress::MacStringSize];
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];

            NN_LOG("             +++ Connection Status +++\n");
            NN_LOG("             STATE           : %s\n", WlanConnectionStateStr[pStatus->state]);
            NN_LOG("             CAUSE           : %d\n", pStatus->cause);
            NN_LOG("             CHANNEL         : %d\n", pStatus->channel);
            NN_LOG("             SSID            : %s\n", pStatus->ssid.GetHexString(ssidStr));
            NN_LOG("             BSSID           : %s\n", pStatus->bssid.GetString(macStr));
            NN_LOG("             AID             : %d\n", pStatus->aid);
            NN_LOG("             REASON CODE     : %d\n", pStatus->statusReasonCode);
            NN_LOG("             CAPABILITY INFO : 0x%04X\n", pStatus->capabilityInfo);
            NN_LOG("             BEACON INTERVAL : %d\n", pStatus->beaconInterval);
            NN_LOG("             +++++++++++++++++++++++++\n");
        }

        static void StopScanThread(void* arg) NN_NOEXCEPT
        {
            nn::os::EventType* pStartEvent = static_cast<nn::os::EventType*>(arg);
            bool isStartEvent;
            int32_t ThreadWaitTime = 10;
            int32_t ThreadTimeOut = 30000 / ThreadWaitTime;
            int32_t ScanWaintTime = 100;

            for (int tmVaue = 0; tmVaue < ThreadTimeOut; tmVaue += ThreadWaitTime)
            {
                isStartEvent = nn::os::TimedWaitEvent(pStartEvent, nn::TimeSpan::FromMilliSeconds(ThreadWaitTime));
                if (isStartEvent != true)
                {
                    continue;
                }

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ScanWaintTime));
                NN_LOG("             StopScan Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::StopScan());
                NN_LOG("             StopScan End\n");

                nn::os::ClearEvent(pStartEvent);
                break;
            }
        }

        static void CancelConnectThread(void* arg) NN_NOEXCEPT
        {
            nn::os::EventType* pStartEvent = static_cast<nn::os::EventType*>(arg);
            bool isStartEvent;
            int32_t threadWaitTime = 10;
            int32_t threadTimeOut = 30000 / threadWaitTime;
            int32_t ConnectWaintTime = 100;
            nn::wlan::ConnectionStatus connectionStatus;

            for (int tmVaue = 0; tmVaue < threadTimeOut; tmVaue += threadWaitTime)
            {
                isStartEvent = nn::os::TimedWaitEvent(pStartEvent, nn::TimeSpan::FromMilliSeconds(threadWaitTime));
                if (isStartEvent != true)
                {
                    continue;
                }

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ConnectWaintTime));
                NN_LOG("             WIFI Chancel Connect Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::CancelConnect());
                NN_LOG("             WIFI Chancel Connect End\n");
                nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             WlanTest: CHANCEL OK!!!\n");
                    break;
                }

                nn::os::ClearEvent(pStartEvent);
                break;
            }
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            int randRet;
            if (0 == oldRandtime)
            {
                oldRandtime = time(NULL);
                srand(static_cast<unsigned int>(oldRandtime));
            }
            if (oldRandtime != time(NULL))
            {
                oldRandtime = time(NULL);
                srand(static_cast<unsigned int>(oldRandtime));
            }
            randRet = min + (int)(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            NN_LOG("             rand : %d\n", randRet);
            return randRet;
        }

        static unsigned long inet_addr(const char* cp) NN_NOEXCEPT
        {
            struct in_addr      val;

            if (nn::socket::InetAton(cp, &val))
            {
                return val.s_addr;
            }
            else
            {
                NN_LOG("             IP Addr err\n");
                return INADDR_NONE;
            }
        }

        static unsigned short checksum(unsigned short *buf, int bufsz) NN_NOEXCEPT
        {
            unsigned long sum = 0;

            while (bufsz > 1)
            {
                sum += *buf;
                buf++;
                bufsz -= 2;
            }

            if (bufsz == 1)
            {
                sum += *reinterpret_cast<unsigned char *>(buf);
            }

            sum = (sum & 0xffff) + (sum >> 16);
            sum = (sum & 0xffff) + (sum >> 16);

            return ~sum;
        }

        bool PingStart(int data_size = 0, uint16_t icd_seq = 0) NN_NOEXCEPT
        {
            int sock;
            sockaddr_in addr;
            int n;
            int ret = true;

            struct timeval tv;
            memset(&tv, 0, sizeof(tv));
            tv.tv_sec = 30;

            struct packet {
                icmp hdr;
                char data[ping_data_size_1023];
            };
            packet senddata;
            int data_len = 0;

            if (0 < data_size)
            {
                data_len = sizeof(senddata.hdr) + data_size;
            }
            else
            {
                data_len = sizeof(senddata.hdr);
            }

            char buf[2000];
            struct icmp *icmpptr;
            struct ip *iphdrptr;

            NN_LOG("             PING\n");
            ConfigureInterface(WLAN_INTERFACE_NAME, true);
            addr.sin_family = AF_INET;
            addr.sin_addr.s_addr = inet_addr(IpAdder);
            NN_LOG("             IP Addr : %s\n", IpAdder);

            /* RAWソケットを作成します */
            sock = nn::socket::Socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
            if (sock < 0)
            {
                NN_LOG("             Socket error : %d\n", sock);
                return false;
            }

            // 受信タイムアウトを設定
            n = nn::socket::SetSockOpt(sock, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
            if (0 > n)
            {
                NN_LOG("             SetSockOpt error : %d\n", nn::socket::GetLastErrno());
                sock = nn::socket::Close(sock);
                if (0 > sock)
                {
                    NN_LOG("             Close error : %d\n", sock);
                }
                return false;
            }

            memset(&senddata.hdr, 0, sizeof(senddata.hdr));

            /* ICMPヘッダ */
            senddata.hdr.icmp_type = ICMP_ECHO;
            senddata.hdr.icmp_code = 0;
            senddata.hdr.icmp_cksum = 0;
            senddata.hdr.icmp_hun.ih_idseq.icd_id = 0;
            senddata.hdr.icmp_hun.ih_idseq.icd_seq = icd_seq;

            /* Data */
            if (0 < data_size)
            {
                memcpy(senddata.data, ping_data, data_size);
            }

            /* ICMPヘッダのチェックサムを計算 */
            senddata.hdr.icmp_cksum = checksum(reinterpret_cast<unsigned short *>(&senddata.hdr), data_len);

            /* ICMPヘッダだけのICMPパケットを送信 */
            n = nn::socket::SendTo(sock, (char *)&senddata, data_len, 0,
                    reinterpret_cast<sockaddr*>(&addr), sizeof(addr));
            if (n < 1)
            {
                NN_LOG("             SendTo error : %d\n", nn::socket::GetLastErrno());
            }

            /* ICMP ECHO REPLY受信部分 */
            memset(buf, 0, sizeof(buf));

            /* 相手ホストからのICMP ECHO REPLYを待ち */
            n = nn::socket::Recv(sock, buf, sizeof(buf), 0);
            if (n < 1)
            {
                NN_LOG("             Recv error : %d\n", nn::socket::GetLastErrno());
                ret = false;
            }

            /* 受信データからIPヘッダ部分へのポインタを取得 */
            iphdrptr = (struct ip *)buf;

            /* 受信データからICMPヘッダ部分へのポインタを取得 */
            icmpptr = (struct icmp *)(buf + (iphdrptr->ip_hl * 4));

            /* ICMPヘッダからICMPの種類を特定 */
            if (true == ret)
            {
                if (icmpptr->icmp_type == ICMP_ECHOREPLY)
                {
                    NN_LOG("             received ICMP ECHO REPLY\n");

                    if ((0 < icd_seq) && (icd_seq != icmpptr->icmp_hun.ih_idseq.icd_seq))
                    {
                        NN_LOG("             Seq No error Send Seq : %d, Req Seq : &d\n",
                                icd_seq, icmpptr->icmp_hun.ih_idseq.icd_seq);
                        ret = false;
                    }
                }
                else
                {
                    NN_LOG("             received ICMP %d\n", icmpptr->icmp_type);
                    ret = false;
                }
            }

            /* 終了 */
            sock = nn::socket::Close(sock);
            if (0 > sock)
            {
                NN_LOG("             Close error : %d\n", sock);
            }
            return ret;
        }

        static void TcpSend(void* arg) NN_NOEXCEPT
        {
            int* wait_time = static_cast<int*>(arg);
            tcp_Send.Send(*wait_time);
        }

        static void TcpRecv(void* arg) NN_NOEXCEPT
        {
            tcp_Recv.Receive();
        }

        static void iperfServer(void* arg) NN_NOEXCEPT
        {
            int    argC;
            char** argV;
            const  char* defaultArguments[] = { "iperf", "-s", "-i", "10", "-w", "262144" };
            argV = const_cast<char**>(defaultArguments);
            argC = sizeof(defaultArguments) / sizeof(defaultArguments[0]);

            testMain(argC, argV);
        }

        static void iperfClient(void* arg) NN_NOEXCEPT
        {
            int    argC;
            char** argV;
            const  char* defaultArguments[] = { "iperf", "-c", IpAdder, "-w","262144", "-t", "28800", "-i", "10" };
            argV = const_cast<char**>(defaultArguments);
            argC = sizeof(defaultArguments) / sizeof(defaultArguments[0]);

            testMain(argC, argV);
        }

        static void WlanStateThread(void* arg) NN_NOEXCEPT
        {
            bool* loopTime = static_cast<bool*>(arg);
            nn::wlan::WlanState new_status;
            nn::wlan::WlanState old_status;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&old_status));
            while (NN_STATIC_CONDITION(1))
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&new_status));
                if (old_status != new_status)
                {
                    NN_LOG("             State : %s\n", WlanStateStr[new_status]);
                }
                old_status = new_status;

                if (false == *loopTime)
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time100m));
            }
        }

    };

    class InfraInitStressTest : public ::testing::Test
    {
    protected:
        const int KLocalConcurrencyLimit = 4;
        const bool IsEnabled = true;

        nn::wlan::WlanState State;
        int64_t agingTime = 0;

    protected:
        InfraInitStressTest() NN_NOEXCEPT
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Infra);

            agingTime = GetAgintTime();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                KLocalConcurrencyLimit));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(WLAN_INTERFACE_NAME));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Finalize());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
        }

        bool ConnectWait(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& bssid,
            int16_t channel, const nn::wlan::Security& security, bool autoKeepAlive,
            int beaconLostTimeout = 10, int64_t msecWaitTimeout = WiFiConnectTimeout) NN_NOEXCEPT
        {
            const int64_t sleepTime = Time3s;

            nn::os::SystemEvent connectionEvent;
            nn::wlan::ConnectionStatus status;
            int64_t waitTime = std::min(msecWaitTimeout, sleepTime);
            nn::os::Tick sysTick;
            nn::Result result;
            bool isConnect = false;

            nn::wlan::Infra::GetConnectionEvent(connectionEvent.GetBase());
            sysTick = nn::os::GetSystemTick();

            while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= msecWaitTimeout)
            {
                result = nn::wlan::Infra::Connect(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             WiFi Connect Error Code(0x%08x) %s %d Line\n",
                        result.GetInnerValueForDebug(), __FILE__, __LINE__);
                    InfraTraceConnectInfo(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
                    break;
                }

                if (connectionEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(waitTime)) != true)
                {
                    continue;
                }

                result = nn::wlan::Infra::GetConnectionStatus(&status);
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             WiFi Connect Status Error Code(0x%08x) %s %d Line\n",
                        result.GetInnerValueForDebug(), __FILE__, __LINE__);
                    InfraTraceConnectInfo(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
                    break;
                }

                if (status.state == nn::wlan::ConnectionState_Connected)
                {
                    isConnect = true;
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitTime));
            }

            if (isConnect != true && result.IsSuccess() == true)
            {
                NN_LOG("             WiFi Connect Time Out %s %d\n",
                    __FILE__, __LINE__);
                InfraTraceConnectInfo(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
            }
            return isConnect;
        }

        bool Disconnect() NN_NOEXCEPT
        {
            nn::os::Tick waitTick, timeoutTick;
            nn::os::SystemEvent conEvent;
            nn::wlan::ConnectionStatus status;
            bool isDisconnect = false;
            nn::Result result;

            nn::wlan::Infra::GetConnectionEvent(conEvent.GetBase());
            if (result.IsSuccess() != true)
            {
                NN_LOG("             GetConnectionEvent Status Error Code(0x%08x) %s %d Line\n",
                    result.GetInnerValueForDebug(), __FILE__, __LINE__);
                return isDisconnect;
            }

            result = nn::wlan::Infra::GetConnectionStatus(&status);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             WiFi Connect Status Error Code(0x%08x) %s %d Line\n",
                    result.GetInnerValueForDebug(), __FILE__, __LINE__);
                return isDisconnect;
            }

            if (status.state != nn::wlan::ConnectionState_Connected)
            {
                return true;
            }

            result = nn::wlan::Infra::Disconnect();
            if (result.IsSuccess() != true)
            {
                NN_LOG("             WiFi Disconnect Error Code(0x%08x) %s %d Line\n",
                    result.GetInnerValueForDebug(), __FILE__, __LINE__);
                return isDisconnect;
            }

            waitTick    = nn::os::GetSystemTick();
            timeoutTick = waitTick;
            while ((nn::os::GetSystemTick() - timeoutTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
            {
                // イベントシグナル状態の確認
                if (conEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(Time20msec)) == true)
                {
                    waitTick = nn::os::GetSystemTick();
                    result = nn::wlan::Infra::GetConnectionStatus(&status);
                    if (result.IsSuccess() != true)
                    {
                        continue;
                    }

                    if ((nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds() > Time100msec)
                    {
                        NN_LOG("             Connection Status 100msec over!!\n");
                        WLANTEST_EXPECT_TRUE((nn::os::GetSystemTick() - waitTick).ToTimeSpan().GetMilliSeconds() > Time100msec);
                        break;
                    }

                    if (status.state != nn::wlan::ConnectionState_Connected)
                    {
                        NN_LOG("             WlanTest: DIS CONNECTED!!!\n");
                        isDisconnect = true;
                        break;
                    }
                }
            }

            if (result.IsSuccess() == true && isDisconnect != true)
            {
                NN_LOG("             Disconnect Status Wait TimeOut\n");
            }

            return isDisconnect;
        }
    };

    TEST_F(InfraInitStressTest, SleepWakeupTest) NN_NOEXCEPT
    {
        const nn::wlan::Ssid ssid(ApSetting2_1_Ssid);
        const nn::wlan::MacAddress mac = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const nn::wlan::Security seq = {
            nn::wlan::SecurityMode_Wpa2Aes, nn::wlan::SecurityMode_Wpa2Aes,
            0, "123456789012345678901234567890123456789012345678901234567890123"
        };

        nn::os::Tick startTick;
        startTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // AP接続 - スリープ/スリープ復帰試験
            NN_LOG("             SleepWakeupTest - WiFi-AP Connect Test Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
            // WiFi-AP接続
            WLANTEST_ASSERT_TRUE(ConnectWait(ssid, mac, -1, seq, true));

            // Sleepリクエスト
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::RequestSleep());
            // インフラステート取得
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            WLANTEST_EXPECT_TRUE(nn::wlan::WlanState_Sleep == State);

            // スリープ状態で待機
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time30s));

            // Wakeupリクエスト
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::RequestWakeUp());
            // インフラステート取得
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            WLANTEST_EXPECT_TRUE(nn::wlan::WlanState_Sleep != State);

            // WiFi-AP切断
            WLANTEST_ASSERT_TRUE(Disconnect());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
            NN_LOG("             SleepWakeupTest - WiFi-AP Connect Test End\n\n");


            // 非AP接続 - スリープ/スリープ復帰試験
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
            NN_LOG("             SleepWakeupTest - WiFi-AP No Connect Test Start\n");
            // Sleepリクエスト
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::RequestSleep());
            // インフラステート取得
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            WLANTEST_EXPECT_TRUE(nn::wlan::WlanState_Sleep == State);

            // スリープ状態で待機
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time30s));

            // Wakeupリクエスト
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::RequestWakeUp());
            // インフラステート取得
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            WLANTEST_EXPECT_TRUE(nn::wlan::WlanState_Sleep != State);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
            NN_LOG("             SleepWakeupTest - WiFi-AP No Connect Test End\n\n");
        }
    }


    // 全チャネルを指定時間連続してスキャンし続けることができること
    TEST_F(InfraStressTest, N111)
    {
        uint16_t count = 1;
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        sysTick = nn::os::GetSystemTick();
        while (NN_STATIC_CONDITION(1))
        {
            // 指定時間連続でScanを行う
            if ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
            {
                NN_LOG("             StartStan %dCnt\n", count);
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(),
                        scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
                DumpScanResultWithReader(pTestScanData.get());
                count++;
            }
            else
            {
                break;
            }
        }
        NN_LOG("             N111_2 is executed.\n");
        NN_LOG("             Before execution, change AP setting, add AP, turn off AP power\n");
    }

    // 同一のWPA2-PSK(AES)設定のAPに指定時間連続で接続と切断を繰り返し
    TEST_F(InfraStressTest, N112)
    {
        int cnt = 0;
        sysTick = nn::os::GetSystemTick();

        while (NN_STATIC_CONDITION(1))
        {
            // 指定時間連続で接続/切断を行う
            if ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
            {
                Connect(AP2_3_WPA2AES);
                Disconnect();
                cnt++;
                NN_LOG("             CNT %d\n", cnt);
                // 再接続に1～1000msecの間隔をあける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(1, 1000)));
            }
            else
            {
                break;
            }
        }
    }

    // WPA-PSK(TKIP)とWPA2-PSK(AES)設定のAPを交互に指定時間連続でスキャン、接続と切断、スキャンを繰り返せること
    TEST_F(InfraStressTest, N126)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        sysTick = nn::os::GetSystemTick();
        nn::os::Tick sysTick_20s = nn::os::GetSystemTick();
        int apkind = 0;
        bool time_chk = false;

        while (NN_STATIC_CONDITION(1))
        {
            // 指定時間連続で接続/切断を行う
            if ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
            {
                // Scan実施
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(),
                        scanTestParam[SCAN_N2_TEST].scanSize, scanTestParam[SCAN_N2_TEST].scanParam));

                if (1 == apkind)
                {
                    // 前回の接続開始から20秒以上であればNGとする
                    if (true == time_chk)
                    {
                        if ((nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds() >= Time20s)
                        {
                            NN_LOG("             Time:%d msec\n",
                                    (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds());
                            WLANTEST_ASSERT_TRUE(false);
                        }
                        sysTick_20s = nn::os::GetSystemTick();
                    }
                    NN_LOG("             WPA-PSK(TKIP)\n");
                    Connect(AP3_1_WPATKIP, true, Time15s);
                    apkind = 0;
                }
                else {
                    // 前回の接続開始から20秒以上であればNGとする
                    if (true == time_chk)
                    {
                        if ((nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds() >= Time20s)
                        {
                            NN_LOG("             Time:%d msec\n",
                                    (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds());
                            WLANTEST_ASSERT_TRUE(false);
                        }
                        sysTick_20s = nn::os::GetSystemTick();
                    }
                    else {
                        time_chk = true;
                    }
                    NN_LOG("             WPA2-PSK(AES)\n");
                    Connect(AP2_3_WPA2AES, true, Time15s);
                    apkind = 1;
                }
                Disconnect();
                // 再接続に1～20msecの間隔をあける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(1, 20)));
            }
            else
            {
                break;
            }
        }
    }

};
