﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <ctime>
#include <memory>

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/socket.h>
#include <nn/socket/socket_TypesPrivate.h>
#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"


NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C"
{
    int testMain(int argc, char **argv);
    void cleanup( void );
}

namespace {
    #define WLANTEST_ACTIVESCAN         nn::wlan::ScanType::ScanType_Active
    #define WLANTEST_PASSIVESCAN        nn::wlan::ScanType::ScanType_Passive
    #define WLANTEST_BROADCASTADDRESS   nn::wlan::MacAddress::CreateBroadcastMacAddress()
    #define WLAN_INTERFACE_NAME "wl0"
    #define STATIC_IP           false
    #define STATIC_SUBNET_MASK  "255.255.255.0"
    #define STATIC_DNS_1        "8.8.8.8"
    #define STATIC_DNS_2        "8.8.8.9"

    //#define WLAN_TEST_STATE_STOP // 100msec毎のGetStateを有効
    bool config_status = false;

    // ジョイコンペアリング
    bool g_IsShort = false;

    struct ScanTestPattern
    {
        uint32_t scanSize;
        nn::wlan::ScanParameters scanParam;
    };
    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    ScanTestPattern scanTestParam[] = {
        // N1
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {1}, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N2
        { BufferSize100k,
            {WLANTEST_PASSIVESCAN, {1}, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N3
        { BufferSize100k,
            {WLANTEST_PASSIVESCAN, {36}, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N6
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {13}, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N12以降
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {1, 6, 11}, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N117
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {1}, 1, 100, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N118,8,9,10,11
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {1}, 0, 100, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N15
#ifdef SDEVEDEV
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {48}, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
#else
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {140}, 1, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
#endif
        // N49
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13}, 13, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        { BufferSize100k,
            {WLANTEST_ACTIVESCAN, {36, 40, 44, 48}, 4, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
        // N129
        { BufferSize100k,
            {WLANTEST_PASSIVESCAN, {1}, 0, 120, -1, nullptr, 0, WLANTEST_BROADCASTADDRESS} },
    };

    enum SCANTESTKIND
    {
        SCAN_N1_TEST,
        SCAN_N2_TEST,
        SCAN_N3_TEST,
        SCAN_N6_TEST,
        SCAN_N12_TEST,
        SCAN_N117_TEST,
        SCAN_N118_TEST,
        SCAN_N15_TEST,
        SCAN_N49_1_TEST,
        SCAN_N49_2_TEST,
        SCAN_N129_TEST,
    };

    const size_t ThreadStackSize = 20480;
    const uint16_t  PortNo = 5001;
    const uint16_t  PortNo2 = 5000;
    const int Echoserverbacklog = 5;

    const int MaxPacketNum = ((60 * 60) * 5) * 1.10;
    const int MinPacketNum = ((60 * 60) * 5) * 0.90;

    NN_OS_ALIGNAS_THREAD_STACK char  ThreadStack[ThreadStackSize];
    NN_OS_ALIGNAS_THREAD_STACK char  ThreadStack2[ThreadStackSize];
#ifdef WLAN_TEST_STATE_STOP
    NN_OS_ALIGNAS_THREAD_STACK char  ThreadStack3[ThreadStackSize];
#endif

    enum GetChannel
    {
        GetChannel_1   = 1,
        GetChannel_36  = 36,
        GetChannel_48  = 48,
        GetChannel_52  = 52,
        GetChannel_56  = 56,
        GetChannel_100 = 100,
        GetChannel_128 = 128,
        GetChannel_140 = 140,
    };

    bool g_isBtSet = false;
};

namespace WlanTest{

    class TCPConnecter
    {
    protected:
        int sockOpen;
        int sockTcp;
        const int Bufsize = 128 * 1024;
        std::unique_ptr<char[]> sendBuf;
        std::unique_ptr<char[]> recvBuf;

    public:
        TCPConnecter() NN_NOEXCEPT
        {
            std::unique_ptr<char[]> sendmessage(new char[Bufsize]);
            sendBuf = std::move(sendmessage);

            std::unique_ptr<char[]> recvmessage(new char[Bufsize]);
            recvBuf = std::move(recvmessage);
        }

        void ConfigureInterface(const char* interfaceName, bool enable) NN_NOEXCEPT
        {
            if(true == config_status) return;
            nn::Result result;
            struct ifreq ifr = {WLAN_INTERFACE_NAME};
            ifr.ifr_flags = IFF_UP;
            int socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
            NN_ABORT_UNLESS(socket >= 0);
            nn::socket::Ioctl(socket, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocSIfFlags), &ifr, sizeof(struct ifreq));

            if (enable)
            {
                nn::bsdsocket::cfg::IfSettings ifcfg;
                memset(&ifcfg, 0, sizeof(ifcfg));

                #if STATIC_IP
                // static ip
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                ifcfg.mtu  = 1500;
                nn::socket::InetAton(IpAdder, &ifcfg.u.modeStatic.addr);
                nn::socket::InetAton(GWIpAdder, &ifcfg.u.modeStatic.gatewayAddr);
                nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
                ifcfg.u.modeStatic.broadcastAddr.S_addr =
                        (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
                            ~ifcfg.u.modeStatic.subnetMask.S_addr;
                nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
                nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
                #else
                // dhcp
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                ifcfg.mtu  = 1500;
                #endif

                result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
                if (result.IsFailure())
                {
                    NN_LOG("             failed to configure interface %s - %d:%d\n",
                               interfaceName,
                               result.GetModule(),
                               result.GetDescription());
                }
                else{
                    config_status = true;
                }
            }
            else
            {
                nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
            }
        }

        static unsigned long inet_addr(const char* cp) NN_NOEXCEPT
        {
            nn::socket::InAddr      val;

            if(nn::socket::InetAton(cp, &val))
            {
                return val.S_addr;
            }
            else
            {
                NN_LOG("             IP Addr err\n");
                return nn::socket::InAddr_None;
            }
        }

        void Connect() NN_NOEXCEPT
        {
            int ret;

            ConfigureInterface(WLAN_INTERFACE_NAME, true);
            nn::socket::TimeVal tv;
            memset(&tv, 0, sizeof(tv));
            tv.tv_sec = 20000;

            int bufsize = 128 * 1024;
            int time_wait = 1;

            /* ソケットを作成します */
            sockOpen = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            if(0 > sockOpen)
            {
                NN_LOG("             Socket error : %d\n", sockOpen);
                return;
            }

            // TIME_WAIT状態対策
            ret = nn::socket::SetSockOpt(sockOpen, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, (const char *)&time_wait, sizeof(time_wait));
            if(0 > ret)
            {
                NN_LOG("             SetSockOpt ReuseAddr error : %d\n", nn::socket::GetLastError());
                Disconnect();
                return;
            }

            // 送信タイムアウトを設定
            ret = nn::socket::SetSockOpt(sockOpen, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, &tv, sizeof(tv));
            if(0 > ret)
            {
                NN_LOG("             SetSockOpt time error : %d\n", nn::socket::GetLastError());
                Disconnect();
                return;
            }

            // 受信タイムアウトを設定
            ret = nn::socket::SetSockOpt(sockOpen, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, &tv, sizeof(tv));
            if(0 > ret)
            {
                NN_LOG("             SetSockOpt time error : %d\n", nn::socket::GetLastError());
                Disconnect();
                return;
            }

            // 送信サイズを設定
            ret = nn::socket::SetSockOpt(sockOpen, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, (char*)&bufsize, sizeof(bufsize));
            if(0 > ret)
            {
                NN_LOG("             SetSockOpt bufsize error : %d\n", nn::socket::GetLastError());
                Disconnect();
                return;
            }

            // 受信サイズを設定
            ret = nn::socket::SetSockOpt(sockOpen, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, (char*)&bufsize, sizeof(bufsize));
            if(0 > ret)
            {
                NN_LOG("             SetSockOpt bufsize error : %d\n", nn::socket::GetLastError());
                Disconnect();
                return;
            }
        }

        void Disconnect() NN_NOEXCEPT
        {
            // ソケットを解放.
            if(0 <= sockTcp)
            {
                nn::socket::Shutdown(sockTcp, nn::socket::ShutdownMethod::Shut_RdWr);
                sockTcp = nn::socket::Close(sockTcp);
                if(0 > sockTcp)
                {
                    NN_LOG("             Close error : %d\n", nn::socket::GetLastError());
                }
                sockTcp = 0;
            }

            // ソケットを解放.
            if(0 <= sockOpen)
            {
                nn::socket::Shutdown(sockOpen, nn::socket::ShutdownMethod::Shut_RdWr);
                sockOpen = nn::socket::Close(sockOpen);
                if(0 > sockOpen)
                {
                    NN_LOG("             Close error : %d\n", nn::socket::GetLastError());
                }
                sockOpen = 0;
            }
            config_status = false;
        }

        void Send(int wait_time) NN_NOEXCEPT
        {
            int ret;
            int err_cnt = 0;
            nn::socket::SockAddrIn server = { 0 };

            server.sin_family = nn::socket::Family::Af_Inet;
            server.sin_port = nn::socket::InetHtons(PortNo);
            nn::socket::InAddr outAddr = { 0 };
            nn::socket::InetAton(IpAdder, &outAddr);
            server.sin_addr = outAddr;

            ret = nn::socket::Connect(sockOpen, reinterpret_cast<nn::socket::SockAddr *>(&server), sizeof(server));
            if(0 > ret)
            {
                NN_LOG("             Connect error : %d\n", nn::socket::GetLastError());
                return;
            }

            while( NN_STATIC_CONDITION(1) )
            {
                if(0 == sockOpen) break;
                int len = nn::socket::Send( sockOpen, (const char*)sendBuf.get(), Bufsize, nn::socket::MsgFlag::Msg_None );
                if(0 > len)
                {
                    NN_LOG("             SendTo error : %d\n", nn::socket::GetLastError());
                    err_cnt ++;
                    if(100 <= err_cnt) break;
                }
                if(0 < wait_time)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(wait_time));
                }
            }
        }

        void Receive() NN_NOEXCEPT
        {
            int ret;
            nn::socket::SockAddrIn client = { 0 };
            nn::socket::SockAddrIn addr = { 0 };
            nn::socket::SockLenT len;

            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(PortNo2);
            addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

            // サーバーソケットに名前をつける
            ret = nn::socket::Bind( sockOpen, reinterpret_cast<nn::socket::SockAddr *>(&addr), sizeof( addr ) );
            if(0 > ret)
            {
                NN_LOG("             Bind error : %d\n", nn::socket::GetLastError());
                return;
            }

            // ソケットを受信待機モードにし，保留接続キューのサイズを確保
            ret = nn::socket::Listen( sockOpen, Echoserverbacklog );
            if(0 > ret)
            {
                NN_LOG("             Listen error : %d\n", nn::socket::GetLastError());
                return;
            }
            len = sizeof( client );

            // サーバーソケットを接続要求待ち状態
            sockTcp = nn::socket::Accept( sockOpen, reinterpret_cast<nn::socket::SockAddr *>(&client), &len );
            if(0 > sockTcp)
            {
                NN_LOG("             Server Socket error : %d\n", nn::socket::GetLastError());
                return;
            }

            // 受信処理.
            int status;
            int err_cnt = 0;
            while( NN_STATIC_CONDITION(1) )
            {
                if(0 == sockTcp) break;
                status = nn::socket::Recv(sockTcp, recvBuf.get(), Bufsize, nn::socket::MsgFlag::Msg_None);
                if(0 == status)
                {
                    NN_LOG("             Disconnect\n");
                    break;
                }
                else if(0 > status)
                {
                    NN_LOG("             Recv failed (error %d)\n", nn::socket::GetLastError());
                    err_cnt ++;
                    if(100 <= err_cnt){
                        break;
                    }
                }
            }
        }
    };

    // インフラ試験ミューテックスクラス（ローカル変数で自動解放使用）
    class InfraLock
    {
    protected:
        nn::os::MutexType* pinfraLock;

    public:
        explicit InfraLock(nn::os::MutexType* pMutex) NN_NOEXCEPT
        {
            pinfraLock = nullptr;

            // MutexTypeポインタチェック且つ、初期化済み判定
            if (pMutex != nullptr &&
                pMutex->_state == nn::os::MutexType::State::State_Initialized)
            {
                // MutexTypeポインタをメンバー変数に格納しロック（デストラクタでロック解放）
                pinfraLock = pMutex;
                nn::os::LockMutex(pinfraLock);
            }
        }

        ~InfraLock() NN_NOEXCEPT
        {
            // メンバー変数のMutexTypeポインタチェック且つ、初期化済み判定
            if (pinfraLock != nullptr &&
                pinfraLock->_state == nn::os::MutexType::State::State_Initialized)
            {
                // メンバー変数に格納されたMutexTypeポインタでロック解放（デストラクタでロック解放）
                nn::os::UnlockMutex(pinfraLock);
                pinfraLock = nullptr;
            }
        }
    };

    // Wowlスリープ試験TCPクラス
    class WowlSocket
    {
    protected:
        struct BufferPoolMngMngDetail
        {
            size_t bufferSize;
            size_t dataSize;
            uint8_t* pData;
        };

        struct BufferPoolMng
        {
            uint32_t maxCount;
            uint32_t writeIdx;
            uint32_t readIdx;
            std::unique_ptr<uint8_t[]> buffer;
            size_t bufferSize;
            size_t packetSize;
            std::unique_ptr<BufferPoolMngMngDetail[]> info;
        };

        BufferPoolMng recvPool;
        nn::os::MutexType recvLock;

        nn::socket::InAddr pcIpAddress;
        IpV4Address pcAddress;

        int32_t wowlSock;
        uint16_t wowlPort;
        bool isConnect;
        int64_t recvCount;

        nn::os::ThreadType recvType;
        nn::os::ThreadType parseRecvType;
        LocalAutoEvent recvEvent;
        LocalAutoEvent recvStopEvent;
        bool isRecvStart;
        bool isRecvStop;

        nn::wlan::WowlWakeCount wowlWakeCount;
        nn::wlan::WowlSleepStats wowlSleepSts;
        nn::wlan::WowlWakeReason wowlReason;
        uint32_t reason = 0;
    public:
        WowlSocket() NN_NOEXCEPT
        :   wowlSock        (-1),
            wowlPort        (45678),
            isConnect       (false),
            isRecvStart     (false),
            isRecvStop      (true)
        {
            // TCP受信イベント初期化
            recvEvent.SetEvent(false, nn::os::EventClearMode::EventClearMode_ManualClear);
            recvStopEvent.SetEvent(false, nn::os::EventClearMode::EventClearMode_ManualClear);

            // TCP受信ミューテックス初期化
            nn::os::InitializeMutex(&recvLock, false, 0);
        }

        ~WowlSocket() NN_NOEXCEPT
        {
            // TCP切断
            WowlDisconnect();

            // TCP受信ミューテックス解放
            nn::os::FinalizeMutex(&recvLock);
        }

        // TCP接続初期設定。dnsを設定する必要がない場合はデフォルトゲートウェイでdns設定（Wowlスリープ試験）
        bool Init(const char* ipAddress, const char* subNetMask, const char* gateway) NN_NOEXCEPT
        {
            return Init(ipAddress, subNetMask, gateway, gateway, "0.0.0.0");
        }

        // TCP接続初期設定。dnsを設定する場合はこちらを呼び出す（Wowlスリープ試験）
        bool Init(const char* ipAddress, const char* subNetMask, const char* defaultGateway,
            const char* dns1, const char* dns2) NN_NOEXCEPT
        {
            if (ipAddress == nullptr || subNetMask == nullptr || defaultGateway == nullptr ||
                dns1 == nullptr || dns2 == nullptr)
            {
                return false;
            }

            WowlDisconnect();

            nn::util::Strlcpy(pcAddress.strIpAddress, ipAddress, INET_ADDRSTRLEN);
            nn::util::Strlcpy(pcAddress.strSubNetMask, subNetMask, INET_ADDRSTRLEN);
            nn::util::Strlcpy(pcAddress.strGateWay, defaultGateway, INET_ADDRSTRLEN);
            nn::util::Strlcpy(pcAddress.strDns1, dns1, INET_ADDRSTRLEN);
            nn::util::Strlcpy(pcAddress.strDns2, dns2, INET_ADDRSTRLEN);
            pcAddress.isDhsp    = false;
            isConnect           = false;
            isRecvStart         = false;
            isRecvStop          = true;

            // 実機のIPアドレス設定
            WowConfigureInterface(WLAN_INTERFACE_NAME, pcAddress, true);

            // Socket生成
            return CreateWowlSocket();
        }

        // TCP受信バッファプール設定（Wowlスリープ試験）
        bool SetPoolMngSetting(const size_t bufferSize, const size_t packetSize) NN_NOEXCEPT
        {
            InfraLock lock(&recvLock);

            // プールできるサイズ分の数を算出
            recvPool.maxCount   = bufferSize / packetSize;
            recvPool.buffer.reset(new uint8_t[bufferSize]);
            recvPool.info.reset(new BufferPoolMngMngDetail[recvPool.maxCount]);

            // 確保バッファチェック
            if (recvPool.buffer.get() == nullptr || recvPool.info.get() == nullptr)
            {
                recvPool.bufferSize = 0;
                return false;
            }

            uint8_t* pBufferPos     = recvPool.buffer.get();
            size_t actSize          = bufferSize;
            recvPool.readIdx        = 0;
            recvPool.writeIdx       = 0;
            recvPool.bufferSize     = bufferSize;
            BufferPoolMngMngDetail* pInfo = recvPool.info.get();

            // 算出したプールサイズ分設定
            for (int32_t i = 0; i < recvPool.maxCount; i++)
            {
                pInfo[i].pData = pBufferPos;
                pInfo[i].bufferSize = packetSize;
                pInfo[i].dataSize = 0;
                actSize -= packetSize;
                pBufferPos += packetSize;
            }

            return true;
        }

        // TCP受信バッファプール取得。プール先頭バッファ取得（Wowlスリープ試験）
        BufferPoolMngMngDetail* GetFirstPoolMng() NN_NOEXCEPT
        {
            if (recvPool.info.get() == nullptr)
            {
                return nullptr;
            }

            recvPool.writeIdx = 0;

            recvPool.info.get()[0].dataSize = 0;

            return &recvPool.info.get()[0];
        }

        // TCP受信バッファプール取得。プール次のバッファ取得（Wowlスリープ試験）
        BufferPoolMngMngDetail* GetNextPoolMng() NN_NOEXCEPT
        {
            if (recvPool.info.get() == nullptr)
            {
                return nullptr;
            }

            ++recvPool.writeIdx %= recvPool.maxCount;

            recvPool.info.get()[recvPool.writeIdx].dataSize = 0;

            return &recvPool.info.get()[recvPool.writeIdx];
        }

        // WowFunctionTestServerツールとのTcp接続。ポート引数固定省略オーバーロード（Wowlスリープ試験）
        bool WowlConnect(const char* ipAddress) NN_NOEXCEPT
        {
            const uint16_t port = 45678;
            return WowlConnect(ipAddress, port);
        }

        // WowFunctionTestServerツールとのTcp接続。ポート指定引数オーバーロード（Wowlスリープ試験）
        bool WowlConnect(const char* ipAddress, const uint16_t port) NN_NOEXCEPT
        {
            if (wowlSock == -1 || ipAddress == nullptr)
            {
                return false;
            }

            if (isConnect == true)
            {
                return true;
            }

            nn::socket::InAddr pcIpAddress;
            nn::socket::SockAddrIn destAddr = {};

            nn::socket::InetAton(ipAddress, &pcIpAddress);
            wowlPort = port;

            destAddr.sin_addr = pcIpAddress;
            destAddr.sin_port = nn::socket::InetHtons(static_cast<uint16_t>(wowlPort));
            destAddr.sin_family = nn::socket::Family::Af_Inet;

            int result = nn::socket::Connect(wowlSock,
                                    reinterpret_cast<nn::socket::SockAddr*>(&destAddr), sizeof(destAddr));
            if (result < 0)
            {
                return false;
            }

            WowlConnectInfo();

            return (isConnect = true);
        }

        // WowFunctionTestServerツールとのTcp接続待機。ポート省略オーバーロード（Wowlスリープ試験）
        bool WowlConnectWait(const char* ipAddress, const int64_t waitTime) NN_NOEXCEPT
        {
            const uint16_t port = 45678;
            return WowlConnectWait(ipAddress, port, waitTime);
        }

        // WowFunctionTestServerツールとのTcp接続待機。接続待機単位msec（Wowlスリープ試験）
        bool WowlConnectWait(const char* ipAddress, const uint16_t port, const int64_t waitTime) NN_NOEXCEPT
        {
            const int64_t oneWait = 1000;
            auto startTick = nn::os::GetSystemTick();
            auto logoutTick = nn::os::GetSystemTick();
            int64_t sleepTime = std::min(waitTime, oneWait);
            bool isWaitConnect = false;

            // // WowFunctionTestServerツールとのTcp接続もしくは、待機時間までループ条件
            while ((nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds() < waitTime)
            {
                // Socketが生成されていないもしくはループ２週目以降のSocket生成条件
                if (wowlSock == -1)
                {
                    CreateWowlSocket();
                }

                // TCP接続
                isWaitConnect = WowlConnect(ipAddress, port);
                if (isWaitConnect == true)
                {
                    break;
                }
                // TCP切断
                WowlDisconnect();
                if ((nn::os::GetSystemTick() - logoutTick).ToTimeSpan().GetMilliSeconds() > 10000)
                {
                    NN_LOG("               ### Wait Server Port Open(Connect Wait) ###\n");
                    logoutTick = nn::os::GetSystemTick();
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTime));
            }

            return isWaitConnect;
        }

        // Tcp情報取得及び、SetTcpSessionInformation設定。Wowlスリープ前に呼び出し必須（Wowlスリープ試験）
        bool WowlConnectInfo() NN_NOEXCEPT
        {
            if (wowlSock == -1)
            {
                return false;
            }

            nn::bsdsocket::cfg::IfState state;
            nn::bsdsocket::cfg::GetIfState(WLAN_INTERFACE_NAME, &state);
            nn::socket::TcpInfo ti;
            nn::socket::SockAddrIn saiServer;
            nn::socket::SockAddrIn saiClient;

            if (GetSocketInfo(ti, saiServer, saiClient) != true)
            {
                return false;
            }

            if (saiClient.sin_addr.S_addr == nn::socket::InAddr_Any)
            {
                saiClient.sin_addr.S_addr = state.addr.S_addr;
            }

            uint8_t remoteHardwareAddress[nn::socket::Ether_Addr_Len];

            // FIXME: static routes
            nn::socket::SockAddrIn saiNext;
            if ((saiServer.sin_addr.S_addr & state.subnetMask.S_addr) ==
                        (state.gatewayAddr.S_addr & state.subnetMask.S_addr))
            {
                saiNext = saiServer;
            }
            else
            {
                saiNext.sin_addr = state.gatewayAddr;
            }

            nn::Result result = nn::bsdsocket::cfg::LookupArpEntry(remoteHardwareAddress,
                                                            nn::socket::Ether_Addr_Len, saiNext.sin_addr);
            if (result.IsFailure())
            {
                NN_LOG("             LookupArpEntry failed (result 0x%08x)\n",
                        result.GetInnerValueForDebug());
                return false;
            }

            NN_LOG("             seq: % 16u[0x%08x]\n", ti.tcpi_snd_nxt, ti.tcpi_snd_nxt);
            NN_LOG("             ack: % 16u[0x%08x]\n", ti.tcpi_rcv_nxt, ti.tcpi_rcv_nxt);
            NN_LOG("             rwin: % 16u\n", ti.tcpi_rcv_space);
            NN_LOG("             local: % 16s:%u\n", nn::socket::InetNtoa(saiClient.sin_addr),
                                                    nn::socket::InetNtohs(saiClient.sin_port));
            NN_LOG("             remote: % 16s:%u\n", nn::socket::InetNtoa(saiServer.sin_addr),
                                                    nn::socket::InetNtohs(saiServer.sin_port));
            NN_LOG("             next addr: % 16s[%02x:%02x:%02x:%02x:%02x:%02x]\n",
                nn::socket::InetNtoa(saiNext.sin_addr),
                remoteHardwareAddress[0], remoteHardwareAddress[1], remoteHardwareAddress[2],
                remoteHardwareAddress[3], remoteHardwareAddress[4], remoteHardwareAddress[5]);

            nn::wlan::WlanIpv4Address src;
            nn::wlan::WlanIpv4Address dst;

            StringAddressToArray(src, nn::socket::InetNtoa(saiClient.sin_addr));
            StringAddressToArray(dst, nn::socket::InetNtoa(saiServer.sin_addr));

            nn::wlan::MacAddress dstMac(remoteHardwareAddress);
            result = nn::wlan::Infra::SetTcpSessionInformation(dstMac,
                src, dst, nn::socket::InetNtohs(saiClient.sin_port),
                                                nn::socket::InetNtohs(saiServer.sin_port),
                ti.tcpi_rcv_nxt, static_cast<uint16_t>(ti.tcpi_rcv_space));
            if (result.IsFailure())
            {
                NN_LOG("             SetTcpSessionInformation failed.\n");
                return false;
            }

            return true;
        }

        // Wowlステータスクリア（Wowlスリープ試験）
        void WowlClearWakeupState() NN_NOEXCEPT
        {
            nn::Result result = nn::wlan::Infra::ClearWowlStats(true, true);
            if (result.IsSuccess() != true)
            {
                NN_LOG("               nn::wlan::Infra::ClearWowlStats Result : 0x%08x\n",
                    result.GetInnerValueForDebug());
            }
        }

        // Wowlステータス取得及びログ出力（Wowlスリープ試験）
        void WowlGetWakeupState(nn::wlan::WowlWakeCount* pCounts,
                                    nn::wlan::WowlSleepStats* pStats) NN_NOEXCEPT
        {
            NN_LOG("             #### Wowl Status Info ####\n");
            nn::Result result = nn::wlan::Infra::GetWowlStats(pCounts, pStats);
            if (result.IsSuccess() == true)
            {
                NN_LOG("                 Wake Count system                      : %lu\n",
                                pCounts->system);
                NN_LOG("                 Wake Count magic Packet                : %lu\n",
                                pCounts->magic);
                NN_LOG("                 Wake Count data Pattern                : %lu\n",
                                pCounts->netPattern);
                NN_LOG("                 Wake Count disconnect Recv Packet      : %lu\n",
                                pCounts->disconnect);
                NN_LOG("                 Wake Count beacon Lost                 : %lu\n",
                                pCounts->beaconLost);
                NN_LOG("                 Wake Count gtk Fail                    : %lu\n",
                                pCounts->gtkFail);
                NN_LOG("                 Wake Count tcp data recv               : %lu\n",
                                pCounts->tcpData);
                NN_LOG("                 Wake Count other                       : %lu\n\n",
                                pCounts->other);
                NN_LOG("                 Wake Status cpu Wakeup Count           : %lu\n",
                                pStats->cpuWakes);
                NN_LOG("                 Wake Status cpu Active Time            : %lu\n",
                                pStats->cpuActiveTime);
                NN_LOG("                 Wake Status total Measured Time        : %lu\n",
                                pStats->totalMeasuredTime);
                NN_LOG("                 Wake Status dtim proccess Count        : %lu\n",
                                pStats->dtim);
                NN_LOG("                 Wake Status tx Send Packet Count       : %lu\n",
                                pStats->txPackets);
                NN_LOG("                 Wake Status tx Total Send Size         : %lu\n",
                                pStats->txBytes);
                NN_LOG("                 Wake Status tx Total Send Time         : %lu\n",
                                pStats->txActiveTime);
                NN_LOG("                 Wake Status rx Recv Packet Count       : %lu\n",
                                pStats->rxPackets);
                NN_LOG("                 Wake Status rx Total Recv Size         : %lu\n",
                                pStats->rxBytes);
                NN_LOG("                 Wake Status tx Total Recv Time         : %lu\n",
                                pStats->rxActiveTime);
                NN_LOG("                 Wake Status total idle Time            : %lu\n",
                                pStats->idleTime);
                NN_LOG("                 Wake Status arp reply Count            : %lu\n",
                                pStats->arpReply);
                NN_LOG("                 Wake Status tcp keep alive reply Count : %lu\n",
                                pStats->tcpKaAck);
                NN_LOG("                 Wake Status gtk update reply Count     : %lu\n\n",
                                pStats->gtkRenewal);
            }

            result = nn::wlan::Infra::GetWakeupReasonRaw(&reason);
            if (result.IsSuccess() == true)
            {
                NN_LOG("                 Wake Reason                            : 0x%08x\n\n",
                                reason);
            }

            result = nn::wlan::Infra::GetWakeupReason(&wowlReason);
            if (result.IsSuccess() == true)
            {
                size_t arrayCount = (sizeof(WakeupReasonState) / sizeof(WakeupReasonState[0]));
                if (wowlReason >= 0 && wowlReason < arrayCount)
                {
                    NN_LOG("                 Wake Reason Status                     : %s\n",
                                    WakeupReasonState[wowlReason]);
                }
                else
                {
                    NN_LOG("                 Wake Reason Status                     : Unknown\n");
                }
            }
            NN_LOG("             #### Wowl Status Info ####\n\n");
        }

        // TCP受信スレッド生成から起動（Wowlスリープ試験）
        bool WowlReceiveStart(const size_t poolSize, const size_t packetSize) NN_NOEXCEPT
        {
            // Socket生成状態チェック
            if (wowlSock == -1 || isConnect != true)
            {
                NN_LOG("             WowlReceiveStart State Fail : %d\n", __LINE__);
                return false;
            }

            // TCP受信バッファプール領域確保及び設定
            bool isResult = SetPoolMngSetting(poolSize, packetSize);
            if (isResult != true)
            {
                NN_LOG("             WowlReceiveStart Buffer Pool Setting Fail : %d\n", __LINE__);
                return false;
            }

            static NN_ALIGNAS(4096) uint8_t wowlParseRecvStack[ThreadStackSize];

            // TCP受信解析スレッド生成から起動（受信解析スレッドが無くても受信スレッドがあれば動作するようにreturnさせない）
            nn::Result createResult = nn::os::CreateThread(&parseRecvType, WowlParseRecvThread,
                        this, wowlParseRecvStack, ThreadStackSize, nn::os::DefaultThreadPriority);
            if (createResult.IsSuccess() == true)
            {
                recvStopEvent.ClearSygnal();
                isRecvStop  = false;
                isRecvStart = true;
                nn::os::StartThread(&parseRecvType);
            }
            else
            {
                NN_LOG("             WowlReceiveStart WowlParseRecvThread Failed : 0x%08x\n",
                        createResult.GetInnerValueForDebug());
            }

            static NN_ALIGNAS(4096) uint8_t wowlRecvStack[ThreadStackSize];

            // TCP受信スレッド生成から起動
            createResult = nn::os::CreateThread(&recvType, WowlRecvThread,
                        this, wowlRecvStack, ThreadStackSize, nn::os::DefaultThreadPriority);
            if (createResult.IsSuccess() == true)
            {
                recvStopEvent.ClearSygnal();
                isRecvStop  = false;
                isRecvStart = true;
                nn::os::StartThread(&recvType);
            }
            else
            {
                NN_LOG("             WowlReceiveStart WowlRecvThread Failed : 0x%08x\n",
                    createResult.GetInnerValueForDebug());
            }

            return createResult.IsSuccess();
        }

        // TCP受信カウントクリア（Wowlスリープ試験）
        void WowlClearRecvCount() NN_NOEXCEPT
        {
            recvCount = 0;
        }

        // TCP受信カウント取得（Wowlスリープ試験）
        int64_t WowlGetRecvCount() NN_NOEXCEPT
        {
            return recvCount;
        }

        // TCP受信待機。単位msec（Wowlスリープ試験）
        bool WowlReceiveWait(int64_t waitTime) NN_NOEXCEPT
        {
            if (isRecvStart != true)
            {
                NN_LOG("               WowlReceiveWait State Fail : %d\n", __LINE__);
                return false;
            }

            return recvStopEvent.WaitSygnal(waitTime);
        }

        // TCP受信停止（Wowlスリープ試験）
        bool WowlReceiveStop() NN_NOEXCEPT
        {
            if (isRecvStart != true)
            {
                return false;
            }

            if (wowlSock != -1)
            {
                int nResult = nn::socket::Shutdown(wowlSock, nn::socket::ShutdownMethod::Shut_RdWr);
                NN_LOG("               nn::socket::Shutdown\n");
                if (nResult != 0)
                {
                    NN_LOG("               nn::socket::Shutdown : %d\n", nn::socket::GetLastError());
                }
            }

            isRecvStop  = true;
            isRecvStart = false;

            nn::os::WaitThread(&recvType);
            nn::os::DestroyThread(&recvType);

            nn::os::WaitThread(&parseRecvType);
            nn::os::DestroyThread(&parseRecvType);

            return true;
        }

        // TCP Socket切断（Wowlスリープ試験）
        bool WowlDisconnect() NN_NOEXCEPT
        {
            if (wowlSock == -1)
            {
                return false;
            }

            NN_LOG("               Tcp Socket Close Start\n");
            auto result = nn::socket::Close(wowlSock);
            NN_LOG("               Tcp Socket Close End\n");
            if (result < 0)
            {
                return false;
            }

            wowlSock = -1;
            isConnect = false;

            return true;
        }

        // Socket設定オプション取得（Wowlスリープ試験）
        bool GetSocketInfo(nn::socket::TcpInfo& tcpInfo, nn::socket::SockAddrIn& serverSoAdd,
                            nn::socket::SockAddrIn& clientSoAdd) NN_NOEXCEPT
        {
            nn::socket::SockLenT len = sizeof(nn::socket::TcpInfo);

            int nResult = nn::socket::GetSockOpt(wowlSock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Info,
                                    reinterpret_cast<void*>(&tcpInfo), &len);
            if (nResult < 0)
            {
                NN_LOG("          GetSocketInfo -> GetSockOpt failed (error %d)\n",
                            nn::socket::GetLastError());
                return false;
            }

            len = sizeof(nn::socket::SockAddrIn);
            nResult = nn::socket::GetSockName(wowlSock,
                                    reinterpret_cast<nn::socket::SockAddr*>(&clientSoAdd), &len);
            if (nResult < 0)
            {
                NN_LOG("          GetSocketInfo -> GetSockName failed (error %d)\n",
                            nn::socket::GetLastError());
                return false;
            }

            len = sizeof(nn::socket::SockAddrIn);
            nResult = nn::socket::GetPeerName(wowlSock,
                                    reinterpret_cast<nn::socket::SockAddr*>(&serverSoAdd), &len);
            if (nResult < 0)
            {
                NN_LOG("          GetSocketInfo -> GetPeerName failed (error %d)\n",
                            nn::socket::GetLastError());
                return false;
            }

            return true;
        }

    protected:

        // Socket生成及びオプション設定（Wowlスリープ試験）
        bool CreateWowlSocket() NN_NOEXCEPT
        {
            WowlDisconnect();

            wowlSock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            NN_ABORT_UNLESS(wowlSock >= 0);

            int ret, opt;

            opt = 1;
            ret = nn::socket::SetSockOpt(wowlSock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, &opt, sizeof(opt));
            if (ret < 0)
            {
                return false;
            }

            // disable SACK/WindowScaling/TimeStamp
            opt = 1;
            ret = nn::socket::SetSockOpt(wowlSock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoOpt, &opt, sizeof(opt));
            if (ret < 0)
            {
                return false;
            }

            opt = 4096;
            ret = nn::socket::SetSockOpt(wowlSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &opt, sizeof(opt));
            if (ret  < 0)
            {
                return false;
            }

            return true;
        }

        // 実機のIPアドレス設定（Wowlスリープ試験）
        void WowConfigureInterface(const char* interfaceName, IpV4Address ipAddr, bool enable) NN_NOEXCEPT
        {
            nn::Result result;

            nn::socket::IfReq ifr = { WLAN_INTERFACE_NAME };
            ifr.ifr_flags = nn::socket::IfrFlag::Iff_Up;
            int socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
            NN_ABORT_UNLESS(socket >= 0);
            nn::socket::Ioctl(socket, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocSIfFlags), &ifr, sizeof(ifr));

            if (enable)
            {
                nn::bsdsocket::cfg::IfSettings ifcfg;
                memset(&ifcfg, 0, sizeof(ifcfg));

                if (ipAddr.isDhsp != true)
                {
                    // static ip
                    ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                    ifcfg.mtu = 1500;
                    ifcfg.isManualDns = true;

                    nn::socket::InetAton(ipAddr.strIpAddress, &ifcfg.u.modeStatic.addr);
                    nn::socket::InetAton(ipAddr.strGateWay, &ifcfg.u.modeStatic.gatewayAddr);
                    nn::socket::InetAton(ipAddr.strSubNetMask, &ifcfg.u.modeStatic.subnetMask);
                    ifcfg.u.modeStatic.broadcastAddr.S_addr =
                        (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
                        ~ifcfg.u.modeStatic.subnetMask.S_addr;
                    nn::socket::InetAton(ipAddr.strDns1, &ifcfg.dnsAddrs[0]);
                    nn::socket::InetAton(ipAddr.strDns2, &ifcfg.dnsAddrs[1]);
                }
                else
                {
                    // dhcp
                    ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                    ifcfg.mtu = 1500;
                    ifcfg.isManualDns = false;
                }

                result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
                if (result.IsFailure())
                {
                    NN_LOG("             failed to configure interface %s - %d:%d\n",
                        interfaceName,
                        result.GetModule(),
                        result.GetDescription());
                }
            }
            else
            {
                nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
            }

            nn::socket::Close(socket);
        }

        // IPアドレスから配列値変換（Wowlスリープ試験）
        bool StringAddressToArray(nn::wlan::WlanIpv4Address& ipArray,
                const char Address[INET_ADDRSTRLEN]) NN_NOEXCEPT
        {
            char* pSearch   = const_cast<char*>(Address);
            char* pPos      = nullptr;

            for (int i = 0; i < 4; i++)
            {
                pPos = std::strtok(pSearch, ".");
                if (pPos == nullptr)
                {
                    return false;
                }
                ipArray.addr[i] = std::atoi(pPos);
                pSearch         = nullptr;
            }

            return true;
        }

        // ダンプ出力文字列変換（Wowlスリープ試験）
        char WowlNibbleToAscii(uint8_t nibble) NN_NOEXCEPT
        {
            nibble &= 0xf;
            return nibble < 0xa ? nibble + '0' : nibble - 0xa + 'A';
        }

        // 受信バッファのダンプ出力（Wowlスリープ試験）
        void WowlTraceBuffer(const uint8_t* pBuffer, const size_t traceSize) NN_NOEXCEPT
        {
            const size_t AddressByteWidth   = 2;
            const size_t BytesInLine        = 16;
            const size_t LineWidth          = AddressByteWidth * 2 + 1 + 3 * BytesInLine * 2 + 2;
            const size_t LineBufferSize     = 36 + BytesInLine * 3 + 1 + BytesInLine + 1 + 1;

            std::array<char, LineWidth> lineBuffer;
            lineBuffer.back() = '\0';

            if (pBuffer == nullptr)
            {
                return;
            }

            NN_LOG("              recv\n");

            char format[LineBufferSize + 1];
            nn::util::SNPrintf(format, sizeof(format), "              %%-%ds "
                            "+0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +A +B +C +D +E +F 0123456789ABCDEF\n",
                            AddressByteWidth * 2);

            NN_LOG(format, "addr");

            int totalReceivedCount = 0;
            char* lineHexWritePtr;
            char* lineAsciiWritePtr;

            auto readPtr = pBuffer;
            auto readEnd = readPtr + traceSize;

            totalReceivedCount = 0;

            for (; readPtr < readEnd; ++readPtr)
            {

                if (!(totalReceivedCount & 0xf))
                {
                    auto lineWritePtr = lineBuffer.begin();
                    lineHexWritePtr = lineWritePtr + AddressByteWidth * 2 + 1;
                    lineAsciiWritePtr = lineHexWritePtr + 3 * BytesInLine;

                    for (int i = AddressByteWidth * 2; i--;)
                    {
                        *lineWritePtr++ = WowlNibbleToAscii(totalReceivedCount >> (i * 4));
                    }

                    while (lineWritePtr < lineBuffer.end())
                    {
                        *lineWritePtr++ = ' ';
                    }
                }

                *(lineHexWritePtr) = WowlNibbleToAscii(*readPtr >> 4);
                *(lineHexWritePtr + 1) = WowlNibbleToAscii(*readPtr);
                lineHexWritePtr += 3;

                *lineAsciiWritePtr++ = *readPtr >= 0x20 && *readPtr < 0x80 ? *readPtr : '.';

                if (!(~totalReceivedCount & 0xf))
                {
                    NN_LOG("              %s\n", lineBuffer.data());
                }

                ++totalReceivedCount;
            }

            if (totalReceivedCount & 0xf)
            {
                NN_LOG("              %s\n", lineBuffer.data());
            }
        }

        // TCP受信スレッド（Wowlスリープ試験）
        static void WowlRecvThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);

            WlanTest::WowlSocket* pThis     = static_cast<WowlSocket*>(arg);
            int64_t totalSize               = 0;
            int32_t recvSize                = 0;
            BufferPoolMngMngDetail* pPool   = pThis->GetFirstPoolMng();
            pThis->recvCount                = 0;

            NN_LOG("             ### Wowl Tcp Recv Thread Start ###\n");

            // 受信バッファプールが存在且つ、受信ストップしていない条件
            while (pPool != nullptr && pThis->isRecvStop != true)
            {
                recvSize = nn::socket::Recv(pThis->wowlSock, pPool->pData, pPool->bufferSize, nn::socket::MsgFlag::Msg_None);
                if (recvSize > 0)
                {
                    pPool->dataSize = recvSize;
                    totalSize       += recvSize;
                    pPool           = pThis->GetNextPoolMng();
                    pThis->recvEvent.SetSygnal();
                    pThis->recvCount++;
                }
                else
                {
                    break;
                }
            }

            NN_LOG("             ### Wowl Tcp Recv Thread End ###n\n");
            // 受信ストップ時のイベントシグナル設定
            pThis->recvStopEvent.SetSygnal();
        }

        // TCP受信バッファプール解析スレッド（Wowlスリープ試験）
        static void WowlParseRecvThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);

            WlanTest::WowlSocket* pThis = static_cast<WowlSocket*>(arg);

            bool isSygnal = false;

            while (pThis->isRecvStop != true)
            {
                // 受信イベント待機（TCP受信毎にシグナルになる）
                isSygnal = pThis->recvEvent.WaitSygnal(100);
                if (isSygnal != true)
                {
                    continue;
                }
                pThis->recvEvent.ClearSygnal();

                uint32_t writePos   = pThis->recvPool.writeIdx;
                uint32_t parsePos   = pThis->recvPool.readIdx;
                uint32_t parseRange = 0;

                // バッファプール範囲条件：読み込み位置、書き込み位置
                if (parsePos > writePos)
                {
                    // バッファリング1周前後の算出
                    parseRange = pThis->recvPool.maxCount - parsePos + writePos;
                }
                else
                {
                    // バッファリング1周前の算出
                    parseRange = writePos - parsePos;
                }

                // バッファプールの書き込みが早い場合受信イベントを待たずに書き込み直前まで読み出し
                for (int32_t i = 0; i < parseRange; i++)
                {
                    BufferPoolMngMngDetail* pPoolData = &pThis->recvPool.info.get()[parsePos];

                    // 受信内容をログ出力
                    NN_LOG("             **** Wowl Tcp Recv Data ****\n");
                    pThis->WowlTraceBuffer(pPoolData->pData, pPoolData->dataSize);
                    NN_LOG("             **** Wowl Tcp Recv Data ****\n\n");

                    InfraLock lock(&pThis->recvLock);
                    ++pThis->recvPool.readIdx %= pThis->recvPool.maxCount;
                    parsePos = pThis->recvPool.readIdx;
                }
            }

        }
    };

    TCPConnecter tcp_Send;
    TCPConnecter tcp_Recv;
    class InfraFuncTest : public ::testing::Test
    {
    protected:
        nn::os::SystemEventType connectionEvent;
        nn::wlan::ConnectionStatus connectionStatus;
        const int KLocalConcurrencyLimit = 4;
        nn::os::Tick sysTick;

        nn::os::ThreadType scanStopThread;
        nn::os::EventType scanStopEvent;

        nn::os::ThreadType connectCancelThread;
        nn::os::EventType connectCancelEvent;

        nn::os::ThreadType agingn117Thread;
        nn::os::EventType agingn117Event;

        nn::os::ThreadType agingn118Thread;
        nn::os::EventType agingn118Event;

        nn::os::ThreadType tcpThread;
        nn::os::EventType tcpEvent;

        nn::os::ThreadType tcpSendThread;

        nn::os::ThreadType tcpRecvThread;

        nn::os::ThreadType iperfServerThread;
        nn::os::EventType iperfServerEvent;

        nn::os::ThreadType iperfClientThread;
        nn::os::EventType iperfClientEvent;

        nn::os::ThreadType wlanStateThread;

        nn::os::ThreadType wlanConnectStateThread;

        int connectionTime = 0;
        const int IPaddresstime = 3000;

        std::unique_ptr<char[]> wlanVertionMsg;

        nn::wlan::WlanState State;

        static const int Time5m = 5;
        static const int Time16m = 16;
        static const int Time20m = 20;
        static const int Time100m = 100;
        static const int Time300m = 300;
        static const int Time1s = 1000;
        static const int Time3s = Time1s * 3;
        static const int Time5s = Time1s * 5;
        static const int Time10s = Time1s * 10;
        static const int Time15s = Time1s * 15;
        static const int Time20s = Time1s * 20;
        static const int Time60s = Time1s * 60;
        static const int Time1h = 3600000;
        static const int Time4h = 3600000 * 4;
        static const int Time12h = 3600000 * 12;

        unsigned oldRandtime = 0;
        bool confSet = false;

        bool loopTime = false;
        const bool IsEnabled = true;
    protected:
        InfraFuncTest() NN_NOEXCEPT
        {
            std::unique_ptr<char[]> wlanMessage(new char[512]);
            wlanVertionMsg = std::move(wlanMessage);
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Infra);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                                                                  nn::socket::DefaultSocketMemoryPoolSize,
                                                                  nn::socket::MinSocketAllocatorSize,
                                                                  KLocalConcurrencyLimit));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            if((nn::wlan::WlanState_InfraIdle!=State) && (nn::wlan::WlanState_InfraIdleScan!=State))
            {
                NN_LOG("              Wlan State:%d\n", State);
                WLANTEST_ASSERT_TRUE(false);
            }

            WLANTEST_ASSERT_TRUE(wlanVertionMsg.get() != nullptr);
            char* pWlanVer = wlanVertionMsg.get();
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetFwVersion(pWlanVer, 512));
            NN_LOG("             WLAN VER : %s", pWlanVer);
            confSet = false;
            WlanState();
            loopTime = true;
#ifdef WLAN_TEST_STATE_STOP
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&wlanStateThread, WlanStateThread, &loopTime, ThreadStack3, ThreadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&wlanStateThread);
#endif
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
#ifdef WLAN_TEST_STATE_STOP
            loopTime = false;
            nn::os::WaitThread(&wlanStateThread);
            nn::os::DestroyThread(&wlanStateThread);
#endif
            nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(WLAN_INTERFACE_NAME));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Finalize());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
        }

        static void DumpScanResultWithReader(void* pBuf) NN_NOEXCEPT
        {
            if( pBuf == nullptr )
            {
                return;
            }

            nn::wlan::BeaconScanResultReader resultReader(pBuf);
            uint32_t bssCount = resultReader.GetCount();
            char macStr[nn::wlan::MacAddress::MacStringSize];
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
            const nn::wlan::VendorInfoElementReader* vie;
            nn::Bit8 matchingOui1[nn::wlan::MacAddress::OuiSize] = {0x00, 0x50, 0xf2};
            nn::Bit8 matchingOui2[nn::wlan::MacAddress::OuiSize] = {0x00, 0x07, 0x40};
            uint8_t matchType = 1;

            NN_LOG("\n             ***** SCAN RESULT ***** \n");
            NN_LOG("             BSS count : %d\n", bssCount);
            NN_LOG("             Scan Result Total Length : %d\n", resultReader.GetTotalResultLength());

            for(uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();

                NN_LOG("             -----------------------%d BSS------------------------\n", i + 1);
                NN_LOG("             BSS size      : %d\n", beacon.GetLength());
                NN_LOG("             IE count      : %d\n", beacon.GetIeCount());
                NN_LOG("             Channel       : %d\n", beacon.GetChannel());
                NN_LOG("             RSSI          : %d[dBm]\n", beacon.GetRssi());
                NN_LOG("             LINKLEVEL     : %d\n", beacon.GetLinkLevel());
                NN_LOG("             Beacon Period : %d[ms]\n", beacon.GetInterval());
                NN_LOG("             BSSID : %s\n", beacon.GetBssid().GetString(macStr));
                NN_LOG("             SSID : %s\n", beacon.GetSsid().GetHexString(ssidStr));

                // WPA/WPA2
                if( beacon.IsWpaSupported() == true )
                {
                    const nn::wlan::WpaInfoElementReader<nn::wlan::WifiInfoElementReader>* pWpa
                        = beacon.GetWpaIe();

                    NN_LOG("                 == WPA: version[%d]\n", pWpa->GetVersion());
                    for( uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
                    {
                        NN_LOG("                    PairCipher(%d) %s\n", k, CipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
                    }
                    NN_LOG("                    GroupCipher %s\n", CipherTypeString[pWpa->GetGroupCipherSuite()]);
                    NN_LOG("                    AKM %s\n", AkmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
                }
                if (beacon.IsWpa2Supported())
                {
                    const nn::wlan::WpaInfoElementReader<nn::wlan::InfoElementReader>* pWpa = beacon.GetWpa2Ie();

                    NN_LOG("                 == WPA2: version[%d]\n", pWpa->GetVersion());
                    for( uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
                    {
                        NN_LOG("                    PairCipher(%d) %s\n", k, CipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
                    }
                    NN_LOG("                    GroupCipher %s\n", CipherTypeString[pWpa->GetGroupCipherSuite()]);
                    NN_LOG("                    AKM %s\n", AkmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
                }

                // VIE matching
                vie = beacon.GetVendorIeWithOui(matchingOui1);
                if( vie != nullptr )
                {
                    NN_LOG("             --- OUI matched!\n");
                    NN_LOG("             IE element id : %d\n", vie->GetElementId());
                    NN_LOG("             IE length     : %d\n", vie->GetLength());
                    NN_LOG("             IE OUI        : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
                }
                else
                {
                    NN_LOG("             --- OUI not matched...\n");
                }
                vie = beacon.GetVendorIeWithOuiAndType(matchingOui2, matchType);
                if( vie != nullptr )
                {
                    NN_LOG("             --- OUI and type matched!\n");
                    NN_LOG("             IE element id : %d\n", vie->GetElementId());
                    NN_LOG("             IE length     : %d\n", vie->GetLength());
                    NN_LOG("             IE OUI        : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
                    NN_LOG("             IE type       : %d\n", vie->GetVendorData()[0]);
                }
                else
                {
                    NN_LOG("             --- OUI and type not matched...\n");
                }
            }
        }

        void WlanState()
        {
             nn::wlan::WlanState State;
             WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
             NN_LOG("             %s\n", WlanStateStr[State]);
        }

        void ElapsedTimeChk(int eltim) NN_NOEXCEPT
        {
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > eltim )
            {
                // 指定秒以上経過はNGとする
                NN_LOG("             Connect Time : %d msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
                NN_LOG("             Connect Time : %d msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
            }
        }

        void Connect(APKIND Apkind, bool timechk = false, int time = Time5s) NN_NOEXCEPT
        {
            nn::os::Tick sysTick2;
            int count = 0;
            memset(IpAdder, NULL, sizeof(IpAdder));
            memset(GWIpAdder, NULL, sizeof(GWIpAdder));
            std::strcpy(IpAdder, &IPAddr[Apkind][0]);
            std::strcpy(GWIpAdder, &IPAddr[Apkind][0]);
            char *GWstr = &GWIpAdder[0];
            connectionTime = time;
            for(int i = 0; i < 3; i++)
            {
                GWstr = std::strchr(GWstr, '.');
                GWstr ++;
                if(i == 2)
                {
                    memset(GWstr, 0, 3);
                    *GWstr = '1';
                }
            }

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
            while( NN_STATIC_CONDITION(1) )
            {
                NN_LOG("             WiFi Connect Start\n");
                if(true == timechk)
                {
                    sysTick = nn::os::GetSystemTick();
                }
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[Apkind].ssid,
                                                                        ConnectParams[Apkind].bssid,
                                                                        ConnectParams[Apkind].channel,
                                                                        ConnectParams[Apkind].security,
                                                                        ConnectParams[Apkind].autoKeepAlive,
                                                                        ConnectParams[Apkind].beaconLostTimeout));

                NN_LOG("             WiFi Connected\n");

                nn::os::WaitSystemEvent(&connectionEvent);
                NN_LOG("             WiFi WaitSystemEvent Signal\n");

                WlanState();
                sysTick2 = nn::os::GetSystemTick();
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                if( (nn::os::GetSystemTick() - sysTick2).ToTimeSpan().GetMilliSeconds() > Time100m )
                {
                    NN_LOG("             Connection Status 100msec over!!\n");
                    WLANTEST_ASSERT_TRUE(false);
                }
                if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    if(true == timechk)
                    {
                        ElapsedTimeChk(connectionTime);  // 接続までの時間チェック
                    }
                    NN_LOG("             WlanTest: CONNECTED!!!\n");
                    break;
                }
                else
                {
                    NN_LOG("             WlanTest: NOT CONNECTED!!! STATE %d\n",connectionStatus.state);
                    PrintConnectionStatus(&connectionStatus);
                    count ++;
                    if(100 <= count){
                        nn::os::DestroySystemEvent(&connectionEvent);
                        NN_LOG("             WlanTest: 100 failures\n");
                        WLANTEST_ASSERT_TRUE(false);
                        return;
                    }
                }

                // 3秒待って再試行
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
            }
            nn::os::DestroySystemEvent(&connectionEvent);
        }

        void Disconnect() NN_NOEXCEPT
        {
            nn::os::Tick sysTick2;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
            while( NN_STATIC_CONDITION(1) )
            {
                // イベントシグナル状態の確認
                if( nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true )
                {
                    WlanState();
                    sysTick2 = nn::os::GetSystemTick();
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                    if( (nn::os::GetSystemTick() - sysTick2).ToTimeSpan().GetMilliSeconds() > Time100m )
                    {
                        NN_LOG("             Connection Status 100msec over!!\n");
                        WLANTEST_ASSERT_TRUE(false);
                    }
                    if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
                    {
                        NN_LOG("             WlanTest: DIS CONNECTED!!!\n");
                        break;
                    }
                }
            }
            nn::os::DestroySystemEvent(&connectionEvent);
        }

        void ConfigureInterface(const char* interfaceName, bool enable) NN_NOEXCEPT
        {
            if(true == confSet)
            {
                return;
            }

            nn::Result result;
            struct ifreq ifr = {WLAN_INTERFACE_NAME};
            ifr.ifr_flags = IFF_UP;
            int socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
            NN_ABORT_UNLESS(socket >= 0);
            nn::socket::Ioctl(socket, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocSIfFlags), &ifr, sizeof(ifr));

            if (enable)
            {
                nn::bsdsocket::cfg::IfSettings ifcfg;
                memset(&ifcfg, 0, sizeof(ifcfg));

                #if STATIC_IP
                // static ip
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                ifcfg.mtu  = 1500;
                nn::socket::InetAton(IpAdder, &ifcfg.u.modeStatic.addr);
                nn::socket::InetAton(GWIpAdder, &ifcfg.u.modeStatic.gatewayAddr);
                nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
                ifcfg.u.modeStatic.broadcastAddr.S_addr =
                        (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
                            ~ifcfg.u.modeStatic.subnetMask.S_addr;
                nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
                nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
                #else
                // dhcp
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                ifcfg.mtu  = 1500;
                #endif

                result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
                if (result.IsFailure())
                {
                    NN_LOG("             failed to configure interface %s - %d:%d\n",
                               interfaceName,
                               result.GetModule(),
                               result.GetDescription());
                }
            }
            else
            {
                nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
            }
            confSet = true;
        }

        void PrintConnectionStatus(nn::wlan::ConnectionStatus* pStatus) NN_NOEXCEPT
        {
            char macStr[nn::wlan::MacAddress::MacStringSize];
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];

            NN_LOG("             +++ Connection Status +++\n");
            NN_LOG("             STATE           : %s\n", WlanConnectionStateStr[pStatus->state]);
            NN_LOG("             CAUSE           : %d\n", pStatus->cause);
            NN_LOG("             CHANNEL         : %d\n", pStatus->channel);
            NN_LOG("             SSID            : %s\n", pStatus->ssid.GetHexString(ssidStr));
            NN_LOG("             BSSID           : %s\n", pStatus->bssid.GetString(macStr));
            NN_LOG("             AID             : %d\n", pStatus->aid);
            NN_LOG("             REASON CODE     : %d\n", pStatus->statusReasonCode);
            NN_LOG("             CAPABILITY INFO : 0x%04X\n", pStatus->capabilityInfo);
            NN_LOG("             BEACON INTERVAL : %d\n", pStatus->beaconInterval);
            NN_LOG("             +++++++++++++++++++++++++\n");
        }

        static void StopScanThread(void* arg) NN_NOEXCEPT
        {
            nn::os::EventType* pStartEvent = static_cast<nn::os::EventType*>(arg);
            bool isStartEvent;
            int32_t ThreadWaitTime = 10;
            int32_t ThreadTimeOut  = 30000 / ThreadWaitTime;
            int32_t ScanWaintTime = 100;

            for ( int tmVaue = 0; tmVaue < ThreadTimeOut; tmVaue += ThreadWaitTime )
            {
                isStartEvent = nn::os::TimedWaitEvent(pStartEvent, nn::TimeSpan::FromMilliSeconds(ThreadWaitTime));
                if (isStartEvent != true)
                {
                    continue;
                }

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ScanWaintTime));
                NN_LOG("             StopScan Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::StopScan());
                NN_LOG("             StopScan End\n");

                nn::os::ClearEvent(pStartEvent);
                break;
            }
        }

        static void CancelConnectThread(void* arg) NN_NOEXCEPT
        {
            nn::os::EventType* pStartEvent = static_cast<nn::os::EventType*>(arg);
            bool isStartEvent;
            int32_t threadWaitTime = 10;
            int32_t threadTimeOut  = 30000 / threadWaitTime;
            int32_t ConnectWaintTime = 100;
            nn::wlan::ConnectionStatus connectionStatus;

            for ( int tmVaue = 0; tmVaue < threadTimeOut; tmVaue += threadWaitTime )
            {
                isStartEvent = nn::os::TimedWaitEvent(pStartEvent, nn::TimeSpan::FromMilliSeconds(threadWaitTime));
                if (isStartEvent != true)
                {
                    continue;
                }

                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ConnectWaintTime));
                NN_LOG("             WIFI Chancel Connect Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::CancelConnect());
                NN_LOG("             WIFI Chancel Connect End\n");
                nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
                {
                    NN_LOG("             WlanTest: CHANCEL OK!!!\n");
                    break;
                }

                nn::os::ClearEvent(pStartEvent);
                break;
            }
        }

        int GetRandom(int min,int max) NN_NOEXCEPT
        {
            int randRet;
            if(0 == oldRandtime)
            {
                oldRandtime = time(NULL);
                srand(static_cast<unsigned int>(oldRandtime));
            }
            if (oldRandtime != time(NULL))
            {
                oldRandtime = time(NULL);
                srand(static_cast<unsigned int>(oldRandtime));
            }
            randRet = min + (int)(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            NN_LOG("             rand : %d\n",randRet);
            return randRet;
        }

        static unsigned long inet_addr(const char* cp) NN_NOEXCEPT
        {
            nn::socket::InAddr      val;

            if(nn::socket::InetAton(cp, &val))
            {
                return val.S_addr;
            }
            else
            {
                NN_LOG("             IP Addr err\n");
                return nn::socket::InAddr_None;
            }
        }

        static unsigned short checksum(unsigned short *buf, int bufsz) NN_NOEXCEPT
        {
            unsigned long sum = 0;

            while(bufsz > 1)
            {
                sum += *buf;
                buf++;
                bufsz -= 2;
            }

            if(bufsz == 1)
            {
                sum += *reinterpret_cast<unsigned char *>(buf);
            }

            sum = (sum & 0xffff) + (sum >> 16);
            sum = (sum & 0xffff) + (sum >> 16);

            return ~sum;
        }

        bool PingStart(int data_size=0, uint16_t icd_seq=0) NN_NOEXCEPT
        {
            int sock;
            nn::socket::SockAddrIn addr;
            int n;
            int ret = true;

            nn::socket::TimeVal tv;
            memset(&tv, 0, sizeof(tv));
            tv.tv_sec = 30;

            struct packet {
                icmp hdr;
                char data[ping_data_size_1023];
            };
            packet senddata;
            int data_len = 0;

            if(0 < data_size)
            {
                data_len = sizeof(senddata.hdr) + data_size;
            }
            else
            {
                data_len = sizeof(senddata.hdr);
            }

            char buf[2000];
            struct icmp *icmpptr;
            struct ip *iphdrptr;

            NN_LOG("             PING\n");
            ConfigureInterface(WLAN_INTERFACE_NAME, true);
            addr.sin_family = nn::socket::Family::Pf_Inet;
            addr.sin_addr.S_addr = inet_addr(IpAdder);
            NN_LOG("             IP Addr : %s\n", IpAdder);

            /* RAWソケットを作成します */
            sock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Raw, nn::socket::Protocol::IpProto_Icmp);
            if(sock < 0)
            {
                NN_LOG("             Socket error : %d\n", sock);
                return false;
            }

            // 受信タイムアウトを設定
            n = nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, &tv, sizeof(tv));
            if(0 > n)
            {
                NN_LOG("             SetSockOpt error : %d\n", nn::socket::GetLastError());
                sock = nn::socket::Close(sock);
                if(0 > sock)
                {
                    NN_LOG("             Close error : %d\n", sock);
                }
                return false;
            }

            memset(&senddata.hdr, 0, sizeof(senddata.hdr));

            /* ICMPヘッダ */
            senddata.hdr.icmp_type = ICMP_ECHO;
            senddata.hdr.icmp_code = 0;
            senddata.hdr.icmp_cksum = 0;
            senddata.hdr.icmp_hun.ih_idseq.icd_id = 0;
            senddata.hdr.icmp_hun.ih_idseq.icd_seq = icd_seq;

            /* Data */
            if(0 < data_size)
            {
                memcpy(senddata.data, ping_data, data_size);
            }

            /* ICMPヘッダのチェックサムを計算 */
            senddata.hdr.icmp_cksum = checksum(reinterpret_cast<unsigned short *>(&senddata.hdr), data_len);

            /* ICMPヘッダだけのICMPパケットを送信 */
            n = nn::socket::SendTo(sock, (char *)&senddata, data_len, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr));
            if(n < 1)
            {
                NN_LOG("             SendTo error : %d\n", nn::socket::GetLastError());
            }

            /* ICMP ECHO REPLY受信部分 */
            memset(buf, 0, sizeof(buf));

            /* 相手ホストからのICMP ECHO REPLYを待ち */
            n = nn::socket::Recv(sock, buf, sizeof(buf), nn::socket::MsgFlag::Msg_None);
            if(n < 1)
            {
                NN_LOG("             Recv error : %d\n", nn::socket::GetLastError());
                ret = false;
            }

            /* 受信データからIPヘッダ部分へのポインタを取得 */
            iphdrptr = (struct ip *)buf;

            /* 受信データからICMPヘッダ部分へのポインタを取得 */
            icmpptr = (struct icmp *)(buf + (iphdrptr->ip_hl * 4));

            /* ICMPヘッダからICMPの種類を特定 */
            if(true == ret)
            {
                if(icmpptr->icmp_type == ICMP_ECHOREPLY)
                {
                    NN_LOG("             received ICMP ECHO REPLY\n");

                    if((0 < icd_seq) && (icd_seq != icmpptr->icmp_hun.ih_idseq.icd_seq))
                    {
                        NN_LOG("             Seq No error Send Seq : %d, Req Seq : &d\n", icd_seq, icmpptr->icmp_hun.ih_idseq.icd_seq);
                        ret = false;
                    }
                }
                else
                {
                    NN_LOG("             received ICMP %d\n", icmpptr->icmp_type);
                    ret = false;
                }
            }

            /* 終了 */
            sock = nn::socket::Close(sock);
            if(0 > sock)
            {
                NN_LOG("             Close error : %d\n", sock);
            }
            return ret;
        }

        static void aging_n117(void* arg) NN_NOEXCEPT
        {
            nn::os::Tick sysTick = nn::os::GetSystemTick();
            std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

            while( NN_STATIC_CONDITION(1) )
            {
                // 1時間連続
                if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h )
                {
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N117_TEST].scanSize, scanTestParam[SCAN_N117_TEST].scanParam));
                    DumpScanResultWithReader(pTestScanData.get());

                    // 100msec間隔でSCANを行う
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                }
                else
                {
                    break;
                }
            }
        }

        static void aging_n118(void* arg) NN_NOEXCEPT
        {
            nn::os::Tick sysTick = nn::os::GetSystemTick();
            std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

            while( NN_STATIC_CONDITION(1) )
            {
                // 1時間連続
                if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h )
                {
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
                    DumpScanResultWithReader(pTestScanData.get());

                    // 3分間隔でPING送信を行う
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time10s * 18));
                }
                else
                {
                    break;
                }
            }
        }

        static void TcpSend(void* arg) NN_NOEXCEPT
        {
            int* wait_time = static_cast<int*>(arg);
            tcp_Send.Send(*wait_time);
        }

        static void TcpRecv(void* arg) NN_NOEXCEPT
        {
            tcp_Recv.Receive();
        }

        static void iperfServer(void* arg) NN_NOEXCEPT
        {
            int    argC;
            char** argV;
            const  char* defaultArguments[] = { "iperf", "-s", "-i", "10", "-w", "262144" };
            argV = const_cast<char**>(defaultArguments);
            argC = sizeof(defaultArguments) / sizeof(defaultArguments[0]);

            testMain(argC, argV);
        }

        static void iperfClient(void* arg) NN_NOEXCEPT
        {
            int    argC;
            char** argV;
            const  char* defaultArguments[] = { "iperf", "-c", IpAdder, "-w","262144", "-t", "28800", "-i", "10" };
            argV = const_cast<char**>(defaultArguments);
            argC = sizeof(defaultArguments) / sizeof(defaultArguments[0]);

            testMain(argC, argV);
        }

        static void WlanStateThread(void* arg) NN_NOEXCEPT
        {
            bool* loopTime = static_cast<bool*>(arg);
            nn::wlan::WlanState new_status;
            nn::wlan::WlanState old_status;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&old_status));
            while( NN_STATIC_CONDITION(1) )
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&new_status));
                if(old_status != new_status)
                {
                    NN_LOG("             State : %s\n", WlanStateStr[new_status]);
                }
                old_status = new_status;

                if( false == *loopTime )
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time100m));
            }
        }

    };

    // インフラローカル通信クラス
    class InfraLocalFuncTest : public ::testing::Test
    {
    protected:
        const int64_t Time5m                = 5;
        const int64_t Time10m               = Time5m * 2;
        const int64_t Time20m               = Time10m * 2;
        const int64_t Time100m              = Time20m * 5;
        const int64_t Time1s                = Time100m * 10;
        const int64_t Time5s                = Time1s * 5;
        const int64_t Time30s               = Time5s * 6;
        const int64_t Time60s               = Time30s * 2;
        const int64_t Time300s              = Time60s * 5;
        const int64_t Time600s              = Time300s * 2;
        const int64_t Time1800s             = Time600s * 3;
        const int64_t Time1h                = Time1800s * 2;

        const size_t BufferSize200k         = 200 * 1024;
        const size_t RecvPacketSize2k       = 2 * 1024;
        const size_t SendPacketSize1400B    = 1400;
        const uint32_t RecvPacketMaxCount   = 0xFFFFFFFF;
        const int KLocalConcurrencyLimit    = 4;
        const bool IsEnabled = true;

        enum WlanTestState
        {
            WlanTestNone = 0,
            WlanTestInfraInit,
            WlanTestInfraOpen,
            WlanTestLocalInit,
            WlanTestLocalMasterOpen,
            WlanTestLocalClientOpen,
        };

        nn::wlan::WowlWakeCount wowlCount;
        nn::wlan::WowlSleepStats wowlStatus;

        WlanTestState testState;
        LocalApiClass localClass;
        nn::os::SystemEvent connectEvent;

        std::unique_ptr<char[]> wlanVertionMsg;

    protected:
        InfraLocalFuncTest() NN_NOEXCEPT
        {
            std::unique_ptr<char[]> wlanMessage(new char[512]);
            wlanVertionMsg = std::move(wlanMessage);
            testState = WlanTestNone;
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Infra);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            LogoutWlanVertion();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                KLocalConcurrencyLimit));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(WLAN_INTERFACE_NAME));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        // Wlanバージョン出力
        void LogoutWlanVertion() NN_NOEXCEPT
        {
            const size_t verSize = 512;
            nn::Result result;
            std::unique_ptr<char[]> verBuffer(new char[verSize]);
            if (verBuffer == nullptr)
            {
                NN_LOG("             Get Vertion Buffer Allocate Failed\n");
                return;
            }

            result = FunctionTestInfraOpen();
            if (result.IsSuccess() != true)
            {
                return;
            }

            result = nn::wlan::Infra::GetFwVersion(verBuffer.get(), verSize);
            if (result.IsSuccess() == true)
            {
                NN_LOG("             WLAN VER : %s", verBuffer.get());
            }
            else
            {
                NN_LOG("             GetFwVersion Failed : 0x%08x\n", result.GetInnerValueForDebug());
            }

            result = FunctionTestInfraClose();
            if (result.IsSuccess() == true)
            {
                NN_LOG("             FunctionTestInfraClose Failed : 0x%08x\n",
                    result.GetInnerValueForDebug());
            }
        }

        // Wlanインフラモードオープン
        nn::Result FunctionTestInfraOpen() NN_NOEXCEPT
        {
            nn::Result result;

            if (testState == WlanTestNone)
            {
                result = nn::wlan::InitializeInfraManager();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             Infra Initialize Failed : 0x%08x\n", result.GetInnerValueForDebug());
                    return result;
                }
                testState = WlanTestInfraInit;
            }

            if (testState == WlanTestInfraInit)
            {
                result = nn::wlan::Infra::OpenMode();
                if (result.IsSuccess() != true)
                {
                    nn::wlan::FinalizeInfraManager();
                    NN_LOG("             Infra OpenMode Failed : 0x%08x\n", result.GetInnerValueForDebug());
                    return result;
                }
                testState = WlanTestInfraOpen;

                nn::wlan::Infra::GetConnectionEvent(connectEvent.GetBase());
            }

            return nn::ResultSuccess();
        }

        // Wlanインフラモードステータスログ出力
        void FunctionTestInfraState()
        {
            nn::wlan::WlanState State;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetState(&State));
            NN_LOG("             %s\n", WlanStateStr[State]);
        }

        // Wlanインフラモード接続
        bool FunctionTestInfraConnect(TestConnectParams& ConParam, const int64_t msecWait) NN_NOEXCEPT
        {
            int64_t waitTime = std::min(msecWait, static_cast<int64_t>(Time20m));
            nn::wlan::ConnectionStatus connectionStatus;
            nn::os::Tick timeOut;
            bool isConnect = false;
            bool isWaitSygnal = false;
            nn::Result result;

            timeOut = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - timeOut).ToTimeSpan().GetMilliSeconds() < msecWait)
            {
                result = nn::wlan::Infra::Connect(
                    ConParam.ssid, ConParam.bssid, ConParam.channel,
                    ConParam.security, ConParam.autoKeepAlive, ConParam.beaconLostTimeout
                );
                if (result.IsSuccess() != true)
                {
                    break;
                }

                isWaitSygnal = connectEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(waitTime));
                if (isWaitSygnal != true)
                {
                    continue;
                }

                result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                if (result.IsSuccess() == true)
                {
                    if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                    {
                        isConnect = true;
                        break;
                    }
                }
            }
            return isConnect;
        }

        // Wlanインフラモード接続及びTCP接続
        bool FunctionTestInfraTcpConnect(TestConnectParams& ConParam, const int64_t msecWait,
                    WowlSocket& wowlClass, const StringIpv4* pClientIpv4, const char* pServerIp,
                    const int64_t tcpTimeOut) NN_NOEXCEPT
        {
            bool isSuccess = FunctionTestInfraConnect(ConParam, msecWait);
            if (isSuccess != true)
            {
                FunctionTestInfraTcpDisconnect(wowlClass);
                return false;
            }

            isSuccess = wowlClass.Init(pClientIpv4->ipAddress, pClientIpv4->mask,
                            pClientIpv4->gateWay, pClientIpv4->dns1, pClientIpv4->dns2);
            if (isSuccess != true)
            {
                FunctionTestInfraTcpDisconnect(wowlClass);
                NN_LOG("               wowlInfra init Fail : %d\n", __LINE__);
                return false;
            }

            // TCPでPC側のWowFunctionTestServerと接続（PC側のIPアドレスがDHCPの場合は都度ヘッダのIPAddrを変更）
            isSuccess = wowlClass.WowlConnectWait(pServerIp, tcpTimeOut);
            if (isSuccess != true)
            {
                FunctionTestInfraTcpDisconnect(wowlClass);
                FunctionTestInfraDisconnect();
                NN_LOG("               wowlInfra WowlConnectWait Fail : %d\n", __LINE__);
            }

            return isSuccess;
        }

        // Wowlスリープ待機・起床待機
        bool WowlSleepWait(WowlSocket& wowlInfra, const char* pMsg, const int32_t wakeLimmit)
        {
            nn::wlan::WowlWakeCount wowlWakeCount;
            nn::wlan::WowlSleepStats wowlSleepSts;
            bool isResult = false;
            int32_t wakeCount = 0;

            // Wowl起床要因ステータスをクリア
            wowlInfra.WowlClearWakeupState();
            // ステータスクリア後のWowl起床要因ステータス確認
            wowlInfra.WowlGetWakeupState(&wowlWakeCount, &wowlSleepSts);
            NN_LOG("              Start WowlFunctionTestServer on PC in Advance.\n");
            NN_LOG("              Press Power button on SDEV Or EDEV to Sleep State.\n");
            NN_LOG("              %s\n", pMsg);
            // 実機のスリープ状態解除後のコントローラでHOMEキーが押された回数でループから終了する条件
            while (wakeCount < wakeLimmit)
            {
                // ここでWowlスリープ待機。実機の電源ボタン押下でWowlスリープ状態にする。
                // Wowlスリープ状態解除後、HOMEキーを押下しないと復帰しない
                isResult = wowlInfra.WowlReceiveWait(Time1s);
                wowlInfra.WowlGetWakeupState(&wowlWakeCount, &wowlSleepSts);
                if (isResult == true)
                {
                    // 切断されてしまった場合
                    isResult = false;
                    NN_LOG("              Wakeup State Fail.\n");
                    break;
                }

                // 起床カウントされた条件(スリープ解除後にHOMEキー押下で復帰)
                if (wowlSleepSts.cpuWakes != 0)
                {
                    wowlInfra.WowlGetWakeupState(&wowlWakeCount, &wowlSleepSts);
                    wowlCount = wowlWakeCount;
                    wowlStatus = wowlSleepSts;
                    wowlInfra.WowlClearWakeupState();
                    wakeCount++;
                    isResult = true;
                    NN_LOG("              Home Key Wakeup.\n");
                }
            }

            return isResult;
        }

        // Wowlスリープ起床１回テスト
        bool WowlSleepTest(const char* testNo, const int32_t mode, const StringIpv4* pClientIpv4,
            const char* pServerIp, const char* pMsg, const int64_t tcpTimeOut) NN_NOEXCEPT
        {
            return WowlSleepTest(testNo, mode, pClientIpv4, pServerIp, pMsg, tcpTimeOut, 1);
        }

        // Wowlスリープ起床指定回数テスト
        bool WowlSleepTest(const char* testNo, const int32_t mode, const StringIpv4* pClientIpv4,
                             const char* pServerIp, const char* pMsg, const int64_t tcpTimeOut, const int32_t wakeLimmit) NN_NOEXCEPT
        {
            WowlSocket wowlInfra;
            bool isResult = false;

            if (testNo == nullptr || pClientIpv4 == nullptr ||
                pServerIp == nullptr || pMsg == nullptr)
            {
                NN_LOG("               WowlSleepTest Argment Fail : %d\n", __LINE__);
                return false;
            }

            NN_LOG("             *** Infra %s WiFi AP%s Test Start ***\n", testNo, ConnectParams[mode].SettingNo);

            // 対象のWiFi-APで接続
            isResult = FunctionTestInfraConnect(ConnectParams[mode], Time1h);
            if (isResult != true)
            {
                NN_LOG("               Connect Fail : %d\n", __LINE__);
                return false;
            }

            isResult = wowlInfra.Init(pClientIpv4->ipAddress, pClientIpv4->mask, pClientIpv4->gateWay,
                pClientIpv4->dns1, pClientIpv4->dns2);
            if (isResult != true)
            {
                FunctionTestInfraDisconnect();
                NN_LOG("               wowlInfra init Fail : %d\n", __LINE__);
                return false;
            }

            // TCPでPC側のWowFunctionTestServerと接続（PC側のIPアドレスがDHCPの場合は都度ヘッダのIPAddrを変更）
            wowlInfra.WowlConnectWait(pServerIp, tcpTimeOut);

            isResult = wowlInfra.WowlReceiveStart(WlanWowlPoolSize, WlanWowlPakcetSize);
            WLANTEST_EXPECT_TRUE(isResult);
            if (isResult == true)
            {
                // Wowlスリープ待機
                isResult = WowlSleepWait(wowlInfra, pMsg, wakeLimmit);
            }
            // TCP受信停止
            WLANTEST_EXPECT_TRUE(wowlInfra.WowlReceiveStop());
            // TCP切断終了
            WLANTEST_EXPECT_TRUE(wowlInfra.WowlDisconnect());
            // WiFi-AP接続切断
            FunctionTestInfraDisconnect();

            NN_LOG("             *** Infra %s WiFi AP%s Test End ***\n", testNo, ConnectParams[mode].SettingNo);

            return isResult;
        }

        // Wowlスリープ起床後TCP再接続受信テスト
        bool WowlSleepAndTcpRestartTest(const char* testNo, const int32_t mode, const StringIpv4* pClientIpv4,
            const char* pServerIp, const char* pMsg, const int64_t tcpTimeOut,
            const int32_t wakeLimmit, const int64_t stopRecvPacket) NN_NOEXCEPT
        {
            WowlSocket wowlInfra;
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
            bool isResult = false;

            if (testNo == nullptr || pClientIpv4 == nullptr ||
                pServerIp == nullptr || pMsg == nullptr)
            {
                NN_LOG("               WowlSleepTest Argment Fail : %d\n", __LINE__);
                return false;
            }

            NN_LOG("             *** Infra %s WiFi AP%s Test Start ***\n", testNo, ConnectParams[mode].SettingNo);
            NN_LOG("               SSID : %s\n", ConnectParams[mode].ssid.GetHexString(ssidStr));
            // 対象のWiFi-APで接続及びTCP接続
            isResult = FunctionTestInfraTcpConnect(ConnectParams[mode], Time1h,
                            wowlInfra, pClientIpv4, pServerIp, tcpTimeOut);
            if (isResult != true)
            {
                NN_LOG("               Infra And Tcp Connect Fail : %d Line\n", __LINE__);
                return false;
            }
            NN_LOG("               Infra And Tcp Connected : %d Line\n", __LINE__);

            // TCP受信開始
            isResult = wowlInfra.WowlReceiveStart(WlanWowlPoolSize, WlanWowlPakcetSize);

            // Wowlスリープ待機
            isResult = WowlSleepWait(wowlInfra, pMsg, wakeLimmit);

            // 対象のWiFi-APからの切断及びTCP切断
            FunctionTestInfraTcpDisconnect(wowlInfra);
            if (isResult != true)
            {
                NN_LOG("              WowlSleepWait Fail : %d Line\n", __LINE__);
                return false;
            }

            // 対象のWiFi-APで接続及びTCP接続
            isResult = FunctionTestInfraTcpConnect(ConnectParams[mode], Time1h,
                wowlInfra, pClientIpv4, pServerIp, tcpTimeOut);
            if (isResult != true)
            {
                NN_LOG("               Infra And Tcp Connect Fail : %d Line\n", __LINE__);
                return false;
            }
            NN_LOG("               Infra And Tcp Connected : %d Line\n", __LINE__);

            // TCP受信開始
            isResult = wowlInfra.WowlReceiveStart(WlanWowlPoolSize, WlanWowlPakcetSize);
            WLANTEST_EXPECT_TRUE(isResult);
            if (isResult == true)
            {
                // Wowlスリープ後のTCP通信
                wowlInfra.WowlClearRecvCount();
                while ((isResult == true) &&
                    wowlInfra.WowlGetRecvCount() < stopRecvPacket)
                {
                    isResult = wowlInfra.WowlReceiveWait(Time1s);
                    if (isResult == true)
                    {
                        // 切断されてしまった場合
                        isResult = false;
                        NN_LOG("               WowlReceiveWait Fail : %d\n", __LINE__);
                        break;
                    }
                    isResult = true;
                    NN_LOG("               WowlReceive Count : %d\n", wowlInfra.WowlGetRecvCount());
                }
            }
            // 対象のWiFi-APからの切断及びTCP切断
            FunctionTestInfraTcpDisconnect(wowlInfra);

            NN_LOG("             *** Infra %s WiFi AP%s Test End ***\n", testNo, ConnectParams[mode].SettingNo);

            return isResult;
        }

        // Wowlスリープ起床後TCP受信テスト
        bool WowlSleepAndTcpTest(const char* testNo, const int32_t mode, const StringIpv4* pClientIpv4,
                                const char* pServerIp, const char* pMsg, const int64_t tcpTimeOut,
                                const int32_t wakeLimmit, const int64_t stopRecvPacket) NN_NOEXCEPT
        {
            WowlSocket wowlInfra;
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
            bool isResult = false;

            if (testNo == nullptr || pClientIpv4 == nullptr ||
                pServerIp == nullptr || pMsg == nullptr)
            {
                NN_LOG("               WowlSleepTest Argment Fail : %d\n", __LINE__);
                return false;
            }

            NN_LOG("             *** Infra %s WiFi AP%s Test Start ***\n", testNo, ConnectParams[mode].SettingNo);
            NN_LOG("               SSID : %s\n", ConnectParams[mode].ssid.GetHexString(ssidStr));
            // 対象のWiFi-APで接続
            isResult = FunctionTestInfraConnect(ConnectParams[mode], Time1h);
            if (isResult != true)
            {
                NN_LOG("               Infra Connect Fail : %d Line\n", __LINE__);
                return false;
            }
            NN_LOG("               Infra Connected : %d Line\n", __LINE__);

            isResult = wowlInfra.Init(pClientIpv4->ipAddress, pClientIpv4->mask, pClientIpv4->gateWay,
                pClientIpv4->dns1, pClientIpv4->dns2);
            if (isResult != true)
            {
                FunctionTestInfraDisconnect();
                NN_LOG("               wowlInfra init Fail : %d\n", __LINE__);
                return false;
            }

            // TCPでPC側のWowFunctionTestServerと接続（PC側のIPアドレスがDHCPの場合は都度ヘッダのIPAddrを変更）
            wowlInfra.WowlConnectWait(pServerIp, tcpTimeOut);

            isResult = wowlInfra.WowlReceiveStart(WlanWowlPoolSize, WlanWowlPakcetSize);
            WLANTEST_EXPECT_TRUE(isResult);
            if (isResult == true)
            {
                // Wowlスリープ待機
                isResult = WowlSleepWait(wowlInfra, pMsg, wakeLimmit);

                // Wowlスリープ後のTCP通信
                wowlInfra.WowlClearRecvCount();
                while ((isResult == true) &&
                    wowlInfra.WowlGetRecvCount() < stopRecvPacket)
                {
                    isResult = wowlInfra.WowlReceiveWait(Time1s);
                    if (isResult == true)
                    {
                        // 切断されてしまった場合
                        isResult = false;
                        NN_LOG("               WowlReceiveWait Fail : %d\n", __LINE__);
                        break;
                    }
                    isResult = true;
                    NN_LOG("               WowlReceive Count : %d\n", wowlInfra.WowlGetRecvCount());
                }
            }
            // TCP受信停止
            WLANTEST_EXPECT_TRUE(wowlInfra.WowlReceiveStop());
            // TCP切断終了
            WLANTEST_EXPECT_TRUE(wowlInfra.WowlDisconnect());
            // WiFi-AP接続切断
            FunctionTestInfraDisconnect();

            NN_LOG("             *** Infra %s WiFi AP%s Test End ***\n", testNo, ConnectParams[mode].SettingNo);

            return isResult;
        }

        // Wlanローカルマスター通信テスト（L106試験同等）
        bool WowlLocalMasterTest(const nn::wlan::MasterBssParameters & bssPrm,
                                        const int64_t testTime) NN_NOEXCEPT
        {
            LocalApiClass localMaster;
            size_t matchCount = sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]);
            LocalConnectsInfo connectsInfo;
            nn::os::Tick testTick, logOutTime;
            uint32_t connectCount = 0;
            bool isResult = false;

            // MASTERセットアップ
            isResult = localMaster.SetupLocalMaster(bssPrm, TestMatchInfo, matchCount);
            if (isResult != true)
            {
                NN_LOG("               LocalMaster Setup Fail : %d\n", __LINE__);
                return isResult;
            }

            // MASTER受信スレッド起動
            isResult = localMaster.LocalReceiveFrameStart(BufferSize200k, RecvPacketSize2k,
                                                            1, 1, RecvPacketMaxCount);
            if (isResult != true)
            {
                NN_LOG("               LocalMaster LocalReceiveFrameStart Fail : %d\n", __LINE__);
                return isResult;
            }

            // CLIENT接続待機
            isResult = localMaster.LocalReceiveWait(Time1800s);
            if (isResult != true)
            {
                NN_LOG("               LocalMaster LocalReceiveWait Fail : %d\n", __LINE__);
                return isResult;
            }

            // 送信スレッド起動
            isResult = localMaster.LocalSendFrameStart(SendPacketSize1400B, testTime, 0, true);
            if (isResult != true)
            {
                NN_LOG("               LocalMaster LocalSendFrameStart Fail : %d\n", __LINE__);
                return isResult;
            }

            testTick = nn::os::GetSystemTick();
            logOutTime = testTick;

            localMaster.LocalGetConnectInfo(&connectsInfo);
            while ((nn::os::GetSystemTick() - testTick).ToTimeSpan().GetMilliSeconds() <= testTime)
            {
                isResult = nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent,
                                                            nn::TimeSpan::FromMilliSeconds(Time5s));
                // シグナル判定
                if (isResult == true)
                {
                    connectCount = localMaster.GetClientMacAddress(localMaster.m_clientMacArray,
                                                            nn::wlan::ConnectionState_Connected);
                    WLANTEST_EXPECT_TRUE(connectCount != 0);
                    if (connectCount == 0)
                    {
                        // クライアントから切断されたらNG判定
                        isResult = false;
                        break;
                    }
                    continue;
                }

                // 通信状態ログ
                if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
                {
                    logOutTime = nn::os::GetSystemTick();
                    localMaster.LocalGetConnectInfo(&connectsInfo);
                    localMaster.LocalTraceConnectsInfo(true);
                }
                // 試験時間まで実行ならPass判定
                isResult = true;
            }

            // 送信スレッド停止
            localMaster.LocalSendFrameStop();

            // MASTER受信スレッド停止
            localMaster.LocalReceiveFrameStop();

            localMaster.LocalTraceConnectsInfo(true);

            return isResult;
        }

        // Wlanローカルクライアント通信テスト（L106試験同等）
        bool WowlLocalClientTest(const TestLocalConnectParams & ConnectParam) NN_NOEXCEPT
        {
            LocalApiClass localClient;
            size_t matchCount = sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]);
            LocalConnectsInfo connectsInfo;
            nn::os::Tick logOutTime;
            nn::wlan::ConnectionStatus clientStatus;
            nn::Result result;
            bool isResult = false;

            // CLIENTセットアップ
            isResult = localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security,
                                                    TestMatchInfo, matchCount, true, false);
            if (isResult != true)
            {
                NN_LOG("               localClient Setup Fail : %d\n", __LINE__);
                return isResult;
            }

            isResult = localClient.LocalClientConnect(Time1800s, ConnectParam.channel);
            if (isResult != true)
            {
                NN_LOG("               localClient LocalClientConnect Fail : %d\n", __LINE__);
                return isResult;
            }

            // 受信スレッド起動
            isResult = localClient.LocalReceiveFrameStart(BufferSize200k, RecvPacketSize2k,
                                                            1, 1, RecvPacketMaxCount);
            if (isResult != true)
            {
                NN_LOG("               localClient LocalReceiveFrameStart Fail : %d\n", __LINE__);
                return isResult;
            }

            // 通信開始パケット受信待機
            isResult = localClient.LocalReceiveWait(Time1800s, Time1s, true);
            if (isResult != true)
            {
                NN_LOG("               localClient LocalReceiveWait Fail : %d\n", __LINE__);
                return isResult;
            }

            // 送信スレッド起動
            isResult = localClient.LocalSendFrameStart(WlanTestRatePacketBufferSize, Time1h, 0, false);
            if (isResult != true)
            {
                NN_LOG("               localClient LocalSendFrameStart Fail : %d\n", __LINE__);
                return isResult;
            }
            logOutTime = nn::os::GetSystemTick();

            while (NN_STATIC_CONDITION(1))
            {
                // MASTERからの接続が切られるまで送信し続ける
                if (nn::os::TimedWaitSystemEvent(&localClient.m_connectEvent,
                        nn::TimeSpan::FromMilliSeconds(Time5s)) == true)
                {
                    nn::wlan::Local::GetConnectionStatus(&clientStatus);
                    if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }
                    continue;
                }

                // 通信ログ
                if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    logOutTime = nn::os::GetSystemTick();
                    localClient.LocalGetConnectInfo(&connectsInfo);
                }
            }

            // 送信スレッド停止
            localClient.LocalSendFrameStop();
            // 受信スレッド停止
            localClient.LocalReceiveFrameStop();
            // 通信結果ログ
            localClient.LocalTraceConnectsInfo(true);
            // 接続切断
            isResult = localClient.LocalClientDisconnect(Time30s);

            return isResult;
        }

        // Wlanインフラモード切断
        void FunctionTestInfraDisconnect() NN_NOEXCEPT
        {
            nn::os::Tick sysTick2;
            nn::os::SystemEvent conEvent;
            nn::wlan::ConnectionStatus connectionStatus;
            nn::Result result;
            NN_LOG("               FunctionTestInfraDisconnect Start\n");

            result = nn::wlan::Infra::GetConnectionEvent(conEvent.GetBase());
            WLANTEST_EXPECT_RESULT_SUCCESS(result);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("               nn::wlan::Infra::Disconnect Start\n");
                result = nn::wlan::Infra::Disconnect();
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                NN_LOG("               nn::wlan::Infra::Disconnect End\n");
                if (result.IsSuccess() != true)
                {
                    return;
                }
            }
            sysTick2 = nn::os::GetSystemTick();

            while (result.IsSuccess() == true)
            {
                // イベントシグナル状態の確認
                if (conEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(Time20m)) == true)
                {
                    FunctionTestInfraState();
                }

                result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                if (result.IsSuccess() != true)
                {
                    return;
                }

                if ((nn::os::GetSystemTick() - sysTick2).ToTimeSpan().GetMilliSeconds() > Time100m)
                {
                    NN_LOG("             Connection Status 100msec over!!\n");
                    WLANTEST_EXPECT_TRUE(false);
                    return;
                }
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             WlanTest: DIS CONNECTED!!!\n");
                    break;
                }
            }
            NN_LOG("               FunctionTestInfraDisconnect End\n");
        }

        // Wlanインフラモード切断及びTCP切断
        void FunctionTestInfraTcpDisconnect(WowlSocket& wowlClass) NN_NOEXCEPT
        {
            wowlClass.WowlReceiveStop();
            wowlClass.WowlDisconnect();
            FunctionTestInfraDisconnect();
        }

        // Wlanインフラモードクローズ
        nn::Result FunctionTestInfraClose() NN_NOEXCEPT
        {
            nn::Result result;

            if (testState == WlanTestInfraOpen)
            {
                result = nn::wlan::Infra::CloseMode();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             Infra CloseMode Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return result;
                }
                testState = WlanTestInfraInit;
            }

            if (testState == WlanTestInfraInit)
            {
                result = nn::wlan::FinalizeInfraManager();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             Infra Finalize Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return result;
                }
                testState = WlanTestNone;
            }

            return nn::ResultSuccess();
        }

        // Wlanローカルマスターモードオープン（BT指定）
        bool FunctionTestLocalMasterOpen(LocalApiBtNode btNode) NN_NOEXCEPT
        {
            nn::Result result;
            bool isResult;

            if (testState == WlanTestNone)
            {
                result = nn::wlan::InitializeLocalManager();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalMaster InitializeLocalManager Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestLocalInit;
            }

            if (testState == WlanTestLocalInit)
            {
                // BT APIのnn::btm::RegisterSystemEventForConnectedDeviceConditionを繰り返し呼び出すとアボート停止する。
                // そのためため、従来通りの初回のみ実施
                if (g_isBtSet != true)
                {
                    localClass.InitializeBtm();
                    isResult = localClass.SetBtMode(btNode);
                    if (isResult != true)
                    {
                        NN_LOG("             LocalMaster SetBtMode Failed Line : %d\n", __LINE__);
                        return false;
                    }
                    g_isBtSet = true;
                }

                result = nn::wlan::Local::OpenMasterMode();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             OpenMasterMode Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestLocalMasterOpen;
            }

            return true;
        }

        // Wlanローカルマスターモードクローズ
        bool FunctionTestLocalMasterClose() NN_NOEXCEPT
        {
            nn::Result result;

            if (testState == WlanTestLocalMasterOpen)
            {
                result = nn::wlan::Local::CloseMasterMode();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             CloseMasterMode Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }

                localClass.LocalRelease();
                testState = WlanTestLocalInit;
            }

            if (testState == WlanTestLocalInit)
            {
                result = nn::wlan::FinalizeLocalManager();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalMaster FinalizeLocalManager Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestNone;
            }

            return true;
        }

        // Wlanローカルクライアントモードオープン（BT指定）
        bool FunctionTestLocalClientOpen(LocalApiBtNode btNode) NN_NOEXCEPT
        {
            nn::Result result;
            bool isResult;

            if (testState == WlanTestNone)
            {
                result = nn::wlan::InitializeLocalManager();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalClient InitializeLocalManager Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestLocalInit;
            }

            if (testState == WlanTestLocalInit)
            {
                // BT APIを初回以外で呼び出すと応答がなくなるため、従来通りの初回のみ実施
                if (g_isBtSet != true)
                {
                    localClass.InitializeBtm();
                    isResult = localClass.SetBtMode(btNode);
                    if (isResult != true)
                    {
                        NN_LOG("             LocalClient SetBtMode Failed Line : %d\n", __LINE__);
                        return false;
                    }
                    g_isBtSet = true;
                }

                result = nn::wlan::Local::OpenClientMode();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             OpenClientMode Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestLocalClientOpen;
            }

            return true;
        }

        // Wlanローカルクライアントモードクローズ
        bool FunctionTestLocalClientClose() NN_NOEXCEPT
        {
            nn::Result result;

            if (testState == WlanTestLocalClientOpen)
            {
                result = nn::wlan::Local::CloseClientMode();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             CloseClientMode Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestLocalInit;
            }

            if (testState == WlanTestLocalInit)
            {
                result = nn::wlan::FinalizeLocalManager();
                if (result.IsSuccess() != true)
                {
                    NN_LOG("             LocalClient FinalizeLocalManager Failed : 0x%08x\n",
                            result.GetInnerValueForDebug());
                    return false;
                }
                testState = WlanTestNone;
            }

            return true;
        }
    };

    // 無線ドライバの初期化/終了処理を繰り返し
    TEST_F(InfraFuncTest, G2)
    {
        int i = 0;

        // 試験時間短縮フラグを確認
        int time = Time1h;
        if (g_IsShort) time = Time60s * 10;

        sysTick = nn::os::GetSystemTick();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
        while( NN_STATIC_CONDITION(1) )
        {
            // 1時間連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= time )
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
                i++;
            }
            else{
                break;
            }
        }
        NN_LOG("             Cnt %d\n", i);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());

        Connect(AP1_1_WPAAES);
        Disconnect();
    }

    // 無線ドライバの初期化/終了処理 任意の単チャネルのアクティブスキャン
    TEST_F(InfraFuncTest, G1_N1)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N1_TEST].scanSize, scanTestParam[SCAN_N1_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
    }

    // 任意の単チャネルのパッシブスキャン
    TEST_F(InfraFuncTest, N2)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N2_TEST].scanSize, scanTestParam[SCAN_N2_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
    }

    // 36chのステルス設定APをパッシブスキャン
    TEST_F(InfraFuncTest, N3)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N3_TEST].scanSize, scanTestParam[SCAN_N3_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
    }

    // 同一設定のAPが複数ある場合、各APを検出
    TEST_F(InfraFuncTest, N6)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N6_TEST].scanSize, scanTestParam[SCAN_N6_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
    }

    // 無線ネットワークがxxxある場合、全ての無線ネットワークを検出できること
    TEST_F(InfraFuncTest, N8)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
        WlanState();
        DumpScanResultWithReader(pTestScanData.get());
    }

    // 前回のスキャンから、直近のスキャンの間にAPのSSIDが変わったものがある場合、新しいAP設定としてスキャンできていること
    // 前回のスキャンから、直近のスキャンの間にAPの動作チャネルが変わったものがある場合、新しいAP設定としてスキャンできていること
    // 前回のスキャンから、直近のスキャンの間にAPの認証/暗号設定が変わったものがある場合、新しいAP設定としてスキャンできていること
    TEST_F(InfraFuncTest, N9_10_11)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
        WlanState();
        DumpScanResultWithReader(pTestScanData.get());
    }

    // スキャン中に、スキャンのキャンセル処理
    TEST_F(InfraFuncTest, N12)
    {
        int sleep_tim = 0;
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        nn::os::InitializeEvent(&scanStopEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanStopThread, StopScanThread, &scanStopEvent, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        // 1～10msecでスリープを行い、キャンセルのタイミングをばらけさせる
        sleep_tim = GetRandom(1, 10);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleep_tim));
        nn::os::StartThread(&scanStopThread);

        nn::os::SignalEvent(&scanStopEvent);
        NN_LOG("             StartScan start\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N12_TEST].scanSize, scanTestParam[SCAN_N12_TEST].scanParam));
        NN_LOG("             StartScan End\n");

        nn::os::WaitThread(&scanStopThread);
        nn::os::DestroyThread(&scanStopThread);
        nn::os::FinalizeEvent(&scanStopEvent);
    }

    // APと接続中に特定の1chのアクティブスキャンを行えること
    TEST_F(InfraFuncTest, N13)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

        Connect(AP1_3_WPA2TKIP);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N1_TEST].scanSize, scanTestParam[SCAN_N1_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
        Disconnect();
    }

    // APと接続中に特定の1chのパッシブスキャン
    TEST_F(InfraFuncTest, N14)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

        Connect(AP1_3_WPA2TKIP);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N2_TEST].scanSize, scanTestParam[SCAN_N2_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
        Disconnect();
    }

    // APと接続してTCPの連続通信中に特定の1chのスキャン
    TEST_F(InfraFuncTest, N15)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

        Connect(AP1_1_WPAAES);
        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s / 2));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N15_TEST].scanSize, scanTestParam[SCAN_N15_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());

        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);

        Disconnect();
    }

    // APと接続中に全チャネルのスキャン
    TEST_F(InfraFuncTest, N16)
    {
        int16_t channel;
        uint32_t bssCount;
        bool channelRet[6] = {false};
        Connect(AP1_1_WPAAES);

        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N12_TEST].scanSize, scanTestParam[SCAN_N12_TEST].scanParam));

        nn::wlan::BeaconScanResultReader resultReader(pTestScanData.get());
        bssCount = resultReader.GetCount();
        for(uint32_t i = 0; i < bssCount; i++)
        {
            nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
            channel = beacon.GetChannel();

            switch(channel)
            {
            case GetChannel_1:
             channelRet[0] = true;
            break;
            case GetChannel_48:
             channelRet[1] = true;
            break;
            case GetChannel_52:
             channelRet[2] = true;
            break;
            case GetChannel_100:
             channelRet[3] = true;
            break;
            case GetChannel_128:
             channelRet[4] = true;
            break;
            case GetChannel_140:
             channelRet[5] = true;
            break;
            // defaultは何もしない
            default:
            break;
            }
        }
#ifdef SDEVEDEV
        if((false == channelRet[0]) || (false == channelRet[1]) || (false == channelRet[2]))
        {   // Channel 1,48,52のいずれかが見つからない場合はNG
            if(false == channelRet[0])
            {
                 NN_LOG("             Channel 1 NG\n");
            }
            else if(false == channelRet[1])
            {
                 NN_LOG("             Channel 48 NG\n");
            }
            else if(false == channelRet[2])
            {
                 NN_LOG("             Channel 52 NG\n");
            }
            WLANTEST_ASSERT_TRUE(false);
        }
#else
        if((false == channelRet[0]) || (false == channelRet[1]) || (false == channelRet[2]) ||
           (false == channelRet[3]) || (false == channelRet[4]) || (false == channelRet[5]))
        {  // Channel 1,48,52,100,128,140のいずれかが見つからない場合はNG
            if(false == channelRet[0])
            {
                 NN_LOG("             Channel 1 NG\n");
            }
            else if(false == channelRet[1])
            {
                 NN_LOG("             Channel 48 NG\n");
            }
            else if(false == channelRet[2])
            {
                 NN_LOG("             Channel 52 NG\n");
            }
            else if(false == channelRet[3])
            {
                 NN_LOG("             Channel 100 NG\n");
            }
            else if(false == channelRet[4])
            {
                 NN_LOG("             Channel 128 NG\n");
            }
            else if(false == channelRet[5])
            {
                 NN_LOG("             Channel 140 NG\n");
            }
            WLANTEST_ASSERT_TRUE(false);
        }
#endif
        Disconnect();
    }

    // APと接続してTCPの連続通信中に特定の全チャネルのスキャン
    TEST_F(InfraFuncTest, N17)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

        Connect(AP1_1_WPAAES);

        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s / 2));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());

        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);

        Disconnect();
    }

    // OPEN/PLAIN設定のAPに接続 能動的にAPとの接続を切断
    TEST_F(InfraFuncTest, N18_N56)
    {
        Connect(N18TEST_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // OPEN/WEP64設定のAPに接続できること
    TEST_F(InfraFuncTest, N19)
    {
        Connect(AP1_2_WEP64, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // OPEN/WEP128設定のAPに接続できること
    TEST_F(InfraFuncTest, N20)
    {
        Connect(AP1_4_WEP128, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // SHARED/WEP64設定のAPに接続できること(暗号鍵が一致している場合)
    TEST_F(InfraFuncTest, N21)
    {
        Connect(AP5_1_SHAREDWEP64, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // SHARED/WEP128設定のAPに接続できること(暗号鍵が一致している場合)
    TEST_F(InfraFuncTest, N22)
    {
        Connect(AP5_2_SHAREDWEP128, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // SHARED/WEP64設定のAPに接続しないこと(暗号鍵が不一致の場合)
    TEST_F(InfraFuncTest, N23)
    {
        int count = 0;

        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N23_SHAREDWEP64].ssid,
                                                                    ConnectParams[N23_SHAREDWEP64].bssid,
                                                                    ConnectParams[N23_SHAREDWEP64].channel,
                                                                    ConnectParams[N23_SHAREDWEP64].security,
                                                                    ConnectParams[N23_SHAREDWEP64].autoKeepAlive,
                                                                    ConnectParams[N23_SHAREDWEP64].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
               count ++;
               if(3 <= count) break;
            }
        }
        nn::os::DestroySystemEvent(&connectionEvent);
    }

    // SHARED/WEP128設定のAPに接続できること(暗号鍵が不一致の場合)
    TEST_F(InfraFuncTest, N24)
    {
        int count = 0;

        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N24_SHAREDWEP128].ssid,
                                                                    ConnectParams[N24_SHAREDWEP128].bssid,
                                                                    ConnectParams[N24_SHAREDWEP128].channel,
                                                                    ConnectParams[N24_SHAREDWEP128].security,
                                                                    ConnectParams[N24_SHAREDWEP128].autoKeepAlive,
                                                                    ConnectParams[N24_SHAREDWEP128].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
               count ++;
               if(3 <= count) break;
            }
        }
        nn::os::DestroySystemEvent(&connectionEvent);
    }

    // WPA-PSK(TKIP)設定のAPに接続(パスフレーズが一致している場合)
    TEST_F(InfraFuncTest, N25)
    {
        Connect(AP3_1_WPATKIP, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // WPA-PSK(TKIP)設定のAPに接続(パスフレーズが不一致の場合)
    TEST_F(InfraFuncTest, N26)
    {
        int count = 0;

        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N26TEST_WPATKIP].ssid,
                                                                    ConnectParams[N26TEST_WPATKIP].bssid,
                                                                    ConnectParams[N26TEST_WPATKIP].channel,
                                                                    ConnectParams[N26TEST_WPATKIP].security,
                                                                    ConnectParams[N26TEST_WPATKIP].autoKeepAlive,
                                                                    ConnectParams[N26TEST_WPATKIP].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
               count ++;
               if(3 <= count) break;
            }
        }
        nn::os::DestroySystemEvent(&connectionEvent);
    }

    // WPA-PSK(AES)設定のAPに接続(パスフレーズが一致している場合)
    TEST_F(InfraFuncTest, N27)
    {
        Connect(AP1_1_WPAAES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // WPA-PSK(AES)設定のAPに接続しないこと(パスフレーズが不一致の場合)
    TEST_F(InfraFuncTest, N28)
    {
        int count = 0;

        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N28TEST_WPAPSK_PASSNG].ssid,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].bssid,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].channel,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].security,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].autoKeepAlive,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
               count ++;
               if(3 <= count) break;
            }
        }
        nn::os::DestroySystemEvent(&connectionEvent);
    }

    // WPA2-PSK(AES)設定のAPに接続(パスフレーズが一致している場合)
    TEST_F(InfraFuncTest, N29_N33)
    {
        Connect(AP4_1_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // WPA2-PSK(AES)設定のAPに接続(パスフレーズが不一致の場合)
    TEST_F(InfraFuncTest, N30)
    {
        int count = 0;

        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N30TEST_WPA2AES].ssid,
                                                                    ConnectParams[N30TEST_WPA2AES].bssid,
                                                                    ConnectParams[N30TEST_WPA2AES].channel,
                                                                    ConnectParams[N30TEST_WPA2AES].security,
                                                                    ConnectParams[N30TEST_WPA2AES].autoKeepAlive,
                                                                    ConnectParams[N30TEST_WPA2AES].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
               count ++;
               if(3 <= count) break;
            }
        }
        nn::os::DestroySystemEvent(&connectionEvent);
    }

    // WPA/WPA2-PSK(AES)設定のAPに対して、WPA2-PSK(AES)設定で接続すること
    TEST_F(InfraFuncTest, N31)
    {
        Connect(AP1_1_WPAAES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // WPA/WPA2-PSK(MIX)設定のAPに対して、WPA2-PSK(AES)設定で接続すること
    TEST_F(InfraFuncTest, N32)
    {
        Connect(N32TEST_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // パスフレーズ63文字設定でAPとWPA2-PSK(AES)接続
    TEST_F(InfraFuncTest, N34)
    {
        Connect(N34TEST_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 接続処理中に、接続動作をキャンセル
    TEST_F(InfraFuncTest, N36)
    {
        int sleepTim = 0;
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        nn::os::InitializeEvent(&connectCancelEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&connectCancelThread, CancelConnectThread, &connectCancelEvent, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&connectCancelThread);

        nn::os::SignalEvent(&connectCancelEvent);
        // 1～10msecでスリープを行い、キャンセルのタイミングをばらけさせる
        sleepTim = GetRandom(1, 10);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTim));

        // キャンセルされても ConnectionEvent がシグナルされない仕様のため直接Connectを呼ぶ
        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WlanTest: Connect Start!!!\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[AP2_1_WPA2AES].ssid,
                                                                    ConnectParams[AP2_1_WPA2AES].bssid,
                                                                    ConnectParams[AP2_1_WPA2AES].channel,
                                                                    ConnectParams[AP2_1_WPA2AES].security,
                                                                    ConnectParams[AP2_1_WPA2AES].autoKeepAlive,
                                                                    ConnectParams[AP2_1_WPA2AES].beaconLostTimeout));
            NN_LOG("             WlanTest: Connect End!!!\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("             WlanTest: CONNECTED!!!\n");
                break;
            }
            // 3秒待って再試行
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }
        Disconnect();

        nn::os::WaitThread(&connectCancelThread);
        nn::os::DestroyThread(&connectCancelThread);
        nn::os::FinalizeEvent(&connectCancelEvent);
    }

    // WPA2-PSKのAPに5秒以内に接続できる
    TEST_F(InfraFuncTest, N125)
    {
        Connect(AP4_1_WPA2AES, true);

        Disconnect();
    }

    // 1文字のSSIDを持つAPに接続できること
    TEST_F(InfraFuncTest, N37)
    {
        Connect(AP1_3_WPA2TKIP, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 32文字のSSIDを持つAPに接続
    TEST_F(InfraFuncTest, N38)
    {
        Connect(N38TEST_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11n 20MHz OPEN設定のAPに、802.11n 20MHz設定で接続
    TEST_F(InfraFuncTest, N39)
    {
        Connect(N18TEST_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11n 40MHz OPEN設定のAPに、802.11n 40MHz設定で接続
    TEST_F(InfraFuncTest, N40)
    {
        Connect(N40TEST_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11ac 20MHz OPEN設定のAPに、802.11ac 20MHz設定で接続
    TEST_F(InfraFuncTest, N41)
    {
        Connect(N41TEST_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11ac 40MHz OPEN設定のAPに、802.11ac 40MHz設定で接続
    TEST_F(InfraFuncTest, N42)
    {
        Connect(N42TEST_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11ac 80MHz OPEN/PLAIN設定のAPに、802.11ac 80MHz設定で接続すること
    TEST_F(InfraFuncTest, N43)
    {
        Connect(AP2_4_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11n 40MHz WPA2-PSK(AES)設定のAPに、802.11n 40MHz設定で接続
    TEST_F(InfraFuncTest, N44)
    {
        Connect(AP2_1_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 802.11ac 80MHz WPA2-PSK(AES)設定のAPに、802.11ac 80MHz設定で接続すること
    TEST_F(InfraFuncTest, N45)
    {
        Connect(AP2_3_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // ステルス設定のOPEN/PLAIN設定のAPに接続できること
    TEST_F(InfraFuncTest, N46)
    {
        Connect(AP2_2_OPEN, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // ステルス設定のWPA2-PSK(AES)設定のAPに接続できること
    TEST_F(InfraFuncTest, N47)
    {
        Connect(AP2_3_WPA2AES, true);

        WLANTEST_ASSERT_TRUE(PingStart());

        Disconnect();
    }

    // 指定したAPが、スキャン対象のチャネル外に存在する場合、APと接続しないこと
    // 試験実施にはAP1-3を36ch、AP2-3を56chに設定しておく
    TEST_F(InfraFuncTest, N49)
    {
        int16_t channel;
        uint32_t bssCount;
        int count = 0;
        TestConnectParams ConnectParamsTest[] = {
            // AP設定 1-3
            { "1-3", nn::wlan::Ssid(ApSetting1_3_Ssid), nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
                { nn::wlan::SecurityMode_Wpa2Tkip, nn::wlan::SecurityMode_Wpa2Tkip, 0, "12345678" }, true, 10 },
            { "2-3",nn::wlan::Ssid(ApSetting2_3_Ssid), nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
                { nn::wlan::SecurityMode_Wpa2Aes, nn::wlan::SecurityMode_Wpa2Aes, 0, "AbcdefghijklmnopqrstuvwxyZ" }, true, 10 },
        };

        NN_LOG("             1-13ch\n");
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N49_1_TEST].scanSize, scanTestParam[SCAN_N49_1_TEST].scanParam));

        nn::wlan::BeaconScanResultReader resultReader(pTestScanData.get());
        bssCount = resultReader.GetCount();
        // 1-13chスキャンで36chが見つからないこと
        for(uint32_t i = 0; i < bssCount; i++)
        {
            nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
            channel = beacon.GetChannel();
            if(GetChannel_36 == channel)
            {
                WLANTEST_ASSERT_TRUE(false);
            }
        }
        // 指定したAPが、スキャン対象のチャネル外に存在する場合、APと接続しないこと
        for(int i = 0; i < scanTestParam[SCAN_N49_1_TEST].scanParam.channelCount; i++)
        {
            ConnectParamsTest[0].channel = scanTestParam[SCAN_N49_1_TEST].scanParam.channelList[i];
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

            while( NN_STATIC_CONDITION(1) )
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParamsTest[0].ssid,
                                                                        ConnectParamsTest[0].bssid,
                                                                        ConnectParamsTest[0].channel,
                                                                        ConnectParamsTest[0].security,
                                                                        ConnectParamsTest[0].autoKeepAlive,
                                                                        ConnectParamsTest[0].beaconLostTimeout));
                nn::os::WaitSystemEvent(&connectionEvent);
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    Disconnect();
                    WLANTEST_ASSERT_TRUE(false);
                    break;
                }
                else
                {
                   count ++;
                   if(3 <= count) break;
                }
                // 3秒待って再試行
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
            }
            nn::os::DestroySystemEvent(&connectionEvent);
            count = 0;
        }

        NN_LOG("             36-48ch\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N49_2_TEST].scanSize, scanTestParam[SCAN_N49_2_TEST].scanParam));

        nn::wlan::BeaconScanResultReader resultReader2(pTestScanData.get());
        bssCount = resultReader2.GetCount();
        // 36-48chスキャンで56chが見つからないこと
        for(uint32_t i = 0; i < bssCount; i++)
        {
            nn::wlan::BeaconDescriptionReader beacon2 = resultReader2.GetNextDescription();
            channel = beacon2.GetChannel();
            if(GetChannel_56 == channel)
            {
                WLANTEST_ASSERT_TRUE(false);
            }
        }
        // 指定したAPが、スキャン対象のチャネル外に存在する場合、APと接続しないこと
        for(int i = 0; i < scanTestParam[SCAN_N49_2_TEST].scanParam.channelCount; i++)
        {
            ConnectParamsTest[1].channel = scanTestParam[SCAN_N49_2_TEST].scanParam.channelList[i];
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

            while( NN_STATIC_CONDITION(1) )
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParamsTest[1].ssid,
                                                                        ConnectParamsTest[1].bssid,
                                                                        ConnectParamsTest[1].channel,
                                                                        ConnectParamsTest[1].security,
                                                                        ConnectParamsTest[1].autoKeepAlive,
                                                                        ConnectParamsTest[1].beaconLostTimeout));
                nn::os::WaitSystemEvent(&connectionEvent);
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    Disconnect();
                    WLANTEST_ASSERT_TRUE(false);
                    break;
                }
                else
                {
                   count ++;
                   if(3 <= count) break;
                }
                // 3秒待って再試行
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
            }
            nn::os::DestroySystemEvent(&connectionEvent);
            count = 0;
        }
    }

    // 2.4GHz帯に同一設定のAPが2台ある場合、SSID指定を指定すると電波が強いAPに接続
    TEST_F(InfraFuncTest, N50)
    {
        Connect(AP1_1_WPAAES);
        PrintConnectionStatus(&connectionStatus);
        Disconnect();
    }

    // 5GHz帯に同一設定のAPが2台ある場合、SSID指定を指定すると電波が強いAPに接続
    TEST_F(InfraFuncTest, N51)
    {
        Connect(AP2_3_WPA2AES);
        PrintConnectionStatus(&connectionStatus);
        Disconnect();
    }

    // 2.4GHzと5GHz帯に同一設定のAPがそれぞれある場合、SSID指定を指定すると電波が強いAPに接続
    TEST_F(InfraFuncTest, N52)
    {
        Connect(AP2_3_WPA2AES);
        PrintConnectionStatus(&connectionStatus);
        Disconnect();
    }

    // 同一設定のAPが2台存在する環境で、電波が強いAPがステルス設定の場合も、SSID指定を指定すると電波が強いAPに接続
    TEST_F(InfraFuncTest, N53)
    {
        Connect(AP2_3_WPA2AES);
        PrintConnectionStatus(&connectionStatus);
        Disconnect();
    }

    // 同一設定のAPが2台存在する環境で、BSSIDを使って接続するAPを選択
    TEST_F(InfraFuncTest, N54)
    {
        Connect(N54TEST_OPEN);
        PrintConnectionStatus(&connectionStatus);
        Disconnect();
    }

    // 接続中のAPの情報を知ることが出来る
    TEST_F(InfraFuncTest, N55)
    {
        int32_t rssi;
        nn::wlan::LinkLevel linkLevel;

        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        char macStr[nn::wlan::MacAddress::MacStringSize];

        Connect(AP2_1_WPA2AES);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetRssi(&rssi));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetLinkLevel(&linkLevel));

        NN_LOG("             +++ Connection Status +++\n");
        NN_LOG("             STATE           : %s\n", WlanConnectionStateStr[connectionStatus.state]);
        NN_LOG("             CAUSE           : %d\n", CauseOfInfo[connectionStatus.cause]);
        NN_LOG("             CHANNEL         : %d\n", connectionStatus.channel);
        NN_LOG("             SSID            : %s\n", connectionStatus.ssid.GetHexString(ssidStr));
        NN_LOG("             BSSID           : %s\n", connectionStatus.bssid.GetString(macStr));
        NN_LOG("             AID             : %d\n", connectionStatus.aid);
        NN_LOG("             REASON CODE     : %d\n", connectionStatus.statusReasonCode);
        NN_LOG("             CAPABILITY INFO : 0x%04X\n", connectionStatus.capabilityInfo);
        NN_LOG("             BEACON INTERVAL : %d\n", connectionStatus.beaconInterval);
        NN_LOG("             RSSI            : %d\n", rssi);
        NN_LOG("             LINK LEVEL      : %s\n", LinkLevel[linkLevel]);
        NN_LOG("             +++++++++++++++++++++++++\n");

        Disconnect();
    }

    // LDPCが無効となっていること キャプチャ取得
    TEST_F(InfraFuncTest, N121)
    {
        Connect(AP2_3_WPA2AES);
        Disconnect();
    }

    // GetConnectionStatusの値が即時に反映されること
    TEST_F(InfraFuncTest, N122)
    {
        sysTick = nn::os::GetSystemTick();
        nn::os::Tick sysTick2 = nn::os::GetSystemTick();
        int cnt = 0;

        while( NN_STATIC_CONDITION(1) )
        {
            Connect(AP2_1_WPA2AES);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(10, 100)));
            Disconnect();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(10, 100)));

            // 30分連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time1h / 2) )
            {
                break;
            }
            // 1分でログ表示
            if( (nn::os::GetSystemTick() - sysTick2).ToTimeSpan().GetMilliSeconds() > (Time60s) )
            {
                 cnt ++;
                 NN_LOG("             elapsed time:%dMinute\n", cnt);
                 sysTick2 = nn::os::GetSystemTick();
            }
        }
    }

    // APからの切断要求受信時、即座に切断が通知される
    TEST_F(InfraFuncTest, N57)
    {
        Connect(AP2_1_WPA2AES);
        NN_LOG("             Set automatic AP disconnection setting\n");

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        nn::os::DestroySystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("             Disconnection event\n");
        }
    }

    // Beaconタイムアウト設定が機能すること
    TEST_F(InfraFuncTest, N58)
    {
        // 10秒
        Connect(AP2_1_WPA2AES);

        // APロストさせる（電源断）
        NN_LOG("             Let AP Lost\n");
        sysTick = nn::os::GetSystemTick();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        nn::os::DestroySystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("             Disconnection event\n");
        }

        // 15秒 AP1
        Connect(N58TEST_WPAAES);

        // APロストさせる（電源断）
        NN_LOG("             Let AP Lost\n");
        sysTick = nn::os::GetSystemTick();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        nn::os::DestroySystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("             Disconnection event\n");
        }
    }

    // APと自発的な切断後、自動でAPと再接続しないこと
    TEST_F(InfraFuncTest, N59)
    {
        Connect(AP2_1_WPA2AES);
        Disconnect();

        sysTick = nn::os::GetSystemTick();
        NN_LOG("             Connection status check for 1 minute\n");
        while( NN_STATIC_CONDITION(1) )
        {
            // 1分間接続状態を確認する
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s )
            {
                nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    WLANTEST_ASSERT_TRUE(false);
                }
                // 300msec待つ
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300m));
            }
            else
            {
                break;
            }
        }
    }

    // APからの切断後、自動でAPと再接続しないこと
    TEST_F(InfraFuncTest, N60)
    {
        Connect(AP2_1_WPA2AES);
        NN_LOG("             Set automatic AP disconnection setting\n");

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        nn::os::DestroySystemEvent(&connectionEvent);

        sysTick = nn::os::GetSystemTick();
        NN_LOG("             Connection status check for 1 minute\n");
        while( NN_STATIC_CONDITION(1) )
        {
            // 1分間接続状態を確認する
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s )
            {
                nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
                if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    WLANTEST_ASSERT_TRUE(false);
                }
                // 300msec待つ
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300m));
            }
            else
            {
                break;
            }
        }
    }

    // DHCPサーバからIPアドレスが取得することができる
    TEST_F(InfraFuncTest, N61)
    {
        nn::bsdsocket::cfg::IfState ifState;

        Connect(AP1_1_WPAAES);
        ConfigureInterface(WLAN_INTERFACE_NAME, true);

        nn::Result result = nn::Result();

        sysTick = nn::os::GetSystemTick();

        result = nn::bsdsocket::cfg::GetIfState(WLAN_INTERFACE_NAME, &ifState);
        if(result.IsSuccess())
        {
            ElapsedTimeChk(IPaddresstime);  // 接続までの時間チェック
            NN_LOG("             GetIfState(%s):\n", WLAN_INTERFACE_NAME);
            NN_LOG("               mode          = %s\n", (ifState.mode == nn::bsdsocket::cfg::IfIpAddrMode_Dhcp) ? "DHCP" : "STATIC");
            NN_LOG("               metric        = %d\n", ifState.metric);
            NN_LOG("               mtu           = %d\n", ifState.mtu);
            NN_LOG("               addr          = %s\n", nn::socket::InetNtoa(ifState.addr));
            NN_LOG("               subnetMask    = %s\n", nn::socket::InetNtoa(ifState.subnetMask));
            NN_LOG("               gatewayAddr   = %s\n", nn::socket::InetNtoa(ifState.gatewayAddr));
            NN_LOG("               broadcastAddr = %s\n", nn::socket::InetNtoa(ifState.broadcastAddr));
            NN_LOG("               dnsAddrs[0]   = %s\n", nn::socket::InetNtoa(ifState.dnsAddrs[0]));
            NN_LOG("               dnsAddrs[1]   = %s\n", nn::socket::InetNtoa(ifState.dnsAddrs[1]));
            NN_LOG("               state         = %s\n", ifState.u.modeDhcp.currentState);
        }
        else{
            NN_LOG("             IP Address NG (Module      %d)\n", result.GetModule());
            NN_LOG("             IP Address NG (Description %d)\n", result.GetDescription());
            WLANTEST_ASSERT_TRUE(false);
        }
        Disconnect();
    }

    // 1秒間隔の1023BytesのPINGの導通が5分途切れず行える
    TEST_F(InfraFuncTest, N62)
    {
        int err_cnt = 0;

        Connect(AP2_1_WPA2AES);

        sysTick = nn::os::GetSystemTick();
        while( NN_STATIC_CONDITION(1) )
        {
            if( false == PingStart(ping_data_size_1023))
            {
                err_cnt ++;
            }
            if( 2 <= err_cnt)
            {
                WLANTEST_ASSERT_TRUE(false);
            }

            // 1秒間隔でPING送信を行う
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            // 指定時間後に抜ける
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time60s * 5) )
            {
                break;
            }
        }

        Disconnect();
    }

    // 16msec間隔の512BytesのPINGが5分途切れず行えること
    TEST_F(InfraFuncTest, N63)
    {
        int err_cnt = 0;
        int count   = 0;

        Connect(AP2_1_WPA2AES);

        sysTick = nn::os::GetSystemTick();
        nn::os::Tick sysTick2;

        while( NN_STATIC_CONDITION(1) )
        {
            if( false == PingStart(ping_data_size_512))
            {
                err_cnt ++;
            }
            else
            {
                count ++;
            }
            if( 2 <= err_cnt)
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            // 5msec間隔でPING送信を行う
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5m));

            // 指定時間後に抜ける
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time60s * 5) )
            {
                break;
            }
        }

        NN_LOG("             pkt count:%d\n", count);
        // 60pkt/sec 増減10%以外であればNG
        if((MaxPacketNum < count) || (MinPacketNum > count))
        {
            WLANTEST_ASSERT_TRUE(false);
        }
        Disconnect();

    }

    // TCPの連続送信を5分間行い、60Mbps以上(前回の結果からの変動幅10%以内)の通信が途切れずおこなえること
    TEST_F(InfraFuncTest, N64_N127)
    {
        int waitTime = 0;
        Connect(AP2_3_WPA2AES);

        tcp_Send.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&iperfClientThread, iperfClient, &waitTime, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&iperfClientThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        NN_LOG("             ging for 5 minutes\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s * 5));

        tcp_Send.Disconnect();
        nn::os::WaitThread(&iperfClientThread);
        nn::os::DestroyThread(&iperfClientThread);

        Disconnect();
    }

    // DTIM設定が4のAPを通して、1秒間隔の64BytesのPINGの導通が5分途切れず行えること
    TEST_F(InfraFuncTest, N65)
    {
        int cnt = 0;
        int err_cnt = 0;
        Connect(AP2_1_WPA2AES);
        sysTick = nn::os::GetSystemTick();
        while( NN_STATIC_CONDITION(1) )
        {
            // 5分連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= (Time60s * 5) )
            {
                if( false == PingStart(ping_data_size_64, cnt))
                {
                    err_cnt ++;
                }
                if( 2 <= err_cnt)
                {
                    WLANTEST_ASSERT_TRUE(false);
                }
                cnt ++;

                // 1秒間隔で接続確認とPing送信
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            }
            else
            {
                break;
            }
        }
        Disconnect();
    }

    // オートキープアライブ機能が使用できる
    TEST_F(InfraFuncTest, N66)
    {
        Connect(AP2_1_WPA2AES);
        NN_LOG("             Keep waiting for 1 minute for auto-keepalive frame confirmation\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(60000));
        Disconnect();
    }

    // OPEN/WEP128設定で、WEPキーの設定がAPと異なる場合、データ通信しないこと
    TEST_F(InfraFuncTest, N68)
    {
        Connect(N68TEST_PASSNG);

        PingStart();

        Disconnect();
    }

    // WPA2-PSK(AES)設定で、APとの切断した後、再度同じAPと接続
    TEST_F(InfraFuncTest, N69)
    {
        Connect(AP2_1_WPA2AES);
        Disconnect();

        Connect(AP2_1_WPA2AES);
        Disconnect();
    }

    // WPA2-PSK(AES)設定で、APとの切断した後、異なるWPA-PSK(TKIP)設定のAP接続
    TEST_F(InfraFuncTest, N70)
    {
        Connect(AP2_1_WPA2AES);
        Disconnect();

        Connect(AP3_1_WPATKIP);
        Disconnect();
    }

    // WPA認証処理中にAPと切断した後に、再度WPA認証ができる
    TEST_F(InfraFuncTest, N102)
    {
        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        NN_LOG("             Unplug the AP when this comment is acquired\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[AP2_1_WPA2AES].ssid,
                                                                ConnectParams[AP2_1_WPA2AES].bssid,
                                                                ConnectParams[AP2_1_WPA2AES].channel,
                                                                ConnectParams[AP2_1_WPA2AES].security,
                                                                ConnectParams[AP2_1_WPA2AES].autoKeepAlive,
                                                                ConnectParams[AP2_1_WPA2AES].beaconLostTimeout));

        NN_LOG("             WiFi Connected\n");
        nn::os::WaitSystemEvent(&connectionEvent);
        NN_LOG("             WiFi WaitSystemEvent Signal\n");

        nn::os::DestroySystemEvent(&connectionEvent);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("             Disconnection event\n");
        }

        NN_LOG("             Wait one minute for AP restart\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        Connect(AP2_1_WPA2AES);
        Disconnect();
    }

    // WPAのパスフレーズを間違えて接続試行した後、正しいパスフレーズで再接続できること
    TEST_F(InfraFuncTest, N103)
    {
        int count = 0;

        // 接続に変化があった際に通知を受けるイベントオブジェクト
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

        while( NN_STATIC_CONDITION(1) )
        {
            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N28TEST_WPAPSK_PASSNG].ssid,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].bssid,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].channel,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].security,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].autoKeepAlive,
                                                                    ConnectParams[N28TEST_WPAPSK_PASSNG].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            else
            {
               count ++;
               if(3 <= count) break;
            }
        }
        nn::os::DestroySystemEvent(&connectionEvent);

        Connect(AP1_1_WPAAES);
        Disconnect();
    }

    // AP接続後、過多な負荷通信を行い、通信中に自発的な切断処理
    TEST_F(InfraFuncTest, N104)
    {
        Connect(AP2_3_WPA2AES);

        // 負荷試験開始
        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);

        NN_LOG("             Cut it after 1 hour\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1h));

        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);

        // 切断
        Disconnect();
    }

    // AP接続後、過多な負荷通信を行い、通信中にAPから切断された際、即座に切断が通信される
    // AP接続後、過多な負荷通信を行い、通信中にAPから切断された後、APと再接続
    TEST_F(InfraFuncTest, N105_N106)
    {
        Connect(AP2_3_WPA2AES);

        // 負荷試験開始
        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);
        NN_LOG("             Set automatic AP disconnection setting\n");

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        nn::os::DestroySystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("             Disconnection event\n");
        }

        // 再接続させる
        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);

        NN_LOG("             Reconnect after 2 minutes\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s * 2));
        Connect(AP2_1_WPA2AES);
        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);

        NN_LOG("             Communicate for 3 minutes\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s * 3));

        Disconnect();
        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);
    }

    // AP接続後、過多な負荷通信を行い、通信中にAPをロストした際、規定のBMISS時間経過後に切断が通信される
    TEST_F(InfraFuncTest, N107)
    {
        Connect(AP2_3_WPA2AES);

        // 負荷試験開始
        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);

        // APロストさせる（電源断）
        NN_LOG("             Turn off the power of AP to lost AP after 10 minutes\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s * 10));
        NN_LOG("             Turn off the power of AP\n");
        sysTick = nn::os::GetSystemTick();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        nn::os::DestroySystemEvent(&connectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("             Disconnection event\n");
        }
        int time = (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds();
        if(ConnectParams[AP2_1_WPA2AES].beaconLostTimeout < time)
        {
            NN_LOG("             Cutting time %dmsec\n", time);
            WLANTEST_ASSERT_TRUE(false);
        }

        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);
    }

    // APの信号強度を取得
    // APとの距離に応じて、リアルタイムに信号強度の値が変化する
    TEST_F(InfraFuncTest, N108_N109)
    {
        int32_t rssi;
        nn::wlan::LinkLevel linkLevel;
        sysTick = nn::os::GetSystemTick();

        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        Connect(AP2_1_WPA2AES);

        while( NN_STATIC_CONDITION(1) )
        {
            // 5分間で500msec毎にRSSI変化確認を行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s * 5 )
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetRssi(&rssi));
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetLinkLevel(&linkLevel));

                NN_LOG("             +++ RSSI LINK LEVEL +++\n");
                NN_LOG("             RSSI            : %d\n", rssi);
                NN_LOG("             LINK LEVEL      : %s\n", LinkLevel[linkLevel]);
                NN_LOG("             +++++++++++++++++++++++++\n");

                // 500msecWait
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s / 2));
            }
            else
            {
                break;
            }
        }

        Disconnect();
    }

    // 新規追加Wowl
    TEST_F(InfraLocalFuncTest, N131_AP2G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N131",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N131_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N132_AP2G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N132",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N132_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N132_AP5G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_3_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_3_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N132",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N132_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N133_AP2G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N133",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N133_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N133_AP5G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_3_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_3_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N133",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N133_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N134_AP2G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N134",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N134_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N134_AP5G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_3_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_3_WPA2AES];

        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        WLANTEST_EXPECT_TRUE(WowlSleepTest("N134",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N134_WowlTestMsg], Time600s));

        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N135_AP2G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N135",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N135_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    TEST_F(InfraLocalFuncTest, N135_AP5G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_3_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_3_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // Tcp接続及びWowlSleep処理待機
        WLANTEST_EXPECT_TRUE(WowlSleepTest("N135",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N135_WowlTestMsg], Time600s));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    // WiFi-AP 2.4Gのみ
    // WoWL Sleep状態させて、APとサーバ間のケーブルを外して、TCPサーバーからTCPセッションをきる。
    // 起床させて、TCPセッションの確立、また通信ができる。
    TEST_F(InfraLocalFuncTest, N136_AP2G)
    {
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        // インフラモードオープン
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

        // WowlSleep処理待機及び起床後のTCP通信
        WLANTEST_EXPECT_TRUE(WowlSleepAndTcpRestartTest("N136",
            mode, pTestIpv4, pServerIp, WowlTestMsg[N136_WowlTestMsg], Time600s, 1, 10));

        // インフラモードクローズ
        WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());
    }

    // WiFi-AP 2.4Gのみ
    // WoWL Sleep状態からのJoycon操作試験（詳細はチェックシート）
    TEST_F(InfraLocalFuncTest, N139_AP2G)
    {
        // Joycon試験５回*２回（handheld１回＋BT１回 ）
        const int32_t joyTestCount = 5 * 2;
        const int32_t mode = static_cast<int32_t>(APKIND::AP2_1_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N131N139_WowlTest];
        const char* pServerIp = IPAddr[AP2_1_WPA2AES];

        for (int32_t i = 0; i < joyTestCount; i++)
        {
            if ((i + 1) % 2 != 0)
            {
                NN_LOG("             #### Joycon Handheld Test Start ####\n");
            }
            else
            {
                NN_LOG("             #### Joycon BT Test       Start ####\n");
            }

            // インフラモードオープン
            WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

            // Tcp接続及びWowlSleep処理待機
            WLANTEST_EXPECT_TRUE(WowlSleepTest("N139",
                mode, pTestIpv4, pServerIp, WowlTestMsg[N139_WowlTestMsg], Time600s));

            // インフラモードクローズ
            WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());

            if ((i + 1) % 2 != 0)
            {
                NN_LOG("             #### Joycon Handheld Test End ####\n\n");
            }
            else
            {
                NN_LOG("             #### Joycon BT Test       End ####\n\n");
            }
        }
    }

    // WiFi-AP 5G、ローカルモード2.4Gマスター
    // WoWL Sleep状態からのローカル通信試験（詳細はチェックシート）
    TEST_F(InfraLocalFuncTest, N140_AP5G_MASTER)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" },
            nn::wlan::Ssid(LocalMasterTest2SSID), 10
        };

        const int32_t testCount     = 3;
        const int32_t mode          = static_cast<int32_t>(APKIND::AP2_3_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N140_WowlMasterTest];
        const char* pServerIp       = IPAddr[AP2_3_WPA2AES];
        std::string strTestNo;

        for (int32_t i = 0; i < testCount; i++)
        {
            // インフラモードオープン
            WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

            // Tcp接続及びWowlSleep処理待機
            strTestNo = "N140 - Master Test Count ";
            strTestNo.append(std::to_string(i + 1)).append("/").append(std::to_string(testCount));
            WLANTEST_EXPECT_TRUE(WowlSleepTest(strTestNo.c_str(),
                mode, pTestIpv4, pServerIp, WowlTestMsg[N140_WowlTestMsg], Time1800s));

            // インフラモードクローズ
            WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());


            // ローカルマスターモードオープン
            WLANTEST_ASSERT_TRUE(FunctionTestLocalMasterOpen(LocalApiBtNode::LocalApiBtNode2));

            // ローカルマスター処理
            WLANTEST_EXPECT_TRUE(WowlLocalMasterTest(fncTestMasterBss, Time600s));

            // ローカルマスターモードクローズ
            WLANTEST_ASSERT_TRUE(FunctionTestLocalMasterClose());
        }
    }

    // WiFi-AP 5G、ローカルモード2.4Gクライアント
    // WoWL Sleep状態からのローカル通信試験（詳細はチェックシート）
    TEST_F(InfraLocalFuncTest, N140_AP5G_CLIENT)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            11, { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" },
            true, nn::wlan::BeaconIndication_Enable, 10
        };

        const int32_t testCount     = 3;
        const int32_t mode          = static_cast<int32_t>(APKIND::AP2_3_WPA2AES);
        const StringIpv4* pTestIpv4 = &TestSdevIpv4[N140_WowlClientTest];
        const char* pServerIp       = IPAddr[AP2_3_WPA2AES];
        std::string strTestNo;

        for (int32_t i = 0; i < testCount; i++)
        {
            // インフラモードオープン
            WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraOpen());

            // Tcp接続及びWowlSleep処理待機
            strTestNo = "N140 - Client Test Count ";
            strTestNo.append(std::to_string(i + 1)).append("/").append(std::to_string(testCount));
            WLANTEST_EXPECT_TRUE(WowlSleepTest(strTestNo.c_str(),
                mode, pTestIpv4, pServerIp, WowlTestMsg[N140_WowlTestMsg], Time1800s));

            // インフラモードクローズ
            WLANTEST_ASSERT_RESULT_SUCCESS(FunctionTestInfraClose());


            // ローカルクライアントモードオープン
            WLANTEST_ASSERT_TRUE(FunctionTestLocalClientOpen(LocalApiBtNode::LocalApiBtNode2));

            // ローカルクライアント処理
            WLANTEST_EXPECT_TRUE(WowlLocalClientTest(ConnectParam));

            // ローカルクライアントモードクローズ
            WLANTEST_ASSERT_TRUE(FunctionTestLocalClientClose());
        }
    }

    // 特定の1つのチャネルを1時間連続してスキャンし続ける
    TEST_F(InfraFuncTest, N110)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        sysTick = nn::os::GetSystemTick();
        while( NN_STATIC_CONDITION(1) )
        {
            // 1時間連続でScanを行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h )
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N1_TEST].scanSize, scanTestParam[SCAN_N1_TEST].scanParam));
                DumpScanResultWithReader(pTestScanData.get());
            }
            else
            {
                break;
            }
        }
    }

    // 全チャネルを1時間連続してスキャンし続けることができること
    TEST_F(InfraFuncTest, N111)
    {
        uint16_t count = 1;
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        sysTick = nn::os::GetSystemTick();
        while( NN_STATIC_CONDITION(1) )
        {
            // 1時間連続でScanを行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h )
            {
                NN_LOG("             StartStan %dCnt\n", count);
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
                DumpScanResultWithReader(pTestScanData.get());
                count ++;
            }
            else
            {
                break;
            }
        }
        NN_LOG("             N111_2 is executed.\n");
        NN_LOG("             Before execution, change AP setting, add AP, turn off AP power\n");
    }

    TEST_F(InfraFuncTest, N111_2)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());
    }

    // BT6台接続した状態で、全チャンネルをアクティブスキャンをして、特定のAPを見つけられること
    TEST_F(InfraFuncTest, N128)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

        for(int i = 0; i < 300; i ++)
        {
            NN_LOG("\n             %dCnt\n", i + 1);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
            DumpScanResultWithReader(pTestScanData.get());
        }
    }

    // BT6台接続した状態で、全チャンネルをパッシブスキャンをして、特定のAPを見つけられること
    TEST_F(InfraFuncTest, N129)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);

        for(int i = 0; i < 300; i ++)
        {
            NN_LOG("\n             %dCnt\n", i + 1);
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N129_TEST].scanSize, scanTestParam[SCAN_N129_TEST].scanParam));
            DumpScanResultWithReader(pTestScanData.get());
        }
    }

    // 同一のWPA2-PSK(AES)設定のAPに1時間連続で接続と切断を繰り返し
    TEST_F(InfraFuncTest, N112)
    {
        int cnt = 0;

        // 試験時間短縮フラグを確認
        int time = Time1h;
        if (g_IsShort) time = Time60s * 10;

        sysTick = nn::os::GetSystemTick();
        while( NN_STATIC_CONDITION(1) )
        {
            // 1時間連続で接続/切断を行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= time )
            {
                Connect(AP2_3_WPA2AES);
                Disconnect();
                cnt ++;
                NN_LOG("             CNT %d\n", cnt);
                // 再接続に1～1000msecの間隔をあける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(1,1000)));
            }
            else
            {
                break;
            }
        }
    }

    // WPA-PSK(TKIP)とWPA2-PSK(AES)設定のAPを交互に4時間連続で接続と切断を繰り返せる
    TEST_F(InfraFuncTest, N113)
    {
        nn::os::Tick sysTick = nn::os::GetSystemTick();
        nn::os::Tick sysTick_20s = nn::os::GetSystemTick();
        int ap_kind = 0;
        bool time_chk = false;
        while( NN_STATIC_CONDITION(1) )
        {
            // 4時間連続で接続/切断を行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time4h )
            {
                if(1 == ap_kind)
                {
                    // 前回の接続開始から20秒以上であればNGとする
                    if(true == time_chk)
                    {
                        if( (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds() >= Time20s )
                        {
                            NN_LOG("             Time:%d msec\n",  (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds());
                            WLANTEST_ASSERT_TRUE(false);
                        }
                        sysTick_20s = nn::os::GetSystemTick();
                    }
                    NN_LOG("             WPA-PSK(TKIP)\n");
                    Connect(AP1_3_WPA2TKIP, true, Time15s);
                    ap_kind = 0;
                }
                else{
                    // 前回の接続開始から20秒以上であればNGとする
                    if(true == time_chk)
                    {
                        if( (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds() >= Time20s )
                        {
                            NN_LOG("             Time:%d msec\n",  (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds());
                            WLANTEST_ASSERT_TRUE(false);
                        }
                        sysTick_20s = nn::os::GetSystemTick();
                    }
                    else{
                        time_chk = true;
                    }
                    NN_LOG("             WPA2-PSK(AES)\n");
                    Connect(AP4_1_WPA2AES, true, Time15s);
                    ap_kind = 1;
                }
                Disconnect();
                // 再接続に1～1000msecの間隔をあける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(1,1000)));
            }
            else
            {
                break;
            }
        }
    }

    // WPA-PSK(TKIP)とWPA2-PSK(AES)設定のAPを交互に4時間連続でスキャン、接続と切断、スキャンを繰り返せること
    TEST_F(InfraFuncTest, N126)
    {
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        nn::os::Tick sysTick = nn::os::GetSystemTick();
        nn::os::Tick sysTick_20s = nn::os::GetSystemTick();
        int apkind = 0;
        bool time_chk = false;

        // 試験時間短縮フラグを確認
        int time = Time60s * 10;
        if (g_IsShort) time = Time60s * 2;

        while( NN_STATIC_CONDITION(1) )
        {
            // 4時間連続で接続/切断を行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= time )
            {
                // Scan実施
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N2_TEST].scanSize, scanTestParam[SCAN_N2_TEST].scanParam));

                if(1 == apkind)
                {
                    // 前回の接続開始から20秒以上であればNGとする
                    if(true == time_chk)
                    {
                        if( (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds() >= Time20s )
                        {
                            NN_LOG("             Time:%d msec\n",  (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds());
                            WLANTEST_ASSERT_TRUE(false);
                        }
                        sysTick_20s = nn::os::GetSystemTick();
                    }
                    NN_LOG("             WPA-PSK(TKIP)\n");
                    Connect(AP1_3_WPA2TKIP, true, Time15s);
                    apkind = 0;
                }
                else{
                    // 前回の接続開始から20秒以上であればNGとする
                    if(true == time_chk)
                    {
                        if( (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds() >= Time20s )
                        {
                            NN_LOG("             Time:%d msec\n",  (nn::os::GetSystemTick() - sysTick_20s).ToTimeSpan().GetMilliSeconds());
                            WLANTEST_ASSERT_TRUE(false);
                        }
                        sysTick_20s = nn::os::GetSystemTick();
                    }
                    else{
                        time_chk = true;
                    }
                    NN_LOG("             WPA2-PSK(AES)\n");
                    Connect(AP4_1_WPA2AES, true, Time15s);
                    apkind = 1;
                }
                Disconnect();
                // 再接続に1～20msecの間隔をあける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(1,20)));
            }
            else
            {
                break;
            }
        }
    }

    // 存在しないAPへの接続を繰り返して問題がないこと
    TEST_F(InfraFuncTest, N123)
    {
        sysTick = nn::os::GetSystemTick();

        while( NN_STATIC_CONDITION(1) )
        {
            // 接続に変化があった際に通知を受けるイベントオブジェクト
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionEvent(&connectionEvent));

            NN_LOG("             WiFi Connect Start\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::Connect(ConnectParams[N123TEST].ssid,
                                                                    ConnectParams[N123TEST].bssid,
                                                                    ConnectParams[N123TEST].channel,
                                                                    ConnectParams[N123TEST].security,
                                                                    ConnectParams[N123TEST].autoKeepAlive,
                                                                    ConnectParams[N123TEST].beaconLostTimeout));
            NN_LOG("             WiFi Connected\n");
            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("             WiFi WaitSystemEvent Signal\n");
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            nn::os::DestroySystemEvent(&connectionEvent);

            // 10分連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time1h / 6) )
            {
                break;
            }
        }

        Connect(AP2_1_WPA2AES);
        Disconnect();
    }

    // 隣接チャネルのAPに接続しないこと
    TEST_F(InfraFuncTest, N124)
    {
        sysTick = nn::os::GetSystemTick();

        while( NN_STATIC_CONDITION(1) )
        {
            Connect(N124TEST);

            if(5 == connectionStatus.channel)
            {
                WLANTEST_ASSERT_TRUE(false);
            }
            Disconnect();

            // 10分連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time1h / 6) )
            {
                break;
            }
        }
    }

    // BT6台接続した状態で、特定のAPに接続できること
    TEST_F(InfraFuncTest, N130)
    {
        for(int i = 0; i < 300; i ++)
        {
            Connect(AP2_3_WPA2AES);
            Disconnect();
            NN_LOG("             %dCnt\n", i);
            // 再接続に1～1000msecの間隔をあける
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(GetRandom(1,1000)));
        }
    }

    // APと接続後、自発的な通信が無い状態で30分間、APとのリンクがきれないこと
    TEST_F(InfraFuncTest, N114)
    {
        Connect(AP1_3_WPA2TKIP);

        // 30分待たせる
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             Let me wait 30 minutes\n");
        while( NN_STATIC_CONDITION(1) )
        {
            // 1分間隔でステート確認
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));
            WlanState();

            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time1h / 2) )
            {
                break;
            }
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
        if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
        {
            WLANTEST_ASSERT_TRUE(false);
        }

        Disconnect();
    }

    // APと接続後、毎秒1回の512BytesのPINGを4時間連続
    TEST_F(InfraFuncTest, N115)
    {
        for(int i=0; i<4; i++)
        {
            Connect(AP2_4_OPEN);
            sysTick = nn::os::GetSystemTick();

            while( NN_STATIC_CONDITION(1) )
            {
                // 10秒連続で1秒間隔でPINGを行う
                if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time10s )
                {
                    PingStart(ping_data_size_512);

                    // 300msec間隔で接続確認とPing送信
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300m));
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                    if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
                    {
                        NN_LOG("             ERROR STATE : %s\n", WlanConnectionStateStr[connectionStatus.state]);
                        WLANTEST_ASSERT_TRUE(false);
                    }

                    // 1秒間隔でPING送信を行う
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s - Time300m));
                }
                else
                {
                    break;
                }
            }

            Disconnect();

            Connect(AP2_4_OPEN);
            sysTick = nn::os::GetSystemTick();
            while( NN_STATIC_CONDITION(1) )
            {
                // 4時間連続
                if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h )
                {
                    PingStart(ping_data_size_512);

                    // 300msec間隔で接続確認とPing送信
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300m));
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
                    if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
                    {
                        NN_LOG("             ERROR STATE : %s\n", WlanConnectionStateStr[connectionStatus.state]);
                        WLANTEST_ASSERT_TRUE(false);
                    }

                    // 1秒間隔でPING送信を行う
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s - Time300m));
                }
                else
                {
                    break;
                }
            }
            Disconnect();
        }
    }

    // APと接続後、8時間連続で100Mbps以上のTCP通信を途切れずおこなえること。APとの信号強度も1秒に1回取得する。
    TEST_F(InfraFuncTest, N116)
    {
        int32_t rssi;
        int waitTime = 0;
        nn::wlan::LinkLevel linkLevel;
        sysTick = nn::os::GetSystemTick();

        Connect(AP2_3_WPA2AES);

        tcp_Send.Connect();
        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpSendThread, TcpSend, &waitTime, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, &waitTime, ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority));

        nn::os::StartThread(&tcpSendThread);
        nn::os::StartThread(&tcpRecvThread);

        while( NN_STATIC_CONDITION(1) )
        {
            // 1秒間隔でAP信号強度確認
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetRssi(&rssi));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetLinkLevel(&linkLevel));

            NN_LOG("             +++ RSSI LINK LEVEL +++\n");
            NN_LOG("             RSSI            : %d\n", rssi);
            NN_LOG("             LINK LEVEL      : %s\n", LinkLevel[linkLevel]);
            NN_LOG("             +++++++++++++++++++++++++\n");
            WlanState();

            // 8時間連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > Time1h * 8 )
            {
                break;
            }
        }

        tcp_Send.Disconnect();
        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpSendThread);
        nn::os::DestroyThread(&tcpSendThread);
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);

        Disconnect();
    }

    // 連続通信中に、任意の1chに対する既接続スキャンを行っても通信が途切れない
    TEST_F(InfraFuncTest, N117)
    {
        uint16_t cnt = 1;
        sysTick = nn::os::GetSystemTick();
        Connect(AP2_1_WPA2AES);
        ConfigureInterface(WLAN_INTERFACE_NAME, true);

        nn::os::InitializeEvent(&agingn117Event, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&agingn117Thread, aging_n117, &agingn117Event, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&agingn117Thread);
        nn::os::SignalEvent(&agingn117Event);

        while( NN_STATIC_CONDITION(1) )
        {
            // 300msec間隔で接続確認とPing送信
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300m));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("             ERROR STATE : %s\n", WlanConnectionStateStr[connectionStatus.state]);
                WLANTEST_ASSERT_TRUE(false);
            }

            PingStart(ping_data_size_512, cnt);
            cnt ++;

            // 1時間連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > Time1h )
            {
                break;
            }
        }

        nn::os::WaitThread(&agingn117Thread);
        nn::os::DestroyThread(&agingn117Thread);
        nn::os::FinalizeEvent(&agingn117Event);

        Disconnect();
    }

    // 連続通信中に、全チャネルに対する既接続スキャンを行っても通信が途切れない
    TEST_F(InfraFuncTest, N118)
    {
        uint16_t cnt = 1;
        sysTick = nn::os::GetSystemTick();
        Connect(AP2_4_OPEN);
        ConfigureInterface(WLAN_INTERFACE_NAME, true);

        nn::os::InitializeEvent(&agingn118Event, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&agingn118Thread, aging_n118, &agingn118Event, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&agingn118Thread);
        nn::os::SignalEvent(&agingn118Event);

        while( NN_STATIC_CONDITION(1) )
        {
            // 300msec間隔で接続確認とPing送信
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300m));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::GetConnectionStatus(&connectionStatus));
            if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("             ERROR STATE : %s\n", WlanConnectionStateStr[connectionStatus.state]);
                WLANTEST_ASSERT_TRUE(false);
            }

            PingStart(ping_data_size_512, cnt);
            cnt ++;

            // 1時間連続
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > Time1h )
            {
                break;
            }
        }

        nn::os::WaitThread(&agingn118Thread);
        nn::os::DestroyThread(&agingn118Thread);
        nn::os::FinalizeEvent(&agingn118Event);

        // 1時間後にScan、PINGができること
        std::unique_ptr<uint8_t[]> pTestScanData(new uint8_t[BufferSize100k]);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestScanData.get(), scanTestParam[SCAN_N118_TEST].scanSize, scanTestParam[SCAN_N118_TEST].scanParam));
        DumpScanResultWithReader(pTestScanData.get());

        PingStart(ping_data_size_512);

        Disconnect();
    }

    // 夏季休業エージング tcp send
    TEST_F(InfraFuncTest, tcp_send)
    {
        int waitTime = 0;
        Connect(AP2_3_WPA2AES);

        tcp_Send.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpSendThread, TcpSend, &waitTime, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpSendThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        NN_LOG("             4 hour aging\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time4h));

        tcp_Send.Disconnect();
        nn::os::WaitThread(&tcpSendThread);
        nn::os::DestroyThread(&tcpSendThread);

        Disconnect();
    }

    // 夏季休業エージング tcp recv
    TEST_F(InfraFuncTest, tcp_recv)
    {
        Connect(AP2_3_WPA2AES);

        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&tcpRecvThread, TcpRecv, nullptr, ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&tcpRecvThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        NN_LOG("             4 hour aging\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time4h));

        tcp_Recv.Disconnect();
        nn::os::WaitThread(&tcpRecvThread);
        nn::os::DestroyThread(&tcpRecvThread);

        Disconnect();
    }

    // iperf send
    TEST_F(InfraFuncTest, iperf_send)
    {
        int waitTime = 0;
        Connect(AP2_3_WPA2AES);

        tcp_Send.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&iperfClientThread, iperfClient, &waitTime, ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&iperfClientThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        NN_LOG("             4 hour aging\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time4h));

        tcp_Send.Disconnect();
        nn::os::WaitThread(&iperfClientThread);
        nn::os::DestroyThread(&iperfClientThread);

        Disconnect();
    }

    // iperf recv
    TEST_F(InfraFuncTest, iperf_recv)
    {
        Connect(AP2_1_WPA2AES);

        tcp_Recv.Connect();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&iperfServerThread, iperfServer, nullptr, ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&iperfServerThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        NN_LOG("             4 hour aging\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time4h));

        tcp_Recv.Disconnect();
        nn::os::WaitThread(&iperfServerThread);
        nn::os::DestroyThread(&iperfServerThread);

        Disconnect();
    }

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();
    NN_LOG("argc=%d\n", argc);
    for (int i = 0; i < argc; i++)
    {
        if (!strcmp(argv[i], "--short"))
        {
            g_IsShort = true;
        }
        NN_LOG("argv[%d]=%s\n", i, argv[i]);
    }

    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}

} // namespace WlanTest
