﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <string>

#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_StandardSteadyClock.h>


namespace WlanTest {

#define WLANTEST_GTESTLIB  // gtest有効/無効切り替え
#define WLAN_TEST_LEFTSHIFT(num)     (1 << num)

#define WLAN_TEST_LOSTFRAME_TRACE
//#define WLAN_TEST_RECV_TRACE        // SIGLO-33534
//#define WLAN_TEST_SENDRECVDATA_TRACE        // SIGLO-33371
//#define WLAN_TEST_STATE_ASSER_STOP  // SIGLO-32857
//#define WLAN_TEST_LINKLEVEL_ERROR   // SIGLO-33371
#define WLAN_TEST_LINKLEVEL_WAIT        // SIGLO-33371
//#define WLAN_TEST_SCAN2GONECH           // SIGLO-49901
#define WLAN_TEST_SEQBACK_TRACE         // SIGLO-42135

#ifdef WLAN_TEST_RECV_TRACE
    static const uint8_t TestTraceMasterMac[6] =
    {
        0x40, 0xD2, 0x8A, 0xB4, 0x81, 0x81
    };
#endif

    struct IpV4Address
    {
        char strIpAddress[INET_ADDRSTRLEN];
        char strSubNetMask[INET_ADDRSTRLEN];
        char strGateWay[INET_ADDRSTRLEN];
        char strDns1[INET_ADDRSTRLEN];
        char strDns2[INET_ADDRSTRLEN];
        uint16_t port;
        bool isDhsp;
    };

    const IpV4Address tcpMaster = {
            "192.168.11.2",
            "255.255.255.0",
            "192.168.11.1",
            "8.8.8.8",
            "8.8.8.9",
            5001,
            false
    };

    const uint32_t LocalApiEtherMinValue        = 0x0801;
    const uint32_t LocalApiEtherMaxCount        = 30;

    const uint32_t LocalApiPoolMax              = 100;
    const uint32_t LocalApiClientMaxCount       = 40;

    const size_t LocalApiFramePacketSize        = 512;
    const int64_t LocalApiBroadFrameInterval    = 16;
    const int64_t LocalApiUniFrameInterval      = 22;
    const int64_t LocalApiActionFrameInterval   = 200;
    const int32_t LocalApiRetryStartFrame       = 2;

    const int32_t LocalTcpGetMacAddressMax      = nn::wlan::ConnectableClientsCountMax + 1;
    const int64_t LocalApiClientWait            = 100;
    const int64_t LocalApiClientWaitTimeOut     = 10 * 1000;
    const int64_t LocalApiBtmWaitTime           = 30 * 1000;

    enum WlanTestMode
    {
        TestMode_Init = 0,
        TestMode_MasterInit,
        TestMode_MasterCreateBss,
        TestMode_ClientInit,
        TestMode_ClientConnect,
        TestMode_SpectatorInit,
        TestMode_SpectatorConnect,
        TestMode_GetConnectEvent,
        TestMode_CreateThread,
        TestMode_CreateEntryId,
        TestMode_CreateActionEntryId,
        TestMode_ReceiveFrameThread,
        TestMode_ReceiveActionFrameThread,
        TestMode_ParseFrameThread,
        TestMode_SendFrameThread,
        TestMode_TcpMasterInit,
        TestMode_TcpMasterAccept,
        TestMode_TcpClientInit
    };

    enum LocalEvent
    {
        // LocalMaster Event
        EventType_Init = 0,
        EventType_MasterSetActionFrameWithBeacon,
        EventType_MasterGetActionFrame,

        EventType_ClientConnect,
        EventType_ClientGetFrame,
        EventType_ClientGetActionFrame,

        // Common Event
        EventType_PutFrame,
        EventType_GetFrame,
        EventType_CancelFrame,
        EventType_CancelActionFrame,
        EventType_Exit
    };

    struct LocalEtherTypeInfo
    {
        uint16_t etherTypes[LocalApiEtherMaxCount];
        uint16_t reserve;
        uint32_t typeCount;
    };

    enum GetLocalFrameType
    {
        FrameType_Invalid = 0,
        FrameType_Frame,
        FrameType_ActionFrame,
    };

    enum FrameReqType
    {
        ReqType_Frame = 0,
        ReqType_MacAddressReq,
        ReqType_MacAddressRes,
        // NTD追加定義
        ReqType_RawPingReq,
        ReqType_RawPingRes,
    };

    enum LocalParseType
    {
        ParseType_Parse = 0,
        ParseType_End
    };

    enum LocalOpenModeType
    {
        OpenMode_Init = 0,
        OpenMode_Master,
        OpenMode_Client,
        OpenMode_Spectator,
    };

    enum LocalApiBtNode
    {
        LocalApiBtNodeNone = 0,
        LocalApiBtNode1,
        LocalApiBtNode2,
        LocalApiBtNode3,
        LocalApiBtNode4,
        LocalApiBtNode5,
        LocalApiBtNode6,
        LocalApiBtNode7,
        LocalApiBtNode8,
    };

    struct LocalClientConnectParameter
    {
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress bssid;
        int16_t channel;
        nn::wlan::Security security;
        bool autoKeepAlive;
        nn::wlan::BeaconIndication indication;
        int beaconLostTimeout;
    };

    struct LocalPoolMngDetail
    {
        size_t bufferSize;
        size_t dataSize;
        uint8_t* pData;
        GetLocalFrameType frameType;
    };

    struct LocalPoolMng
    {
        uint32_t maxCount;
        uint32_t writeIdx;
        uint32_t readIdx;
        LocalPoolMngDetail info[LocalApiPoolMax];
        uint8_t* pBuffer;
        size_t bufferSize;
        size_t packetSize;
    };

    struct LocalClientInfoDetail
    {
        GetLocalFrameType frameType;
        nn::wlan::MacAddress fromMacAddres;
        nn::wlan::MacAddress toMacAddres;
        uint8_t etherType[2];

        // NTD追加定義
        uint16_t rttMin;
        uint16_t rttMax;
        uint16_t rttCount;
        double   rttAverage;
        uint32_t seqNumberRttRecent;
        uint32_t seqNumberRttGap;

        uint32_t seqNum;
        uint32_t getFrameCount;
        uint32_t lostFrameCount;
        uint64_t totalFrameSize;
        uint32_t seqNumberGap;
        double perRate;
        bool isConnected;
        bool isGetFrame;
        bool isRttPkt;
    };

    struct LocalFrameHeader
    {
        uint8_t toMacAddress[nn::wlan::MacAddress::MacAddressSize];
        uint8_t fromMacAddress[nn::wlan::MacAddress::MacAddressSize];
        uint8_t etherType[2];
        uint8_t data[1];
    };

    struct LocalPayloadData
    {
        uint8_t matchData[sizeof(nn::wlan::ReceivedDataMatchInfo::matchData)];
        uint32_t seq;
        uint16_t reqType;
        uint8_t data[1];
    };

    struct LocalActionFrameData
    {
        uint8_t categry;
        uint8_t oui[3];
        uint8_t frameType;
        uint8_t reserve;
        uint8_t fromMacAddress[nn::wlan::MacAddress::MacAddressSize];
        uint8_t etherType[2];
        uint8_t data[1];
    };

    struct LocalActionFramePayloadData
    {
        uint8_t matchData[sizeof(nn::wlan::ReceivedDataMatchInfo::matchData)];
        uint32_t seq;
        uint16_t reqType;
        uint8_t data[1];
    };

    struct LocalPayLoadMacReqData
    {
        uint16_t len;
        uint8_t macAddress[LocalApiClientMaxCount][nn::wlan::MacAddress::MacAddressSize];
    };

    struct LocalConnectsInfo
    {
        uint32_t connectCount;
        LocalClientInfoDetail facing[LocalApiClientMaxCount];
        uint64_t totalReceiveSize;
        uint64_t totalSendSize;
        double dbRecvRate;
        double dbSendRate;
        nn::os::Tick startRecvTick;
        nn::os::Tick startSendTick;
        bool isReceiveStart;
        bool isSendStart;
    };

    struct SocketClientInfo
    {
        int32_t accept;
        nn::socket::SockAddrIn clientAddr;
    };

    struct SocketInfo
    {
        int32_t socket;
        IpV4Address masterIp;
        IpV4Address clientIp;
        uint32_t connectCount;
        uint32_t waitCount;
        SocketClientInfo clients[nn::wlan::ConnectableClientsCountMax];
        nn::os::ThreadType tcpThread;
        nn::os::EventType* pConnectEvent;
        uint8_t* pTcpBuffer;
        size_t tcpSize;
        bool isThreadStop;
    };

    struct LoaclMacAddressInfo
    {
        uint32_t macCount;
        nn::wlan::MacAddress macArray[LocalApiClientMaxCount];
    };


    struct TestLocalConnectParams
    {
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress bssid;
        int16_t channel;
        nn::wlan::Security security;
        bool autoKeepAlive;
        nn::wlan::BeaconIndication indication;
        int beaconLostTimeout;
    };

    typedef LocalPayLoadMacReqData  TcpMacAddressPacketFmt;
    typedef void (*LocalStateCallBack)(nn::wlan::WlanState oldState, nn::wlan::WlanState newState);

    class AutoLock
    {
    public:
        explicit AutoLock(nn::os::MutexType* plock)
        {
            m_pLock = plock;
            if (m_pLock != nullptr)
            {
                nn::os::LockMutex(m_pLock);
            }
        }

        ~AutoLock()
        {
            if (m_pLock != nullptr)
            {
                nn::os::UnlockMutex(m_pLock);
            }
        }
    private:
        nn::os::MutexType* m_pLock;

    };

    class LocalAutoEvent
    {
    public:
        explicit LocalAutoEvent() NN_NOEXCEPT;
        ~LocalAutoEvent() NN_NOEXCEPT;
        void SetEvent(nn::os::EventType&) NN_NOEXCEPT;
        void SetEvent(bool, nn::os::EventClearMode) NN_NOEXCEPT;
        void SetSygnal() NN_NOEXCEPT;
        void ClearSygnal() NN_NOEXCEPT;
        bool WaitSygnal() NN_NOEXCEPT;
        bool WaitSygnal(int64_t waitTime) NN_NOEXCEPT;
        bool TrySygnal() NN_NOEXCEPT;
        nn::os::EventType* GetEvent() { return &m_event; }
        void Release() NN_NOEXCEPT;

    protected:
        nn::os::EventType m_event;
        nn::os::MutexType m_lock;
    };

    class LocalStateMonitoring
    {
    public:
        explicit LocalStateMonitoring(bool isManagerInit = true) NN_NOEXCEPT;
        ~LocalStateMonitoring() NN_NOEXCEPT;

        void StateThreadStart(const int64_t interval, LocalStateCallBack pFunction) NN_NOEXCEPT;
        void StateThreadStop() NN_NOEXCEPT;

        nn::os::MutexType* GetLock() NN_NOEXCEPT;
        nn::Result GetState(nn::wlan::WlanState* pState) NN_NOEXCEPT;
    private:
        static void StateThread(void* arg) NN_NOEXCEPT;

    private:
        bool m_isApiInit;
        bool m_isStateThread;
        nn::os::MutexType m_lock;
        LocalStateCallBack m_func;
        nn::os::ThreadType m_stateId;
        int64_t m_interval;
    };

    class LocalApiClass : public LocalStateMonitoring
    {
    public:
        explicit LocalApiClass(bool isManagerInit = true) NN_NOEXCEPT;
        ~LocalApiClass() NN_NOEXCEPT;

        void InitializeBtm() NN_NOEXCEPT;
        bool SetBtMode(int32_t btCount) NN_NOEXCEPT;
        bool SetBtMode(int32_t btCount, int64_t waitMsecTime) NN_NOEXCEPT;

        bool AddEtherType(const uint16_t etherType) NN_NOEXCEPT;
        void DeleteEhterType(const uint16_t etherType) NN_NOEXCEPT;

        void SetChannel(int16_t channel) NN_NOEXCEPT;

        LocalOpenModeType GetLocalOpenMode() NN_NOEXCEPT;
        LocalOpenModeType GetLocalOpenMode(nn::wlan::WlanState& state) NN_NOEXCEPT;
        LocalOpenModeType GetLocalModeType() NN_NOEXCEPT;
        std::string GetStringOpenMode(const LocalOpenModeType mode) NN_NOEXCEPT;

        bool SetupLocalMaster(const nn::wlan::MasterBssParameters & bssParamer, const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount, bool isActionFrame = false) NN_NOEXCEPT;
        bool SetupLocalClient(const nn::wlan::Ssid& ssid, const nn::wlan::Security& security, const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount, bool isFrame, bool isActionFrame) NN_NOEXCEPT;
        bool SetupLocalClient(const nn::wlan::Ssid& ssid, const nn::wlan::Security& security) NN_NOEXCEPT;
        void SetCommandTime(uint32_t msecTime) NN_NOEXCEPT;

        bool LocalPassiveScanResult(nn::wlan::Ssid compareSsid, uint16_t channel) NN_NOEXCEPT;
        bool LocalPassiveScanResult(nn::wlan::Ssid compareSsid, uint16_t channel, int32_t scnanTime) NN_NOEXCEPT;
        bool LocalPassiveAllChannelScanResult(nn::wlan::Ssid compareSsid, uint16_t channel) NN_NOEXCEPT;

        bool GetScanResult(nn::wlan::Ssid& ssid, uint8_t pScanBuffer[], size_t bufferSize) NN_NOEXCEPT;
        bool GetScanResult(nn::wlan::Ssid& ssid, uint8_t pScanBuffer[], size_t bufferSize, uint16_t channel) NN_NOEXCEPT;
        bool LocalClientConnect(int64_t waitMsecTime, int16_t channel = -1) NN_NOEXCEPT;
        bool LocalClientConnect(LocalClientConnectParameter& conParam, int64_t waitTime) NN_NOEXCEPT;
        bool LocalClientAsyncConnect(int16_t channel = -1) NN_NOEXCEPT;
        bool LocalClientDisconnect(int64_t waitMsecTime) NN_NOEXCEPT;
        bool LocalGetFrame(size_t size) NN_NOEXCEPT;
        bool LocalSetActionFrameWithBeacon(uint8_t buffer[], size_t size) NN_NOEXCEPT;
        void LocalRelease() NN_NOEXCEPT;
        void LocalRelease(bool isSetup) NN_NOEXCEPT;

        uint32_t LocalMasterConnectWait(int64_t waitMsecTime, uint32_t connectCount) NN_NOEXCEPT;
        uint32_t LocalMasterConnectWait(nn::wlan::MacAddress clientArray[nn::wlan::ConnectableClientsCountMax], int64_t waitMsecTime, uint32_t connectCount) NN_NOEXCEPT;
        uint32_t LocalMasterDisConnectWait(nn::wlan::MacAddress clientArray[nn::wlan::ConnectableClientsCountMax], int64_t waitMsecTime) NN_NOEXCEPT;
        uint32_t LocalMasterDisConnectWait(int64_t waitMsecTime) NN_NOEXCEPT;
        uint32_t GetStateClientCount(nn::wlan::ConnectionState State) NN_NOEXCEPT;
        uint32_t GetClientMacAddress(nn::wlan::MacAddress clientMacAddress[nn::wlan::ConnectableClientsCountMax], nn::wlan::ConnectionState State) NN_NOEXCEPT;

        bool GetMasterTcpMacAddress(nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax], uint32_t &macCount,
                                        IpV4Address masterIpAddr, uint32_t connectCount) NN_NOEXCEPT;

        bool GetClientTcpMacAddress(nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax], uint32_t &macCount,
                                        IpV4Address masterIpAddr, IpV4Address clientIpAddr) NN_NOEXCEPT;

        bool SetupTcpMaster(IpV4Address masterIpAddr, uint32_t connectCount) NN_NOEXCEPT;
        bool SetupTcpClient(IpV4Address masterIpAddr, IpV4Address clientIpAddr) NN_NOEXCEPT;
        bool TcpMasterAcceptWait(int64_t waitMsecTime) NN_NOEXCEPT;
        bool TcpMasterSend(uint8_t buffer[], size_t bufferSize) NN_NOEXCEPT;
        bool TcpClientConnectAndRecv(uint8_t* buffer, size_t bufferSize, int64_t waitMsecTime) NN_NOEXCEPT;
        void TcpRelease();

        bool LocalGetConnectInfo(LocalConnectsInfo* pInfo) NN_NOEXCEPT;
        bool LocalGetConnectInfo(LocalConnectsInfo* pInfo, const GetLocalFrameType type) NN_NOEXCEPT;

        bool LocalGetMacList(LoaclMacAddressInfo& listInfo) NN_NOEXCEPT;

        static bool LocalMakeFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
            const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const uint32_t seqNumber) NN_NOEXCEPT;
        static bool LocalMakeFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac, const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo) NN_NOEXCEPT;

        // NTD追加関数
        static bool LocalMakeFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
            const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const uint32_t seqNumber, bool isNeedReply) NN_NOEXCEPT;

        static bool LocalSeqNumberAddFrame(uint8_t frameBuffer[], size_t size, const uint32_t seqNumber) NN_NOEXCEPT;
        static bool LocalMakeActionFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac,
            const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const nn::wlan::ActionFrameType actionType, const uint32_t seqNumber) NN_NOEXCEPT;
        static bool LocalMakeActionFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress& toMac, const nn::wlan::MacAddress& fromMac, const uint16_t etherType, const nn::wlan::ReceivedDataMatchInfo& matchInfo, const nn::wlan::ActionFrameType actionType) NN_NOEXCEPT;
        static bool LocalSeqNumberAddActionFrame(uint8_t frameBuffer[], size_t size, const uint32_t seqNumber) NN_NOEXCEPT;
        static bool LocalMakeResMacFrame(uint8_t frameBuffer[], size_t size, const nn::wlan::MacAddress setMacArray[nn::wlan::ConnectableClientsCountMax], const uint32_t macCount) NN_NOEXCEPT;
        static bool LocalMakeReqMacFrame(uint8_t frameBuffer[], size_t size) NN_NOEXCEPT;

        void LocalSetElapseTime() NN_NOEXCEPT;

        bool LocalPutFrame(uint8_t buffer[], size_t size) NN_NOEXCEPT;
        bool LocalCancelFrameStart(uint32_t waitMsecTime) NN_NOEXCEPT;
        void LocalCancelFrameStop() NN_NOEXCEPT;

        bool LocalCancelActionFrameStart(uint32_t waitMsecTime) NN_NOEXCEPT;
        void LocalCancelActionFrameStop() NN_NOEXCEPT;

        bool LocalReceiveFrameStart(const size_t bufferSize, const size_t packetSize, const uint32_t waitConnectCount, const uint32_t endConnectCount, const uint32_t endPacketCount) NN_NOEXCEPT;
        bool LocalReceiveWait(uint64_t waitMsecTime, bool isStart = true);
        bool LocalReceiveWait(uint64_t waitMsecTime, uint64_t sendTime, bool isStart = true);
        void LocalReceiveFrameStop() NN_NOEXCEPT;

        bool LocalSendFrameStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, bool isClientToClientSend) NN_NOEXCEPT;
        bool LocalSendFrameOneShotWaitStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, bool isClientToClientSend) NN_NOEXCEPT;

        // NTD追加関数
        bool LocalSendFrameOneShotWaitStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, bool isClientToClientSend, bool isNeedReply) NN_NOEXCEPT;

        bool LocalBroadcastSendFrameStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval) NN_NOEXCEPT;
        bool LocalSendActionFrameStart(const size_t sendSize, const int64_t sendTime, const int64_t sendInterval, nn::wlan::ActionFrameType frameType) NN_NOEXCEPT;
        void LocalSendFrameStop() NN_NOEXCEPT;

        void LocalClearSeqNumber() NN_NOEXCEPT;

        void LocalTraceBssInfo() NN_NOEXCEPT;
        void LocalTraceCurrentInfo() NN_NOEXCEPT;
        void LocalTracceClientInfo(LocalConnectsInfo& info) NN_NOEXCEPT;
        void LocalTraceConnectsInfo(bool isModeLog = true) NN_NOEXCEPT;

        // NTD追加ログ（受け入れ検証と分離）
        void LocalTraceConnetsRttInfo(bool isModeLog = true) NN_NOEXCEPT;

    protected:
        static void CommandThread(void* arg) NN_NOEXCEPT;
        void SetCommandEvent(LocalEvent eventType) NN_NOEXCEPT;
        void SetEventMode(LocalEvent eventType) NN_NOEXCEPT;
        void SetState(uint32_t state) NN_NOEXCEPT;

        bool SetPoolMngSetting(LocalPoolMng& testBufferMng, const size_t packetSize) NN_NOEXCEPT;
        LocalPoolMngDetail* GetFirstWritePool(LocalPoolMng& testBufferMng, const GetLocalFrameType frameType) NN_NOEXCEPT;
        LocalPoolMngDetail* GetNextWritePool(LocalPoolMng& testBufferMng, const GetLocalFrameType frameType) NN_NOEXCEPT;

        bool LocalStartTriggerPutFrame(uint8_t frameBuffer[], size_t size, nn::wlan::MacAddress& toMac, nn::wlan::MacAddress& fromMac) NN_NOEXCEPT;
        bool LocalStartTriggerPutActionFrame(uint8_t frameBuffer[], size_t size, nn::wlan::MacAddress& toMac, nn::wlan::MacAddress& fromMac) NN_NOEXCEPT;
        void LocalResGetMacAddress(LocalPayloadData* pPayLoad, LoaclMacAddressInfo* pMacInfo) NN_NOEXCEPT;
        void LocalResGetMacAddress(LocalActionFramePayloadData* pPayLoad, LoaclMacAddressInfo* pMacInfo) NN_NOEXCEPT;

        bool LocalParseFrame(LocalConnectsInfo& connectsInfo, uint8_t frameBuffer[], size_t size) NN_NOEXCEPT;
        bool LocalMasterParseFrame(LocalClientInfoDetail *pDetailInfo, LocalFrameHeader *pHeader, LocalPayloadData* pPayload, bool isFirstFrame) NN_NOEXCEPT;
        bool LocalClientParseFrame(LocalPayloadData* pPayload) NN_NOEXCEPT;
        bool LocalParseActionFrame(LocalConnectsInfo& connectsInfo, uint8_t frameBuffer[], size_t size) NN_NOEXCEPT;
        bool LocalMasterParseActionFrame(LocalClientInfoDetail *pDetailInfo, LocalActionFrameData *pHeader, LocalActionFramePayloadData* pPayload, bool isFirstFrame) NN_NOEXCEPT;
        bool LocalClientParseActionFrame(LocalActionFramePayloadData* pPayload) NN_NOEXCEPT;

        bool LocalClientSendActionFrame(uint8_t* pPutBuffer, size_t putSize,
            uint16_t channel, nn::wlan::ActionFrameType frameType) NN_NOEXCEPT;

        bool LocalMasterSendActionFrame(uint8_t* pPutBuffer, size_t putSize,
            uint16_t channel, nn::wlan::ActionFrameType frameType) NN_NOEXCEPT;

        void ConfigureInterface(const char* interfaceName, IpV4Address ipAddr, bool isDhcp) NN_NOEXCEPT;

        static void LocalSendFrameThread(void* arg) NN_NOEXCEPT;
        static void LocalSendFrameOneShotWaitThread(void* arg) NN_NOEXCEPT;
        static void LocalBroadcastSendFrameThread(void* arg) NN_NOEXCEPT;
        static void LocalSendActionFrameThread(void* arg) NN_NOEXCEPT;
        static void LocalReceiveFrameThread(void* arg) NN_NOEXCEPT;
        static void LocalReceiveActionFrameThread(void* arg) NN_NOEXCEPT;
        static void LocalParseFrameThread(void* arg) NN_NOEXCEPT;

        static void TcpMasterThread(void* arg) NN_NOEXCEPT;
        static void TcpClientThread(void* arg) NN_NOEXCEPT;
    public:

        bool m_isManagerInit;
        uint32_t m_setupMode;
        int64_t m_waitTime;
        int64_t m_waitActionTime;

        uint32_t m_entRxId;
        uint32_t m_actionRxId;

        // LocalMasterメンバー
        nn::wlan::MacAddress m_macAddress;
        nn::wlan::MasterBssParameters m_bssParam;
        nn::wlan::ClientStatus m_clientStatus[nn::wlan::ConnectableClientsCountMax];
        nn::wlan::MacAddress m_clientMacArray[nn::wlan::ConnectableClientsCountMax];
        nn::wlan::MacAddress m_actionFrameMacAddress[LocalApiClientMaxCount];
        nn::Bit32 m_clientStatusBitMap;
        uint32_t m_connectCount;
        uint32_t m_actionMaxConnect;
        LocalConnectsInfo m_connectsInfo;

        LocalEvent m_eventType;
        nn::os::ThreadType m_commandThread;
        nn::os::EventType m_commandEvent;
        uint32_t m_commandWaitTime;

        // LocalClientメンバー
        nn::os::SystemEventType m_connectEvent;
        nn::wlan::Ssid m_ssid;
        nn::wlan::MacAddress m_bssid;
        nn::wlan::Security m_security;
        int16_t m_channel;
        nn::wlan::BeaconIndication m_indication;

        // 共通メンバー
        nn::Result m_ThreadResult;
        std::unique_ptr<uint8_t[]> m_putBuffer;
        std::unique_ptr<uint8_t[]> m_startPutBuffer;
        std::unique_ptr<uint8_t[]> m_getBuffer;
        LocalEtherTypeInfo m_etherList;
        size_t m_putSize;
        size_t m_startPutSize;
        size_t m_getSize;

        // スレッドメンバー
        nn::Result m_sendResult;
        bool m_isActionFrame;
        bool m_isFrame;
        bool m_isCancelFrame;
        bool m_isCancelActionFrame;
        bool m_isSendFrame;
        bool m_isSendActionFrame;
        bool m_isClientToClient;

        // NTD追加定義
        bool m_isNeedReply;
        bool m_isRttSnd;

        nn::os::MutexType m_localApiLock;
        nn::os::MutexType m_localRcvLock;

        LocalPoolMng m_poolFrameMng;

        nn::os::EventType m_parseFrameEvent;
        nn::os::EventType m_receiveFrameEvent;
        nn::os::EventType m_tcpConnectEvent;

        nn::os::ThreadType m_receiveThread;
        nn::os::ThreadType m_receiveActionFrameThread;
        bool m_isReceiveFrame;

        nn::os::ThreadType m_parseThread;
        LocalParseType m_parseType;

        LoaclMacAddressInfo m_resFrameMacAddress;

        uint32_t m_waitConnectCount;
        uint32_t m_endConnectCount;
        uint32_t m_endPacketCount;
        size_t m_packetSize;

        int64_t m_sendTime;
        int64_t m_sendInterval;
        nn::os::ThreadType m_sendThread;
        nn::os::ThreadType m_sendActionThread;

        uint32_t m_sendTimeOut;
        nn::wlan::ActionFrameType m_frameType;

        uint32_t m_seqNumber;
        uint32_t m_seqArray[LocalApiClientMaxCount];

        nn::os::Tick m_elapsedTick;

        nn::btm::WlanMode m_btmMode;

        SocketInfo m_tcpInfo;
        LocalAutoEvent m_tcpMasterEvent;

#ifdef WLAN_TEST_RECV_TRACE
        uint32_t m_masterToClientCount;
        uint32_t m_clientToClientCount;
#endif
    };

}  // namespace WlanTest
