﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt.h>

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/socket.h>
#include <nn/nn_Macro.h>
#include <nn/TargetConfigs/build_Base.h>

namespace WlanTest {

        // SSID
    const int Wlantest_ssidlengthmax = 32;
    static const char WtestSsid[nn::wlan::ScanningSsidCountMax][Wlantest_ssidlengthmax] = {
        "AP2-XXXX-2G",
        "AP1-WEP64",
        "AP5-SharedWEP64",
        "AP1-WEP128",
        "AP5-SharedWEP128",
        "AP3-WPA-PSK_TKIP",
        " ",
        "WPA-PSK(AES)",
        "AP2-WPA2_AES",
    };

    // WiFi Beaconロストタイムアウト
    const int WiFiBeaconLostTimeout         = 10;

    // WiFi-AP接続タイムアウト
    const int64_t WiFiConnectTimeout        = 30000;
    const int64_t WiFiManualConnectTimeout  = 10 * 60000;

    // Connectパラメータ
    const int Wlantest_keylengthmax = 64;
    static const nn::Bit8 Default_key[nn::wlan::ScanningSsidCountMax][Wlantest_keylengthmax] = {
        "",
        "3031323334",
        "1010101010",
        "30313233343536373839616263",
        "31323334353637383930313233",
        "123456789012345678901234567890123456789012345678901234567890123",
        "12345678",
        "Passphrase- [!#$%&'()=^~@;+:*,.<>/?`@\"']|",
        "AbcdefghijklmnopqrstuvwxyZ",
    };

    static const nn::wlan::SecurityMode Default_privacyMode[nn::wlan::ScanningSsidCountMax] = {
        nn::wlan::SecurityMode::SecurityMode_Open,
        nn::wlan::SecurityMode::SecurityMode_Wep64Open,
        nn::wlan::SecurityMode::SecurityMode_Wep64Shared,
        nn::wlan::SecurityMode::SecurityMode_Wep128Open,
        nn::wlan::SecurityMode::SecurityMode_Wep128Shared,
        nn::wlan::SecurityMode::SecurityMode_WpaTkip,
        nn::wlan::SecurityMode::SecurityMode_Wpa2Tkip,
        nn::wlan::SecurityMode::SecurityMode_WpaAes,
        nn::wlan::SecurityMode::SecurityMode_Wpa2Aes,
        nn::wlan::SecurityMode::SecurityMode_StaticAes
    };

    static const nn::wlan::SecurityMode Default_groupPrivacyMode[nn::wlan::ScanningSsidCountMax] = {
        nn::wlan::SecurityMode::SecurityMode_Open,
        nn::wlan::SecurityMode::SecurityMode_Wep64Open,
        nn::wlan::SecurityMode::SecurityMode_Wep64Shared,
        nn::wlan::SecurityMode::SecurityMode_Wep128Open,
        nn::wlan::SecurityMode::SecurityMode_Wep128Shared,
        nn::wlan::SecurityMode::SecurityMode_WpaTkip,
        nn::wlan::SecurityMode::SecurityMode_Wpa2Tkip,
        nn::wlan::SecurityMode::SecurityMode_WpaAes,
        nn::wlan::SecurityMode::SecurityMode_Wpa2Aes,
        nn::wlan::SecurityMode::SecurityMode_StaticAes
    };

    static const char* CipherTypeString[] =
    {
        "USE_GROUPKEY",
        "WEP_40BIT",
        "TKIP",
        "AES_OCB",
        "AES_CCMP",
        "WEP_104BIT",
        "NONE"
    };

    static const char*  AkmTypeString[] =
    {
        "NO_WPA",
        "WPA_EAP",
        "WPA_PSK"
    };

    static const uint32_t Default_keyIdx[nn::wlan::ScanningSsidCountMax] = {
        0,
        0,
        0,
        0,
        0,
        0,
        0,
        0,
        0,
        0
    };

    static const char* WlanStateStr[] =
    {
        "WlanState_Stop",
        "WlanState_Ready",
        "WlanState_InfraIdle",
        "WlanState_InfraSta",
        "WlanState_InfraIdleScan",
        "WlanState_InfraStaScan",
        "WlanState_LocalMasterIdle",
        "WlanState_LocalMasterBss",
        "WlanState_LocalMasterIdleScan",
        "WlanState_LocalMasterBssScan",
        "WlanState_LocalClientIdle",
        "WlanState_LocalClient",
        "WlanState_LocalClientIdleScan",
        "WlanState_LocalClientScan",
        "WlanState_LocalSpectatorIdle",
        "WlanState_LocalSpectator",
        "WlanState_LocalSpectatorIdleScan",
        "WlanState_LocalSpectatorScan",
        "WlanState_LocalLcsMasterIdle",
        "WlanState_LocalLcsMasterBss",
        "WlanState_LocalLcsMasterIdleScan",
        "WlanState_LocalLcsMasterBssScan",
        "WlanState_LocalLcsClientIdle",
        "WlanState_LocalLcsClient",
        "WlanState_LocalLcsClientIdleScan",
        "WlanState_LocalLcsClientScan",
        "WlanState_Sleep",
        "WlanState_Exit",
        "WlanState_Num"
    };

#define WLANTEST_ADDRESS_TO_STR(address) #address

    enum WlanTestState
    {
        WlanTestState_Infra = 0,
        WlanTestState_Local,
        WlanTestState_None,
    };

    class WlanUnitTest
    {
    public:
        WlanUnitTest() NN_NOEXCEPT;
        void UnitTest(const char* fileName, const char* testName, void(*WlanTestFunc)(void)) NN_NOEXCEPT;
        void SetTestCaseName(const char* testCaseName, int32_t* testCaseNo) NN_NOEXCEPT;
        bool Compare(uint64_t condition, uint64_t actualValue) NN_NOEXCEPT;
        bool Compare(int64_t condition, int64_t actualValue) NN_NOEXCEPT;
        bool Compare(uint32_t condition, uint32_t actualValue) NN_NOEXCEPT;
        bool Compare(int32_t condition, int32_t actualValue) NN_NOEXCEPT;
        bool Compare(uint16_t condition, uint16_t actualValue) NN_NOEXCEPT;
        bool Compare(int16_t condition, int16_t actualValue) NN_NOEXCEPT;
        bool Compare(uint8_t condition, uint8_t actualValue) NN_NOEXCEPT;
        bool Compare(int8_t condition, int8_t actualValue) NN_NOEXCEPT;
        bool CompareTrue(bool actualValue) NN_NOEXCEPT;
        bool CompareFalse(bool actualValue) NN_NOEXCEPT;
        bool SuccessResult(nn::Result actualResult) NN_NOEXCEPT;
        bool FailureResult(nn::Result actualResult) NN_NOEXCEPT;
        void StateSet(WlanTestState wlanTestState) NN_NOEXCEPT;
        void HideState() NN_NOEXCEPT;
    protected:
        bool SetCompareResult(bool isExpectRsult) NN_NOEXCEPT;

        int32_t  successCount;
        int32_t  failCount;
        char*    pTestName;
        int32_t* pTestNo;
        WlanTestState   test_State = WlanTestState_None;
    };

    WlanUnitTest* GetResultClass();
    void SystemInitialize() NN_NOEXCEPT;

    bool InfraConnectWait(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& bssid,
        int16_t channel, const nn::wlan::Security& security, bool autoKeepAlive,
        int beaconLostTimeout = 10, int64_t msecWaitTimeout = WiFiConnectTimeout) NN_NOEXCEPT;

    void InfraTraceConnectInfo(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& bssid,
        int16_t channel, const nn::wlan::Security& security, bool autoKeepAlive,
        int beaconLostTimeout) NN_NOEXCEPT;

#define WLANTEST_UNIT_TEST(testName, testFunc)                                              \
    GetResultClass()->UnitTest(__FILE__, testName, testFunc);                               \



#ifdef WLANTEST_GTESTLIB

#define WLANTEST_SET_TESTNAME(testCaseName, testCaseNo)                                     \
    GetResultClass()->SetTestCaseName(testCaseName, testCaseNo);

#define WLANTEST_STATE_SET(wlanTestState)                                               \
        GetResultClass()->StateSet(wlanTestState);


#define WLANTEST_EXPECT_RESULT_SUCCESS(actualResult)                                        \
    {                                                                                       \
        nn::Result temporaryResult = (actualResult);                                        \
        if (temporaryResult.IsSuccess() != true)                                            \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        NNT_EXPECT_RESULT_SUCCESS(temporaryResult);                                         \
    }

#define WLANTEST_EXPECT_RESULT_NOTSUCCESS(actualResult)                                     \
    {                                                                                       \
        nn::Result temporaryResult = (actualResult);                                        \
        if (temporaryResult.IsSuccess() == true)                                            \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        else                                                                                \
        {                                                                                   \
            NN_LOG("%s %d Line Expect Result Failler code(0x%08x) %d:%d\n",__FUNCTION__ ,   \
                        __LINE__, temporaryResult.GetInnerValueForDebug(),                  \
                        temporaryResult.GetModule(), temporaryResult.GetDescription());     \
        }                                                                                   \
        EXPECT_FALSE(temporaryResult.IsSuccess());                                          \
    }

#define WLANTEST_ASSERT_RESULT_SUCCESS(actualResult)                                        \
    {                                                                                       \
        nn::Result temporaryResult = (actualResult);                                        \
        if (temporaryResult.IsSuccess() != true)                                            \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        NNT_ASSERT_RESULT_SUCCESS(temporaryResult);                                         \
    }

#define WLANTEST_ASSERT_RESULT_NOTSUCCESS(actualResult)                                     \
    {                                                                                       \
        nn::Result temporaryResult = (actualResult);                                        \
        if (temporaryResult.IsSuccess() == true)                                            \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        ASSERT_FALSE(temporaryResult.IsSuccess());                                          \
    }

#define WLANTEST_EXPECT_TRUE(actualValue)                                                   \
    {                                                                                       \
        bool temporaryIsResult = (actualValue);                                             \
        if(temporaryIsResult != true)                                                       \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        EXPECT_TRUE(temporaryIsResult);                                                     \
    }

#define WLANTEST_ASSERT_TRUE(actualValue)                                                   \
    {                                                                                       \
        bool temporaryIsResult = (actualValue);                                             \
        if(temporaryIsResult != true)                                                       \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        ASSERT_TRUE(temporaryIsResult);                                                     \
    }

#define WLANTEST_EXPECT_FALSE(actualValue)                                                  \
    {                                                                                       \
        bool temporaryIsResult = (actualValue);                                             \
        if(temporaryIsResult == true)                                                       \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        EXPECT_FALSE(temporaryIsResult);                                                    \
    }

#define WLANTEST_ASSERT_FALSE(actualValue)                                                  \
    {                                                                                       \
        bool temporaryIsResult = (actualValue);                                             \
        if(temporaryIsResult == true)                                                       \
        {                                                                                   \
            GetResultClass()->HideState();                                                  \
        }                                                                                   \
        ASSERT_FALSE(temporaryIsResult);                                                    \
    }

#define WLANTEST_EXPECT_DEATH_TEST(statement)                                               \
        EXPECT_DEATH(statement, "");

#define WLANTEST_ASSERT_DEATH_TEST(statement)                                               \
        ASSERT_DEATH(statement, "");

#define WLANTEST_SCOPED_TRACE(message)                                                      \
        SCOPED_TRACE(message);

#else   // WLANTEST_GTESTLIB

#define WLANTEST_SET_TESTNAME(testCaseName, testCaseNo)                                     \
    GetResultClass()->SetTestCaseName(testCaseName, testCaseNo);

#define WLANTEST_STATE_SET(wlanTestState)                                               \
        GetResultClass()->StateSet(wlanTestState);

#define WLANTEST_EXPECT_RESULT_SUCCESS(actualResult)                                    \
    {                                                                                   \
        nn::Result temporaryResult = (actualResult);                                    \
        if (GetResultClass()->SuccessResult(temporaryResult) != true)                   \
        {                                                                               \
            NN_LOG("%s %d Line Success Result Faillercode(0x%08x)\n",__FUNCTION__,      \
                        __LINE__, temporaryResult.GetInnerValueForDebug());             \
            GetResultClass()->HideState();                                              \
        }                                                                               \
    }

#define WLANTEST_EXPECT_RESULT_NOTSUCCESS(actualResult)                                 \
    {                                                                                   \
        nn::Result temporaryResult = (actualResult);                                    \
        if (GetResultClass()->SuccessResult(temporaryResult) == true)                   \
        {                                                                               \
            NN_LOG("%s %d Line Expect Result Failler code(0x%08x)\n",__FUNCTION__,      \
                        __LINE__, temporaryResult.GetInnerValueForDebug());             \
            GetResultClass()->HideState();                                              \
        }                                                                               \
    }

#define WLANTEST_ASSERT_RESULT_SUCCESS(actualResult)                                    \
    {                                                                                   \
        nn::Result temporaryResult = (actualResult);                                    \
        if (GetResultClass()->SuccessResult(temporaryResult) != true)                   \
        {                                                                               \
            NN_LOG("%s %d Line Assert Result Failler code(0x%08x)\n",__FUNCTION__,      \
                        __LINE__, temporaryResult.GetInnerValueForDebug());             \
            GetResultClass()->HideState();                                              \
            return;                                                                     \
        }                                                                               \
    }

#define WLANTEST_ASSERT_RESULT_NOTSUCCESS(actualResult)                                 \
    {                                                                                   \
        nn::Result temporaryResult = (actualResult);                                    \
        if (GetResultClass()->SuccessResult(temporaryResult) == true)                   \
        {                                                                               \
            NN_LOG("%s %d Line Not Assert Result Failler code(0x%08)x\n",__FUNCTION__,  \
                        __LINE__, temporaryResult.GetInnerValueForDebug());             \
            GetResultClass()->HideState();                                              \
            return;                                                                     \
        }                                                                               \
    }

#define WLANTEST_EXPECT_TRUE(actualValue)                                               \
    {                                                                                   \
        bool temporaryIsResult = (actualValue);                                         \
        if(WlanTest::GetResultClass()->CompareTrue(temporaryIsResult) != true)          \
        {                                                                               \
            NN_LOG("%s %d Line Not Expect Result Failler\n",__FUNCTION__, __LINE__);    \
            GetResultClass()->HideState();                                              \
        }                                                                               \
    }

#define WLANTEST_ASSERT_TRUE(actualValue)                                               \
    {                                                                                   \
        bool temporaryIsResult = (actualValue);                                         \
        if(GetResultClass()->CompareTrue(temporaryIsResult) != true)                    \
        {                                                                               \
            NN_LOG("%s %d Line Not Assert Result Failler\n",__FUNCTION__, __LINE__);    \
            GetResultClass()->HideState();                                              \
            return;                                                                     \
        }                                                                               \
    }

#define WLANTEST_EXPECT_FALSE(actualValue)                                              \
    {                                                                                   \
        bool temporaryIsResult = (actualValue);                                         \
        if(GetResultClass()->CompareFalse(temporaryIsResult) != true)                   \
        {                                                                               \
            NN_LOG("%s %d Line Not Expect Result Failler\n",__FUNCTION__, __LINE__);    \
            GetResultClass()->HideState();                                              \
        }                                                                               \
    }


#define WLANTEST_ASSERT_FALSE(actualValue)                                              \
    {                                                                                   \
        bool temporaryIsResult = (actualValue);                                         \
        if(GetResultClass()->CompareFalse(temporaryIsResult) != true)                   \
        {                                                                               \
            NN_LOG("%s %d Line Not Assert Result Failler\n",__FUNCTION__, __LINE__);    \
            GetResultClass()->HideState();                                              \
            return;                                                                     \
        }                                                                               \
    }

#define WLANTEST_EXPECT_DEATH_TEST(statement)

#define WLANTEST_ASSERT_DEATH_TEST(statement)

#define WLANTEST_SCOPED_TRACE(message)


#endif  //WLANTEST_GTESTLIB

}  // namespace WlanTest

