﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mbuf/mbuf_Mbuf.h>
#include "testWlan_UnitTest.h"

namespace WlanTest {

WlanUnitTest wlanClassResult;

// Wlan単体試験クラス
WlanUnitTest::WlanUnitTest() NN_NOEXCEPT
    :   successCount    (0),
        failCount       (0),
        pTestName       (nullptr),
        pTestNo         (nullptr)
{
}

// 単体試験関数を呼び出す関数
void WlanUnitTest::UnitTest(const char* fileName, const char* testName, void (*WlanTestFunc)(void)) NN_NOEXCEPT
{
    this->successCount = 0;
    this->failCount = 0;
    if (WlanTestFunc != nullptr)
    {
        NN_LOG("\n\n %s \n", fileName);
        NN_LOG(" [==========] Runnning %s Start \n", testName);
        WlanTestFunc();
        if (this->failCount > 0)
        {
            NN_LOG(" [  FAILED  ] %s \n", testName);
        }

        if (this->successCount > 0 && this->failCount == 0)
        {
            NN_LOG(" [  PASSED  ] %s \n", testName);
        }
        NN_LOG(" [==========] Runnning %s End \n", testName);
    }
}

void WlanUnitTest::SetTestCaseName(const char* testCaseName, int32_t* testCaseNo) NN_NOEXCEPT
{
    if ((testCaseName != nullptr) && (testCaseNo != nullptr))
    {
        this->pTestName = const_cast<char*>(testCaseName);
        this->pTestNo   = testCaseNo;
        NN_LOG("\n\n[ RUN      ] %s TestNo.%d \n", this->pTestName, *(this->pTestNo));
        *(this->pTestNo) += 1 ;
    }
}

// 関数結果をIsSuccess、IsFailler別にカウントする関数
bool WlanUnitTest::SuccessResult(nn::Result actualResult) NN_NOEXCEPT
{
    return SetCompareResult(actualResult.IsSuccess() == true);
}

// 比較結果が一致するか不一致するかでカウントする関数
// 一致：isSuccessカウント
// 不一致：isFaillerカウント
bool WlanUnitTest::Compare(uint64_t condition, uint64_t actualValue) NN_NOEXCEPT
{
    return SetCompareResult((condition == actualValue));
}

bool WlanUnitTest::Compare(uint32_t condition, uint32_t actualValue) NN_NOEXCEPT
{
    return SetCompareResult((condition == actualValue));
}

bool WlanUnitTest::Compare(uint16_t condition, uint16_t actualValue) NN_NOEXCEPT
{
    return SetCompareResult((condition == actualValue));
}

bool WlanUnitTest::Compare(uint8_t condition, uint8_t actualValue) NN_NOEXCEPT
{
    return SetCompareResult((condition == actualValue));
}

bool WlanUnitTest::Compare(int8_t condition, int8_t actualValue) NN_NOEXCEPT
{
    return SetCompareResult((condition == actualValue));
}


// bool型の結果でカウントする関数
// true:isSuccessカウント
// false:isFaillerカウント
bool WlanUnitTest::CompareTrue(bool actualValue) NN_NOEXCEPT
{
    return SetCompareResult((actualValue == true));
}

// bool型の結果でカウントする関数
// true:isFaillerカウント
// false:isSuccessカウント
bool WlanUnitTest::CompareFalse(bool actualValue) NN_NOEXCEPT
{
    return SetCompareResult((actualValue == false));
}


// 引数の結果でisSuccess/isFailler別にカウントするprivate関数
bool WlanUnitTest::SetCompareResult(bool isExpectResult) NN_NOEXCEPT
{
    if (isExpectResult == true)
    {
        this->successCount++;
    }
    else
    {
        this->failCount++;
    }

    if (this->pTestName != nullptr)
    {
        if (isExpectResult == true)
        {
            NN_LOG("\n\n [       OK ] %s TestNo.%d \n", this->pTestName, *(this->pTestNo) - 1);
        }
        else
        {
            NN_LOG("\n\n [       NG ] %s TestNo.%d \n", this->pTestName, *(this->pTestNo) - 1);
        }
        this->pTestName = nullptr;
        this->pTestNo = nullptr;
    }

    return isExpectResult;
}

void InfraTraceConnectInfo(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& bssid,
    int16_t channel, const nn::wlan::Security& security, bool autoKeepAlive,
    int beaconLostTimeout)
{
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    char macStr[nn::wlan::MacAddress::MacStringSize];
    nn::wlan::ConnectionStatus conStatus;
    nn::Result result = nn::wlan::Infra::GetConnectionStatus(&conStatus);
    if (result.IsSuccess() == true)
    {
        const char* pstrState[] = { "Idle", "Connected", "Dissconnect", "Join", "Invalid State"};
        int pos = 0;

        switch (conStatus.state)
        {
        case nn::wlan::ConnectionState::ConnectionState_Idle:
            pos = 0;
            break;
        case nn::wlan::ConnectionState::ConnectionState_Connected:
            pos = 1;
            break;
        case nn::wlan::ConnectionState::ConnectionState_Disconnected:
            pos = 2;
            break;
        case nn::wlan::ConnectionState::ConnectionState_Joined:
            pos = 3;
            break;
        default:
            pos = 4;
            break;
        }

        NN_LOG("\n");
        NN_LOG("             *** INFRA CONNECT STATUS ***\n");
        NN_LOG("             STATUS          : %s\n", pstrState[pos]);
        NN_LOG("             SSID            : %s\n", ssid.GetHexString(ssidStr));
        NN_LOG("             BSSID           : %s\n", bssid.GetString(macStr));
        NN_LOG("             CHANNEL         : %d\n", channel);
        NN_LOG("             PRIVACYMODE     : %d\n", security.privacyMode);
        NN_LOG("             PRIVACYMODE     : %d\n", security.groupPrivacyMode);
        NN_LOG("             KEY INDEX       : %d\n", security.keyIdx);
        NN_LOG("             KEY             : %s\n", security.key);
        NN_LOG("             AUTOKEEPALIVE   : %s\n", autoKeepAlive == true ? "ON": "OFF");
        NN_LOG("             BCONLOSTTIMEOUT : %d\n", beaconLostTimeout);
        NN_LOG("             *** INFRA CONNECT STATUS ***\n\n");
    }
}

void WlanUnitTest::StateSet(WlanTestState wlanTestState) NN_NOEXCEPT
{
    test_State = wlanTestState;
}

// 結果表示の際にWLANの状態を表示させる
void WlanUnitTest::HideState() NN_NOEXCEPT
{
    nn::wlan::WlanState wlanState = nn::wlan::WlanState_Stop;
    if(WlanTestState_Infra == test_State)
    {
        nn::wlan::Infra::GetState(&wlanState);
    }
    else if(WlanTestState_Local == test_State)
    {
        nn::wlan::Local::GetState(&wlanState);
    }
    NN_LOG("WlanState:%s\n", WlanStateStr[wlanState]);
}

// Wlan単体試験クラスのインスタンスポインタを返す関数
WlanUnitTest* GetResultClass() NN_NOEXCEPT
{
    return &wlanClassResult;
}

// Wlan Local通信用の初期化処理
void SystemInitialize() NN_NOEXCEPT
{
    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabled(false);
}

// Infra WiFi-AP接続処理
bool InfraConnectWait(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& bssid,
    int16_t channel, const nn::wlan::Security& security, bool autoKeepAlive,
    int beaconLostTimeout, int64_t msecWaitTimeout) NN_NOEXCEPT
{
    const int64_t sleepTime = 3000;

    nn::wlan::ConnectionStatus status;
    int64_t waitTime = std::min(msecWaitTimeout, sleepTime);
    bool isConnect = false;
    nn::os::Tick sysTick;
    nn::Result result;

    nn::os::SystemEvent connectionEvent;
    nn::wlan::Infra::GetConnectionEvent(connectionEvent.GetBase());
    sysTick = nn::os::GetSystemTick();

    while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= msecWaitTimeout)
    {
        result = nn::wlan::Infra::Connect(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             WiFi Connect Error Code(0x%08x) %s %d Line\n",
                result.GetInnerValueForDebug(), __FILE__, __LINE__);
            InfraTraceConnectInfo(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
            break;
        }

        if (connectionEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(waitTime)) != true)
        {
            continue;
        }

        result = nn::wlan::Infra::GetConnectionStatus(&status);
        if (result.IsSuccess() != true)
        {
            NN_LOG("             WiFi Connect Status Error Code(0x%08x) %s %d Line\n",
                result.GetInnerValueForDebug(), __FILE__, __LINE__);
            InfraTraceConnectInfo(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
            break;
        }

        if (status.state == nn::wlan::ConnectionState_Connected)
        {
            isConnect = true;
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitTime));
    }

    if (isConnect != true && result.IsSuccess() == true)
    {
        NN_LOG("             WiFi Connect Time Out %s %d\n",
            __FILE__, __LINE__);
        InfraTraceConnectInfo(ssid, bssid, channel, security, autoKeepAlive, beaconLostTimeout);
    }
    return isConnect;
}

}
