﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdint.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>

#include <nne/vcc/vcc_API.h>
#include <nne/vcc/vccapi.h>

#include <nnt/gtest/gtest.h>

#include <nn/dd.h>

#include <nn/fs/fs_MemoryManagement.h>
#include <cstdlib>

//const uintptr_t MAX_RATES_COUNT = 32;
const uintptr_t PLLP_BASE_0 = 0xa0;
const uintptr_t PLLP_BASE_0_BYPASS_SHIFT = 31;
const uintptr_t PLLP_BASE_0_BYPASS_MASK = 0x1;
const uintptr_t PLLP_BASE_0_DIVN_SHIFT = 10;
const uintptr_t PLLP_BASE_0_DIVN_MASK = 0xff;
const uintptr_t PLLP_BASE_0_DIVM_SHIFT = 0;
const uintptr_t PLLP_BASE_0_DIVM_MASK = 0xff;
//const uintptr_t PLLP_RESHIFT_0 = 0x528;
//const uintptr_t PLLP_RESHIFT_0_RATIO_SHIFT = 2;
//const uintptr_t PLLP_RESHIFT_0_RATIO_MASK = 0xff;
const uintptr_t PLLC4_BASE_0 = 0x5a4;
const uintptr_t PLLC4_BASE_0_BYPASS_SHIFT = 31;
const uintptr_t PLLC4_BASE_0_BYPASS_MASK = 0x1;
const uintptr_t PLLC4_BASE_0_DIVN_SHIFT = 8;
const uintptr_t PLLC4_BASE_0_DIVN_MASK = 0xff;
const uintptr_t PLLC4_BASE_0_DIVM_SHIFT = 0;
const uintptr_t PLLC4_BASE_0_DIVM_MASK = 0xff;
const uintptr_t PLLC4_BASE_0_DIVP_SHIFT = 19;
const uintptr_t PLLC4_BASE_0_DIVP_MASK = 0xf; // only 16 divp entries are defined in this test
const uintptr_t CLK_SOURCE_SDMMC1_0 = 0x150;
const uintptr_t CLK_SOURCE_SDMMC2_0 = 0x154;
const uintptr_t CLK_SOURCE_SDMMC3_0 = 0x1bc;
const uintptr_t CLK_SOURCE_SDMMC4_0 = 0x164;
const uint32_t  CLK_SOURCE_SDMMCX_0_SRC_SHIFT = 29;
const uint32_t  CLK_SOURCE_SDMMCX_0_SRC_MASK = 0x7;
const uint32_t  CLK_SOURCE_SDMMCX_0_DIV_SHIFT = 0;
const uint32_t  CLK_SOURCE_SDMMCX_0_DIV_MASK = 0xff;
const nn::dd::PhysicalAddress ClockResetControllerRegistersPhysicalAddress = 0x60006000ull;
const size_t ClockResetControllerRegistersSize = 0x1000;

// fixed osc value
const uint32_t  OSC_DIV_CLK = 38400000;

const size_t g_HeapSize = 16 * 1024 * 1024;

extern "C" int NvOsDrvOpen(const char *pathname, int flags);

enum {
    PLL_INVALID,
    PLL_C4,
    PLL_P,
    //    PLL_CLKM,
};

enum {
    OUT_INVALID,
    OUT_0,
    OUT_1,
    OUT_2,
    OUT_0_LJ,
    OUT_1_LJ,
    OUT_2_LJ,
};

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(g_HeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", g_HeapSize);
}

uint32_t RegRead(uintptr_t address)
{
    uint32_t* ptr = reinterpret_cast<uint32_t*>(address);
    return *ptr;
}

void RegWrite(uintptr_t address, uint32_t value)
{
    uint32_t* ptr = reinterpret_cast<uint32_t*>(address);
    *ptr = value;
}

uint32_t CalculatePLLP(uint32_t out)
{
    uintptr_t registersAddress;
    registersAddress = nn::dd::QueryIoMappingAddress(ClockResetControllerRegistersPhysicalAddress,
                                                     ClockResetControllerRegistersSize);

    uint32_t base = RegRead(registersAddress + PLLP_BASE_0);
    //uint32_t reshift = RegRead(registersAddress + PLLP_RESHIFT_0);
    uint32_t vco;
    uint32_t out0;

    if ( (base >> PLLP_BASE_0_BYPASS_SHIFT) & PLLP_BASE_0_BYPASS_MASK )
    {
        vco = OSC_DIV_CLK;
    }
    else
    {
        uint32_t divn = (base >> PLLP_BASE_0_DIVN_SHIFT) & PLLP_BASE_0_DIVN_MASK;
        uint32_t divm = (base >> PLLP_BASE_0_DIVM_SHIFT) & PLLP_BASE_0_DIVM_MASK;
        vco = (divn * OSC_DIV_CLK) / divm;
    }

    //uint32_t ratio = (reshift >> PLLP_RESHIFT_0_RATIO_SHIFT) & PLLP_RESHIFT_0_RATIO_MASK;

    //out0 = vco / (ratio / 2 + 1);

    out0 = vco;

    if ( out == OUT_0 )
    {
        return out0;
    }
    else
    {
        return 0;
    }
}

uint32_t CalculatePLLC4(uint32_t out)
{
    uintptr_t registersAddress;
    registersAddress = nn::dd::QueryIoMappingAddress(ClockResetControllerRegistersPhysicalAddress,
                                                     ClockResetControllerRegistersSize);

    uint32_t base = RegRead(registersAddress + PLLC4_BASE_0);
    //uint32_t misc = RegRead(registersAddress + PLLC4_MISC_0);
    uint32_t vco;
    uint32_t out0;
    uint32_t out1;
    uint32_t out2;
    //uint32_t out3;

    if ( (base >> PLLC4_BASE_0_BYPASS_SHIFT) & PLLC4_BASE_0_BYPASS_MASK )
    {
        vco = OSC_DIV_CLK;
    }
    else
    {
        uint32_t divn = (base >> PLLC4_BASE_0_DIVN_SHIFT) & PLLC4_BASE_0_DIVN_MASK;
        uint32_t divm = (base >> PLLC4_BASE_0_DIVM_SHIFT) & PLLC4_BASE_0_DIVM_MASK;
        vco = (divn * OSC_DIV_CLK) / divm;
    }
    uint32_t divp = (base >> PLLC4_BASE_0_DIVP_SHIFT) & PLLC4_BASE_0_DIVP_MASK;
    uint32_t pl[] = { 1, 2, 3, 4, 5, 6, 8, 9, 10, 12, 15, 16, 18, 20, 24, 30 };
    //uint32_t ratio = (base >> PLLC4_BASE_0_RATIO_SHIFT) & PLLC4_RATIO_MASK;

    out0 = vco / pl[divp];
    out1 = vco / 3;
    out2 = vco / 5;
    /*
    if ((base >> PLLC4_BASE_0_DIV_BYP_SHIFT) & PLLC4_DIV_BYP_MASK )
    {
        out3 = out0 / (ratio / 2 + 1);
    }
    */

    if ( out == OUT_0 || out == OUT_0_LJ )
    {
        return out0;
    }
    else if ( out == OUT_1 || out == OUT_1_LJ )
    {
        return out1;
    }
    else if ( out == OUT_2 || out == OUT_2_LJ )
    {
        return out2;
    }
    else
    {
        return 0;
    }
}

uint32_t CalculateClockSDMMC(uint32_t id)
{
    uintptr_t registersAddress;
    uint32_t parentPll;
    uint32_t out;
    uint32_t clockSource;
    uint32_t src;
    uint32_t div;

    registersAddress = nn::dd::QueryIoMappingAddress(ClockResetControllerRegistersPhysicalAddress,
                                                     ClockResetControllerRegistersSize);

    if ( id == 1 )
    {
        clockSource = RegRead(registersAddress + CLK_SOURCE_SDMMC1_0);
    }
    else if ( id == 2 )
    {
        clockSource = RegRead(registersAddress + CLK_SOURCE_SDMMC2_0);
    }
    else if ( id == 3 )
    {
        clockSource = RegRead(registersAddress + CLK_SOURCE_SDMMC3_0);
    }
    else if ( id == 4 )
    {
        clockSource = RegRead(registersAddress + CLK_SOURCE_SDMMC4_0);
    }
    else
    {
        NN_LOG("invalid argument %d\n", id);
        return 0;
    }

    NN_LOG("sdmmc %d, ", id);

    NN_LOG("clockSource 0x%08x, ", clockSource);

    src = (clockSource >> CLK_SOURCE_SDMMCX_0_SRC_SHIFT) & CLK_SOURCE_SDMMCX_0_SRC_MASK;
    div = (clockSource >> CLK_SOURCE_SDMMCX_0_DIV_SHIFT) & CLK_SOURCE_SDMMCX_0_DIV_MASK;

    NN_LOG("src 0x%x, div 0x%x\n", src, div);

    // sdmmc 1
    // 0: PLLP_OUT0
    // 1: PLLA_OUT --- not supported in this test ---
    // 2: PLLC_OUT0 --- not supported in this test ---
    // 3: PLLC4_OUT2
    // 4: PLLM_OUT0 --- not supported in this test ---
    // 5: PLLE_OUT0 --- not supported in this test ---
    // 6: CLK_M --- not supported in this test ---
    // 7: PLLC4_OUT0

    // sdmmc 2 or 4
    // 0: PLLP_OUT0
    // 1: PLLC4_OUT2_LJ
    // 2: PLLC4_OUT0_LJ
    // 3: PLLC4_OUT2
    // 4: PLLC4_OUT1
    // 5: PLLC4_OUT1_LJ
    // 6: CLK_M --- not supported in this test ---
    // 7: PLLC4_OUT0

    // sdmmc 3
    // 0: PLLP_OUT0
    // 1: PLLA_OUT --- not supported in this test ---
    // 2: PLLC_OUT0 --- not supported in this test ---
    // 3: PLLC4_OUT2
    // 4: PLLC4_OUT1
    // 5: PLLE_OUT0 --- not supported in this test ---
    // 6: CLK_M --- not supported in this test ---
    // 7: PLLC4_OUT0

    if ( id == 1 )
    {
        switch ( src )
        {
        case 0: parentPll = PLL_P; out = OUT_0; break;
        case 3: parentPll = PLL_C4; out = OUT_2; break;
            //case 6: parentPll = PLL_CLKM; out = OUT_0; break;
        case 7: parentPll = PLL_C4; out = OUT_0; break;
        default: parentPll = PLL_INVALID; out = OUT_INVALID; break;
        }
    }
    else if ( id == 2 || id == 4 )
    {
        switch ( src )
        {
        case 0: parentPll = PLL_P; out = OUT_0; break;
        case 1: parentPll = PLL_C4; out = OUT_2_LJ; break;
        case 2: parentPll = PLL_C4; out = OUT_0_LJ; break;
        case 3: parentPll = PLL_C4; out = OUT_2; break;
        case 4: parentPll = PLL_C4; out = OUT_1; break;
        case 5: parentPll = PLL_C4; out = OUT_1_LJ; break;
            //case 6: parentPll = PLL_CLKM; out = OUT_0; break;
        case 7: parentPll = PLL_C4; out = OUT_0; break;
        default: parentPll = PLL_INVALID; out = OUT_INVALID; break;
        }
    }
    else if ( id == 3 )
    {
        switch ( src )
        {
        case 0: parentPll = PLL_P; out = OUT_0; break;
        case 3: parentPll = PLL_C4; out = OUT_2; break;
        case 4: parentPll = PLL_C4; out = OUT_1; break;
            //case 6: parentPll = PLL_CLKM; out = OUT_0; break;
        case 7: parentPll = PLL_C4; out = OUT_0; break;
        default: parentPll = PLL_INVALID; out = OUT_INVALID; break;
        }
    }
    else
    {
        parentPll = PLL_INVALID; out = OUT_INVALID;
    }

    if ( parentPll == PLL_C4 ) NN_LOG("pllc4 - ");
    else if ( parentPll == PLL_P ) NN_LOG("pllp - ");
    else NN_LOG("invalid pll - ");

    if ( out == OUT_0 ) NN_LOG("out0\n");
    else if ( out == OUT_1 ) NN_LOG("out1\n");
    else if ( out == OUT_1 ) NN_LOG("out2\n");
    else if ( out == OUT_0_LJ ) NN_LOG("out0 low jitter\n");
    else if ( out == OUT_1_LJ ) NN_LOG("out1 low jitter\n");
    else if ( out == OUT_2_LJ ) NN_LOG("out2 low jitter\n");
    else NN_LOG("invalid out\n");

    if ( parentPll == PLL_C4 )
    {
        return CalculatePLLC4(out) / (div / 2 + 1);
    }
    else if ( parentPll == PLL_P )
    {
        return CalculatePLLP(out) / (div / 2 + 1);
    }
    //else if ( parentPll == PLL_CLKM )
    //{
    //    return CalculateCLKM(out) / (div / 2 + 1);
    //}
    else
    {
        return 0;
    }
}

TEST( VccTestSdmmcClk, All )
{
    IP sdmmcIp[] = {IP_SDMMC1, IP_SDMMC2, IP_SDMMC3, IP_SDMMC4};
    uint32_t index, rateIndex;
    uint32_t rates[] = {
        199680000,
        204000000,
        102000000,
        99840000,
    };
    int32_t count = 0;
    nn::pcv::IPSettingsQuery settings;

    nne::vcc::Initialize();

    for ( index = 0; index < 4; index++ )
    {
        nne::vcc::QuerySettings(&settings);
        NN_LOG("Expected %d Hz - Actual %d Hz.\n", rates[index], settings.f_Hz);

        // Clock-gating is not supported by Raptor yet as of 10/19/2015
        //NN_LOG("enable clock sdmmc %d.\n", index);
        //ASSERT_TRUE(nne::vcc::SetClockEnabled(sdmmcIp[index], true).IsSuccess());

        count = 0;

        //NN_LOG("query posslble rate of sdmmc %d.\n", index);
        //ASSERT_TRUE(nne::vcc::QueryPossibleRates(sdmmcIp[index], rates, &count).IsSuccess());
        for ( rateIndex = 0; rateIndex < 4; rateIndex++ )
        {
            NN_LOG("set sdmmc %d rate %d Hz.\n", index, rates[rateIndex]);
            EXPECT_TRUE(nne::vcc::SetClockRate(sdmmcIp[index], rates[rateIndex]) == nne::vcc::Result_Vcc_Success);
            settings.id = sdmmcIp[index];
            settings.f_Hz = 0;
            NN_LOG("query current rate of sdmmc %d.\n", index);
            ASSERT_TRUE(nne::vcc::QuerySettings(&settings) == nne::vcc::Result_Vcc_Success);
            NN_LOG("Expected %d Hz - Actual %d Hz.\n", rates[rateIndex], settings.f_Hz);

            // verify settings
            ASSERT_TRUE(settings.f_Hz == rates[rateIndex]);

            uint32_t id = 0;
            if ( sdmmcIp[index] == IP_SDMMC1 ) id = 1;
            else if ( sdmmcIp[index] == IP_SDMMC2 ) id = 2;
            else if ( sdmmcIp[index] == IP_SDMMC3 ) id = 3;
            else if ( sdmmcIp[index] == IP_SDMMC4 ) id = 4;
            ASSERT_TRUE(id != 0);

            uint32_t realFreq = CalculateClockSDMMC(id);
            NN_LOG("Expected %d Hz - Actual %d Hz(from registers).\n",
                   rates[rateIndex], realFreq);
            ASSERT_TRUE(rates[rateIndex] == realFreq);
        }
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    static auto alloc = [](size_t size) -> void* {
        return std::malloc(size);
    };
    static auto dealloc = [](void* p, size_t size) {
        NN_UNUSED(size);
        std::free(p);
    };

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(alloc, dealloc);

    RUN_ALL_TESTS();
}
