﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <stdint.h>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nne/vcc/vcc_API.h>
#include <nne/vcc/vccapi.h>
#include <nnt/gtest/gtest.h>
#include <nn/dd.h>

#include <nn/fs/fs_MemoryManagement.h>
#include <cstdlib>

const uint8_t MAX_DIVP_SETTINGS = 17;

/* PDIV Settings to PDIV value
 * DIVP = Post divider setting (PDIV setting)
 * Relation between value and setting is as follows:
 * PDIV setting: 0, 1, 2, 3, 4, 5, 6, 7,  8,  9, 10, 11, 12, 13, 14, 15, 16
 * DIVP values:  1, 2, 3, 4, 5, 6, 8, 9, 10, 12, 15, 16, 18, 20, 24, 30, 32
 */
const uint32_t DIVP_VALUES[MAX_DIVP_SETTINGS] = {1,2,3,4,5,6,8,9,10,12,15,16,18,20,24,30,32};

// PLL register offsets
const uintptr_t PLLA1_BASE_0         = 0x6a4;
const uintptr_t PLLA1_MISC_0         = 0x6a8;
const uintptr_t PLLA1_MISC_1         = 0x6ac;
const uintptr_t PLLA1_MISC_2         = 0x6b0;
const uintptr_t PLLA1_MISC_3         = 0x6b4;
const uintptr_t ACLK_BURST_POLICY_0  = 0x6e0;
const uintptr_t SUPER_ACLK_DIVIDER_0 = 0x6e4;

// PLL physical address
const nn::dd::PhysicalAddress ClockResetControllerRegistersPhysicalAddress = 0x60006000ull;
const size_t ClockResetControllerRegistersSize = 0x1000;

// fixed osc value
const uint32_t  OSC_DIV_CLK = 38400000;

// NVOsDrvOpen is defined in nv_hos_drv.c
// and Is defined in libnvn_src.a

extern "C" int NvOsDrvOpen(const char *pathname, int flags);
const size_t heapSize = 16 * 1024 * 1024;

uint32_t RegRead(uintptr_t address)
{
    uint32_t* ptr = reinterpret_cast<uint32_t*>(address);
    return *ptr;
}

void RegWrite(uintptr_t address, uint32_t value)
{
    uint32_t* ptr = reinterpret_cast<uint32_t*>(address);
    *ptr = value;
}

uint32_t GetClockRegisterValue(uintptr_t registerOffset)
{

    uintptr_t registerBaseAddress;
    registerBaseAddress = nn::dd::QueryIoMappingAddress(ClockResetControllerRegistersPhysicalAddress,
                                                     ClockResetControllerRegistersSize);
    uint32_t registerValue = RegRead(registerBaseAddress + registerOffset);
    return registerValue;
}

void PrintRegister(const char *registerName, uintptr_t registerOffset, uint32_t registerValue)
{
    NN_LOG("0x%08x %s 0x%08x\n", registerOffset, registerName, registerValue);
}

void ReadAndPrintClockRegister(const char *registerName, uintptr_t registerOffset)
{
    uint32_t registerValue = GetClockRegisterValue(registerOffset);
    PrintRegister(registerName, registerOffset, registerValue);
}

uint32_t ExtractBitField(uint32_t word, int highBit, int lowBit)
{
    uint32_t highBitsToClear = 31 - highBit;
    uint32_t field = word << highBitsToClear; // shift left to clear high bits
    field >>= lowBit + highBitsToClear;   // shift right to clear low bits
    return field;
}

void PrintField(uint32_t word, const char *fieldName, int highBit, int lowBit, const char *desc)
{
    uint32_t fieldValue = ExtractBitField(word, highBit, lowBit);
    if (strlen(desc) < 50)
    {
        // if desc is short, print it all on one line
        NN_LOG(" %s %d %s\n", fieldName, fieldValue, desc);
    }
    else
    {
        // if desc is long, then print it on it's own line and add a separator
        NN_LOG(" %s %d\n", fieldName, fieldValue);
        NN_LOG(" %s\n", desc);
        NN_LOG("\n", desc);
    }
}

/*
 * Read divM, divN and divP from calculate final frequency output.
 *
 * Refer to TRM section 5.1.4. "PLL Programming Constraints" (on or near page 49)
 *
 * Returns 0 if divP setting index lookup exceeds MAX_DIVP_SETTINGS
 */
uint32_t CalculatePLLA1OutputFrequencyFromRegisterValues()
{
    uint32_t finalFrequencyOutput;
    uint32_t base = GetClockRegisterValue(PLLA1_BASE_0);

    if (ExtractBitField(base, 31, 31) == 1)
    {
        NN_LOG("BYPASS ENABLED");
        finalFrequencyOutput = OSC_DIV_CLK;
    }
    else
    {
        uint32_t divP = ExtractBitField(base,24,20);
        uint32_t divN = ExtractBitField(base,17,10);
        uint32_t divM = ExtractBitField(base, 7, 0);
        uint32_t cf  = OSC_DIV_CLK / divM;
        uint32_t vco = cf * divN;
        if (divP < MAX_DIVP_SETTINGS)
            finalFrequencyOutput = vco / DIVP_VALUES[divP];
        else
            return 0;
        NN_LOG("divP %d divN %d divM %d cf %d vco %d ffo %d\n",
                    divP, divN, divM, cf, vco, finalFrequencyOutput);
    }

    return finalFrequencyOutput;
}

void ReadAndPrintAclkBurstPolicy0()
{
    uint32_t burstPolicyValue = GetClockRegisterValue(ACLK_BURST_POLICY_0);
    PrintRegister("5.1.279 ACLK_BURST_POLICY_0",  ACLK_BURST_POLICY_0, burstPolicyValue);
    PrintField(burstPolicyValue,"ADSP_STATE",                31,28,"0=STNDBY; 1=IDLE; 2=RUN; 4=IRQ; 8=FIQ");
    PrintField(burstPolicyValue,"COP_AUTO_AWAKEUP_FROM_FIQ", 27,27,"0=NOP; 1=Burst on COP FIQ");
    PrintField(burstPolicyValue,"CPU_AUTO_AWAKEUP_FROM_FIQ", 26,26,"0=NOP; 1=Burst on CPU FIQ");
    PrintField(burstPolicyValue,"COP_AUTO_AWAKEUP_FROM_IRQ", 25,25,"0=NOP; 1=Burst on COP IRQ");
    PrintField(burstPolicyValue,"CPU_AUTO_AWAKEUP_FROM_IRQ", 24,24,"0=NOP; 1=Burst on CPU IRQ");

    const char *wakeupSrc = "0=PLLA1_OUT0; 1=PLLC_OUT0; 2=PLLP_OUT0; 3=PLLA_OUT0; 4=PLLC2_OUT0; "
                            "5=PLLC3_OUT0; 6=CLK_M; 7=PLLA_OUT; 8=PLLA1_OUT_LJ; 15=PLLC2_OUT_LJ";

    PrintField(burstPolicyValue,"AWAKEUP_FIQ_SOURCE",15,12, wakeupSrc);
    PrintField(burstPolicyValue,"AWAKEUP_IRQ_SOURCE",11,8, wakeupSrc);
    PrintField(burstPolicyValue,"AWAKEUP_RUN_SOURCE", 7,4, wakeupSrc);
    PrintField(burstPolicyValue,"AWAKEUP_IDLE_SOURCE",3,0, wakeupSrc);
}

void ReadAndPrintPlla1Base()
{
    uint32_t base = GetClockRegisterValue(PLLA1_BASE_0);
    PrintRegister("5.1.264 PLLA1_BASE_0",  PLLA1_BASE_0, base);
    PrintField(base ,"BYPASS", 31,31,"");
    PrintField(base ,"ENABLE", 30,30,"");
    PrintField(base ,"REF_DIS", 29,29,"");
    PrintField(base ,"FREQ_LOCK", 27,27,"");
    PrintField(base ,"LOCK", 26,26,"");
    PrintField(base ,"DIVP", 24,20,"");
    PrintField(base ,"DIVN", 17,10,"");
    PrintField(base ,"DIVM", 7,0,"");
}

void PrintAllAudioClockRegisters()
{
    ReadAndPrintAclkBurstPolicy0();
    ReadAndPrintPlla1Base();
    ReadAndPrintClockRegister("PLLA1_BASE_0",  PLLA1_BASE_0);
    ReadAndPrintClockRegister("PLLA1_MISC_0",  PLLA1_MISC_0);
    ReadAndPrintClockRegister("PLLA1_MISC_1",  PLLA1_MISC_1);
    ReadAndPrintClockRegister("PLLA1_MISC_2",  PLLA1_MISC_2);
    ReadAndPrintClockRegister("PLLA1_MISC_3",  PLLA1_MISC_3);
    ReadAndPrintClockRegister("SUPER_ACLK_DIVIDER_0", SUPER_ACLK_DIVIDER_0);
}

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(heapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", heapSize);
}

TEST( VccTestAudioClk, All )
{
    NN_LOG("Audio clock registers before vcc::Initialize()\n");
    PrintAllAudioClockRegisters();

    NN_LOG("About to call vcc::Initialize()\n");
    nne::vcc::Initialize();

    NN_LOG("Audio clock registers after vcc::Initialize()\n");
    PrintAllAudioClockRegisters();

    uint32_t registerFrequency = CalculatePLLA1OutputFrequencyFromRegisterValues();
    NN_LOG("Calculated PLLA1 rate after vcc::Initialize(): %d KHz\n", registerFrequency / 1000);
    ASSERT_GT(registerFrequency, 0);
    static const int MAX_RATES_NUM = 32;
    uint32_t rates[MAX_RATES_NUM];
    int numRates = MAX_RATES_NUM;
    IP moduleId = IP_AUDIO_DSP;

    if (nne::vcc::QueryPossibleRates(moduleId, rates, &numRates) == nne::vcc::Result_Vcc_Success )
    {
        NN_LOG("Got %d rates for module: IP_AUDIO_DSP\n", numRates);

        for (int rateIndex = 0; rateIndex < numRates; rateIndex++)
        {
            uint32_t rate = rates[rateIndex];
            nne::vcc::IPSettingsRequest request;
            request.id = moduleId;
            request.f_Hz = rate;

            NN_LOG("\n");
            NN_LOG("Setting Rate to %d kHz rates for IP_AUDIO_DSP\n", rate / 1000);
            nne::vcc::ApplySettings(&request, 1);

            uint32_t registerFrequency = CalculatePLLA1OutputFrequencyFromRegisterValues();
            ASSERT_TRUE(registerFrequency > (uint32_t) 0);
            NN_LOG("Calculated PLLA1 Rate is %d kHz \n", registerFrequency / 1000);
            EXPECT_EQ(rate, registerFrequency);
            if (rate != registerFrequency)
            {
                NN_LOG("ERROR. %d != %d\n", rate / 1000, registerFrequency / 1000);
                NN_LOG("Printing All Clock Registers\n");
                PrintAllAudioClockRegisters();
            }
        }
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    static auto alloc = [](size_t size) -> void* {
        return std::malloc(size);
    };
    static auto dealloc = [](void* p, size_t size) {
        NN_UNUSED(size);
        std::free(p);
    };

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(alloc, dealloc);

    RUN_ALL_TESTS();
}
