﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief   nn::util の文字エンコード API のテストです。
 * @details 以下の関数のテストをします。
 *          - ConvertStringUtf8ToUtf16Native (srcLength あり)
 *          - ConvertStringUtf16NativeToUtf8 (srcLength あり)
 *          - GetLengthOfConvertedStringUtf8ToUtf16Native
 *          - GetLengthOfConvertedStringUtf16NativeToUtf8
 */

////////////////////////////////////////////////////////////////
/**
 * 文字エンコーディングライブラリのテストにおける[テスト観点-因子表]
 *
 *  --------------+-----------------------------
 *   [テスト観点] | [因子]
 *  ============================================
 *   境界値       | サイズ
 *                | 特殊文字
 *  --------------+-----------------------------
 *   異常値       | 共通 (nullptr,負値等の区分)
 *                | 不正文字
 *                | エラーコード
 *                | 異常終了
 *  --------------+-----------------------------
 *   データ       | ストレス
 *                | 特殊文字
 *  --------------+-----------------------------
 *
 */
////////////////////////////////////////////////////////////////

#include <cstdlib>
#include <nnt/nntest.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <algorithm>
#include <nn/fs.h>
#include <nn/os.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX // numeric_limits の min/max との衝突を回避。
#endif
#include <nn/nn_Windows.h>
#endif  // #if defined(NN_BUILD_CONFIG_OS_WIN32)

namespace
{

// 静的配列の長さを取得します
template<typename T, std::size_t U>
std::size_t GetArrayLength(const T (&staticArray)[U])
{
    NN_UNUSED(staticArray);
    return U;
}

// 任意の文字列を必要な長さぶん生成できるテンプレートクラス
template<typename T>
class BufferArray
{
private:
    int m_Length;
    T* m_Ptr;

    /**
     * @brief   m_Ptr に src のデータを srcLength 区切りで繰り返して m_Length までコピーします。
     * @details 例）
     *          src = {1, 2, 3}, srcLength = 3, m_Length = 5 の場合、
     *          m_Ptr = {1, 2, 3, 1, 2} とコピーされます。
     */
    void FillRepeatedDataPattern(const T* src, int srcLength) const
    {
        int copies = 0;
        for (int i1=0; i1<m_Length; i1+=copies)
        {
            copies = std::min( m_Length - i1, srcLength );
            std::copy(src, &src[copies], &m_Ptr[i1]);
        }
    }
public:
    explicit BufferArray(int length) :
        m_Length(length),
        m_Ptr(new T[length])
    {
        memset( m_Ptr, 0xAC, length*sizeof(T) );
    }
    BufferArray(const T* src, int length) :
        m_Length(length),
        m_Ptr(new T[length])
    {
        FillRepeatedDataPattern( src, length );
    }
    BufferArray(int length, const T* src, int srcLength) :
        m_Length(length),
        m_Ptr(new T[length])
    {
        FillRepeatedDataPattern( src, srcLength );
    }

    BufferArray(BufferArray&& value)
    {
        m_Ptr = value.m_Ptr;
        value.m_Ptr = nullptr;
        m_Length = value.m_Length;
    }
    ~BufferArray()
    {
        if ( nullptr != m_Ptr )
        {
            delete [] m_Ptr;
        }
        m_Length = 0;
    }
    int Length() const
    {
        return m_Length;
    }
    T* Ptr() const
    {
        return m_Ptr;
    }
    T* Ptr(int index) const
    {
        return &m_Ptr[index];
    }
};

/**
 * @brief 変換時の長さに関する情報をまとめた構造体です。
 */
struct Length
{
    int srcNew;     // 変換元データ配列の長さ
    int srcConvert; // 変換される長さ
    int dstAdd;     // 変換先データ配列の長さの変化量(負値を許容)
};

////////////////////////////////////////////////////////////
// Test data of strings.
// [ABCAAAあいうｱｲｳ ](4-6文字目のAは環境依存文字)
#define UTF8_TEST_CONTENTS \
    '\x41', '\x42', '\x43', '\xC3', '\x80', '\xC3', '\x81', '\xC3',\
    '\x82', '\xE3', '\x81', '\x82', '\xE3', '\x81', '\x84', '\xE3',\
    '\x81', '\x86', '\xEF', '\xBD', '\xB1', '\xEF', '\xBD', '\xB2',\
    '\xEF', '\xBD', '\xB3', '\xF0', '\x90', '\x80', '\x80', '\x00'
#define UTF16_TEST_CONTENTS \
    0x0041, 0x0042, 0x0043, 0x00C0, 0x00C1,\
    0x00C2, 0x3042, 0x3044, 0x3046, 0xFF71,\
    0xFF72, 0xFF73, 0xD800, 0xDC00, 0x0000

// BOM(バイトオーダーマーク)
#define UTF8_BOM         '\xEF', '\xBB', '\xBF'
static const uint16_t UTF16_BOM = 0xFEFF;

// BOMを先頭に付けたテストデータ
#define UTF8_TEST_CONTENTS_WITH_BOM         UTF8_BOM, UTF8_TEST_CONTENTS
#define UTF16_TEST_CONTENTS_WITH_BOM        UTF16_BOM, UTF16_TEST_CONTENTS

// 非文字(変換はできる)
#define UTF8_NON_CHARACTER_UPPER            '\xEF', '\xB7', '\x90'
#define UTF8_NON_CHARACTER_LOWER            '\xEF', '\xB7', '\xAF'
#define UTF8_NON_CHARACTER_INVERSE_BOM      '\xEF', '\xBF', '\xBE'
#define UTF8_NON_CHARACTER_END              '\xEF', '\xBF', '\xBF'
static const uint16_t UTF16_NON_CHARACTER_UPPER = 0xFDD0;  // Unicode 3.1 からの非文字上限
static const uint16_t UTF16_NON_CHARACTER_LOWER = 0xFDEF;  // Unicode 3.1 からの非文字下限
static const uint16_t UTF16_NON_CHARACTER_INVERSE_BOM = 0xFFFE;
static const uint16_t UTF16_NON_CHARACTER_END = 0xFFFF;

// ' 'のUTF8
static const uint8_t UTF8_SPACE = 0x20u;
static const uint16_t UTF16_SPACE = 0x0020;
#define UTF8_VERBOSE_SPACE_A    '\xC0', '\x20'                // 冗長表現
#define UTF8_VERBOSE_SPACE_B    '\xE0', '\x80', '\x20'         // 冗長表現
#define UTF8_VERBOSE_SPACE_C    '\xF0', '\x80', '\x80', '\x20'  // 冗長表現

// '/'のUTF8
static const uint8_t UTF8_SLASH = 0x2Fu;
static const uint16_t UTF16_SLASH = 0x002F;
#define UTF8_VERBOSE_SLASH_A    '\xC0', '\xAF'                // 冗長表現
#define UTF8_VERBOSE_SLASH_B    '\xE0', '\x80', '\xAF'         // 冗長表現
#define UTF8_VERBOSE_SLASH_C    '\xF0', '\x80', '\x80', '\xAF'  // 冗長表現

// '~'のUTF8
static const uint8_t UTF8_TILDE = 0x7Eu;
static const uint16_t UTF16_TILDE = 0x007E;
#define UTF8_VERBOSE_TILDE_A    '\xC1', '\xBE'                // 冗長表現
#define UTF8_VERBOSE_TILDE_B    '\xE0', '\x81', '\xBE'         // 冗長表現
#define UTF8_VERBOSE_TILDE_C    '\xF0', '\x80', '\x81', '\xBE'  // 冗長表現

// サロゲートペア(境界値)
#define UTF8_SURROGATE_LOWER_LOWER  '\xF0', '\x90', '\x80', '\x80'
#define UTF16_SURROGATE_LOWER_LOWER 0xD800, 0xDC00
#define UTF8_SURROGATE_LOWER_UPPER  '\xF0', '\x90', '\x8F', '\xBF'
#define UTF16_SURROGATE_LOWER_UPPER 0xD800, 0xDFFF
#define UTF8_SURROGATE_UPPER_LOWER  '\xF4', '\x8F', '\xB0', '\x80'
#define UTF16_SURROGATE_UPPER_LOWER 0xDBFF, 0xDC00
#define UTF8_SURROGATE_UPPER_UPPER  '\xF4', '\x8F', '\xBF', '\xBF'
#define UTF16_SURROGATE_UPPER_UPPER 0xDBFF, 0xDFFF

// For definition of test data.
#define NEW_TEST_DATA(initialNameSpace, initialUtf8, initialUtf16) \
namespace initialNameSpace {                        \
    const char s_Utf8Data[] = initialUtf8;          \
    const uint16_t s_Utf16Data[] = initialUtf16;    \
    const BufferArray<char> g_Utf8(s_Utf8Data, static_cast<int>(GetArrayLength(s_Utf8Data)));        \
    const BufferArray<uint16_t> g_Utf16(s_Utf16Data, static_cast<int>(GetArrayLength(s_Utf16Data))); \
}

NEW_TEST_DATA(WithoutBom,
    {UTF8_TEST_CONTENTS},
    {UTF16_TEST_CONTENTS})
NEW_TEST_DATA(WithBom,
    {UTF8_TEST_CONTENTS_WITH_BOM},
    {UTF16_TEST_CONTENTS_WITH_BOM})
NEW_TEST_DATA(OnlyBom,
    {UTF8_BOM},
    {UTF16_BOM})
NEW_TEST_DATA(NonCharacterA,
    {UTF8_NON_CHARACTER_UPPER},
    {UTF16_NON_CHARACTER_UPPER})
NEW_TEST_DATA(NonCharacterB,
    {UTF8_NON_CHARACTER_LOWER},
    {UTF16_NON_CHARACTER_LOWER})
NEW_TEST_DATA(NonCharacterC,
    {UTF8_NON_CHARACTER_INVERSE_BOM},
    {UTF16_NON_CHARACTER_INVERSE_BOM})
NEW_TEST_DATA(NonCharacterD,
    {UTF8_NON_CHARACTER_END},
    {UTF16_NON_CHARACTER_END})
NEW_TEST_DATA(Space,
    {UTF8_SPACE},
    {UTF16_SPACE})
NEW_TEST_DATA(VerboseSpaceA,
    {UTF8_VERBOSE_SPACE_A},
    {UTF16_SPACE})
NEW_TEST_DATA(VerboseSpaceB,
    {UTF8_VERBOSE_SPACE_B},
    {UTF16_SPACE})
NEW_TEST_DATA(VerboseSpaceC,
    {UTF8_VERBOSE_SPACE_C},
    {UTF16_SPACE})
NEW_TEST_DATA(Slash,
    {UTF8_SLASH},
    {UTF16_SLASH})
NEW_TEST_DATA(VerboseSlashA,
    {UTF8_VERBOSE_SLASH_A},
    {UTF16_SLASH})
NEW_TEST_DATA(VerboseSlashB,
    {UTF8_VERBOSE_SLASH_B},
    {UTF16_SLASH})
NEW_TEST_DATA(VerboseSlashC,
    {UTF8_VERBOSE_SLASH_C},
    {UTF16_SLASH})
NEW_TEST_DATA(Tilde,
    {UTF8_TILDE},
    {UTF16_TILDE})
NEW_TEST_DATA(VerboseTildeA,
    {UTF8_VERBOSE_TILDE_A},
    {UTF16_TILDE})
NEW_TEST_DATA(VerboseTildeB,
    {UTF8_VERBOSE_TILDE_B},
    {UTF16_TILDE})
NEW_TEST_DATA(VerboseTildeC,
    {UTF8_VERBOSE_TILDE_C},
    {UTF16_TILDE})
NEW_TEST_DATA(SurrogateLowerLower,
    {UTF8_SURROGATE_LOWER_LOWER},
    {UTF16_SURROGATE_LOWER_LOWER})
NEW_TEST_DATA(SurrogateLowerUpper,
    {UTF8_SURROGATE_LOWER_UPPER},
    {UTF16_SURROGATE_LOWER_UPPER})
NEW_TEST_DATA(SurrogateUpperLower,
    {UTF8_SURROGATE_UPPER_LOWER},
    {UTF16_SURROGATE_UPPER_LOWER})
NEW_TEST_DATA(SurrogateUpperUpper,
    {UTF8_SURROGATE_UPPER_UPPER},
    {UTF16_SURROGATE_UPPER_UPPER})

////////////////////////////////////////////////////////////

/**
 * @brief       2つのバッファを先頭から見ていき、完全に一致するかどうか確認します。
 * @param[in]   lhs     比較対象1
 * @param[in]   rhs     比較対象2
 * @param[in]   length  比較する長さ
 * @return      一致した場合は length を返し、そうでない場合は一致しなかった場所を返します。
 */
template<typename T>
int FindNotMatchedPosition(const T* lhs, const T* rhs, int length)
{
    for (int i1=0; i1<length; ++i1)
    {
        if ( lhs[i1] != rhs[i1] )
        {
            // 一致しなかったポジションを返す。
            return i1;
        }
    }
    // Return the input length if the lhs matches the rhs.
    return length;
}

/**
 * @brief       2 つの引数の合計を 0 ～ int の範囲内に収めます
 * @return      合計が int の範囲を超えるなら、 std::numeric_limits<int>::max() を、0 以下なら 0 を、それ以外なら合計値を返します。
 */
int AddInt32WithUnsignedSaturation(int lhs, int rhs)
{
    int64_t sum = static_cast<int64_t>(lhs) + static_cast<int64_t>(rhs);
    if ( std::numeric_limits<int>::max() < sum )
    {
        return std::numeric_limits<int>::max();
    }
    // Return a positive value.
    return static_cast<int>(std::max(sum, static_cast<int64_t>(0)));
}

/**
 * @brief   テスト用ファイルを読み込み、BufferArray 形式で返却します。
 */
BufferArray<char> ReadText()
{
    // ファイルシステムの初期化
    size_t mountRomCacheBufferSize = 0;
    auto result = nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize);
    NN_ASSERT(result.IsSuccess());
    char* mountRomCacheBuffer = new char[mountRomCacheBufferSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] mountRomCacheBuffer;
    };
    result = nn::fs::MountRom("rom", mountRomCacheBuffer, mountRomCacheBufferSize);
    NN_ASSERT(result.IsSuccess());

    nn::fs::FileHandle handle;
    result = nn::fs::OpenFile(&handle, "rom:/utf8str.txt", nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;   // 読み込むファイルのサイズ
    result = nn::fs::GetFileSize(&fileSize, handle);
    NN_ASSERT(result.IsSuccess());

    BufferArray<char> buffer( static_cast<int>(fileSize));
    nn::fs::ReadFile(handle, 0, reinterpret_cast<void*>(buffer.Ptr()), static_cast<size_t>(fileSize));

    // ファイルシステムの終了処理
    nn::fs::CloseFile(handle);
    nn::fs::Unmount("rom");

    BufferArray<char>& bufferReference = buffer;
    return std::move(bufferReference);
}

/**
 * @brief   UTF-16 変換に必要な長さを返します。
 */
int GetLengthToConvertStringUtf8ToUtf16Native(const char* pSrc, int length)
{
    int dstLength = -1;
    nn::util::CharacterEncodingResult result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native( &dstLength, pSrc, length );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, result );
    return dstLength;
}

/**
 * @brief   UTF-8 変換に必要な長さを返します。
 */
int GetLengthToConvertStringUtf16NativeToUtf8(const uint16_t* pSrc, int length)
{
    int dstLength = -1;
    nn::util::CharacterEncodingResult result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &dstLength, pSrc, length );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, result );
    return dstLength;
}

/**
 * @brief   2 つの BufferArray を verifyLength 分一致するかどうか確認します。
 */
template<typename T>
void VerifyConvertedString(const BufferArray<T>& dstBuffer, const BufferArray<T>& correctData, int verifyLength)
{
    // Verify the converted string.
    BufferArray<T> truth(dstBuffer.Length(), correctData.Ptr(), correctData.Length());
    if ( verifyLength < dstBuffer.Length() )
    {
        // 変換範囲外のデータが変換されていないかのチェック
        memset( truth.Ptr(verifyLength), 0xCC, 1 );
        EXPECT_NE( 1, FindNotMatchedPosition( dstBuffer.Ptr(verifyLength), truth.Ptr(verifyLength), 1 ) );
    }
    EXPECT_EQ( verifyLength, FindNotMatchedPosition( dstBuffer.Ptr(), truth.Ptr(), verifyLength ) );
}

/**
 * @brief   checkedData を UTF-16 に変換したものと correctData と比較します。
 * @details checkedData から、 lentgh.srcConvert 分の長さだけ変換し、correctData と比較します。
 */
void TestConvertingUtf8ToUtf16Native(
    const BufferArray<char>& checkedData,
    const BufferArray<uint16_t>& correctData,
    const Length& length)
{
    // srcBuffer には checkedData の 0 ～ checkedData.Length() までの文字列パターンが繰り返された
    // 長さ length.srcNew 分の文字列が格納される
    BufferArray<char> srcBuffer(length.srcNew, checkedData.Ptr(), checkedData.Length());
    NN_ASSERT( length.srcConvert <= srcBuffer.Length() );

    // Get lengths to convert and verify the string.
    int dstLength    = GetLengthToConvertStringUtf8ToUtf16Native( srcBuffer.Ptr(), srcBuffer.Length() );
    ASSERT_LE( 0, dstLength );
    int verifyLength = GetLengthToConvertStringUtf8ToUtf16Native( srcBuffer.Ptr(), length.srcConvert );
    ASSERT_LE( 0, verifyLength );
    dstLength   = AddInt32WithUnsignedSaturation( dstLength, length.dstAdd );
    ASSERT_LE( verifyLength, dstLength );

    // Convert the string.
    BufferArray<uint16_t> dstBuffer(dstLength);
    nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf8ToUtf16Native(
                dstBuffer.Ptr(), dstBuffer.Length(),
                srcBuffer.Ptr(), length.srcConvert );

    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, result );

    // Verify the converted string.
    VerifyConvertedString( dstBuffer, correctData, verifyLength );
}

void TestConvertingUtf8ToUtf16Native(
    const BufferArray<char>& checkedData,
    const BufferArray<uint16_t>& correctData)
{
    Length length;
    length.dstAdd = 0;
    length.srcNew = length.srcConvert = checkedData.Length();
    TestConvertingUtf8ToUtf16Native( checkedData, correctData, length );
}

/**
 * @brief   checkedData を UTF-8 に変換したものと correctData と比較します。
 * @details checkedData から、 lentgh.srcConvert 分の長さだけ変換し、correctData と比較します。
 */
void TestConvertingUtf16NativeToUtf8(
    const BufferArray<uint16_t>& checkedData,
    const BufferArray<char>& correctData,
    const Length& length)
{
    BufferArray<uint16_t> srcBuffer(length.srcNew, checkedData.Ptr(), checkedData.Length());
    NN_ASSERT( length.srcConvert <= srcBuffer.Length() );

    // Get lengths to convert and verify the string.
    int dstLength    = GetLengthToConvertStringUtf16NativeToUtf8( srcBuffer.Ptr(), srcBuffer.Length() );
    ASSERT_LE( 0, dstLength );
    int verifyLength = GetLengthToConvertStringUtf16NativeToUtf8( srcBuffer.Ptr(), length.srcConvert );
    ASSERT_LE( 0, verifyLength );
    dstLength   = AddInt32WithUnsignedSaturation( dstLength, length.dstAdd );
    ASSERT_LE( verifyLength, dstLength );

    // Convert the string.
    BufferArray<char> dstBuffer(dstLength);
    nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf16NativeToUtf8(
                dstBuffer.Ptr(), dstBuffer.Length(),
                srcBuffer.Ptr(), length.srcConvert );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, result );

    // Verify the converted string.
    VerifyConvertedString( dstBuffer, correctData, verifyLength );
}

void TestConvertingUtf16NativeToUtf8(
    const BufferArray<uint16_t>& checkedData,
    const BufferArray<char>& correctData)
{
    Length length;
    length.dstAdd = 0;
    length.srcNew = length.srcConvert = checkedData.Length();
    TestConvertingUtf16NativeToUtf8( checkedData, correctData, length );
}

/**
 * @brief   1 文字が UTF-8 -> UTF-16 変換できるかどうか確認します。
 */
nn::util::CharacterEncodingResult TryUtf8ToUtf16NativeWithOneCharacter(
    char character,
    nn::util::CharacterEncodingResult (*Utf8ToUtf16Native)(const BufferArray<char>&, int))
{
    const char src[] = {character};
    BufferArray<char> srcBuffer(src, static_cast<int>(GetArrayLength(src)));
    return Utf8ToUtf16Native( srcBuffer, srcBuffer.Length() );
}

/**
 * @brief       2 文字が UTF-16 -> UTF-8 変換できるかどうか確認します。
 * @param[in]   upper   1文字目
 * @param[in]   lower   2文字目
 */
nn::util::CharacterEncodingResult TryUtf16NativeToUtf8WithTwoCharacters(
    uint16_t upper,
    uint16_t lower,
    nn::util::CharacterEncodingResult (*Utf16NativeToUtf8)(const BufferArray<uint16_t>&, int))
{
    const uint16_t src[] = {upper, lower};
    BufferArray<uint16_t> srcBuffer(src, static_cast<int>(GetArrayLength(src)));
    return Utf16NativeToUtf8( srcBuffer, srcBuffer.Length() * 4 );
}

/**
 * @brief   nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native が正常に動作したかどうかだけ確認します。
 * @details コールバック関数です。
 */
nn::util::CharacterEncodingResult TryToGetLengthOfConvertedStringUtf8ToUtf16Native(const BufferArray<char>& srcBuffer, int)
{
    // dstLength はただのバッファ。取得されても使われない。
    int dstLength = -1;
    return nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native( &dstLength, srcBuffer.Ptr(), srcBuffer.Length() );
}

/**
 * @brief   nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8 が正常に動作したかどうかだけ確認します。
 * @details コールバック関数です。
 */
nn::util::CharacterEncodingResult TryToGetLengthOfConvertedStringUtf16NativeToUtf8(const BufferArray<uint16_t>& srcBuffer, int)
{
    // dstLength はただのバッファ。取得されても使われない。
    int dstLength = -1;
    return nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &dstLength, srcBuffer.Ptr(), srcBuffer.Length() );
}

/**
 * @brief   nn::util::ConvertStringUtf8ToUtf16Native が正常に動作したかどうかだけ確認します。
 * @details コールバック関数です。
 */
nn::util::CharacterEncodingResult TryToConvertStringUtf8ToUtf16Native(const BufferArray<char>& srcBuffer, int dstLength)
{
    // dstBuffer はただのバッファ。変換されても使われない。
    BufferArray<uint16_t> dstBuffer( dstLength );
    return nn::util::ConvertStringUtf8ToUtf16Native(
               dstBuffer.Ptr(), dstBuffer.Length(),
               srcBuffer.Ptr(), srcBuffer.Length() );
}

/**
 * @brief   nn::util::ConvertStringUtf16NativeToUtf8 が正常に動作したかどうかだけ確認します。
 * @details コールバック関数です。
 */
nn::util::CharacterEncodingResult TryToConvertStringUtf16NativeToUtf8(const BufferArray<uint16_t>& srcBuffer, int dstLength)
{
    // dstBuffer はただのバッファ。変換されても使われない。
    BufferArray<char> dstBuffer( dstLength );
    return nn::util::ConvertStringUtf16NativeToUtf8(
               dstBuffer.Ptr(), dstBuffer.Length(),
               srcBuffer.Ptr(), srcBuffer.Length() );
}

/**
 * @brief   UTF-8 -> UTF-16 の変換をテストします。
 * @details 不正パターンと dstLength の不足の両方に当てはまる文字列を与え、そのどちらを result で返すかの判定を見ます
 */
void TestErrorEvaluationOrderUtf8ToUtf16Native(int srcLength)
{
    NN_ASSERT( 2 <= srcLength );
    BufferArray<char> srcUtf8( srcLength, WithoutBom::g_Utf8.Ptr(), WithoutBom::g_Utf8.Length() );

    // 十分な変換先の長さを取得。
    int dstUtf16Length = GetLengthToConvertStringUtf8ToUtf16Native( srcUtf8.Ptr(), srcUtf8.Length() );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, TryToConvertStringUtf8ToUtf16Native( srcUtf8, dstUtf16Length ) );

    // 十分な変換先の長さより少し短い長さを取得。
    int dstShortUtf16Length = GetLengthToConvertStringUtf8ToUtf16Native( srcUtf8.Ptr(), srcUtf8.Length() - 2 );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InsufficientLength, TryToConvertStringUtf8ToUtf16Native( srcUtf8, dstShortUtf16Length ) );

    // 末端を不正文字(非基本ラテン文字)に書き換える。
    *srcUtf8.Ptr(srcLength - 1) = 0xF8u;
    EXPECT_EQ( nn::util::CharacterEncodingResult_InsufficientLength, TryToConvertStringUtf8ToUtf16Native( srcUtf8, dstShortUtf16Length ) );

    // もうひとつ末端を不正文字に書き換える。
    *srcUtf8.Ptr(srcLength - 2) = 0xF8u;
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryToConvertStringUtf8ToUtf16Native( srcUtf8, dstShortUtf16Length ) );
}

/**
 * @brief UTF-16 -> UTF-8 の変換をテストします。
 * @details 不正パターンと dstLength の不足の両方に当てはまる文字列を与え、そのどちらを result で返すかの判定を見ます
 */
void TestErrorEvaluationOrderUtf16NativeToUtf8(int srcLength)
{
    NN_ASSERT( 4 <= srcLength );
    BufferArray<uint16_t> srcUtf16( srcLength, WithoutBom::g_Utf16.Ptr(), WithoutBom::g_Utf16.Length() );

    // 十分な変換先の長さを取得。
    int dstUtf8Length = GetLengthToConvertStringUtf16NativeToUtf8( srcUtf16.Ptr(), srcUtf16.Length() );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, TryToConvertStringUtf16NativeToUtf8( srcUtf16, dstUtf8Length ) );

    // 十分な変換先の長さより少し短い長さを取得。
    int dstShortUtf8Length = GetLengthToConvertStringUtf16NativeToUtf8( srcUtf16.Ptr(), srcUtf16.Length() - 3 );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InsufficientLength, TryToConvertStringUtf16NativeToUtf8( srcUtf16, dstShortUtf8Length ) );

    // 末端を不正文字(部分的にサロゲート上位 + サロゲート下位以外の文字)に書き換える。
    *srcUtf16.Ptr(srcLength - 2) = 0xD800;
    *srcUtf16.Ptr(srcLength - 1) = 0x0000;
    EXPECT_EQ( nn::util::CharacterEncodingResult_InsufficientLength, TryToConvertStringUtf16NativeToUtf8( srcUtf16, dstShortUtf8Length ) );

    // もうひとつ末端を不正文字に書き換える。
    *srcUtf16.Ptr(srcLength - 4) = 0xD800;
    *srcUtf16.Ptr(srcLength - 3) = 0x0000;
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryToConvertStringUtf16NativeToUtf8( srcUtf16, dstShortUtf8Length ) );
}

/**
 * @brief   不正なサロゲートペアを入れた場合にエラーが返ることを確認します。
 * @details UTF-8 から UTF-16 への変換に置いて、
 *          サロゲートペア文字の前半数バイトのみを変換にかけ、エラーになることを確認します。
 */
void TestInvalidSurrogatePairUtf16NativeToUtf8(nn::util::CharacterEncodingResult (*ProcessUtf8ToUtf16Native)(const BufferArray<char>&, int))
{
    struct
    {
        const BufferArray<char>* utf8;
        const BufferArray<uint16_t>* utf16;
    } checkedDataList[] =
    {
        { &SurrogateLowerLower::g_Utf8, &SurrogateLowerLower::g_Utf16 },
        { &SurrogateLowerUpper::g_Utf8, &SurrogateLowerUpper::g_Utf16 },
        { &SurrogateUpperLower::g_Utf8, &SurrogateUpperLower::g_Utf16 },
        { &SurrogateUpperUpper::g_Utf8, &SurrogateUpperUpper::g_Utf16 },
    };
    for (auto& checkedData : checkedDataList)
    {
        // テストデータに問題がないか事前に正常系テストでチェックする。
        TestConvertingUtf8ToUtf16Native( *checkedData.utf8, *checkedData.utf16 );
        TestConvertingUtf16NativeToUtf8( *checkedData.utf16, *checkedData.utf8 );

        // 半端なサロゲートペアを入力。
        for (int srcLength = 1; srcLength < checkedData.utf8->Length(); ++srcLength)
        {
            BufferArray<char> srcUtf8( srcLength, checkedData.utf8->Ptr(), checkedData.utf8->Length() );
            EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( srcUtf8, checkedData.utf16->Length() ) );
        }
    }
}

/**
 * @brief   不正パターンの検出に関するテストケースを実行します。
 */
void TestBoundaryInvalidFormat(
    nn::util::CharacterEncodingResult (*ProcessUtf8ToUtf16Native)(const BufferArray<char>&, int),
    nn::util::CharacterEncodingResult (*ProcessUtf16NativeToUtf8)(const BufferArray<uint16_t>&, int) )
{
    // 基本ラテン文字のみ許容するかチェック
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, TryUtf8ToUtf16NativeWithOneCharacter( 0x00u, ProcessUtf8ToUtf16Native ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, TryUtf8ToUtf16NativeWithOneCharacter( 0x7Fu, ProcessUtf8ToUtf16Native ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf8ToUtf16NativeWithOneCharacter( 0x80u, ProcessUtf8ToUtf16Native ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf8ToUtf16NativeWithOneCharacter( 0xFFu, ProcessUtf8ToUtf16Native ) );

    // 冗長表現を使ってはいけない。
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseSpaceA::g_Utf8, VerboseSpaceA::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseSpaceB::g_Utf8, VerboseSpaceB::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseSpaceC::g_Utf8, VerboseSpaceC::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf8ToUtf16Native( Space::g_Utf8, Space::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf16NativeToUtf8( Space::g_Utf16, Space::g_Utf8.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseSlashA::g_Utf8, VerboseSlashA::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseSlashB::g_Utf8, VerboseSlashB::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseSlashC::g_Utf8, VerboseSlashC::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf8ToUtf16Native( Slash::g_Utf8, Slash::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf16NativeToUtf8( Slash::g_Utf16, Slash::g_Utf8.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseTildeA::g_Utf8, VerboseTildeA::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseTildeB::g_Utf8, VerboseTildeB::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( VerboseTildeC::g_Utf8, VerboseTildeC::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf8ToUtf16Native( Tilde::g_Utf8, Tilde::g_Utf16.Length() ) );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf16NativeToUtf8( Tilde::g_Utf16, Tilde::g_Utf8.Length() ) );

    // UTF-8 で 1 コードポイントが複数バイトで表される文字を、途中までしか渡さない
    EXPECT_EQ(nn::util::CharacterEncodingResult_InvalidFormat, TryUtf8ToUtf16NativeWithOneCharacter(0xC0u, ProcessUtf8ToUtf16Native));  // 2 バイト
    EXPECT_EQ(nn::util::CharacterEncodingResult_InvalidFormat, TryUtf8ToUtf16NativeWithOneCharacter(0xE0u, ProcessUtf8ToUtf16Native));  // 3 バイト
    EXPECT_EQ(nn::util::CharacterEncodingResult_InvalidFormat, TryUtf8ToUtf16NativeWithOneCharacter(0xF0u, ProcessUtf8ToUtf16Native));  // 4 バイト以上

    // 半端なサロゲート上位表現は不正文字として扱われる。
    TestInvalidSurrogatePairUtf16NativeToUtf8( ProcessUtf8ToUtf16Native );

    uint16_t highSurrogate[] = {0xD800, 0xDBFF};    // 上位サロゲートの上限と下限
    for (uint16_t& utf16 : highSurrogate)
    {
        // サロゲート上位とサロゲート下位は対となり連続しなければならない。
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0x0000, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xDBFF, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_Success, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xDC00, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_Success, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xDFFF, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xE000, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xFFFF, ProcessUtf16NativeToUtf8 ) );

        // サロゲート上位で終了した場合も解釈できないのでエラーとなる。
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0x0000, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xD7FF, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xD800, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xDBFF, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xDC00, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xDFFF, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xE000, utf16, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( 0xFFFF, utf16, ProcessUtf16NativeToUtf8 ) );
    }

    uint16_t lowSurrogate[] = {0xDC00, 0xDFFF};
    for (uint16_t& utf16 : lowSurrogate)
    {
        // サロゲート上位とサロゲート下位は対となり連続しなければならない。
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0x0000, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xD7FF, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xD800, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xDBFF, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xDC00, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xDFFF, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xE000, ProcessUtf16NativeToUtf8 ) );
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, TryUtf16NativeToUtf8WithTwoCharacters( utf16, 0xFFFF, ProcessUtf16NativeToUtf8 ) );
    }

    // 正常に変換可能な文字列の一部に不正文字を紛れ込ませる
    const int maxInsertionPosition = 1023;
    int insertionPosition[] = {0, 1, maxInsertionPosition};
    for (int& position : insertionPosition)
    {
        BufferArray<char> invalidUtf8( maxInsertionPosition + 1, WithoutBom::g_Utf8.Ptr(), WithoutBom::g_Utf8.Length() );
        int sufficientLength = invalidUtf8.Length();    // UTF8->UTF16なので、変換元の長さが変換先の長さとして十分。
        EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf8ToUtf16Native( invalidUtf8, sufficientLength ) );
        // 非基本ラテン文字に書き換える。
        *invalidUtf8.Ptr(position) = 0xF8u;
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf8ToUtf16Native( invalidUtf8, sufficientLength ) );

        BufferArray<uint16_t> invalidUtf16( maxInsertionPosition + 2, WithoutBom::g_Utf16.Ptr(), WithoutBom::g_Utf16.Length() );
        sufficientLength = invalidUtf16.Length() * 4;    // UTF16->UTF8なので、十分な変換先の長さとして変換元の長さ*4。
        EXPECT_EQ( nn::util::CharacterEncodingResult_Success, ProcessUtf16NativeToUtf8( invalidUtf16, sufficientLength ) );
        // 部分的にサロゲート上位 + サロゲート下位以外の文字(不正なデータの組み合わせ)に書き換える。
        *invalidUtf16.Ptr(position)     = 0xD800;
        *invalidUtf16.Ptr(position + 1) = 0x0000;
        EXPECT_EQ( nn::util::CharacterEncodingResult_InvalidFormat, ProcessUtf16NativeToUtf8( invalidUtf16, sufficientLength ) );
    }
}

/**
 * @brief   不足した文字列長を渡した場合に失敗することを確認します。(UTF-8 -> UTF-16)
 */
void TestInsufficientLengthUtf8ToUtf16Native(
    const BufferArray<char>& checkedData,
    const int srcNewLength,
    const int dstAdd)
{
    // The added length should be negative.
    NN_ASSERT( dstAdd < 0 );

    BufferArray<char> srcBuffer(srcNewLength, checkedData.Ptr(), checkedData.Length());
    int dstLength = GetLengthToConvertStringUtf8ToUtf16Native( srcBuffer.Ptr(), srcBuffer.Length() );
    ASSERT_LE( 0, dstLength );
    // 負値の dstAdd を加算することで、変換先データ配列の長さを短くする。
    dstLength   = AddInt32WithUnsignedSaturation( dstLength, dstAdd );

    BufferArray<uint16_t> dstBuffer(dstLength);
    nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf8ToUtf16Native(
                dstBuffer.Ptr(), dstBuffer.Length(),
                srcBuffer.Ptr(), srcBuffer.Length() );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InsufficientLength, result );
}

/**
 * @brief   不足した文字列長を渡した場合に失敗することを確認します。(UTF-16 -> UTF-8)
 */
void TestInsufficientLengthUtf16NativeToUtf8(
    const BufferArray<uint16_t>& checkedData,
    const int srcNewLength,
    const int dstAdd)
{
    // The added length should be negative.
    NN_ASSERT( dstAdd < 0 );

    BufferArray<uint16_t> srcBuffer(srcNewLength, checkedData.Ptr(), checkedData.Length());
    int dstLength = GetLengthToConvertStringUtf16NativeToUtf8( srcBuffer.Ptr(), srcBuffer.Length() );
    ASSERT_LE( 0, dstLength );
    // 負値の dstAdd を加算することで、変換先データ配列の長さを短くする。
    dstLength   = AddInt32WithUnsignedSaturation( dstLength, dstAdd );

    BufferArray<char> dstBuffer(dstLength);
    nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf16NativeToUtf8(
                dstBuffer.Ptr(), dstBuffer.Length(),
                srcBuffer.Ptr(), srcBuffer.Length() );
    EXPECT_EQ( nn::util::CharacterEncodingResult_InsufficientLength, result );
}

} // namespace

////////////////////////////////////////////////////////////
/**
 *  TEST CASES:
 *   テストケース、名の上部に (テスト観点, 因子) を記述。
 *   因子については2つまで割り当てられる場合がある。
 *
 *  記述例:
 *    (境界値, [特殊文字, サイズ])
 *    TEST(CharacterEncoding, NonCharacter)
 *
 */
////////////////////////////////////////////////////////////

// (境界値, サイズ)
// UTF-8 と UTF-16 の基本的な相互変換のチェック
TEST(EncBasicTest, BoundaryStringLength)
{
    // 変換元データ配列のサイズに境界値と頻度の高い値を設定。
    Length bufferLength[] = { {0, 0, 0}, {1, 1, 0}, {1024, 1024, 0}, {1024 * 1024 * 4, 1024 * 1024 * 4, 0} };
    for (Length& length : bufferLength)
    {
        TestConvertingUtf8ToUtf16Native( WithoutBom::g_Utf8, WithoutBom::g_Utf16, length );
        TestConvertingUtf16NativeToUtf8( WithoutBom::g_Utf16, WithoutBom::g_Utf8, length );
    }
}

// (境界値, [特殊文字, サイズ])
// BOM と NonCharacter が正常に変換できるか
TEST(EncBasicTest, NonCharacter)
{
    // 非文字を変換。
    TestConvertingUtf16NativeToUtf8( WithBom::g_Utf16, WithBom::g_Utf8 );
    TestConvertingUtf16NativeToUtf8( OnlyBom::g_Utf16, OnlyBom::g_Utf8 );
    TestConvertingUtf16NativeToUtf8( NonCharacterA::g_Utf16, NonCharacterA::g_Utf8 );
    TestConvertingUtf16NativeToUtf8( NonCharacterB::g_Utf16, NonCharacterB::g_Utf8 );
    TestConvertingUtf16NativeToUtf8( NonCharacterC::g_Utf16, NonCharacterC::g_Utf8 );
    TestConvertingUtf16NativeToUtf8( NonCharacterD::g_Utf16, NonCharacterD::g_Utf8 );
}

// (境界値, サイズ)
// 指定した長さまでが変換できていること(指定した長さを超えるは変換していない)を確認
TEST(EncBasicTest, OnlyInputLength)
{
    // ここで設定した長さ(srcConvert:2番目のメンバ変数)しか変換されないことを確認する。
    Length convertLength[] = { {1024, 0, 0}, {1024, 1, 0}, {1024, 1023, 0} };
    for (Length& length : convertLength)
    {
        TestConvertingUtf8ToUtf16Native( WithoutBom::g_Utf8, WithoutBom::g_Utf16, length );
        TestConvertingUtf16NativeToUtf8( WithoutBom::g_Utf16, WithoutBom::g_Utf8, length );
    }
}

// (境界値, サイズ)
// 変換先配列の余剰がいくらあっても変換に問題ないことを確認
TEST(EncBasicTest, UsingSufficientBuffer)
{
    // 変換先のデータ配列を余分に長くする。
    //
    // NOTE: メモリ確保に時間がかかるため、7桁の素数を最大値とする。
    Length convertLength[] = { {1024, 1024, 1}, {1024, 1024, 1037339} };
    //Length convertLength[] = { {1024, 1024, 1}, {1024, 1024, std::numeric_limits<int>::max()} };
    for (Length& length : convertLength)
    {
        TestConvertingUtf8ToUtf16Native( WithoutBom::g_Utf8, WithoutBom::g_Utf16, length );
        TestConvertingUtf16NativeToUtf8( WithoutBom::g_Utf16, WithoutBom::g_Utf8, length );
    }
}

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
// DEATH テスト
// (異常値, [共通, 異常終了])
// pDst や pSrc nullptr を渡した場合に落ちることを確認
TEST(EncBasicDeathTest, NullPointer)
{
    const int dummyLength = 1024;
    char c8Dummy[dummyLength];
    uint16_t c16Dummy[dummyLength];
    int dummyOutLength;
    uint16_t* pNullPtr = nullptr;

    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf8ToUtf16Native( pNullPtr, dummyLength, c8Dummy, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf8ToUtf16Native( c16Dummy, dummyLength, nullptr, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf16NativeToUtf8( nullptr, dummyLength, c16Dummy, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf16NativeToUtf8( c8Dummy, dummyLength, pNullPtr, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native( nullptr, c8Dummy, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native( &dummyOutLength, nullptr, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( nullptr, c16Dummy, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &dummyOutLength, pNullPtr, dummyLength ), "" );
}

// (異常値, [共通, 異常終了])
// 長さに負の値が渡された場合に落ちることを確認
TEST(EncBasicDeathTest, NegativeValue)
{
    const int dummyLength = 1024;
    char c8Dummy[dummyLength];
    uint16_t c16Dummy[dummyLength];
    int dummyOutLength;

    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf8ToUtf16Native( c16Dummy, -1, c8Dummy, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf8ToUtf16Native( c16Dummy, dummyLength, c8Dummy, -1 ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf16NativeToUtf8( c8Dummy, -1, c16Dummy, dummyLength ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::ConvertStringUtf16NativeToUtf8( c8Dummy, dummyLength, c16Dummy, -1 ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native( &dummyOutLength, c8Dummy, -1 ), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &dummyOutLength, c16Dummy, -1 ), "" );
}

#endif

// (異常値, [不正文字, エラーコード])
// CharacterEncodingResult_InvalidFormat が返るすべてのパターンを確認
// - pSrc が 1 バイトかつ値が ASCII コード外の範囲
// - 冗長表現
// - 複数バイト必要な文字の途中まで( srcLength の不足)
// - 不正なサロゲートペア
//   - サロゲートペア + 不正文字
//   - 上位／下位サロゲートのみ
//   - 上位と下位の順序が逆
//   - 上位／下位サロゲートの連続
// - 正常に変換可能な文字列の一部に不正文字を紛れ込ませる
TEST(EncBasicTest, BoundaryInvalidFormat)
{
    TestBoundaryInvalidFormat( TryToGetLengthOfConvertedStringUtf8ToUtf16Native, TryToGetLengthOfConvertedStringUtf16NativeToUtf8 );
    TestBoundaryInvalidFormat( TryToConvertStringUtf8ToUtf16Native, TryToConvertStringUtf16NativeToUtf8 );
}

// (異常値, エラーコード)
// dstLength が不足している場合に CharacterEncodingResult_InsufficientLength を返すことを確認
TEST(EncBasicTest, InsufficientLength)
{
    // 短めの変換先データ配列を設定する。
    int dstAddLength[] = {std::numeric_limits<int>::min(), -1};
    for (int& dstAdd : dstAddLength)
    {
        TestInsufficientLengthUtf8ToUtf16Native( WithoutBom::g_Utf8,  1024, dstAdd );
        TestInsufficientLengthUtf16NativeToUtf8( WithoutBom::g_Utf16, 1024, dstAdd );
    }
}

// (異常値, エラーコード)
// 不正パターンと dstLength の不足の両方に当てはまる文字列を与え、
// そのどちらを result で返すかの優先順位を確認します
// CharacterEncodingResult_InvalidFormat が CharacterEncodingResult_InsufficientLength より優先される
TEST(EncBasicTest, ErrorEvaluationOrder)
{
    // エラーコードが返る順序が正しいかを確認する。
    //
    // NOTE: 不完全なデータ列(終端にサロゲート上位etc.)が入力になるのを避けるために、1025という多少恣意的な長さをとっている。
    int srcLength[] = {2, 3, 1025};
    for (int& length : srcLength)
    {
        TestErrorEvaluationOrderUtf8ToUtf16Native(length);
        TestErrorEvaluationOrderUtf16NativeToUtf8(length * 2);
    }
}

// (データ, [ストレス, 特殊文字])
// UTF-8 文字列として正しいテキストファイル(約 100KB)を読み込み変換します
TEST(EncBasicTest, StressfulString)
{
    BufferArray<char> srcBuffer = ReadText();

    // 順変換できるかチェック
    int dstLength = GetLengthToConvertStringUtf8ToUtf16Native( srcBuffer.Ptr(), srcBuffer.Length() );
    ASSERT_LE( 0, dstLength );
    BufferArray<uint16_t> dstBuffer(dstLength);
    nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf8ToUtf16Native(
                dstBuffer.Ptr(), dstBuffer.Length(),
                srcBuffer.Ptr(), srcBuffer.Length() );
    EXPECT_EQ( nn::util::CharacterEncodingResult_Success, result );

    // 逆変換して元データと照合
    TestConvertingUtf16NativeToUtf8( dstBuffer, srcBuffer );
}
