﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// general, neon 共通のテストを書くヘッダ
// このヘッダをインクルードする .cpp 側にて、どちらの実装を使用するかを決定する。

#include <nnt/benchmark/benchmark.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_VectorApi.h>
#include <nn/util/util_QuaternionApi.h>
#include <nn/util/util_Constant.h>

#if NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_GENERAL

    #define BENCHMARK_QUATERNION_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::Vector4fType); \

    #define BENCHMARK_WRAPPER_FUNC(f_) nn::util::general::f_##Wrapper

    #include "Wrapper/testUtil_BenchmarkMatrixApiWrapper.general.h"
    #include "Wrapper/testUtil_BenchmarkVectorApiWrapper.general.h"
    #include "Wrapper/testUtil_BenchmarkQuaternionApiWrapper.general.h"

#elif NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_NEON

    #define BENCHMARK_QUATERNION_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::Vector4fType); \

    #define BENCHMARK_WRAPPER_FUNC(f_) nn::util::neon::f_##Wrapper

    #include "Wrapper/testUtil_BenchmarkMatrixApiWrapper.neon.h"
    #include "Wrapper/testUtil_BenchmarkVectorApiWrapper.neon.h"
    #include "Wrapper/testUtil_BenchmarkQuaternionApiWrapper.neon.h"

#endif

#define BENCHMARK_QUATERNION(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_QUATERNION_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

BENCHMARK_QUATERNION(QuaternionBenchmark, Identity)
{
    nn::util::Vector4fType quaternion;

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionIdentity)(&quaternion);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Add)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionAdd)(&quaternion, quaternion1, quaternion2);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Subtract)
{
    static nn::util::Vector4fType quaternion;

    static  nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionSubtract)(&quaternion, quaternion1, quaternion2);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, MultiplyScalar)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionMultiply)(&quaternion, quaternion1, -1.f / 2.f);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, MultiplyQuaternion)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionMultiply)(&quaternion, quaternion1, quaternion2);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Divide)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionDivide)(&quaternion, quaternion1, quaternion2);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Dot)
{
    static float result;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(QuaternionDot)(quaternion1, quaternion2);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Normalize)
{
    static float result;

    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        result = BENCHMARK_WRAPPER_FUNC(QuaternionNormalize)(&quaternion, quaternion1);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Inverse)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionInverse)(&quaternion, quaternion1);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Exp)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 0.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionExp)(&quaternion, quaternion1);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, LogN)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 0.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionLogN)(&quaternion, quaternion1);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Lerp)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);
    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, 6.f, -2.f, 4.f, -8.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionLerp)(&quaternion, quaternion1, quaternion2, 0.5f);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Slerp)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -0.03f, 0.01f, -0.02f, 0.04f);
    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, 0.06f, -0.02f, 0.04f, -0.08f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionSlerp)(&quaternion, quaternion1, quaternion2, 0.5f);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, Squad)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector3fType axis;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axis, 1.f, 0.5f, 0.3f);
    BENCHMARK_WRAPPER_FUNC(VectorNormalize)(&axis, axis);

    static nn::util::Vector4fType p, a, b, q;
    BENCHMARK_WRAPPER_FUNC(QuaternionRotateAxis)(&p, axis, 0.1f);
    BENCHMARK_WRAPPER_FUNC(QuaternionRotateAxis)(&a, axis, 0.2f);
    BENCHMARK_WRAPPER_FUNC(QuaternionRotateAxis)(&b, axis, 0.3f);
    BENCHMARK_WRAPPER_FUNC(QuaternionRotateAxis)(&q, axis, 0.5f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionSquad)(&quaternion, p, a, b, q, 0.5f);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, MakeClosest)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    static nn::util::Vector4fType quaternion2;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion2, 6.f, -2.f, 4.f, -8.f);


    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionMakeClosest)(&quaternion, quaternion1, quaternion2);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, RotateAxisRadian)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector3fType axis;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axis, -3.f, 1.f, -2.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionRotateAxis)(&quaternion, axis, 60.f / 180.f * nn::util::FloatPi);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, MakeVectorRotation)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::Vector3fType from;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&from, -3.f, 1.f, -2.f);

    static nn::util::Vector3fType to;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&to, -6.f, 2.f, 4.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionMakeVectorRotation)(&quaternion, from, to);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, FromMatrixRowMajor)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::MatrixRowMajor4x3fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        0.9417491555213928f, 0.3213937878608704f, 0.0990685001015663f,
        -0.2249634265899658f, 0.3830222487449646f, 0.8959270715713501f,
        0.2499999701976776f, -0.8660253286361694f, 0.4330127239227295f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionFromMatrix)(&quaternion, matrix1);
    }
}

BENCHMARK_QUATERNION(QuaternionBenchmark, FromMatrixColumnMajor)
{
    static nn::util::Vector4fType quaternion;

    static nn::util::MatrixColumnMajor4x3fType matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        0.9417491555213928f, 0.3213937878608704f, 0.0990685001015663f,
        -0.2249634265899658f, 0.3830222487449646f, 0.8959270715713501f,
        0.2499999701976776f, -0.8660253286361694f, 0.4330127239227295f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning()) {
        BENCHMARK_WRAPPER_FUNC(QuaternionFromMatrix)(&quaternion, matrix1);
    }
}
