﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// general, neon 共通のテストを書くヘッダ
// このヘッダをインクルードする .cpp 側にて、どちらの実装を使用するかを決定する。

#include <nnt/benchmark/benchmark.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_VectorApi.h>
#include <nn/util/util_QuaternionApi.h>
#include <nn/util/util_Constant.h>

#if NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_GENERAL

    #define BENCHMARK_MATRIX3X2_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::MatrixRowMajor3x2fType); \
        BENCHMARK_TEMPLATE(f_, nn::util::general::MatrixColumnMajor3x2fType);

    #define BENCHMARK_MATRIX4X3_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::MatrixRowMajor4x3fType); \
        BENCHMARK_TEMPLATE(f_, nn::util::general::MatrixColumnMajor4x3fType);

    #define BENCHMARK_MATRIX4X4_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::general::MatrixRowMajor4x4fType); \
        BENCHMARK_TEMPLATE(f_, nn::util::general::MatrixColumnMajor4x4fType); \

    #define BENCHMARK_WRAPPER_FUNC(f_) nn::util::general::f_##Wrapper

    #include "Wrapper/testUtil_BenchmarkMatrixApiWrapper.general.h"
    #include "Wrapper/testUtil_BenchmarkVectorApiWrapper.general.h"
    #include "Wrapper/testUtil_BenchmarkQuaternionApiWrapper.general.h"

#elif NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_NEON

    #define BENCHMARK_MATRIX3X2_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::MatrixRowMajor3x2fType); \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::MatrixColumnMajor3x2fType);

    #define BENCHMARK_MATRIX4X3_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::MatrixRowMajor4x3fType); \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::MatrixColumnMajor4x3fType);

    #define BENCHMARK_MATRIX4X4_REGISTER(f_) \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::MatrixRowMajor4x4fType); \
        BENCHMARK_TEMPLATE(f_, nn::util::neon::MatrixColumnMajor4x4fType);

    #define BENCHMARK_WRAPPER_FUNC(f_) nn::util::neon::f_##Wrapper

    #include "Wrapper/testUtil_BenchmarkMatrixApiWrapper.neon.h"
    #include "Wrapper/testUtil_BenchmarkVectorApiWrapper.neon.h"
    #include "Wrapper/testUtil_BenchmarkQuaternionApiWrapper.neon.h"

#endif

#define BENCHMARK_MATRIX3X2(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_MATRIX3X2_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

#define BENCHMARK_MATRIX4X3(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_MATRIX4X3_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

#define BENCHMARK_MATRIX4X4(t_, f_) \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state); \
    BENCHMARK_MATRIX4X4_REGISTER(t_##_##f_); \
    template <typename TypeParam> void t_##_##f_(benchmark::State& state)

//
// 3x2 行列
//
BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Set)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetRow)
{
    static TypeParam matrix;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -7.f, -8.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRow)(&matrix, 1, vector1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetRows)
{
    static TypeParam matrix;

    static nn::util::Vector2fType row0, row1, row2;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row0, -7.f, -8.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row1, -9.f, -10.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row2, -11.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRows)(&matrix, row0, row1, row2);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetRow)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector2fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetRow)(&vector1, matrix, 1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetRows)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector2fType row0, row1, row2;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetRows)(&row0, &row1, &row2, matrix);
    }
}


BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetColumn)
{
    static TypeParam matrix;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -7.f, -9.f, -11.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetColumn)(&matrix, 1, vector1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetColumns)
{
    static TypeParam matrix;

    static nn::util::Vector3fType column0, column1;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column0, -7.f, -9.f, -11.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column1, -8.f, -10.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetColumns)(&matrix, column0, column1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetColumn)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector3fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetColumn)(&vector1, matrix, 1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetColumns)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector3fType column0, column1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetColumns)(&column0, &column1, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetAxisX)
{
    static TypeParam matrix;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -7.f, -8.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisX)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetAxisY)
{
    static TypeParam matrix;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -9.f, -10.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisY)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetAxisZ)
{
    static TypeParam matrix;

    static nn::util::Vector2fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -11.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisZ)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetAxes)
{
    static TypeParam matrix;

    static nn::util::Vector2fType axisX, axisY, axisZ;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisX, -7.f, -8.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisY, -9.f, -10.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisZ, -11.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxes)(&matrix, axisX, axisY, axisZ);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetAxisX)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector2fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisX)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetAxisY)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector2fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisY)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetAxisZ)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector2fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisZ)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, GetAxes)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::Vector2fType axisX, axisY, axisZ;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxes)(&axisX, &axisY, &axisZ, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, LoadFromRowMajor)
{
    static TypeParam matrix;

    static nn::util::FloatRowMajor3x2 value = NN_UTIL_FLOAT_ROW_MAJOR_3X2_INITIALIZER
    (
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f
    );

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLoad)(&matrix, value);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, LoadFromColumnMajor)
{
    static TypeParam matrix;

    static nn::util::FloatColumnMajor3x2 value = NN_UTIL_FLOAT_COLUMN_MAJOR_3X2_INITIALIZER
    (
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f
    );

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLoad)(&matrix, value);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, StoreToRowMajor)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::FloatRowMajor3x2 value;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixStore)(&value, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, StoreToColumnMajor)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    static nn::util::FloatColumnMajor3x2 value;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixStore)(&value, matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Zero)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixZero)(&matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Identity)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixIdentity)(&matrix);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Add)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixAdd)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Subtract)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);


    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSubtract)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, MultiplyScalar)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiply)(&matrix, matrix1, -1.f / 2.f);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, MultiplyMatrix)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiply)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, MultiplyAdd)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiplyAdd)(&matrix, 3.f, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Divide)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixDivide)(&matrix, matrix1, -2.f);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Inverse)
{
    static bool result;

    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning())
    {
        result = BENCHMARK_WRAPPER_FUNC(MatrixInverse)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, Transpose)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixTranspose)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, InverseTranspose)
{
    static bool result;

    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    while (state.KeepRunning())
    {
        result = BENCHMARK_WRAPPER_FUNC(MatrixInverseTranspose)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetTranslate)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static nn::util::Vector2fType translate;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&translate, -3.f, 1.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetTranslate)(&matrix, translate);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetScale)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static nn::util::Vector2fType scale;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&scale, -3.f, 1.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetScale)(&matrix, scale);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetRotate)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static float rotation = 40.f / 180.f * nn::util::FloatPi;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRotate)(&matrix, rotation);
    }
}

BENCHMARK_MATRIX3X2(Matrix3x2Benchmark, SetScaleRotate)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    static nn::util::Vector2fType scale;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&scale, -3.f, 2.f);

    static float rotation = 40.f / 180.f * nn::util::FloatPi;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetScaleRotate)(&matrix, scale, rotation);
    }
}

//
// 4x3 行列
//
BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Set)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetRow)
{
    static TypeParam matrix;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -13.f, -14.f, -15.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRow)(&matrix, 1, vector1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetRows)
{
    static TypeParam matrix;

    static nn::util::Vector3fType row0, row1, row2, row3;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row0, -13.f, -14.f, -15.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row1, -16.f, -17.f, -18.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row2, -19.f, -20.f, -21.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row3, -22.f, -23.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRows)(&matrix, row0, row1, row2, row3);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetRow)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetRow)(&vector1, matrix, 1);
    }
}


BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetRows)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType row0, row1, row2, row3;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetRows)(&row0, &row1, &row2, &row3, matrix);
    }
}


BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetColumn)
{
    static TypeParam matrix;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -13.f, -16.f, -19.f, -22.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetColumn)(&matrix, 1, vector1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetColumns)
{
    static TypeParam matrix;

    static nn::util::Vector4fType column0, column1, column2;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column0, -13.f, -16.f, -19.f, -22.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column1, -14.f, -17.f, -20.f, -23.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column2, -15.f, -18.f, -21.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetColumns)(&matrix, column0, column1, column2);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetColumn)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector4fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetColumn)(&vector1, matrix, 1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetColumns)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector4fType column0, column1, column2;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetColumns)(&column0, &column1, &column2, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetAxisX)
{
    static TypeParam matrix;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -13.f, -14.f, -15.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisX)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetAxisY)
{
    static TypeParam matrix;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -16.f, -17.f, -18.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisY)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetAxisZ)
{
    static TypeParam matrix;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -19.f, -20.f, -21.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisZ)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetAxisW)
{
    static TypeParam matrix;

    static nn::util::Vector3fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -22.f, -23.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxisW)(&matrix, vector1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetAxes)
{
    static TypeParam matrix;

    static nn::util::Vector3fType axisX, axisY, axisZ, axisW;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisX, -13.f, -14.f, -15.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisY, -16.f, -17.f, -18.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisZ, -19.f, -20.f, -21.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&axisW, -22.f, -23.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetAxes)(&matrix, axisX, axisY, axisZ, axisW);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetAxisX)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisX)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetAxisY)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisY)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetAxisZ)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisZ)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetAxisW)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxisW)(&vector1, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, GetAxes)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::Vector3fType axisX, axisY, axisZ, axisW;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetAxes)(&axisX, &axisY, &axisZ, &axisW, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, LoadFromRowMajor)
{
    static TypeParam matrix;

    static nn::util::FloatRowMajor4x3 value = NN_UTIL_FLOAT_ROW_MAJOR_4X3_INITIALIZER
    (
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f
    );

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLoad)(&matrix, value);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, LoadFromColumnMajor)
{
    static TypeParam matrix;

    static nn::util::FloatColumnMajor4x3 value = NN_UTIL_FLOAT_COLUMN_MAJOR_4X3_INITIALIZER
    (
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f
    );

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLoad)(&matrix, value);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, StoreToRowMajor)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::FloatRowMajor4x3 value;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixStore)(&value, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, StoreToColumnMajor)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    static nn::util::FloatColumnMajor4x3 value;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixStore)(&value, matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Zero)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixZero)(&matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Identity)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixIdentity)(&matrix);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Add)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixAdd)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Subtract)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSubtract)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, MultiplyScalar)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiply)(&matrix, matrix1, -1.f / 2.f);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, MultiplyMatrix)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiply)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, MultiplyAdd)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiplyAdd)(&matrix, 3.f, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Divide)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixDivide)(&matrix, matrix1, -2.f);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Inverse)
{
    static bool result;

    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, 0.f, 1.f,
        -1.f, 1.f, 1.f,
        1.f, -1.f, 0.f,
        1.f, 1.f, -1.f);

    while (state.KeepRunning())
    {
        result = BENCHMARK_WRAPPER_FUNC(MatrixInverse)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, Transpose)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixTranspose)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, InverseTranspose)
{
    static bool result;

    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, 0.f, 1.f,
        -1.f, 1.f, 1.f,
        1.f, -1.f, 0.f,
        1.f, 1.f, -1.f);

    while (state.KeepRunning())
    {
        result = BENCHMARK_WRAPPER_FUNC(MatrixInverseTranspose)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, LookAtRightHanded)
{
    static TypeParam matrix;

    static nn::util::Vector3fType position, up, target;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&position, 2.f, -2.f, 4.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&up, 0.f, 1.f, 0.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&target, 0.f, 0.f, 0.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLookAtRightHanded)(&matrix, position, target, up);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, LookAtRightHandedWithTwist)
{
    static TypeParam matrix;

    static nn::util::Vector3fType position, target;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&position, 2.f, -2.f, 4.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&target, 0.f, 0.f, 0.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLookAtRightHanded)(&matrix, position, target, 60.f / 180.f * nn::util::FloatPi);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, CameraRotate)
{
    static TypeParam matrix;

    static nn::util::Vector3fType position, rotation;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&position, -3.f, 1.f, -2.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&rotation, 30.f / 180.f * nn::util::FloatPi, 40.f / 180.f * nn::util::FloatPi, 50.f / 180.f * nn::util::FloatPi);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixCameraRotateZxy)(&matrix, position, rotation);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetTranslate)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static nn::util::Vector3fType translate;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&translate, -3.f, 1.f, -2.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetTranslate)(&matrix, translate);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetScale)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static nn::util::Vector3fType scale;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&scale, -3.f, 1.f, -2.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetScale)(&matrix, scale);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetRotate)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);
    BENCHMARK_WRAPPER_FUNC(QuaternionNormalize)(&quaternion1, quaternion1);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRotate)(&matrix, quaternion1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetRotateXyz)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static nn::util::Vector3fType rotation;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&rotation,
        30.f / 180.f * nn::util::FloatPi,
        40.f / 180.f * nn::util::FloatPi,
        50.f / 180.f * nn::util::FloatPi);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRotateXyz)(&matrix, rotation);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetScaleRotate)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static nn::util::Vector3fType scale;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&scale, -3.f, 1.f, -2.f);

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, -3.f, 1.f, -2.f, 4.f);
    BENCHMARK_WRAPPER_FUNC(QuaternionNormalize)(&quaternion1, quaternion1);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetScaleRotate)(&matrix, scale, quaternion1);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, SetScaleRotateXyz)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    static nn::util::Vector3fType scale;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&scale, -3.f, 1.f, -2.f);

    static nn::util::Vector3fType rotation;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&rotation,
        30.f / 180.f * nn::util::FloatPi,
        40.f / 180.f * nn::util::FloatPi,
        50.f / 180.f * nn::util::FloatPi);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetScaleRotateXyz)(&matrix, scale, rotation);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, TextureProjectionPerspectiveOffCenterRightHanded)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixTextureProjectionPerspectiveOffCenterRightHanded)(&matrix, -2.f, 4.f, -1.f, 3.f, 10.f);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, TextureProjectionPerspectiveFieldOfViewRightHanded)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixTextureProjectionPerspectiveFieldOfViewRightHanded)(&matrix, 60.f / 180.f * nn::util::FloatPi, 16.0f / 9.0f);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, TextureProjectionOrthographicOffCenterRightHanded)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixTextureProjectionOrthographicOffCenterRightHanded)(&matrix, -2.f, 4.f, -1.f, 3.f);
    }
}

BENCHMARK_MATRIX4X3(Matrix4x3Benchmark, FromQuaternion)
{
    static TypeParam matrix;

    static nn::util::Vector4fType quaternion1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&quaternion1, 0.5304983258247376f, 0.0454432815313339f, 0.1645002365112305f, 0.8303288221359253f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixFromQuaternion)(&matrix, quaternion1);
    }
}

//
// 4x4 行列
//
BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Set)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, SetRow)
{
    static TypeParam matrix;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -17.f, -18.f, -19.f, -20.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRow)(&matrix, 1, vector1);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, SetRows)
{
    static TypeParam matrix;

    static nn::util::Vector4fType row0, row1, row2, row3;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row0, -17.f, -18.f, -19.f, -20.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row1, -21.f, -22.f, -23.f, -24.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row2, -25.f, -26.f, -27.f, -28.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&row3, -29.f, -30.f, -31.f, -32.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetRows)(&matrix, row0, row1, row2, row3);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, GetRow)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    static nn::util::Vector4fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetRow)(&vector1, matrix, 1);
    }
}


BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, GetRows)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    static nn::util::Vector4fType row0, row1, row2, row3;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetRows)(&row0, &row1, &row2, &row3, matrix);
    }
}


BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, SetColumn)
{
    static TypeParam matrix;

    static nn::util::Vector4fType vector1;
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&vector1, -17.f, -21.f, -25.f, -29.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetColumn)(&matrix, 1, vector1);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, SetColumns)
{
    static TypeParam matrix;

    static nn::util::Vector4fType column0, column1, column2, column3;

    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column0, -17.f, -21.f, -25.f, -29.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column1, -18.f, -22.f, -26.f, -30.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column2, -19.f, -23.f, -27.f, -31.f);
    BENCHMARK_WRAPPER_FUNC(VectorSet)(&column3, -20.f, -24.f, -28.f, -32.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSetColumns)(&matrix, column0, column1, column2, column3);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, GetColumn)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    static nn::util::Vector4fType vector1;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetColumn)(&vector1, matrix, 1);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, GetColumns)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    static nn::util::Vector4fType column0, column1, column2, column3;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixGetColumns)(&column0, &column1, &column2, &column3, matrix);
    }
}


BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, LoadFromRowMajor)
{
    static TypeParam matrix;

    nn::util::FloatRowMajor4x4 value = NN_UTIL_FLOAT_ROW_MAJOR_4X4_INITIALIZER
    (
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f
    );

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLoad)(&matrix, value);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, LoadFromColumnMajor)
{
    static TypeParam matrix;

    nn::util::FloatColumnMajor4x4 value = NN_UTIL_FLOAT_COLUMN_MAJOR_4X4_INITIALIZER
    (
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f
    );

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixLoad)(&matrix, value);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, StoreToRowMajor)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    static nn::util::FloatRowMajor4x4 value;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixStore)(&value, matrix);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, StoreToColumnMajor)
{
    static TypeParam matrix;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    static nn::util::FloatColumnMajor4x4 value;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixStore)(&value, matrix);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Zero)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixZero)(&matrix);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Identity)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixIdentity)(&matrix);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Add)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        10.f, -12.f, 14.f, -16.f,
        -18.f, 20.f, -22.f, 24.f,
        -26.f, 28.f, -30.f, 32.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixAdd)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Subtract)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        10.f, -12.f, 14.f, -16.f,
        -18.f, 20.f, -22.f, 24.f,
        -26.f, 28.f, -30.f, 32.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixSubtract)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, MultiplyScalar)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiply)(&matrix, matrix1, -1.f / 2.f);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, MultiplyMatrix)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    static TypeParam matrix2;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        -10.f, -12.f, -14.f, -16.f,
        18.f, 20.f, 22.f, 24.f,
        -26.f, -28.f, -30.f, -32.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixMultiply)(&matrix, matrix1, matrix2);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Divide)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixDivide)(&matrix, matrix1, -2.f);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Inverse)
{
    static bool result;

    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, 0.f, 1.f, 2.f,
        -1.f, 1.f, 1.f, 1.f,
        1.f, -1.f, 0.f, 1.f,
        1.f, 1.f, -1.f, -2.f);

    while (state.KeepRunning())
    {
        result = BENCHMARK_WRAPPER_FUNC(MatrixInverse)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, Transpose)
{
    static TypeParam matrix;

    static TypeParam matrix1;
    BENCHMARK_WRAPPER_FUNC(MatrixSet)(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixTranspose)(&matrix, matrix1);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, PerspectiveOffCenterRightHanded)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixPerspectiveOffCenterRightHanded)(&matrix, -2.f, 4.f, -1.f, 3.f, 10.f, 100.f);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, PerspectiveFieldOfViewRightHanded)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixPerspectiveFieldOfViewRightHanded)(&matrix, 60.f / 180.f * nn::util::FloatPi, 16.0f / 9.0f, 0.1f, 100.0f);
    }
}

BENCHMARK_MATRIX4X4(Matrix4x4Benchmark, OrthographicOffCenterRightHanded)
{
    static TypeParam matrix;

    while (state.KeepRunning())
    {
        BENCHMARK_WRAPPER_FUNC(MatrixOrthographicOffCenterRightHanded)(&matrix, -2.f, 4.f, -1.f, 3.f, 10.f, 100.f);
    }
}
