﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

// neon 版ヘッダを単体でインクルード可能なことを確認
#include <nn/util/detail/util_VectorApi.neon.h>
#include <nn/util/util_Arithmetic.h>

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatPi = 3.141592653589793f;
    const float Float2Pi = 6.283185307179586f;

    const nn::util::AngleIndex AngleIndexHalfRound = 0x80000000;
    const nn::util::AngleIndex AngleIndex4Ulp = static_cast<nn::util::AngleIndex>(static_cast<int64_t>(4.f * ::std::numeric_limits<float>::epsilon() * (AngleIndexHalfRound / FloatPi)));

    const float ErrorSinCosEst = 0.00006f;
    const float ErrorTanEst = 0.00006f;
    const float ErrorAtanEst = 0.000006f;

    const float ErrorSinCosTable = 0.00006f;
    const float ErrorTanTable = 0.00006f;
    const float ErrorAtanTable = 0.000006f;

    ::testing::AssertionResult TestAreEqual(float expected, float actual, float error)
    {
        float diff = ::std::abs(expected - actual);
        float max = ::std::max(::std::abs(expected), ::std::abs(actual));

        error = max <= 1 ? error : error * max;

        if (diff <= error)
        {
            return ::testing::AssertionSuccess();
        }
        else
        {
            return ::testing::AssertionFailure()
                << "\n"
                << " expected: " << expected << "\n"
                << "   actual: " << actual << "\n"
                << "     diff: " << diff << "\n"
                << "    error: " << error << "\n"
                << " exceeded: " << diff - error << "\n";
        }
    }
}

TEST(MathVectorDetailNeonTest, Round)
{
    float32x4_t vector = { 0.49f, -1.49f, 0.51f, -0.51f };
    float32x4_t vResult = nn::util::neon::detail::Vector4fRound(vector);

    EXPECT_FLOAT_EQ(nn::util::neon::detail::Vector4fGetX(vResult), 0.f);
    EXPECT_FLOAT_EQ(nn::util::neon::detail::Vector4fGetY(vResult), -1.f);
    EXPECT_FLOAT_EQ(nn::util::neon::detail::Vector4fGetZ(vResult), 1.f);
    EXPECT_FLOAT_EQ(nn::util::neon::detail::Vector4fGetW(vResult), -1.f);
}

TEST(MathVectorDetailNeonTest, SinCosTan)
{
    for (uint32_t i = 0; i < 512; i++)
    {
        SCOPED_TRACE(i);

        nn::util::AngleIndex index = i * (AngleIndexHalfRound >> 8);
        float radian = nn::util::AngleIndexToRadian(index);
        float32x4_t vRadian = vdupq_n_f32(radian);

        float32x4_t vSin = nn::util::neon::detail::Vector4fSin(vRadian);
        float32x4_t vCos = nn::util::neon::detail::Vector4fCos(vRadian);

        EXPECT_NEAR(::std::sin(radian), nn::util::neon::detail::Vector4fGetX(vSin), ErrorSinCosEst);
        EXPECT_NEAR(::std::cos(radian), nn::util::neon::detail::Vector4fGetX(vCos), ErrorSinCosEst);

        float32x4_t vSin2, vCos2;
        nn::util::neon::detail::Vector4fSinCos(&vSin2, &vCos2, vRadian);

        EXPECT_FLOAT_EQ(nn::util::neon::detail::Vector4fGetX(vSin2), nn::util::neon::detail::Vector4fGetX(vSin));
        EXPECT_FLOAT_EQ(nn::util::neon::detail::Vector4fGetX(vCos2), nn::util::neon::detail::Vector4fGetX(vCos));

        // 90 度はスキップ
        if (i % 256 != 128)
        {
            float32x4_t vTan = nn::util::neon::detail::Vector4fTan(vRadian);

            EXPECT_TRUE(TestAreEqual(::std::tan(radian), nn::util::neon::detail::Vector4fGetX(vTan), ErrorTanEst));
        }
    }
}


TEST(MathVectorDetailNeonTest, Atan)
{
    for (uint32_t i = 1; i < 512; i++)
    {
        SCOPED_TRACE(i);

        float tangent = 1.0f / 256 * i;
        float32x4_t vTangentPositive = vdupq_n_f32(tangent);
        float32x4_t vTangentNegative = vdupq_n_f32(-tangent);

        float32x4_t vAtanPositive = nn::util::neon::detail::Vector4fAtan(vTangentPositive);
        float32x4_t vAtanNegative = nn::util::neon::detail::Vector4fAtan(vTangentNegative);

        EXPECT_NEAR(::std::atan(tangent), nn::util::neon::detail::Vector4fGetX(vAtanPositive), ErrorAtanEst);
        EXPECT_NEAR(::std::atan(-tangent), nn::util::neon::detail::Vector4fGetX(vAtanNegative), ErrorAtanEst);
    }
}

TEST(MathVectorDetailNeonTest, Atan2)
{
    EXPECT_NEAR(
        ::std::atan2(0.f, 0.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(0.f), vdupq_n_f32(0.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(1.f, 2.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(1.f), vdupq_n_f32(2.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(2.f, 1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(2.f), vdupq_n_f32(1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(2.f, -1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(2.f), vdupq_n_f32(-1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(1.f, -2.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(1.f), vdupq_n_f32(-2.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(-1.f, -2.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(-1.f), vdupq_n_f32(-2.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(-2.f, -1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(-2.f), vdupq_n_f32(-1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(-2.f, 1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(-2.f), vdupq_n_f32(1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::atan2(-1.f, 2.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAtan2(vdupq_n_f32(-1.f), vdupq_n_f32(2.f))),
        ErrorAtanEst);
}

TEST(MathVectorDetailNeonTest, Asin)
{
    EXPECT_NEAR(
        ::std::asin(0.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(0.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::asin(1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::asin(-1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(-1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::asin(0.1f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(0.1f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::asin(0.9f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(0.9f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::asin(-0.9f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(-0.9f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::asin(-0.1f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAsin(vdupq_n_f32(-0.1f))),
        ErrorAtanEst);
}

TEST(MathVectorDetailNeonTest, Acos)
{
    EXPECT_NEAR(
        ::std::acos(0.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(0.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::acos(1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::acos(-1.f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(-1.f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::acos(0.1f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(0.1f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::acos(0.9f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(0.9f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::acos(-0.9f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(-0.9f))),
        ErrorAtanEst);
    EXPECT_NEAR(
        ::std::acos(-0.1f),
        nn::util::neon::detail::Vector4fGetX(nn::util::neon::detail::Vector4fAcos(vdupq_n_f32(-0.1f))),
        ErrorAtanEst);
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
