﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_QuaternionApi.h>
#include <nn/util/util_VectorApi.h>

#include <cstring>

#include "testUtil_Vector.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.000001f;
}

//
// Float
//

// 2次元 Float
class MathFloat2Test : public ::testing::Test
{
};

TEST_F(MathFloat2Test, Initializer)
{
    nn::util::Float2 f = NN_UTIL_FLOAT_2_INITIALIZER(-3.f, 1.f);

    NNT_UTIL_FLOAT2_EXPECT_EQ(f, -3.f, 1.f);
}

TEST_F(MathFloat2Test, Make)
{
    nn::util::Float2 f = nn::util::MakeFloat2(-3.f, 1.f);

    NNT_UTIL_FLOAT2_EXPECT_EQ(f, -3.f, 1.f);
}

// 3次元 Float
class MathFloat3Test : public ::testing::Test
{
};

TEST_F(MathFloat3Test, Initializer)
{
    nn::util::Float3 f = NN_UTIL_FLOAT_3_INITIALIZER(-3.f, 1.f, -2.f);

    NNT_UTIL_FLOAT3_EXPECT_EQ(f, -3.f, 1.f, -2.f);
}

TEST_F(MathFloat3Test, Make)
{
    nn::util::Float3 f = nn::util::MakeFloat3(-3.f, 1.f, -2.f);

    NNT_UTIL_FLOAT3_EXPECT_EQ(f, -3.f, 1.f, -2.f);
}

// 4次元 Float
class MathFloat4Test : public ::testing::Test
{
};

TEST_F(MathFloat4Test, Initializer)
{
    nn::util::Float4 f = NN_UTIL_FLOAT_4_INITIALIZER(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_FLOAT4_EXPECT_EQ(f, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathFloat4Test, Make)
{
    nn::util::Float4 f = nn::util::MakeFloat4(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_FLOAT4_EXPECT_EQ(f, -3.f, 1.f, -2.f, 4.f);
}

//
// ベクトル
//

// 2次元ベクトル
class MathVector2Test : public ::testing::Test
{
};

TEST_F(MathVector2Test, Initializer)
{
    nn::util::Vector2fType vector = NN_UTIL_VECTOR_2F_INITIALIZER(-3.f, 1.f);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
}

TEST_F(MathVector2Test, Make)
{
    nn::util::Vector2fType vector = nn::util::MakeVector2fType(-3.f, 1.f);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
}

TEST_F(MathVector2Test, Set)
{
    nn::util::Vector2fType vector;

    nn::util::VectorSet(&vector, -3.f, 1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
}

TEST_F(MathVector2Test, SetXy)
{
    float value[2] = { -1.f, -1.f };

    nn::util::Vector2fType vector;
    std::memcpy(&vector, value, sizeof(vector));

    nn::util::VectorSetX(&vector, -3.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, -1.f);

    nn::util::VectorSetY(&vector, 1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
}

TEST_F(MathVector2Test, GetXy)
{
    nn::util::Vector2fType vector;
    nn::util::VectorSet(&vector, -3.f, 1.f);

    EXPECT_EQ(-3.f, nn::util::VectorGetX(vector));
    EXPECT_EQ(1.f, nn::util::VectorGetY(vector));
}

TEST_F(MathVector2Test, Substitution)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2(vector1);

    nn::util::VectorZero(&vector1);

    vector1 = vector2;

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -3.f, 1.f);
}

TEST_F(MathVector2Test, Load)
{
    {
        float value[2] = { -3.f, 1.f };

        nn::util::Vector2fType vector;
        nn::util::VectorLoad(&vector, value);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
    }

    {
        nn::util::Float2 float2Value = NN_UTIL_FLOAT_2_INITIALIZER(-3.f, 1.f);

        nn::util::Vector2fType vector;
        nn::util::VectorLoad(&vector, float2Value);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
    }
}

TEST_F(MathVector2Test, Store)
{
    {
        nn::util::Vector2fType vector;
        nn::util::VectorSet(&vector, -3.f, 1.f);

        float value[2];

        nn::util::VectorStore(value, vector);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, value[0], value[1]);
    }

    {
        nn::util::Vector2fType vector;
        nn::util::VectorSet(&vector, -3.f, 1.f);

        nn::util::Float2 value;

        nn::util::VectorStore(&value, vector);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, value.v[0], value.v[1]);
    }
}

TEST_F(MathVector2Test, Zero)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);

    nn::util::VectorZero(&vector1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 0.f, 0.f);
}

TEST_F(MathVector2Test, IsZero)
{
    nn::util::Vector2fType vector1;

    {
        nn::util::VectorZero(&vector1);
        EXPECT_TRUE(nn::util::VectorIsZero(vector1));

        nn::util::VectorSet(&vector1, 1.f, 0.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1));

        nn::util::VectorSet(&vector1, 0.f, 1.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1));
    }

    {
        float error = 0.0001f;

        nn::util::VectorSet(&vector1, error, -error);
        EXPECT_TRUE(nn::util::VectorIsZero(vector1, error));

        nn::util::VectorSet(&vector1, -2 * error, 0.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1, error));

        nn::util::VectorSet(&vector1, 0.f, 2 * error);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1, error));
    }
}

TEST_F(MathVector2Test, Add)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f);

    nn::util::VectorAdd(&vector1, vector1, vector2);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -9.f, 3.f);
}

TEST_F(MathVector2Test, Subtract)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f);

    nn::util::VectorSubtract(&vector1, vector1, vector2);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 3.f, -1.f);
}

TEST_F(MathVector2Test, Multiply)
{
    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f);

        nn::util::VectorMultiply(&vector1, vector1, -1.f / 2.f);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.5f, -0.5f);
    }

    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f);
        nn::util::Vector2fType vector2;
        nn::util::VectorSet(&vector2, -6.f, 2.f);

        nn::util::VectorMultiply(&vector1, vector1, vector2);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 18.f, 2.f);
    }
}

TEST_F(MathVector2Test, Divide)
{
    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f);

        nn::util::VectorDivide(&vector1, vector1, -2.f);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.5f, -0.5f);
    }

    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f);
        nn::util::Vector2fType vector2;
        nn::util::VectorSet(&vector2, 1.f / -6.f, 1.f / 2.f);

        nn::util::VectorDivide(&vector1, vector1, vector2);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 18.f, 2.f);
    }
}

TEST_F(MathVector2Test, Dot)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f);

    EXPECT_EQ(20.f, nn::util::VectorDot(vector1, vector2));
}

TEST_F(MathVector2Test, Cross)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, 6.f, 2.f);

    EXPECT_EQ(-12.f, nn::util::VectorCross(vector1, vector2));
}

TEST_F(MathVector2Test, Length)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);

    EXPECT_FLOAT_EQ(3.16227766017f, nn::util::VectorLength(vector1));
    EXPECT_EQ(10.f, nn::util::VectorLengthSquared(vector1));

    nn::util::Vector2fType vectorZero;
    nn::util::VectorZero(&vectorZero);

    EXPECT_FLOAT_EQ(0.f, nn::util::VectorLength(vectorZero));
    EXPECT_EQ(0.f, nn::util::VectorLengthSquared(vectorZero));
}

TEST_F(MathVector2Test, Distance)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, 6.f, 2.f);

    EXPECT_FLOAT_EQ(9.05538513814f, nn::util::VectorDistance(vector1, vector2));
    EXPECT_EQ(82.f, nn::util::VectorDistanceSquared(vector1, vector2));

    EXPECT_FLOAT_EQ(0.f, nn::util::VectorDistance(vector1, vector1));
    EXPECT_EQ(0.f, nn::util::VectorDistanceSquared(vector1, vector1));
}

TEST_F(MathVector2Test, Normalize)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);

    float lengthSquared1 = nn::util::VectorNormalize(&vector1, vector1);
    NNT_UTIL_VECTOR2_EXPECT_NEARLY_EQ(vector1, -0.94868329805f, 0.31622776601f, FloatError);
    EXPECT_EQ(10.f, lengthSquared1);

    nn::util::Vector2fType vectorZero;
    nn::util::VectorZero(&vectorZero);

    float lengthSquaredZero = nn::util::VectorNormalize(&vector1, vectorZero);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 0.f, 0.f);
    EXPECT_EQ(0.f, lengthSquaredZero);
}

TEST_F(MathVector2Test, Maximize)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f);

    nn::util::VectorMaximize(&vector1, vector1, vector2);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 6.f, 1.f);
}

TEST_F(MathVector2Test, Minimize)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f);

    nn::util::VectorMinimize(&vector1, vector1, vector2);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -3.f, -2.f);
}

TEST_F(MathVector2Test, Lerp)
{
    nn::util::Vector2fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f);
    nn::util::Vector2fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f);

    nn::util::Vector2fType vector3;

    nn::util::VectorLerp(&vector3, vector1, vector2, 0.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector3, -3.f, 1.f);

    nn::util::VectorLerp(&vector3, vector1, vector2, 1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector3, 6.f, -2.f);

    nn::util::VectorLerp(&vector3, vector1, vector2, 0.5f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector3, 1.5f, -0.5f);
}

TEST_F(MathVector2Test, Transform)
{
    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f);

        nn::util::MatrixRowMajor3x2fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        nn::util::VectorTransform(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 19.f, -26.f);
    }

    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f);

        nn::util::MatrixColumnMajor3x2fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        nn::util::VectorTransform(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 19.f, -26.f);
    }
}

TEST_F(MathVector2Test, TransformNormal)
{
    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f);

        nn::util::MatrixRowMajor3x2fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        nn::util::VectorTransformNormal(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 14.f, -20.f);
    }

    {
        nn::util::Vector2fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f);

        nn::util::MatrixColumnMajor3x2fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        nn::util::VectorTransformNormal(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 14.f, -20.f);
    }
}

// 3次元ベクトル
class MathVector3Test : public ::testing::Test
{
};

TEST_F(MathVector3Test, Initializer)
{
    nn::util::Vector3fType vector = NN_UTIL_VECTOR_3F_INITIALIZER(-3.f, 1.f, -2.f);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3Test, Make)
{
    nn::util::Vector3fType vector = nn::util::MakeVector3fType(-3.f, 1.f, -2.f);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3Test, Set)
{
    nn::util::Vector3fType vector;

    nn::util::VectorSet(&vector, -3.f, 1.f, -2.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3Test, SetXyz)
{
    float value[4] = { -1.f, -1.f, -1.f, -1.f };

    nn::util::Vector3fType vector;
    std::memcpy(&vector, value, sizeof(vector));

    nn::util::VectorSetZ(&vector, -2.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -1.f, -1.f, -2.f);

    nn::util::VectorSetX(&vector, -3.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, -1.f, -2.f);

    nn::util::VectorSetY(&vector, 1.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3Test, GetXyz)
{
    nn::util::Vector3fType vector;
    nn::util::VectorSet(&vector, -3.f, 1.f, -2.f);

    EXPECT_EQ(-3.f, nn::util::VectorGetX(vector));
    EXPECT_EQ(1.f, nn::util::VectorGetY(vector));
    EXPECT_EQ(-2.f, nn::util::VectorGetZ(vector));
}

TEST_F(MathVector3Test, Substitution)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2(vector1);

    nn::util::VectorZero(&vector1);

    vector1 = vector2;

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3Test, Load)
{
    {
        float value[3] = { -3.f, 1.f, -2.f };

        nn::util::Vector3fType vector;
        nn::util::VectorLoad(&vector, value);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
    }

    {
        nn::util::Float3 float3Value = NN_UTIL_FLOAT_3_INITIALIZER(-3.f, 1.f, -2.f);

        nn::util::Vector3fType vector;
        nn::util::VectorLoad(&vector, float3Value);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
    }
}

TEST_F(MathVector3Test, Store)
{
    {
        nn::util::Vector3fType vector;
        nn::util::VectorSet(&vector, -3.f, 1.f, -2.f);

        float value[3];

        nn::util::VectorStore(value, vector);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, value[0], value[1], value[2]);
    }

    {
        nn::util::Vector3fType vector;
        nn::util::VectorSet(&vector, -3.f, 1.f, -2.f);

        nn::util::Float3 value;

        nn::util::VectorStore(&value, vector);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, value.v[0], value.v[1], value.v[2]);
    }
}

TEST_F(MathVector3Test, Zero)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);

    nn::util::VectorZero(&vector1);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 0.f, 0.f, 0.f);
}

TEST_F(MathVector3Test, IsZero)
{
    nn::util::Vector3fType vector1;

    {
        nn::util::VectorZero(&vector1);
        EXPECT_TRUE(nn::util::VectorIsZero(vector1));

        nn::util::VectorSet(&vector1, 1.f, 0.f, 0.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1));

        nn::util::VectorSet(&vector1, 0.f, 1.f, 0.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1));

        nn::util::VectorSet(&vector1, 0.f, 0.f, 1.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1));
    }

    {
        float error = 0.0001f;

        nn::util::VectorSet(&vector1, error, -error, error);
        EXPECT_TRUE(nn::util::VectorIsZero(vector1, error));

        nn::util::VectorSet(&vector1, -2 * error, 0.f, 0.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1, error));

        nn::util::VectorSet(&vector1, 0.f, 2 * error, 0.f);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1, error));

        nn::util::VectorSet(&vector1, 0.f, 0.f, -2 * error);
        EXPECT_FALSE(nn::util::VectorIsZero(vector1, error));
    }
}

TEST_F(MathVector3Test, Add)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f);

    nn::util::VectorAdd( &vector1, vector1, vector2 );
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -9.f, 3.f, 2.f);
}

TEST_F(MathVector3Test, Subtract)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f);

    nn::util::VectorSubtract( &vector1, vector1, vector2 );
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 3.f, -1.f, -6.f);
}

TEST_F(MathVector3Test, Multiply)
{
    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);

        nn::util::VectorMultiply(&vector1, vector1, -1.f / 2.f);
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
        nn::util::Vector3fType vector2;
        nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f);

        nn::util::VectorMultiply( &vector1, vector1, vector2 );
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 18.f, 2.f, -8.f);
    }
}

TEST_F(MathVector3Test, Divide)
{
    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);

        nn::util::VectorDivide(&vector1, vector1, -2.f);
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
        nn::util::Vector3fType vector2;
        nn::util::VectorSet(&vector2, 1.f / -6.f, 1.f / 2.f, 1.f / 4.f);

        nn::util::VectorDivide(&vector1, vector1, vector2);
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 18.f, 2.f, -8.f);
    }
}

TEST_F(MathVector3Test, Dot)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f);

    EXPECT_EQ(12.f, nn::util::VectorDot(vector1, vector2));
}

TEST_F(MathVector3Test, Cross)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f);

    nn::util::Vector3fType vector3;
    nn::util::VectorCross(&vector3, vector1, vector2);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, 8.f, 24.f, 0.f);
}

TEST_F(MathVector3Test, Length)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);

    EXPECT_FLOAT_EQ(3.74165738677f, nn::util::VectorLength(vector1));
    EXPECT_EQ(14.f, nn::util::VectorLengthSquared(vector1));

    nn::util::Vector3fType vectorZero;
    nn::util::VectorZero(&vectorZero);

    EXPECT_FLOAT_EQ(0.f, nn::util::VectorLength(vectorZero));
    EXPECT_EQ(0.f, nn::util::VectorLengthSquared(vectorZero));
}

TEST_F(MathVector3Test, Distance)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f);

    EXPECT_FLOAT_EQ(6.78232998312f, nn::util::VectorDistance(vector1, vector2));
    EXPECT_EQ(46.f, nn::util::VectorDistanceSquared(vector1, vector2));

    EXPECT_FLOAT_EQ(0.f, nn::util::VectorDistance(vector1, vector1));
    EXPECT_EQ(0.f, nn::util::VectorDistanceSquared(vector1, vector1));
}

TEST_F(MathVector3Test, Normalize)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);

    float lengthSquared1 = nn::util::VectorNormalize(&vector1, vector1);
    NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, -0.80178372574f, 0.267261241912705f, -0.53452248382541f, FloatError);
    EXPECT_EQ(14.f, lengthSquared1);

    nn::util::Vector3fType vectorZero;
    nn::util::VectorZero(&vectorZero);

    float lengthSquaredZero = nn::util::VectorNormalize(&vector1, vectorZero);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 0.f, 0.f, 0.f);
    EXPECT_EQ(0.f, lengthSquaredZero);
}

TEST_F(MathVector3Test, Maximize)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f, 4.f);

    nn::util::VectorMaximize(&vector1, vector1, vector2);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 6.f, 1.f, 4.f);
}

TEST_F(MathVector3Test, Minimize)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f, 4.f);

    nn::util::VectorMinimize(&vector1, vector1, vector2);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -3.f, -2.f, -2.f);
}

TEST_F(MathVector3Test, Lerp)
{
    nn::util::Vector3fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f);
    nn::util::Vector3fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f, 4.f);

    nn::util::Vector3fType vector3;

    nn::util::VectorLerp(&vector3, vector1, vector2, 0.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, -3.f, 1.f, -2.f);

    nn::util::VectorLerp(&vector3, vector1, vector2, 1.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, 6.f, -2.f, 4.f);

    nn::util::VectorLerp(&vector3, vector1, vector2, 0.5f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, 1.5f, -0.5f, 1.f);
}

TEST_F(MathVector3Test, Rotate)
{
    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, 3.f, 4.f, -5.f, -6.f);

        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);
        nn::util::VectorRotate(&vector1, vector1, quaternion1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -316.f, 72.f, 556.f);
    }

    {
        nn::util::Vector4fType quaternion1;
        // 軸(1.f, 2.f, -3.f) を中心に 60度 回転させる、正規化されたクォータニオンです。
        nn::util::VectorSet(&quaternion1, 0.1336306184530258f, 0.2672612369060516f, -0.4008918404579163f, 0.8660253882408142f);

        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);
        nn::util::VectorRotate(&vector1, vector1, quaternion1);

        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, 0.1428570747375488f, -6.4917459487915039f, 3.7197880744934082f, FloatError);
    }
}

TEST_F(MathVector3Test, Transform)
{
    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::VectorTransform(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 50.f, -61.f, 72.f);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::Vector4fType vector2;

        nn::util::VectorTransform(&vector2, vector1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector2, 63.f, -74.f, 85.f, -96.f);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixColumnMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::VectorTransform(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 50.f, -61.f, 72.f);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixColumnMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::Vector4fType vector2;

        nn::util::VectorTransform(&vector2, vector1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector2, 63.f, -74.f, 85.f, -96.f);
    }
}

TEST_F(MathVector3Test, TransformNormal)
{
    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::VectorTransformNormal(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 60.f, -72.f, 84.f);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixColumnMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::VectorTransformNormal(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 60.f, -72.f, 84.f);
    }
}

TEST_F(MathVector3Test, TransformCoord)
{
    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::VectorTransformCoord(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, -0.65625f, 0.7708333333333333f, -0.8854166666666666f, FloatError);
    }

    {
        nn::util::Vector3fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f);

        nn::util::MatrixColumnMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::VectorTransformCoord(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, -0.65625f, 0.7708333333333333f, -0.8854166666666666f, FloatError);
    }
}

// 4次元ベクトル
class MathVector4Test : public ::testing::Test
{
};

TEST_F(MathVector4Test, Initializer)
{
    nn::util::Vector4fType vector = NN_UTIL_VECTOR_4F_INITIALIZER(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4Test, Make)
{
    nn::util::Vector4fType vector = nn::util::MakeVector4fType(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4Test, Set)
{
    nn::util::Vector4fType vector;
    nn::util::VectorSet(&vector, -3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4Test, SetXyzw)
{
    nn::util::Vector4fType vector;
    nn::util::VectorSet(&vector, 0.f, 0.f, 0.f, 0.f);

    nn::util::VectorSetX(&vector, -3.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 0.f, 0.f, 0.f);

    nn::util::VectorSetY(&vector, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, 0.f, 0.f);

    nn::util::VectorSetZ(&vector, -2.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 0.f);

    nn::util::VectorSetW(&vector, 4.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4Test, GetXyzw)
{
    nn::util::Vector4fType vector;
    nn::util::VectorSet(&vector, -3.f, 1.f, -2.f, 4.f);

    EXPECT_EQ(-3.f, nn::util::VectorGetX(vector));
    EXPECT_EQ(1.f, nn::util::VectorGetY(vector));
    EXPECT_EQ(-2.f, nn::util::VectorGetZ(vector));
    EXPECT_EQ(4.f, nn::util::VectorGetW(vector));
}

TEST_F(MathVector4Test, Substitution)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2(vector1);

    nn::util::VectorZero(&vector1);

    vector1 = vector2;

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4Test, Load)
{
    {
        nn::util::Uint8x4 value = { { 0, 64, 127, 255 } };

        nn::util::Vector4fType vector;
        nn::util::VectorLoad(&vector, value);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, 0.f, 64.f, 127.f, 255.f);
    }

    {
        nn::util::Unorm8x4 value = { { 0, 64, 127, 255 } } ;

        nn::util::Vector4fType vector;
        nn::util::VectorLoad(&vector, value);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(vector, 0.f, 0.250980407f, 0.498039246f, 1.f, FloatError);
    }

    {
        float value[4] = { -3.f, 1.f, -2.f, 4.f };

        nn::util::Vector4fType vector;
        nn::util::VectorLoad(&vector, value);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
    }

    {
        nn::util::Float4 float3Value = NN_UTIL_FLOAT_4_INITIALIZER(-3.f, 1.f, -2.f, 4.f);

        nn::util::Vector4fType vector;
        nn::util::VectorLoad(&vector, float3Value);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
    }
}

TEST_F(MathVector4Test, Store)
{
    {
        nn::util::Vector4fType vector;
        nn::util::VectorSet(&vector, -2.f, 0.25f, 0.5f, 256.f);

        nn::util::Uint8x4 value;

        nn::util::VectorStore(&value, vector);

        EXPECT_TRUE(value.v[0] == 0 && value.v[1] == 0 && value.v[2] == 1 && value.v[3] == 255);
    }

    {
        nn::util::Vector4fType vector;
        nn::util::VectorSet(&vector, -2.f, 0.25f, 0.5f, 1.1f);

        nn::util::Unorm8x4 value;

        nn::util::VectorStore(&value, vector);

        EXPECT_TRUE(value.v[0] == 0 && value.v[1] == 64 && value.v[2] == 128 && value.v[3] == 255);
    }

    {
        nn::util::Vector4fType vector;
        nn::util::VectorSet(&vector, -3.f, 1.f, -2.f, 4.f);

        float value[4];

        nn::util::VectorStore(value, vector);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, value[0], value[1], value[2], value[3]);
    }

    {
        nn::util::Vector4fType vector;
        nn::util::VectorSet(&vector, -3.f, 1.f, -2.f, 4.f);

        nn::util::Float4 value;

        nn::util::VectorStore(&value, vector);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, value.v[0], value.v[1], value.v[2], value.v[3]);
    }
}

TEST_F(MathVector4Test, Zero)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorZero(&vector1);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 0.f, 0.f, 0.f, 0.f);
}

TEST_F(MathVector4Test, IsZero)
{
    nn::util::Vector4fType vector1;

    nn::util::VectorZero(&vector1);
    EXPECT_TRUE(nn::util::VectorIsZero(vector1));

    nn::util::VectorSet(&vector1, 1.f, 0.f, 0.f, 0.f);
    EXPECT_FALSE(nn::util::VectorIsZero(vector1));

    nn::util::VectorSet(&vector1, 0.f, 1.f, 0.f, 0.f);
    EXPECT_FALSE(nn::util::VectorIsZero(vector1));

    nn::util::VectorSet(&vector1, 0.f, 0.f, 1.f, 0.f);
    EXPECT_FALSE(nn::util::VectorIsZero(vector1));

    nn::util::VectorSet(&vector1, 0.f, 0.f, 0.f, 1.f);
    EXPECT_FALSE(nn::util::VectorIsZero(vector1));
}

TEST_F(MathVector4Test, IsZeroWOne)
{
    nn::util::Vector4fType vector1;

    nn::util::VectorSet(&vector1, 0.f, 0.f, 0.f, 1.f);
    EXPECT_TRUE(nn::util::VectorIsZeroWOne(vector1));

    nn::util::VectorSet(&vector1, 1.f, 0.f, 0.f, 1.f);
    EXPECT_FALSE(nn::util::VectorIsZeroWOne(vector1));

    nn::util::VectorSet(&vector1, 0.f, 1.f, 0.f, 1.f);
    EXPECT_FALSE(nn::util::VectorIsZeroWOne(vector1));

    nn::util::VectorSet(&vector1, 0.f, 0.f, 1.f, 1.f);
    EXPECT_FALSE(nn::util::VectorIsZeroWOne(vector1));

    nn::util::VectorSet(&vector1, 0.f, 0.f, 0.f, 0.f);
    EXPECT_FALSE(nn::util::VectorIsZeroWOne(vector1));
}

TEST_F(MathVector4Test, Add)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f, -8.f);

    nn::util::VectorAdd( &vector1, vector1, vector2 );
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -9.f, 3.f, 2.f, -4.f);
}

TEST_F(MathVector4Test, Subtract)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f, -8.f);

    nn::util::VectorSubtract( &vector1, vector1, vector2 );
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 3.f, -1.f, -6.f, 12.f);
}

TEST_F(MathVector4Test, Multiply)
{
    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);

        nn::util::VectorMultiply(&vector1, vector1, -1.f / 2.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f, -2.f);
    }

    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4fType vector2;
        nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f, -8.f);

        nn::util::VectorMultiply(&vector1, vector1, vector2);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 18.f, 2.f, -8.f, -32.f);
    }
}

TEST_F(MathVector4Test, Divide)
{
    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);

        nn::util::VectorDivide(&vector1, vector1, -2.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f, -2.f);
    }

    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4fType vector2;
        nn::util::VectorSet(&vector2, 1.f / -6.f, 1.f / 2.f, 1.f / 4.f, 1.f / -8.f);

        nn::util::VectorDivide(&vector1, vector1, vector2);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 18.f, 2.f, -8.f, -32.f);
    }
}

TEST_F(MathVector4Test, Dot)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f, -8.f);

    EXPECT_EQ(-20.f, nn::util::VectorDot(vector1, vector2));
}

TEST_F(MathVector4Test, Length)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);

    EXPECT_FLOAT_EQ(5.47722557505f, nn::util::VectorLength(vector1));
    EXPECT_EQ(30.f, nn::util::VectorLengthSquared(vector1));

    nn::util::Vector3fType vectorZero;
    nn::util::VectorZero(&vectorZero);

    EXPECT_FLOAT_EQ(0.f, nn::util::VectorLength(vectorZero));
    EXPECT_EQ(0.f, nn::util::VectorLengthSquared(vectorZero));
}

TEST_F(MathVector4Test, Distance)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f, -8.f);

    EXPECT_FLOAT_EQ(13.7840487520f, nn::util::VectorDistance(vector1, vector2));
    EXPECT_EQ(190.f, nn::util::VectorDistanceSquared(vector1, vector2));

    EXPECT_FLOAT_EQ(0.f, nn::util::VectorDistance(vector1, vector1));
    EXPECT_EQ(0.f, nn::util::VectorDistanceSquared(vector1, vector1));
}

TEST_F(MathVector4Test, Normalize)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vectorZero;
    nn::util::VectorZero(&vectorZero);

    float lengthSquared1 = nn::util::VectorNormalize(&vector1, vector1);
    NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(vector1, -0.54772255750f, 0.182574185835055f, -0.36514837167011f, 0.73029674334022f, FloatError);
    EXPECT_EQ(30.f, lengthSquared1);

    float lengthSquaredZero = nn::util::VectorNormalize(&vector1, vectorZero);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 0.f, 0.f, 0.f, 0.f);
    EXPECT_EQ(0.f, lengthSquaredZero);
}

TEST_F(MathVector4Test, Maximize)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f, 4.f, -8.f);

    nn::util::VectorMaximize(&vector1, vector1, vector2);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 6.f, 1.f, 4.f, 4.f);
}

TEST_F(MathVector4Test, Minimize)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f, 4.f, -8.f);

    nn::util::VectorMinimize(&vector1, vector1, vector2);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -3.f, -2.f, -2.f, -8.f);
}

TEST_F(MathVector4Test, Lerp)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, 6.f, -2.f, 4.f, -8.f);

    nn::util::Vector4fType vector3;

    nn::util::VectorLerp(&vector3, vector1, vector2, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector3, -3.f, 1.f, -2.f, 4.f);

    nn::util::VectorLerp(&vector3, vector1, vector2, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector3, 6.f, -2.f, 4.f, -8.f);

    nn::util::VectorLerp(&vector3, vector1, vector2, 0.5f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector3, 1.5f, -0.5f, 1.f, -2.f);
}

TEST_F(MathVector4Test, Transform)
{
    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f, -8.f);

        nn::util::MatrixRowMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::VectorTransform(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 180.f, -200.f, 220.f, -240.f);
    }

    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f, -8.f);

        nn::util::MatrixColumnMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::VectorTransform(&vector1, vector1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 180.f, -200.f, 220.f, -240.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
