﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/util/util_StringView.h>

namespace nn { namespace util {

TEST( StringViewTest, Empty )
{
    string_view empty;

    EXPECT_TRUE( empty.empty() );
    EXPECT_EQ( NULL, empty.data() );
    EXPECT_EQ( 0, empty.length() );

    empty.clear();

    EXPECT_EQ( NULL, empty.data() );
}

TEST( StringViewTest, Compare )
{
    string_view empty;
    string_view abc( "abc" );
    string_view abc2( "abc" );
    string_view abcde("abcde");
    string_view bcd( "bcd" );

    EXPECT_EQ( abc, abc );
    EXPECT_EQ( abc, abc2 );
    EXPECT_EQ( empty, empty );
    EXPECT_NE( abc, empty );
    EXPECT_NE( abc, abcde );
    EXPECT_NE( abc2, empty );
    EXPECT_TRUE( abc == abc2 );
    EXPECT_FALSE( abc == abcde );
    EXPECT_FALSE( abc == empty );
    EXPECT_EQ( 0, abc.compare( abc ) );
    EXPECT_LT( 0, abc.compare( empty ) );
    EXPECT_GT( 0, empty.compare( abc ) );
    EXPECT_EQ( 0, empty.compare( empty ) );

    EXPECT_GT(0, abcde.compare(0, 3, bcd));
    EXPECT_EQ(0, abcde.compare(1, 3, bcd));
    EXPECT_LT(0, abcde.compare(2, 3, bcd));

    EXPECT_GT(0, abcde.compare(0, 3, "bcd"));
    EXPECT_EQ(0, abcde.compare(1, 3, "bcd"));
    EXPECT_LT(0, abcde.compare(2, 3, "bcd"));

    EXPECT_GT(0, abcde.compare(0, 3, abcde, 1, 3));
    EXPECT_EQ(0, abcde.compare(1, 3, abcde, 1, 3));
    EXPECT_LT(0, abcde.compare(2, 3, abcde, 1, 3));

    EXPECT_GT(0, abcde.compare(0, 3, "bcdef", 3));
    EXPECT_EQ(0, abcde.compare(1, 3, "bcdef", 3));
    EXPECT_LT(0, abcde.compare(2, 3, "bcdef", 3));
}

TEST(StringViewTest, StartsWith)
{
    string_view empty;
    string_view abc("abc");
    string_view abcd("abcd");
    string_view abcde("abcde");
    string_view bcd("bcd");

    EXPECT_TRUE(abcd.starts_with(empty));
    EXPECT_TRUE(abcd.starts_with('a'));
    EXPECT_TRUE(abcd.starts_with("abc"));
    EXPECT_TRUE(abcd.starts_with(abc));
    EXPECT_TRUE(abcd.starts_with(abcd));
    EXPECT_FALSE(abcd.starts_with('b'));
    EXPECT_FALSE(abcd.starts_with("bcd"));
    EXPECT_FALSE(abcd.starts_with(bcd));
    EXPECT_FALSE(abcd.starts_with(abcde));
}

TEST(StringViewTest, EndsWith)
{
    string_view empty;
    string_view abc("abc");
    string_view abcd("abcd");
    string_view abcde("abcde");
    string_view bcd("bcd");

    EXPECT_TRUE(abcd.ends_with(empty));
    EXPECT_TRUE(abcd.ends_with('d'));
    EXPECT_TRUE(abcd.ends_with("bcd"));
    EXPECT_TRUE(abcd.ends_with(bcd));
    EXPECT_TRUE(abcd.ends_with(abcd));
    EXPECT_FALSE(abcd.ends_with('c'));
    EXPECT_FALSE(abcd.ends_with("abc"));
    EXPECT_FALSE(abcd.ends_with(abc));
    EXPECT_FALSE(abcd.ends_with(abcde));
}

TEST( StringViewTest, SubString )
{
    string_view empty;
    string_view abc( "abc" );
    string_view abcde( "abcde" );
    string_view spaced( "   abc" );

    EXPECT_TRUE(abc == abcde.substr(0, 3));
    EXPECT_TRUE(abcde == abcde.substr(0, 10));
    EXPECT_EQ('e', abcde[4]);

    abcde.remove_suffix( 2 );
    EXPECT_TRUE( abc == abcde );
    spaced.remove_prefix( 3 );
    EXPECT_TRUE( abc == spaced );
}

TEST( StringViewTest, Find )
{
    string_view abcabcabc( "abcabcabc" );

    EXPECT_EQ(2, abcabcabc.find( "ca" ) );
    EXPECT_EQ(2, abcabcabc.find( 'c' ) );
    EXPECT_EQ(5, abcabcabc.find( "ca", 3 ) );

    EXPECT_EQ(6, abcabcabc.rfind( "ab" ) );
    EXPECT_EQ(6, abcabcabc.rfind( 'a' ) );
    EXPECT_EQ(3, abcabcabc.rfind( "ab", 5 ) );
    EXPECT_EQ(6, abcabcabc.rfind( "ab", 6 ) ); // std::string と同様 1文字目 'a' で探索に該当する

    EXPECT_EQ(1, abcabcabc.find_first_of( "bc" ) );
    EXPECT_EQ(0, abcabcabc.find_first_not_of( "bc" ) );

    EXPECT_EQ(7, abcabcabc.find_last_of( "ab" ) );
    EXPECT_EQ(7, abcabcabc.find_last_of( "ab", 7 ) );
    EXPECT_EQ(6, abcabcabc.find_last_of( "ab", 6 ) );

    EXPECT_EQ(8, abcabcabc.find_last_not_of( "ab" ) );

    EXPECT_EQ(string_view::npos, abcabcabc.find( "de" ) );
    EXPECT_EQ(string_view::npos, abcabcabc.find( 'f' ) );
    EXPECT_EQ(string_view::npos, abcabcabc.rfind( "de" ) );
    EXPECT_EQ(string_view::npos, abcabcabc.rfind( 'f' ) );
    EXPECT_EQ(string_view::npos, abcabcabc.find_first_of( "de" ) );
    EXPECT_EQ(string_view::npos, abcabcabc.find_first_not_of( "abc" ) );
    EXPECT_EQ(string_view::npos, abcabcabc.find_last_of( "de" ) );
    EXPECT_EQ(string_view::npos, abcabcabc.find_last_not_of( "abc" ) );
}

TEST( StringViewDeathTest, StringView )
{
    string_view abcde( "abcde" );
    ASSERT_DEATH_IF_SUPPORTED( abcde[6], "" );
}

}} // namespace nn::util
