﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/util/util_StringUtil.h>

#include <algorithm>
#include <cstring>

#define NELEMS(array)   (sizeof(array) / sizeof(array[0]))

namespace nn { namespace util {

TEST( StringUtilTest, StrlcpyChar )
{
    char srcString[] = "To be, or not to be: that is the question.";

    const size_t DstBufferCountShort      = NELEMS(srcString) - 5;
    const size_t DstBufferCountCorrect    = NELEMS(srcString);
    const size_t DstBufferCountSufficient = NELEMS(srcString) + 5;

    char dstBuffer[DstBufferCountSufficient];

    size_t i;
    size_t countToCopy;

    // 短すぎるバッファで切りつめられる場合

    countToCopy = std::min(DstBufferCountShort - 1, NELEMS(srcString) - 1);
    EXPECT_EQ(DstBufferCountShort - 1, countToCopy);

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, DstBufferCountShort));
    for( i = 0; i < countToCopy; ++i )
    {
        EXPECT_EQ(srcString[i], dstBuffer[i]);
    }
    EXPECT_EQ('\0', dstBuffer[i]);
    for( i = i + 1; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<char>(~0), dstBuffer[i]);
    }

    // 同じサイズのバッファへぴったりコピーされる場合

    countToCopy = std::min(DstBufferCountCorrect - 1, NELEMS(srcString) - 1);
    EXPECT_EQ(DstBufferCountCorrect - 1, countToCopy);
    EXPECT_EQ(NELEMS(srcString) - 1, countToCopy);

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, DstBufferCountCorrect));
    for( i = 0; i < countToCopy; ++i )
    {
        EXPECT_EQ(srcString[i], dstBuffer[i]);
    }
    EXPECT_EQ('\0', dstBuffer[i]);
    for( i = i + 1; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<char>(~0), dstBuffer[i]);
    }

    // 余裕のあるバッファへコピーされる場合

    countToCopy = std::min(DstBufferCountSufficient - 1, NELEMS(srcString) - 1);
    EXPECT_EQ(NELEMS(srcString) - 1, countToCopy);

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, DstBufferCountSufficient));
    for( i = 0; i < countToCopy; ++i )
    {
        EXPECT_EQ(srcString[i], dstBuffer[i]);
    }
    EXPECT_EQ('\0', dstBuffer[i]);
    for( i = i + 1; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<char>(~0), dstBuffer[i]);
    }

    // コピー先の要素数に 0 を指定された場合

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, 0));
    for( i = 0; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<char>(~0), dstBuffer[i]);
    }
}

TEST( StringUtilTest, StrlcpyWchar )
{
    wchar_t srcString[] = L"To be, or not to be: that is the question.";

    const size_t DstBufferCountShort      = NELEMS(srcString) - 5;
    const size_t DstBufferCountCorrect    = NELEMS(srcString);
    const size_t DstBufferCountSufficient = NELEMS(srcString) + 5;

    wchar_t dstBuffer[DstBufferCountSufficient];

    size_t i;
    size_t countToCopy;

    // 短すぎるバッファで切りつめられる場合

    countToCopy = std::min(DstBufferCountShort - 1, NELEMS(srcString) - 1);
    EXPECT_EQ(DstBufferCountShort - 1, countToCopy);

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, DstBufferCountShort));
    for( i = 0; i < countToCopy; ++i )
    {
        EXPECT_EQ(srcString[i], dstBuffer[i]);
    }
    EXPECT_EQ('\0', dstBuffer[i]);
    for( i = i + 1; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<wchar_t>(~0), dstBuffer[i]);
    }

    // 同じサイズのバッファへぴったりコピーされる場合

    countToCopy = std::min(DstBufferCountCorrect - 1, NELEMS(srcString) - 1);
    EXPECT_EQ(DstBufferCountCorrect - 1, countToCopy);
    EXPECT_EQ(NELEMS(srcString) - 1, countToCopy);

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, DstBufferCountCorrect));
    for( i = 0; i < countToCopy; ++i )
    {
        EXPECT_EQ(srcString[i], dstBuffer[i]);
    }
    EXPECT_EQ('\0', dstBuffer[i]);
    for( i = i + 1; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<wchar_t>(~0), dstBuffer[i]);
    }

    // 余裕のあるバッファへコピーされる場合

    countToCopy = std::min(DstBufferCountSufficient - 1, NELEMS(srcString) - 1);
    EXPECT_EQ(NELEMS(srcString) - 1, countToCopy);

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, DstBufferCountSufficient));
    for( i = 0; i < countToCopy; ++i )
    {
        EXPECT_EQ(srcString[i], dstBuffer[i]);
    }
    EXPECT_EQ('\0', dstBuffer[i]);
    for( i = i + 1; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<wchar_t>(~0), dstBuffer[i]);
    }

    // コピー先の要素数に 0 を指定された場合

    std::memset(dstBuffer, ~0, sizeof(dstBuffer));
    EXPECT_EQ(NELEMS(srcString) - 1, Strlcpy(dstBuffer, srcString, 0));
    for( i = 0; i < NELEMS(dstBuffer); ++i )
    {
        EXPECT_EQ(static_cast<wchar_t>(~0), dstBuffer[i]);
    }
}

TEST( StringUtilTest, StrnlenChar )
{
    char srcString[] = "To be, or not to be: that is the question.";

    const size_t CountShort      = NELEMS(srcString) - 5;
    const size_t CountCorrect    = NELEMS(srcString);
    const size_t CountSufficient = NELEMS(srcString) + 5;

    EXPECT_EQ(0, Strnlen(srcString, 0));
    EXPECT_EQ(CountShort, Strnlen(srcString, CountShort));
    EXPECT_EQ(CountCorrect - 1, Strnlen(srcString, CountCorrect - 1));
    EXPECT_EQ(NELEMS(srcString) - 1, Strnlen(srcString, CountCorrect));
    EXPECT_EQ(NELEMS(srcString) - 1, Strnlen(srcString, CountSufficient));
}

TEST( StringUtilTest, StrnlenWchar )
{
    wchar_t srcString[] = L"To be, or not to be: that is the question.";

    const size_t CountShort      = NELEMS(srcString) - 5;
    const size_t CountCorrect    = NELEMS(srcString);
    const size_t CountSufficient = NELEMS(srcString) + 5;

    EXPECT_EQ(0, Strnlen(srcString, 0));
    EXPECT_EQ(CountShort, Strnlen(srcString, CountShort));
    EXPECT_EQ(CountCorrect - 1, Strnlen(srcString, CountCorrect - 1));
    EXPECT_EQ(NELEMS(srcString) - 1, Strnlen(srcString, CountCorrect));
    EXPECT_EQ(NELEMS(srcString) - 1, Strnlen(srcString, CountSufficient));
}

TEST( StringUtilTest, StrncmpChar )
{
    {
        // '.' < '?'
        char string1[] = "To be, or not to be: that is the question.";
        char string2[] = "To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 2;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strncmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch - 1) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch - 1) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountSufficient) > 0);
    }

    {
        char string1[] = "To be, or not to be:";
        char string2[] = "To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 1;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strncmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountSufficient) > 0);
    }
}

TEST( StringUtilTest, StrncmpWchar )
{
    {
        // '.' < '?'
        wchar_t string1[] = L"To be, or not to be: that is the question.";
        wchar_t string2[] = L"To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 2;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strncmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountSufficient) > 0);
    }

    {
        wchar_t string1[] = L"To be, or not to be:";
        wchar_t string2[] = L"To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 1;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strncmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strncmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strncmp(string2, string1, CountSufficient) > 0);
    }
}

TEST( StringUtilTest, StrnicmpChar )
{
    {
        char string1[] = "to be, or not to be: that is the question.";
        char string2[] = "TO BE, OR NOT TO BE: THAT IS THE QUESTION.";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) == 0);
    }

    {
        char string1[] = "TO BE, OR NOT TO BE: THAT IS THE QUESTION.";
        char string2[] = "to be, or not to be: that is the question.";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) == 0);
    }

    {
        char string1[] = "TO BE, OR NOT TO BE: that is the question.";
        char string2[] = "to be, or not to be: THAT IS THE QUESTION.";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) == 0);
    }

    {
        // '.' < '?'
        char string1[] = "To be, or not to be: that is the question.";
        char string2[] = "To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 2;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch - 1) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch - 1) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) > 0);
    }

    {
        char string1[] = "To be, or not to be:";
        char string2[] = "To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 1;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) > 0);
    }
}

TEST( StringUtilTest, StrnicmpWchar )
{
    {
        wchar_t string1[] = L"to be, or not to be: that is the question.";
        wchar_t string2[] = L"TO BE, OR NOT TO BE: THAT IS THE QUESTION.";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) == 0);
    }

    {
        wchar_t string1[] = L"TO BE, OR NOT TO BE: THAT IS THE QUESTION.";
        wchar_t string2[] = L"to be, or not to be: that is the question.";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) == 0);
    }

    {
        wchar_t string1[] = L"to be, or not to be: THAT IS THE QUESTION.";
        wchar_t string2[] = L"TO BE, OR NOT TO BE: that is the question.";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) == 0);
    }

    {
        // '.' < '?'
        wchar_t string1[] = L"To be, or not to be: that is the question.";
        wchar_t string2[] = L"To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 2;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) > 0);
    }

    {
        wchar_t string1[] = L"To be, or not to be:";
        wchar_t string2[] = L"To be, or not to be: that is the question?";

        const size_t CountShort      = NELEMS(string1) - 5;
        const size_t CountMatch      = NELEMS(string1) - 1;
        const size_t CountCorrect    = NELEMS(string1);
        const size_t CountSufficient = NELEMS(string1) + 5;

        EXPECT_TRUE(Strnicmp(string1, string2, 0) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, 0) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountShort) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountShort) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch) == 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch) == 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountMatch + 1) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountMatch + 1) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountCorrect) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountCorrect) > 0);

        EXPECT_TRUE(Strnicmp(string1, string2, CountSufficient) < 0);
        EXPECT_TRUE(Strnicmp(string2, string1, CountSufficient) > 0);
    }
}

TEST( StringUtilTest, ToLowerChar )
{
    {
        char string1[] = "to be, or not to be: that is the question.";
        char string2[] = "TO BE, OR NOT TO BE: THAT IS THE QUESTION.";
        char string3[NELEMS(string2)] = {};

        int i = 0;
        for (auto c : string2)
        {
            string3[i++] = ToLower(c);
        }

        EXPECT_TRUE(Strncmp(string1, string3, NELEMS(string1)) == 0);
    }

    {
        for (char c1 = CHAR_MIN; c1 < CHAR_MAX; c1++)
        {
            char c2 = ToLower(c1);

            if (c1 >= 'A' && c1 <= 'Z')
            {
                EXPECT_TRUE(c1 != c2);
            }
            else
            {
                EXPECT_TRUE(c1 == c2);
            }
        }

        EXPECT_TRUE(CHAR_MAX == ToLower(CHAR_MAX));
    }
}

TEST( StringUtilTest, ToLowerWchar )
{
    {
        wchar_t string1[] = L"to be, or not to be: that is the question.";
        wchar_t string2[] = L"TO BE, OR NOT TO BE: THAT IS THE QUESTION.";
        wchar_t string3[NELEMS(string2)] = {};

        int i = 0;
        for (auto c : string2)
        {
            string3[i++] = ToLower(c);
        }

        EXPECT_TRUE(Strncmp(string1, string3, NELEMS(string1)) == 0);
    }

    {
        wchar_t wcharMin = 0;
        wchar_t wcharMax = WCHAR_MAX;

        // sizeof (wchar_t) = 4 の時に約 43 億回ループするのを防ぐ。
        if (wcharMax >= 0xFFFF)
        {
            wcharMax = 0xFFFF;
        }

        for (wchar_t c1 = wcharMin; c1 < wcharMax; c1++)
        {
            wchar_t c2 = ToLower(c1);

            if (c1 >= 'A' && c1 <= 'Z')
            {
                EXPECT_TRUE(c1 != c2);
            }
            else
            {
                EXPECT_TRUE(c1 == c2);
            }
        }

        EXPECT_TRUE(wcharMax == ToLower(wcharMax));
    }
}

TEST( StringUtilTest, ToUpperChar )
{
    {
        char string1[] = "to be, or not to be: that is the question.";
        char string2[] = "TO BE, OR NOT TO BE: THAT IS THE QUESTION.";
        char string3[NELEMS(string1)] = {};

        int i = 0;
        for (auto c : string1)
        {
            string3[i++] = ToUpper(c);
        }

        EXPECT_TRUE(Strncmp(string2, string3, NELEMS(string2)) == 0);
    }

    {
        for (char c1 = CHAR_MIN; c1 < CHAR_MAX; c1++)
        {
            char c2 = ToUpper(c1);

            if (c1 >= 'a' && c1 <= 'z')
            {
                EXPECT_TRUE(c1 != c2);
            }
            else
            {
                EXPECT_TRUE(c1 == c2);
            }
        }

        EXPECT_TRUE(CHAR_MAX == ToUpper(CHAR_MAX));
    }
}

TEST( StringUtilTest, ToUpperWchar )
{
    {
        wchar_t string1[] = L"to be, or not to be: that is the question.";
        wchar_t string2[] = L"TO BE, OR NOT TO BE: THAT IS THE QUESTION.";
        wchar_t string3[NELEMS(string1)] = {};

        int i = 0;
        for (auto c : string1)
        {
            string3[i++] = ToUpper(c);
        }

        EXPECT_TRUE(Strncmp(string2, string3, NELEMS(string2)) == 0);
    }

    {
        wchar_t wcharMin = 0;
        wchar_t wcharMax = WCHAR_MAX;

        // sizeof (wchar_t) = 4 の時に約 43 億回ループするのを防ぐ。
        if (wcharMax >= 0xFFFF)
        {
            wcharMax = 0xFFFF;
        }

        for (wchar_t c1 = wcharMin; c1 < wcharMax; c1++)
        {
            wchar_t c2 = ToUpper(c1);

            if (c1 >= 'a' && c1 <= 'z')
            {
                EXPECT_TRUE(c1 != c2);
            }
            else
            {
                EXPECT_TRUE(c1 == c2);
            }
        }

        EXPECT_TRUE(wcharMax == ToUpper(wcharMax));
    }
}

}} // namespace nn::util
