﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Quaternion.h>
#include <nn/util/util_Vector.h>

#include "testUtil_Vector.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.000001f;
}

//
// クオータニオン
//
class MathQuaternionClassTest : public ::testing::Test
{
};

TEST_F(MathQuaternionClassTest, Constant)
{
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Quaternion::Identity(), 0.f, 0.f, 0.f, 1.f);
}

TEST_F(MathQuaternionClassTest, Constractor)
{
    {
        nn::util::Quaternion quaternion(-3.f, 1.f, -2.f, 4.f);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 4.f);
    }

    {
        nn::util::Float4 floatValue = NN_UTIL_FLOAT_4_INITIALIZER(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion(floatValue);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 4.f);
    }

    {
        nn::util::Vector4fType vectorTypeValue = NN_UTIL_VECTOR_4F_INITIALIZER(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion(vectorTypeValue);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 4.f);
    }
}

TEST_F(MathQuaternionClassTest, GetXyzw)
{
    nn::util::Quaternion quaternion(-3.f, 1.f, -2.f, 4.f);

    EXPECT_EQ(-3.f, quaternion.GetX());
    EXPECT_EQ(1.f, quaternion.GetY());
    EXPECT_EQ(-2.f, quaternion.GetZ());
    EXPECT_EQ(4.f, quaternion.GetW());
}

TEST_F(MathQuaternionClassTest, Get)
{
    nn::util::Quaternion vector(-3.f, 1.f, -2.f, 4.f);

    float x, y, z, w;
    vector.Get(&x, &y, &z, &w);

    EXPECT_EQ(-3.f, x);
    EXPECT_EQ(1.f, y);
    EXPECT_EQ(-2.f, z);
    EXPECT_EQ(4.f, w);
}

TEST_F(MathQuaternionClassTest, SetXyzw)
{
    nn::util::Quaternion quaternion(0.f, 0.f, 0.f, 0.f);

    quaternion.SetX(-3.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 0.f, 0.f, 0.f);

    quaternion.SetY(1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, 0.f, 0.f);

    quaternion.SetZ(-2.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 0.f);

    quaternion.SetW(4.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathQuaternionClassTest, Set)
{
    nn::util::Quaternion quaternion;
    quaternion.Set(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathQuaternionClassTest, PlusSign)
{
    nn::util::Quaternion quaternion(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathQuaternionClassTest, MinusSign)
{
    nn::util::Quaternion quaternion(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(-quaternion, 3.f, -1.f, 2.f, -4.f);
}

TEST_F(MathQuaternionClassTest, Add)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 = quaternion1 + quaternion2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -9.f, 3.f, 2.f, -4.f);
    }

    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 += quaternion2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -9.f, 3.f, 2.f, -4.f);
    }
}

TEST_F(MathQuaternionClassTest, Subtract)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 = quaternion1 - quaternion2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 3.f, -1.f, -6.f, 12.f);
    }

    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 -= quaternion2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 3.f, -1.f, -6.f, 12.f);
    }
}

TEST_F(MathQuaternionClassTest, Multiply)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 = quaternion1 * quaternion2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 8.f, 24.f, 32.f, -44.f);
    }

    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 *= quaternion2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 8.f, 24.f, 32.f, -44.f);
    }
}

TEST_F(MathQuaternionClassTest, Divide)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 = quaternion1 / quaternion2;

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.4666666984558106f, 0.0666666850447655f, 0.0000000000000000f, -0.1666666716337204f, FloatError);
    }

    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        quaternion1 /= quaternion2;

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.4666666984558106f, 0.0666666850447655f, 0.0000000000000000f, -0.1666666716337204f, FloatError);
    }
}

TEST_F(MathQuaternionClassTest, Dot)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        EXPECT_EQ(-20.f, quaternion1.Dot(quaternion2));
    }

    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(-6.f, 2.f, 4.f, -8.f);

        EXPECT_EQ(-20.f, nn::util::Quaternion::Dot(quaternion1, quaternion2));
    }
}

TEST_F(MathQuaternionClassTest, Length)
{
    nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);

    EXPECT_FLOAT_EQ(5.47722557505f, quaternion1.Length());
    EXPECT_EQ(30.f, quaternion1.LengthSquared());
}

TEST_F(MathQuaternionClassTest, ToAxisAngle)
{
    // 軸(1.f, 2.f, -3.f) を中心に 60度 回転させる、正規化されたクォータニオンです。
    nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);

    nn::util::Vector3f axisAngle = quaternion1.ToAxisAngle();

    NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(axisAngle, -1.3734008073806763f, -0.19739556312561035f, -0.82321196794509888f, FloatError);
}

TEST_F(MathQuaternionClassTest, Normalize)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);

        bool result = quaternion1.Normalize();

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, -0.54772255750f, 0.182574185835055f, -0.36514837167011f, 0.73029674334022f, FloatError);
        EXPECT_TRUE(result);
    }

    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);

        quaternion1 = nn::util::Quaternion::Normalize(quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, -0.54772255750f, 0.182574185835055f, -0.36514837167011f, 0.73029674334022f, FloatError);
    }

    {
        nn::util::Quaternion quaternion1(0.0001f, -0.0002f, 0.0003f, -0.0004f);

        bool result = quaternion1.Normalize();

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 0.f, 0.f, 0.f, 0.f);
        EXPECT_FALSE(result);
    }

    {
        nn::util::Quaternion quaternion1(0.0001f, -0.0002f, 0.0003f, -0.0004f);

        quaternion1 = nn::util::Quaternion::Normalize(quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 0.f, 0.f, 0.f, 0.f);
    }
}

TEST_F(MathQuaternionClassTest, Inverse)
{
    nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);

    quaternion1 = nn::util::Quaternion::Inverse(quaternion1);

    NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.1000000089406967f, -0.0333333350718021f, 0.0666666701436043f, 0.1333333402872086f, FloatError);
}

TEST_F(MathQuaternionClassTest, Lerp)
{
    nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Quaternion quaternion2(6.f, -2.f, 4.f, -8.f);

    nn::util::Quaternion quaternion3;

    quaternion3 = nn::util::Quaternion::Lerp(quaternion1, quaternion2, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -3.f, 1.f, -2.f, 4.f);

    quaternion3 = nn::util::Quaternion::Lerp(quaternion1, quaternion2, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, 6.f, -2.f, 4.f, -8.f);

    quaternion3 = nn::util::Quaternion::Lerp(quaternion1, quaternion2, 0.5f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, 1.5f, -0.5f, 1.f, -2.f);
}

TEST_F(MathQuaternionClassTest, Slerp)
{
    {
        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Quaternion quaternion2(6.f, -2.f, 4.f, -8.f);

        nn::util::Quaternion quaternion3;

        quaternion3 = nn::util::Quaternion::Slerp(quaternion1, quaternion2, 0.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -3.f, 1.f, -2.f, 4.f);

        quaternion3 = nn::util::Quaternion::Slerp(quaternion1, quaternion2, 1.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -6.f, 2.f, -4.f, 8.f);

        quaternion3 = nn::util::Quaternion::Slerp(quaternion1, quaternion2, 0.5f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -4.5f, 1.5f, -3.f, 6.f);
    }

    {
        nn::util::Quaternion quaternion1(-0.03f, 0.01f, -0.02f, 0.04f);
        nn::util::Quaternion quaternion2(0.06f, -0.02f, 0.04f, -0.08f);

        nn::util::Quaternion quaternion3;

        quaternion3 = nn::util::Quaternion::Slerp(quaternion1, quaternion2, 0.f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion3, -0.0299999993294477f, 0.0099999997764826f, -0.0199999995529652f, 0.0399999991059303f, FloatError);

        quaternion3 = nn::util::Quaternion::Slerp(quaternion1, quaternion2, 1.f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion3, -0.0599999986588955f, 0.0199999995529652f, -0.0399999991059303f, 0.0799999982118607f, FloatError);

        quaternion3 = nn::util::Quaternion::Slerp(quaternion1, quaternion2, 0.5f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion3, -0.0634495541453362f, 0.0211498513817787f, -0.0422997027635574f, 0.0845994055271149f, FloatError);
    }
}

TEST_F(MathQuaternionClassTest, Squad)
{
    nn::util::Quaternion p(-3.f, 1.f, -2.f, 4.f);
    nn::util::Quaternion a(6.f, -2.f, 4.f, -8.f);
    nn::util::Quaternion b(-12.f, 4.f, -8.f, 16.f);
    nn::util::Quaternion q(24.f, -8.f, 16.f, -32.f);

    nn::util::Quaternion quaternion1;

    quaternion1 = nn::util::Quaternion::Squad(p, a, b, q, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -3.f, 1.f, -2.f, 4.f);

    quaternion1 = nn::util::Quaternion::Squad(p, a, b, q, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -24.f, 8.f, -16.f, 32.f);

    quaternion1 = nn::util::Quaternion::Squad(p, a, b, q, 0.5f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -11.25f, 3.75f, -7.5f, 15.f);
}

TEST_F(MathQuaternionClassTest, FromYawPitchRoll)
{
    nn::util::Quaternion quaternion;

    float roll = 40.f / 180.f * nn::util::FloatPi;
    float pitch = 60.f / 180.f * nn::util::FloatPi;
    float yaw = 30.f / 180.f * nn::util::FloatPi;

    quaternion = nn::util::Quaternion::FromYawPitchRoll(yaw, pitch, roll);

    NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion,
        0.5304983854293823f,
        0.0454432815313339f,
        0.1645002514123917f,
        0.8303288221359253f,
        FloatError);
}

TEST_F(MathQuaternionClassTest, FromRotationMatrix)
{
    {
        // 以下のロール、ピッチ、ヨーによる回転行列
        // roll = 40.f / 180.f * PI;
        // pitch = 60.f / 180.f * PI;
        // yaw = 30.f / 180.f * PI;
        nn::util::MatrixRowMajor4x3f matrix1(
            0.9417491555213928f, 0.3213937878608704f, 0.0990685001015663f,
            -0.2249634265899658f, 0.3830222487449646f, 0.8959270715713501f,
            0.2499999701976776f, -0.8660253286361694f, 0.4330127239227295f,
            -10.f, 11.f, -12.f);

        nn::util::Quaternion quaternion1;

        quaternion1 = nn::util::Quaternion::FromRotationMatrix(matrix1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.5304983258247376f, 0.0454432815313339f, 0.1645002365112305f, 0.8303288221359253f, FloatError);
    }
}

TEST_F(MathQuaternionClassTest, FromAxisAngle)
{
    nn::util::Quaternion quaternion1;
    nn::util::Vector3f axis(-3.f, 1.f, -2.f);

    quaternion1 = nn::util::Quaternion::FromAxisAngle(axis, 60.f / 180.f * nn::util::FloatPi);

    NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, -0.4008918404579163f, 0.1336306184530258f, -0.2672612369060517f, 0.8660254478454590f, FloatError);
}

TEST_F(MathQuaternionClassTest, MakeVectorRotation)
{
    {
        nn::util::Vector3f from(-3.f, 1.f, -2.f);

        nn::util::Vector3f to(-6.f, 2.f, 4.f);

        nn::util::Vector4f quaternion1;

        nn::util::QuaternionMakeVectorRotation(&quaternion1, from, to);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 1.5689290761947632f, 4.7067871093750000f, 0.0000000000000000f, 2.5495097637176514f, FloatError);
    }

    {
        nn::util::Vector3f from(-3.f, 1.f, -2.f);

        nn::util::Vector3f to(6.f, -2.f, 4.f);

        nn::util::Vector4f quaternion1;

        nn::util::QuaternionMakeVectorRotation(&quaternion1, from, to);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 1.f, 0.f, 0.f, 0.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
