﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

#include "testUtil_Matrix.h"
#include "testUtil_Vector.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.001f;
}

//
// 行列
//

// 3x2 行列
template <typename T>
class MathMatrix3x2ClassTest : public ::testing::Test
{
};

typedef ::testing::Types<nn::util::MatrixRowMajor3x2f> Matrix3x2Types;
TYPED_TEST_CASE(MathMatrix3x2ClassTest, Matrix3x2Types);

TYPED_TEST(MathMatrix3x2ClassTest, Constant)
{
    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(TypeParam::Zero(),
        0.f, 0.f,
        0.f, 0.f,
        0.f, 0.f);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(TypeParam::Identity(),
        1.f, 0.f,
        0.f, 1.f,
        0.f, 0.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, Constractor)
{
    {
        TypeParam matrix(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::Vector2f axisX(1.f, 2.f);
        nn::util::Vector2f axisY(3.f, 4.f);
        nn::util::Vector2f axisZ(5.f, 6.f);

        TypeParam matrix(axisX, axisY, axisZ);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::FloatRowMajor3x2 floatValue = NN_UTIL_FLOAT_ROW_MAJOR_3X2_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        TypeParam matrix(floatValue);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::MatrixRowMajor3x2fType matrixType = NN_UTIL_MATRIX_ROW_MAJOR_3X2F_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        TypeParam matrix(matrixType);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }
}

TYPED_TEST(MathMatrix3x2ClassTest, GetAxis)
{
    TypeParam matrix1(
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2f vector1;

    vector1 = matrix1.GetAxisX();
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.f, 2.f);

    vector1 = matrix1.GetAxisY();
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 3.f, 4.f);

    vector1 = matrix1.GetAxisZ();
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, GetAxes)
{
    TypeParam matrix1(
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2f axisX, axisY, axisZ;
    matrix1.Get(&axisX, &axisY, &axisZ);

    NNT_UTIL_VECTOR2_EXPECT_EQ(axisX, 1.f, 2.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(axisY, 3.f, 4.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(axisZ, 5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, SetAxis)
{
    TypeParam matrix1(
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2f vector1;

    vector1.Set(-7.f, -8.f);
    matrix1.SetAxisX(vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        3.f, 4.f,
        5.f, 6.f);

    vector1.Set(-9.f, -10.f);
    matrix1.SetAxisY(vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        5.f, 6.f);

    vector1.Set(-11.f, -12.f);
    matrix1.SetAxisZ(vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, SetAxes)
{
    TypeParam matrix1(
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2f axisX(-7.f, -8.f);
    nn::util::Vector2f axisY(-9.f, -10.f);
    nn::util::Vector2f axisZ(-11.f, -12.f);

    matrix1.Set(axisX, axisY, axisZ);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, SetScale)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);


    nn::util::Vector2f scale(-3.f, 1.f);

    matrix1.SetScale(scale);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -3.f, 0.f,
        0.f, 1.f,
        5.f, -6.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, SetTranslate)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::Vector2f translate(-3.f, 1.f);

    matrix1.SetTranslate(translate);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        -3.f, 1.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, SetRotate)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    float rotation = 40.f / 180.f * nn::util::FloatPi;

    matrix1.SetRotate(rotation);

    NNT_UTIL_MATRIX_3X2_EXPECT_NEARLY_EQ(matrix1,
        0.76604444311f, 0.64278760968f,
        -0.64278760968f, 0.76604444311f,
        5.f, -6.f,
        FloatError);
}

TYPED_TEST(MathMatrix3x2ClassTest, Transpose)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    matrix1.Transpose();

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        1.f, -3.f,
        -2.f, 4.f,
        0.f, 0.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, Inverse)
{
    {
        TypeParam matrix1(
            1.f, -2.f,
            -4.f, 8.f,
            5.f, -6.f);

        EXPECT_FALSE(matrix1.Inverse());

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            0.f, 0.f,
            0.f, 0.f,
            0.f, 0.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        EXPECT_TRUE(matrix1.Inverse());

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -2.f, -1.f,
            -1.5f, -0.5f,
            1.f, 2.f);
    }
}

TYPED_TEST(MathMatrix3x2ClassTest, InverseTranspose)
{
    {
        TypeParam matrix1(
            1.f, -2.f,
            -4.f, 8.f,
            5.f, -6.f);

        EXPECT_FALSE(matrix1.InverseTranspose());

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            0.f, 0.f,
            0.f, 0.f,
            0.f, 0.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        EXPECT_TRUE(matrix1.InverseTranspose());

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -2.f, -1.5f,
            -1.f, -0.5f,
            0.f, 0.f);
    }
}

TYPED_TEST(MathMatrix3x2ClassTest, Transform)
{
    nn::util::Vector2f vector1(2.f, -4.f);

    nn::util::MatrixRowMajor3x2f matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    vector1 = matrix1.Transform(vector1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 19.f, -26.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, TransformNormal)
{
    nn::util::Vector2f vector1(2.f, -4.f);

    nn::util::MatrixRowMajor3x2f matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    vector1 = matrix1.TransformNormal(vector1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 14.f, -20.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, PlusSign)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(+matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, MinusSign)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(-matrix1,
        -1.f, 2.f,
        3.f, -4.f,
        -5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, Add)
{
    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2(
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        matrix1 = matrix1 + matrix2;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            3.f, -6.f,
            3.f, 12.f,
            -5.f, 6.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2(
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        matrix1 += matrix2;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            3.f, -6.f,
            3.f, 12.f,
            -5.f, 6.f);
    }
}

TYPED_TEST(MathMatrix3x2ClassTest, Subtract)
{
    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2(
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        matrix1 = matrix1 - matrix2;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -1.f, 2.f,
            -9.f, -4.f,
            15.f, -18.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2(
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        matrix1 -= matrix2;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -1.f, 2.f,
            -9.f, -4.f,
            15.f, -18.f);
    }

}

TYPED_TEST(MathMatrix3x2ClassTest, Multiply)
{
    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        matrix1 = matrix1 * (-1.f / 2.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -0.5f, 1.f,
            1.5f, -2.f,
            -2.5f, 3.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        matrix1 = (-1.f / 2.f) * matrix1;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -0.5f, 1.f,
            1.5f, -2.f,
            -2.5f, 3.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        matrix1 *= (-1.f / 2.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -0.5f, 1.f,
            1.5f, -2.f,
            -2.5f, 3.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2(
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        matrix1 = matrix1 * matrix2;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -10.f, -20.f,
            18.f, 44.f,
            -36.f, -56.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2(
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        matrix1 *= matrix2;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -10.f, -20.f,
            18.f, 44.f,
            -36.f, -56.f);
    }
}

TYPED_TEST(MathMatrix3x2ClassTest, Divide)
{
    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        matrix1 = matrix1 / -2.f;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -0.5f, 1.f,
            1.5f, -2.f,
            -2.5f, 3.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        matrix1 /= -2.f;

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -0.5f, 1.f,
            1.5f, -2.f,
            -2.5f, 3.f);
    }
}

TYPED_TEST(MathMatrix3x2ClassTest, MakeTranslation)
{
    nn::util::Vector2f translate(-3.f, 1.f);

    TypeParam matrix1 = TypeParam::MakeTranslation(translate);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        1.f, 0.f,
        0.f, 1.f,
        -3.f, 1.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, MakeScale)
{
    nn::util::Vector2f scale(-3.f, 1.f);

    TypeParam matrix1 = TypeParam::MakeScale(scale);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -3.f, 0.f,
        0.f, 1.f,
        0.f, 0.f);
}

TYPED_TEST(MathMatrix3x2ClassTest, MakeRotation)
{
    float rotation = 40.f / 180.f * nn::util::FloatPi;

    TypeParam matrix1 = TypeParam::MakeRotation(rotation);

    NNT_UTIL_MATRIX_3X2_EXPECT_NEARLY_EQ(matrix1,
        0.76604444311f, 0.64278760968f,
        -0.64278760968f, 0.76604444311f,
        0.f, 0.f,
        FloatError);
}

TYPED_TEST(MathMatrix3x2ClassTest, MakeTranspose)
{
    TypeParam matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    matrix1 = TypeParam::MakeTranspose(matrix1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        1.f, -3.f,
        -2.f, 4.f,
        0.f, 0.f);
}

// 4x3 行列
template <typename T>
class MathMatrix4x3ClassTest : public ::testing::Test
{
};

typedef ::testing::Types<nn::util::MatrixRowMajor4x3f> Matrix4x3Types;
TYPED_TEST_CASE(MathMatrix4x3ClassTest, Matrix4x3Types);

TYPED_TEST(MathMatrix4x3ClassTest, Constant)
{
    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(TypeParam::Zero(),
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(TypeParam::Identity(),
            1.f, 0.f, 0.f,
            0.f, 1.f, 0.f,
            0.f, 0.f, 1.f,
            0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, Constractor)
{
    {
        TypeParam matrix(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::Vector3f axisX(1.f, 2.f, 3.f);
        nn::util::Vector3f axisY(4.f, 5.f, 6.f);
        nn::util::Vector3f axisZ(7.f, 8.f, 9.f);
        nn::util::Vector3f axisW(10.f, 11.f, 12.f);

        TypeParam matrix(axisX, axisY, axisZ, axisW);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::FloatRowMajor4x3 floatValue = NN_UTIL_FLOAT_ROW_MAJOR_4X3_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        TypeParam matrix(floatValue);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::MatrixRowMajor4x3fType matrixType = NN_UTIL_MATRIX_ROW_MAJOR_4X3F_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        TypeParam matrix(matrixType);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, GetAxis)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3f vector1;

    vector1 = matrix1.GetAxisX();
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.f, 2.f, 3.f);

    vector1 = matrix1.GetAxisY();
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 4.f, 5.f, 6.f);

    vector1 = matrix1.GetAxisZ();
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 7.f, 8.f, 9.f);

    vector1 = matrix1.GetAxisW();
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, GetAxes)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3f axisX, axisY, axisZ, axisW;
    matrix1.Get(&axisX, &axisY, &axisZ, &axisW);

    NNT_UTIL_VECTOR3_EXPECT_EQ(axisX, 1.f, 2.f, 3.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(axisY, 4.f, 5.f, 6.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(axisZ, 7.f, 8.f, 9.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(axisW, 10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, SetAxis)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3f vector1;

    vector1.Set(-13.f, -14.f, -15.f);
    matrix1.SetAxisX(vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

    vector1.Set(-16.f, -17.f, -18.f);
    matrix1.SetAxisY(vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

    vector1.Set(-19.f, -20.f, -21.f);
    matrix1.SetAxisZ(vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            -19.f, -20.f, -21.f,
            10.f, 11.f, 12.f);

    vector1.Set(-22.f, -23.f, -24.f);
    matrix1.SetAxisW(vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            -19.f, -20.f, -21.f,
            -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, SetAxes)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3f axisX(-13.f, -14.f, -15.f);
    nn::util::Vector3f axisY(-16.f, -17.f, -18.f);
    nn::util::Vector3f axisZ(-19.f, -20.f, -21.f);
    nn::util::Vector3f axisW(-22.f, -23.f, -24.f);

    matrix1.Set(axisX, axisY, axisZ, axisW);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            -19.f, -20.f, -21.f,
            -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, SetScale)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3f scale(-3.f, 1.f, -2.f);

    matrix1.SetScale(scale);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -3.f, 0.f, 0.f,
        0.f, 1.f, 0.f,
        0.f, 0.f, -2.f,
        -10.f, 11.f, -12.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, SetTranslate)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3f translate(-3.f, 1.f, -2.f);

    matrix1.SetTranslate(translate);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -3.f, 1.f, -2.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, SetRotate)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::Quaternion quaternion1(-3.f, 1.f, -2.f, 4.f);
        quaternion1.Normalize();

        matrix1.SetRotate(quaternion1);

        NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
            0.6666666269302368f, -0.7333333492279053f, 0.1333333551883698f,
            0.3333333432674408f, 0.1333333253860474f, -0.9333333969116210f,
            0.6666667461395264f, 0.6666667461395264f, 0.3333333134651184f,
            -10.f, 11.f, -12.f,
            FloatError);
    }

    {
        TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

        nn::util::Vector3f rotation(
            30.f / 180.f * nn::util::FloatPi,
            40.f / 180.f * nn::util::FloatPi,
            50.f / 180.f * nn::util::FloatPi);

        matrix1.SetRotate(rotation);

        NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
            0.4923306107521058f, 0.5867368578910828f, -0.6427397727966309f,
            -0.4567770063877106f, 0.8027404546737671f, 0.3829680085182190f,
            0.7407116293907166f, 0.1049940884113312f, 0.6633203029632568f,
            -10.f, 11.f, -12.f,
            FloatError);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, Transpose)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    matrix1.Transpose();

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, -4.f, 7.f,
            -2.f, 5.f, -8.f,
            3.f, -6.f, 9.f,
            0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, Inverse)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        EXPECT_FALSE(matrix1.Inverse());

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1(
            1.f, 0.f, 1.f,
            -1.f, 1.f, 1.f,
            1.f, -1.f, 0.f,
            1.f, 1.f, -1.f);

        EXPECT_TRUE(matrix1.Inverse());

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, -1.f, -1.f,
            1.f, -1.f, -2.f,
            0.f, 1.f, 1.f,
            -2.f, 3.f, 4.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, InverseTranspose)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        EXPECT_FALSE(matrix1.InverseTranspose());

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1(
            1.f, 0.f, 1.f,
            -1.f, 1.f, 1.f,
            1.f, -1.f, 0.f,
            1.f, 1.f, -1.f);

        EXPECT_TRUE(matrix1.InverseTranspose());

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, 1.f, 0.f,
            -1.f, -1.f, 1.f,
            -1.f, -2.f, 1.f,
            0.f, 0.f, 0.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, Transform)
{
    {
        nn::util::Vector3f vector1(2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x3f matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        vector1 = matrix1.Transform(vector1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 50.f, -61.f, 72.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, TransformNormal)
{
    {
        nn::util::Vector3f vector1(2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x3f matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        vector1 = matrix1.TransformNormal(vector1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 60.f, -72.f, 84.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, PlusSign)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(+matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, MinusSign)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(-matrix1,
        -1.f, 2.f, -3.f,
        4.f, -5.f, 6.f,
        -7.f, 8.f, -9.f,
        10.f, -11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, Add)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2(
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        matrix1 = matrix1 + matrix2;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            3.f, -6.f, 9.f,
            4.f, -5.f, 6.f,
            -7.f, 8.f, -9.f,
            -30.f, 33.f, -36.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2(
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        matrix1 += matrix2;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            3.f, -6.f, 9.f,
            4.f, -5.f, 6.f,
            -7.f, 8.f, -9.f,
            -30.f, 33.f, -36.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, Subtract)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2(
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        matrix1 = matrix1 - matrix2;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -1.f, 2.f, -3.f,
            -12.f, 15.f, -18.f,
            21.f, -24.f, 27.f,
            10.f, -11.f, 12.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2(
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        matrix1 -= matrix2;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -1.f, 2.f, -3.f,
            -12.f, 15.f, -18.f,
            21.f, -24.f, 27.f,
            10.f, -11.f, 12.f);
    }

}

TYPED_TEST(MathMatrix4x3ClassTest, Multiply)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        matrix1 = matrix1 * (-1.f / 2.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f,
            2.f, -2.5f, 3.f,
            -3.5f, 4.f, -4.5f,
            5.f, -5.5f, 6.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        matrix1 = (-1.f / 2.f) * matrix1;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f,
            2.f, -2.5f, 3.f,
            -3.5f, 4.f, -4.5f,
            5.f, -5.5f, 6.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        matrix1 *= (-1.f / 2.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f,
            2.f, -2.5f, 3.f,
            -3.5f, 4.f, -4.5f,
            5.f, -5.5f, 6.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2(
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        matrix1 = matrix1 * matrix2;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -56.f, 64.f, -72.f,
            116.f, -130, 144.f,
            -176.f, 196.f, -216.f,
            216.f, -240.f, 264.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2(
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        matrix1 *= matrix2;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -56.f, 64.f, -72.f,
            116.f, -130, 144.f,
            -176.f, 196.f, -216.f,
            216.f, -240.f, 264.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, Divide)
{
    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        matrix1 = matrix1 / -2.f;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f,
            2.f, -2.5f, 3.f,
            -3.5f, 4.f, -4.5f,
            5.f, -5.5f, 6.f);
    }

    {
        TypeParam matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        matrix1 /= -2.f;

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f,
            2.f, -2.5f, 3.f,
            -3.5f, 4.f, -4.5f,
            5.f, -5.5f, 6.f);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, MakeTranslation)
{
    nn::util::Vector3f translate(-3.f, 1.f, -2.f);

    TypeParam matrix1 = TypeParam::MakeTranslation(translate);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        1.f, 0.f, 0.f,
        0.f, 1.f, 0.f,
        0.f, 0.f, 1.f,
        -3.f, 1.f, -2.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, MakeScale)
{
    nn::util::Vector3f scale(-3.f, 1.f, -2.f);

    TypeParam matrix1 = TypeParam::MakeScale(scale);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -3.f, 0.f, 0.f,
        0.f, 1.f, 0.f,
        0.f, 0.f, -2.f,
        0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, MakeRotation)
{
    {
        nn::util::Vector4f quaternion1(-3.f, 1.f, -2.f, 4.f);
        quaternion1.Normalize();

        TypeParam matrix1 = TypeParam::MakeRotation(quaternion1);

        NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
            0.6666666269302368f, -0.7333333492279053f, 0.1333333551883698f,
            0.3333333432674408f, 0.1333333253860474f, -0.9333333969116210f,
            0.6666667461395264f, 0.6666667461395264f, 0.3333333134651184f,
            0.f, 0.f, 0.f,
            FloatError);
    }

    {
        nn::util::Vector3f rotation(
            30.f / 180.f * nn::util::FloatPi,
            40.f / 180.f * nn::util::FloatPi,
            50.f / 180.f * nn::util::FloatPi);

        TypeParam matrix1 = TypeParam::MakeRotation(rotation);

        NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
            0.4923306107521058f, 0.5867368578910828f, -0.6427397727966309f,
            -0.4567770063877106f, 0.8027404546737671f, 0.3829680085182190f,
            0.7407116293907166f, 0.1049940884113312f, 0.6633203029632568f,
            0.f, 0.f, 0.f,
            FloatError);
    }
}

TYPED_TEST(MathMatrix4x3ClassTest, MakeTranspose)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    matrix1 = TypeParam::MakeTranspose(matrix1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, -4.f, 7.f,
            -2.f, 5.f, -8.f,
            3.f, -6.f, 9.f,
            0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3ClassTest, LookAtRightHanded)
{
    TypeParam matrix1;

    nn::util::Vector3f position(2.f, -2.f, 4.f);
    nn::util::Vector3f target(0.f, 0.f, 0.f);
    nn::util::Vector3f up(0.f, 1.f, 0.f);

    matrix1 = TypeParam::LookAtRightHanded(position, target, up);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.8944271802902222f, 0.1825741827487946f, 0.4082482755184174f,
        0.0000000000000000f, 0.9128708839416504f, -0.4082482755184174f,
        -0.4472135901451111f, 0.3651483654975891f, 0.8164965510368347f,
        -0.0000000000000000f, -0.0000000000000000f, -4.8989791870117187f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3ClassTest, LookToRightHanded)
{
    TypeParam matrix1;

    nn::util::Vector3f position(2.f, -2.f, 4.f);
    nn::util::Vector3f direction(1.f, -1.f, 2.f);
    nn::util::Vector3f up(0.f, 1.f, 0.f);

    matrix1 = TypeParam::LookToRightHanded(position, direction, up);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.8944271802902222f, 0.1825741827487946f, 0.4082482755184174f,
        0.0000000000000000f, 0.9128708839416504f, -0.4082482755184174f,
        -0.4472135901451111f, 0.3651483654975891f, 0.8164965510368347f,
        -0.0000000000000000f, -0.0000000000000000f, -4.8989791870117187f,
        FloatError);
}

// 4x4 行列
template <typename T>
class MathMatrix4x4ClassTest : public ::testing::Test
{
};

typedef ::testing::Types<nn::util::MatrixRowMajor4x4f> Matrix4x4Types;
TYPED_TEST_CASE(MathMatrix4x4ClassTest, Matrix4x4Types);

TYPED_TEST(MathMatrix4x4ClassTest, Constant)
{
    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(TypeParam::Zero(),
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(TypeParam::Identity(),
            1.f, 0.f, 0.f, 0.f,
            0.f, 1.f, 0.f, 0.f,
            0.f, 0.f, 1.f, 0.f,
            0.f, 0.f, 0.f, 1.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, Constractor)
{
    {
        TypeParam matrix(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::Vector4f axisX(1.f, 2.f, 3.f, 4.f);
        nn::util::Vector4f axisY(5.f, 6.f, 7.f, 8.f);
        nn::util::Vector4f axisZ(9.f, 10.f, 11.f, 12.f);
        nn::util::Vector4f axisW(13.f, 14.f, 15.f, 16.f);

        TypeParam matrix(axisX, axisY, axisZ, axisW);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::FloatRowMajor4x4 floatValue = NN_UTIL_FLOAT_ROW_MAJOR_4X4_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        TypeParam matrix(floatValue);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::MatrixRowMajor4x4fType matrixType = NN_UTIL_MATRIX_ROW_MAJOR_4X4F_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        TypeParam matrix(matrixType);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }
}

TYPED_TEST(MathMatrix4x4ClassTest, GetAxis)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4f vector1;

    vector1 = matrix1.GetAxisX();
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.f, 2.f, 3.f, 4.f);

    vector1 = matrix1.GetAxisY();
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 5.f, 6.f, 7.f, 8.f);

    vector1 = matrix1.GetAxisZ();
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 9.f, 10.f, 11.f, 12.f);

    vector1 = matrix1.GetAxisW();
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, GetAxes)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4f axisX, axisY, axisZ, axisW;
    matrix1.Get(&axisX, &axisY, &axisZ, &axisW);

    NNT_UTIL_VECTOR4_EXPECT_EQ(axisX, 1.f, 2.f, 3.f, 4.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(axisY, 5.f, 6.f, 7.f, 8.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(axisZ, 9.f, 10.f, 11.f, 12.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(axisW, 13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, SetAxis)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4f vector1;

    vector1.Set(-17.f, -18.f, -19.f, -20.f);
    matrix1.SetAxisX(vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -17.f, -18.f, -19.f, -20.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

    vector1.Set(-21.f, -22.f, -23.f, -24.f);
    matrix1.SetAxisY(vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -17.f, -18.f, -19.f, -20.f,
            -21.f, -22.f, -23.f, -24.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

    vector1.Set(-25.f, -26.f, -27.f, -28.f);
    matrix1.SetAxisZ(vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -17.f, -18.f, -19.f, -20.f,
            -21.f, -22.f, -23.f, -24.f,
            -25.f, -26.f, -27.f, -28.f,
            13.f, 14.f, 15.f, 16.f);

    vector1.Set(-29.f, -30.f, -31.f, -32.f);
    matrix1.SetAxisW(vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -17.f, -18.f, -19.f, -20.f,
            -21.f, -22.f, -23.f, -24.f,
            -25.f, -26.f, -27.f, -28.f,
            -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, SetAxes)
{
    TypeParam matrix1(
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4f axisX(-17.f, -18.f, -19.f, -20.f);
    nn::util::Vector4f axisY(-21.f, -22.f, -23.f, -24.f);
    nn::util::Vector4f axisZ(-25.f, -26.f, -27.f, -28.f);
    nn::util::Vector4f axisW(-29.f, -30.f, -31.f, -32.f);

    matrix1.Set(axisX, axisY, axisZ, axisW);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -17.f, -18.f, -19.f, -20.f,
            -21.f, -22.f, -23.f, -24.f,
            -25.f, -26.f, -27.f, -28.f,
            -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, Transpose)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    matrix1.Transpose();

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            1.f, -5.f, 9.f, -13.f,
            -2.f, 6.f, -10.f, 14.f,
            3.f, -7.f, 11.f, -15.f,
            -4.f, 8.f, -12.f, 16.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, Inverse)
{
    {
        TypeParam matrix1(
            0.f, 0.f, 0.f, 0.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        EXPECT_FALSE(matrix1.Inverse());

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1(
            1.f, 0.f, 1.f, 2.f,
            -1.f, 1.f, 1.f, 1.f,
            1.f, -1.f, 0.f, 1.f,
            1.f, 1.f, -1.f, -2.f);

        EXPECT_TRUE(matrix1.Inverse());

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            1.f, -1.f, -1.f, 0.f,
            -1.f, 2.f, 2.f, 1.f,
            4.f, -5.f, -7.f, -2.f,
            -2.f, 3.f, 4.f, 1.f);
    }
}

TYPED_TEST(MathMatrix4x4ClassTest, InverseTranspose)
{
    {
        TypeParam matrix1(
            0.f, 0.f, 0.f, 0.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        EXPECT_FALSE(matrix1.InverseTranspose());

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1(
            1.f, 0.f, 1.f, 2.f,
            -1.f, 1.f, 1.f, 1.f,
            1.f, -1.f, 0.f, 1.f,
            1.f, 1.f, -1.f, -2.f);

        EXPECT_TRUE(matrix1.InverseTranspose());

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            1.f, -1.f, 4.f, -2.f,
            -1.f, 2.f, -5.f, 3.f,
            -1.f, 2.f, -7.f, 4.f,
            0.f, 1.f, -2.f, 1.f);
    }
}

TYPED_TEST(MathMatrix4x4ClassTest, Add)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        10.f, -12.f, 14.f, -16.f,
        -18.f, 20.f, -22.f, 24.f,
        -26.f, 28.f, -30.f, 32.f);

    nn::util::MatrixAdd(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        3.f, -6.f, 9.f, -12.f,
        5.f, -6.f, 7.f, -8.f,
        -9.f, 10.f, -11.f, 12.f,
        -39.f, 42.f, -45.f, 48.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, Subtract)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        10.f, -12.f, 14.f, -16.f,
        -18.f, 20.f, -22.f, 24.f,
        -26.f, 28.f, -30.f, 32.f);

    nn::util::MatrixSubtract(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -1.f, 2.f, -3.f, 4.f,
        -15.f, 18.f, -21.f, 24.f,
        27.f, -30.f, 33.f, -36.f,
        13.f, -14.f, 15.f, -16.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, Multiply)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, -1.f / 2.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f, 2.f,
            2.5f, -3.f, 3.5f, -4.f,
            -4.5f, 5.f, -5.5f, 6.f,
             6.5f, -7.f, 7.5f, -8.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        TypeParam matrix2;
        nn::util::MatrixSet(&matrix2,
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            180.f, 192.f, 220.f, 224.f,
            -404.f, -416.f, -508.f, -480.f,
            628.f, 640.f, 796.f, 736.f,
            -852.f, -864.f, -1084.f, -992.f);
    }
}

TYPED_TEST(MathMatrix4x4ClassTest, MakeTranspose)
{
    TypeParam matrix1(
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    matrix1 = TypeParam::MakeTranspose(matrix1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            1.f, -5.f, 9.f, -13.f,
            -2.f, 6.f, -10.f, 14.f,
            3.f, -7.f, 11.f, -15.f,
            -4.f, 8.f, -12.f, 16.f);
}

TYPED_TEST(MathMatrix4x4ClassTest, PerspectiveRightHanded)
{
    TypeParam matrix1;

    matrix1 = TypeParam::PerspectiveRightHanded(2.f, 1.f, 0.1f, 100.0f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        0.1000000014901161f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.2000000029802322f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -1.0010010004043579f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.1001001000404358f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x4ClassTest, PerspectiveFieldOfViewRightHanded)
{
    TypeParam matrix1;

    matrix1 = TypeParam::PerspectiveFieldOfViewRightHanded(60.f / 180.f * nn::util::FloatPi, 16.0f / 9.0f, 0.1f, 100.0f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        0.9742785692214966f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 1.7320507764816284f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -1.0010010004043579f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.1001001000404358f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x4ClassTest, PerspectiveOffCenterRightHanded)
{
    TypeParam matrix1;

    matrix1 = TypeParam::PerspectiveOffCenterRightHanded(-2.f, 4.f, -1.f, 3.f, 10.f, 100.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        3.3333334922790527f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 5.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.3333333432674408f, 0.5000000000000000f, -1.1111111640930176f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -11.1111116409301760f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x4ClassTest, OrthographicRightHanded)
{
    TypeParam matrix1;

    matrix1 = TypeParam::OrthographicRightHanded(6.f, 4.f, 10.f, 100.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        0.3333333432674408f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.5000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.0111111113801599f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.1111111119389534f, 1.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x4ClassTest, OrthographicOffCenterRightHanded)
{
    TypeParam matrix1;

    matrix1 = TypeParam::OrthographicOffCenterRightHanded(-2.f, 4.f, -1.f, 3.f, 10.f, 100.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        0.3333333432674408f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.5000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.0111111113801599f, 0.0000000000000000f,
        -0.3333333432674408f, -0.5000000000000000f, -0.1111111119389534f, 1.0000000000000000f,
        FloatError);
}

// 4x3, 4x4 行列を同時に扱うテスト
class MathMatrixClassTest : public ::testing::Test
{
};

TEST_F(MathMatrixClassTest, Convert)
{
    {
        nn::util::MatrixRowMajor4x3f source(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        nn::util::MatrixRowMajor4x4f dest;
        dest = nn::util::MatrixRowMajor4x3f::ToMatrix4x4f(source);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(dest,
            1.f, 2.f, 3.f, 0.f,
            4.f, 5.f, 6.f, 0.f,
            7.f, 8.f, 9.f, 0.f,
            10.f, 11.f, 12.f, 1.f);
    }

    {
        nn::util::MatrixRowMajor4x4f source(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        nn::util::MatrixRowMajor4x3f dest;
        dest = nn::util::MatrixRowMajor4x4f::ToMatrix4x3f(source);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(dest,
            1.f, 2.f, 3.f,
            5.f, 6.f, 7.f,
            9.f, 10.f, 11.f,
            13.f, 14.f, 15.f);
    }
}

TEST_F(MathMatrixClassTest, Multiply)
{
    {
        nn::util::MatrixRowMajor4x3f matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::MatrixRowMajor4x4f matrix2(
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        matrix2 = matrix1 * matrix2;

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix2,
            76.f, 80.f, 100.f, 96.f,
            -166.f, -164.f, -226.f, -192.f,
            256.f, 248.f, 352.f, 288.f,
            -372.f, -360.f, -508.f, -416.f);
    }

    {
        nn::util::MatrixRowMajor4x4f matrix1(
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixRowMajor4x3f matrix2(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        matrix1 = matrix1 * matrix2;

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            140.f, -160.f, 180.f, -8.f,
            100.f, -104.f, 108.f, -16.f,
            -148.f, 152.f, -156.f, 24.f,
            196.f, -200.f, 204.f, -32.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
