﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/nntest.h>

#include <nn/util/util_LockFreeAtomicType.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

namespace {

struct TestType
{
    int32_t a;
    int32_t array[32];
    int32_t b;
};

TEST(testUtil_LockFreeAtomicType, StoreLoad)
{
    nn::util::LockFreeAtomicType<TestType> atomicType;

    {
        TestType data = {};
        nn::util::StoreToLockFreeAtomicType(&atomicType, data);

        auto loadData = nn::util::LoadFromLockFreeAtomicType(&atomicType);
        EXPECT_EQ(0, loadData.a);
        EXPECT_EQ(0, loadData.b);
    }

    {
        TestType data = {};
        data.a = 1;
        for(int i = 0 ; i < NN_ARRAY_SIZE(data.array) ; ++i)
        {
            data.array[i] = i;
        }
        data.b = 2;
        nn::util::StoreToLockFreeAtomicType(&atomicType, data);

        auto loadData = nn::util::LoadFromLockFreeAtomicType(&atomicType);
        EXPECT_EQ(1, loadData.a);
        for(int i = 0 ; i < NN_ARRAY_SIZE(data.array) ; ++i)
        {
            EXPECT_EQ(i, loadData.array[i]);
        }
        EXPECT_EQ(2, loadData.b);
    }
}

nn::util::LockFreeAtomicType<TestType> g_Data;

nn::os::ThreadType g_WriteThread;
NN_OS_ALIGNAS_THREAD_STACK char g_WriteThreadStack[4096];
nn::os::EventType g_ThreadFinishEvent;

void WriteThreadFunction(void*) NN_NOEXCEPT
{
    TestType data = {};

    while(NN_STATIC_CONDITION(true))
    {
        nn::util::StoreToLockFreeAtomicType(&g_Data, data);

        data.a++;
        data.b++;

        if(nn::os::TryWaitEvent(&g_ThreadFinishEvent))
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5)); // ビジーループ回避のため
    }
}

TEST(testUtil_LockFreeAtomicType, LockFreeAtomicType)
{
    TestType data = {};

    nn::util::StoreToLockFreeAtomicType(&g_Data, data);

    nn::os::InitializeEvent( &g_ThreadFinishEvent, false, nn::os::EventClearMode_AutoClear );

    // ReadのビジーループでもWrite処理が止まらないように、Writeスレッドを高優先度で起動
    auto writeThreadPriority = nn::os::GetThreadPriority(nn::os::GetCurrentThread()) - 1;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::os::CreateThread( &g_WriteThread, WriteThreadFunction, nullptr, g_WriteThreadStack, sizeof(g_WriteThreadStack), writeThreadPriority ));
    nn::os::StartThread( &g_WriteThread );

    for(int i = 0 ; i < 10000 ; ++i)
    {
        data = nn::util::LoadFromLockFreeAtomicType(&g_Data);
        ASSERT_EQ(data.a, data.b);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        if(i % 1000 == 0)
        {
            NN_LOG("i:%d\n", i);
        }
    }

    nn::os::SignalEvent( &g_ThreadFinishEvent );
    nn::os::WaitThread( &g_WriteThread );
    nn::os::DestroyThread( &g_WriteThread );
}

}
