﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <iterator>
#include <iostream>
#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_IntrusiveList.h>
#include "testUtil_ListTestUtil.h"

namespace {

// 侵入型リストの要素が複数のノードを持つ場合のテストを行います。
template <typename TListA, typename TListB>
void TestIntrusiveListMultipleNodes()
{
    NN_STATIC_ASSERT((std::is_same<typename TListA::value_type, typename TListB::value_type>::value));
    typedef typename TListA::value_type ValueType;

    // 2つの異なるリストにノードをぶらさげます
    TListA listA;
    TListB listB;
    const int nodeNumber = 10;
    ValueType items[nodeNumber];
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
    }
    for (int i = 0; i < nodeNumber; i++)
    {
        // リストの先頭と末尾に適当に割り振って追加
        if (i % 3 == 0)
        {
            listA.push_back(items[i]);
            listB.push_back(items[i]);
        }
        else
        {
            listA.push_front(items[i]);
            listB.push_front(items[i]);
        }
    }

    // 検証
    EXPECT_EQ(listA.size(), nodeNumber);
    EXPECT_EQ(listB.size(), nodeNumber);
    {
        int numbers[nodeNumber] = { 8, 7, 5, 4, 2, 9, 6, 3, 0, 1 };
        for (int i = 0; i < nodeNumber / 2; i++)
        {
            EXPECT_EQ(listA.front().Get(), numbers[i]);
            listA.pop_front();
            EXPECT_EQ(listB.front().Get(), numbers[i]);
            listB.pop_front();
        }
        for (int i = nodeNumber / 2; i < nodeNumber; i++)
        {
            EXPECT_EQ(listA.back().Get(), numbers[i]);
            listA.pop_back();
            EXPECT_EQ(listB.back().Get(), numbers[i]);
            listB.pop_back();
        }
    }
    EXPECT_EQ(listA.size(), 0);
    EXPECT_EQ(listB.size(), 0);
}

// 侵入型リストの iterator_to が正しいイテレータを返すことのテストを行います。
template <typename TList>
void TestIntrusiveListIteratorTo()
{
    typedef typename TList::value_type ValueType;

    TList list;
    const int nodeNumber = 3;
    ValueType items[nodeNumber];

    for (int pushIndex = 0; pushIndex < nodeNumber; pushIndex++)
    {
        items[pushIndex].Set(pushIndex);
        list.push_back(items[pushIndex]);

        // 要素を追加したら、全ての要素の iterator_to が正しいことを確認
        for (int i = 0; i <= pushIndex; i++)
        {
            EXPECT_EQ(std::next(list.begin(), i), list.iterator_to(items[i]));
            EXPECT_EQ(std::next(list.cbegin(), i), list.iterator_to(static_cast<const ValueType&>(items[i])));
            EXPECT_EQ(i, list.iterator_to(items[i])->Get());
            EXPECT_EQ(i, list.iterator_to(static_cast<const ValueType&>(items[i]))->Get());
        }
    }
}

// ノードをメンバとして定義するスタイルのテスト用のリストアイテムです
class ItemMemberNodeStyle
{
public:
    ItemMemberNodeStyle()
        : m_NodeA()
        , m_NodeB()
        , m_Number(0)
    {
    }

    void Set(int number)
    {
        m_Number = number;
    }

    int Get() const
    {
        return m_Number;
    }

public:
    nn::util::IntrusiveListNode m_NodeA;
    nn::util::IntrusiveListNode m_NodeB;

private:
    int m_Number;
};

typedef nn::util::IntrusiveList<
    ItemMemberNodeStyle, nn::util::IntrusiveListMemberNodeTraits<
        ItemMemberNodeStyle, &ItemMemberNodeStyle::m_NodeA>> IntrusiveListMemberNodeStyleA;
typedef nn::util::IntrusiveList<
    ItemMemberNodeStyle, nn::util::IntrusiveListMemberNodeTraits<
        ItemMemberNodeStyle, &ItemMemberNodeStyle::m_NodeB>> IntrusiveListMemberNodeStyleB;

struct IntrusiveListBaseNodeTagB
{
};

// ノードを基底クラスとして持つスタイルのテスト用のリストアイテムです
class ItemBaseNodeStyle
    : public nn::util::IntrusiveListBaseNode<ItemBaseNodeStyle>
    , public nn::util::IntrusiveListBaseNode<ItemBaseNodeStyle, IntrusiveListBaseNodeTagB>
{
public:
    ItemBaseNodeStyle()
        : m_Number(0)
    {
    }

    void Set(int number)
    {
        m_Number = number;
    }

    int Get() const
    {
        return m_Number;
    }

private:
    int m_Number;
};

typedef nn::util::IntrusiveList<
    ItemBaseNodeStyle, nn::util::IntrusiveListBaseNodeTraits<
        ItemBaseNodeStyle>> IntrusiveListBaseNodeStyleA;
typedef nn::util::IntrusiveList<
    ItemBaseNodeStyle, nn::util::IntrusiveListBaseNodeTraits<
        ItemBaseNodeStyle, IntrusiveListBaseNodeTagB>> IntrusiveListBaseNodeStyleB;

} //anonymous namespace

namespace nn { namespace util {

template class nn::util::IntrusiveList<
    ItemMemberNodeStyle, nn::util::IntrusiveListMemberNodeTraits<
        ItemMemberNodeStyle, &ItemMemberNodeStyle::m_NodeA>>;

template class nn::util::IntrusiveList<
    ItemBaseNodeStyle, nn::util::IntrusiveListBaseNodeTraits<
        ItemBaseNodeStyle>>;

}} //nn::util

namespace {

// 自身と同じ型を要素の型とする IntrusiveList をメンバとして持てることのテスト用のリストアイテムです
class ItemWithChild
    : public nn::util::IntrusiveListBaseNode<ItemWithChild>
{
public:
    void AddChild(ItemWithChild& child)
    {
        m_Children.push_back(child);
    }

private:
    typedef nn::util::IntrusiveList<ItemWithChild, nn::util::IntrusiveListBaseNodeTraits<ItemWithChild>> ChildList;
    ChildList m_Children;
};


// ノードをメンバとして定義するスタイルのテスト

TEST(IntrusiveListMemberNodeStyle, MultipleNodes)
{
    TestIntrusiveListMultipleNodes<IntrusiveListMemberNodeStyleA, IntrusiveListMemberNodeStyleB>();
}

TEST(IntrusiveListMemberNodeStyle, IteratorTo)
{
    TestIntrusiveListIteratorTo<IntrusiveListMemberNodeStyleA>();
}

TEST(IntrusiveListMemberNodeStyle, Iterator)
{
    TestListIterator<IntrusiveListMemberNodeStyleA>();
}

TEST(IntrusiveListMemberNodeStyle, EraseAndInsert)
{
    TestListEraseAndInsert<IntrusiveListMemberNodeStyleA>();
}

TEST(IntrusiveListMemberNodeStyle, Splice)
{
    TestListSplice<IntrusiveListMemberNodeStyleA>();
}

TEST(IntrusiveListMemberNodeStyle, Duplicated)
{
    TestListDuplicatedInsert<IntrusiveListMemberNodeStyleA>();
}

// ノードを基底クラスとして定義するスタイルのテスト

TEST(IntrusiveListBaseNodeStyle, MultipleNodes)
{
    TestIntrusiveListMultipleNodes<IntrusiveListBaseNodeStyleA, IntrusiveListBaseNodeStyleB>();
}

TEST(IntrusiveListBaseNodeStyle, IteratorTo)
{
    TestIntrusiveListIteratorTo<IntrusiveListBaseNodeStyleA>();
}

TEST(IntrusiveListBaseNodeStyle, Iterator)
{
    TestListIterator<IntrusiveListBaseNodeStyleA>();
}

TEST(IntrusiveListBaseNodeStyle, EraseAndInsert)
{
    TestListEraseAndInsert<IntrusiveListBaseNodeStyleA>();
}

TEST(IntrusiveListBaseNodeStyle, Splice)
{
    TestListSplice<IntrusiveListBaseNodeStyleA>();
}

TEST(IntrusiveListBaseNodeStyle, Duplicated)
{
    TestListDuplicatedInsert<IntrusiveListBaseNodeStyleA>();
}

TEST(IntrusiveListBaseNodeStyle, ChildList)
{
    ItemWithChild parent;
    ItemWithChild child;

    parent.AddChild(child);

    // コンパイルできればよい
    GTEST_SUCCEED();
}

} //anonymous namespace
