﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <iterator>
#include <iostream>
#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_IntrusiveDic.h>
#include "testUtil_DicTestUtil.h"

namespace {

// 侵入型辞書の要素が複数のノードを持つ場合のテストを行います。
template <typename TDicA, typename TDicB>
void TestIntrusiveDicMultipleNodes()
{
    NN_STATIC_ASSERT( ( std::is_same< typename TDicA::value_type, typename TDicB::value_type >::value ) );
    typedef typename TDicA::value_type ValueType;
    typedef typename TDicA::iterator IteratorTypeA;
    typedef typename TDicB::iterator IteratorTypeB;
    typedef typename TDicA::traits_type TraitsTypeA;
    typedef typename TDicB::traits_type TraitsTypeB;

    // 2つの異なる辞書にノードをぶらさげます
    TDicA dicA;
    TDicB dicB;
    const int nodeNumber = 10;
    const char* keys[] = { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9" };
    ValueType items[nodeNumber];

    // 追加
    for ( int i = 0; i < nodeNumber; i++ )
    {
        items[i].Set(i);

        std::pair< IteratorTypeA, bool > resultA = dicA.insert( keys[i], items[i]);
        ASSERT_TRUE( resultA.second );
        EXPECT_EQ( i, resultA.first->Get() );
        EXPECT_EQ( keys[i], TraitsTypeA::GetNode( *resultA.first ).GetKey() );

        std::pair< IteratorTypeB, bool > resultB = dicB.insert( keys[i], items[i]);
        EXPECT_TRUE( resultB.second );
        EXPECT_EQ( i, resultB.first->Get() );
        EXPECT_EQ( keys[i], TraitsTypeB::GetNode( *resultB.first ).GetKey() );

        resultA = dicA.insert( keys[i], items[i]);
        EXPECT_FALSE( resultA.second );
        EXPECT_EQ( i, resultA.first->Get() );

        resultB = dicB.insert( keys[i], items[i]);
        EXPECT_FALSE( resultB.second );
        EXPECT_EQ( i, resultB.first->Get() );

        EXPECT_EQ( i + 1, dicA.size() );
        EXPECT_EQ( i + 1, dicB.size() );
    }

    // 削除
    for ( int i = 0; i < nodeNumber; i++ )
    {
        int resultA = dicA.erase( keys[i] );
        EXPECT_EQ( 1, resultA );
        EXPECT_EQ( dicA.end(), dicA.find( keys[i] ) );
        EXPECT_EQ( nodeNumber - i - 1, dicA.size() );

        int resultB = dicB.erase( keys[i] );
        EXPECT_EQ( 1, resultB );
        EXPECT_EQ( dicB.end(), dicB.find( keys[i] ) );
        EXPECT_EQ( nodeNumber - i - 1, dicB.size() );
    }

    EXPECT_EQ( 0, dicA.size() );
    EXPECT_EQ( 0, dicB.size() );
}

// 侵入型辞書の iterator_to が正しいイテレータを返すことのテストを行います。
template <typename TDic>
void TestIntrusiveDicIteratorTo()
{
    typedef typename TDic::value_type ValueType;

    TDic dic;
    const int nodeNumber = 3;
    const char* keys[] = { "0", "1", "2" };
    ValueType items[nodeNumber];

    for (int pushIndex = 0; pushIndex < nodeNumber; pushIndex++)
    {
        items[pushIndex].Set(pushIndex);
        dic.insert( keys[pushIndex], items[pushIndex]);

        // 要素を追加したら、全ての要素の iterator_to が正しいことを確認
        for (int i = 0; i <= pushIndex; i++)
        {
            EXPECT_EQ( dic.iterator_to( items[i] ), dic.find( keys[i] ) );
            EXPECT_EQ( dic.iterator_to( static_cast<const ValueType&>( items[i] ) ), dic.find( keys[i] ) );
            EXPECT_EQ( i, dic.iterator_to( items[i] )->Get() );
            EXPECT_EQ( i, dic.iterator_to( static_cast<const ValueType&>( items[i] ) )->Get() );
        }
    }
}

// ノードをメンバとして定義するスタイルのテスト用の辞書アイテムです
class ItemMemberNodeStyle
{
public:
    ItemMemberNodeStyle()
        : m_NodeA()
        , m_NodeB()
        , m_Number(0)
    {
    }

    void Set(int number)
    {
        m_Number = number;
    }

    int Get() const
    {
        return m_Number;
    }

public:
    nn::util::IntrusiveDicNode m_NodeA;
    nn::util::IntrusiveDicNode m_NodeB;

private:
    int m_Number;
};

typedef nn::util::IntrusiveDic<
    ItemMemberNodeStyle, nn::util::IntrusiveDicMemberNodeTraits<
        ItemMemberNodeStyle, &ItemMemberNodeStyle::m_NodeA>> IntrusiveDicMemberNodeStyleA;
typedef nn::util::IntrusiveDic<
    ItemMemberNodeStyle, nn::util::IntrusiveDicMemberNodeTraits<
        ItemMemberNodeStyle, &ItemMemberNodeStyle::m_NodeB>> IntrusiveDicMemberNodeStyleB;

struct IntrusiveDicBaseNodeTagB
{
};

// ノードを基底クラスとして持つスタイルのテスト用の辞書アイテムです
class ItemBaseNodeStyle
    : public nn::util::IntrusiveDicBaseNode<ItemBaseNodeStyle>
    , public nn::util::IntrusiveDicBaseNode<ItemBaseNodeStyle, IntrusiveDicBaseNodeTagB>
{
public:
    typedef nn::util::IntrusiveDicBaseNode<ItemBaseNodeStyle> BaseA;
    typedef nn::util::IntrusiveDicBaseNode<ItemBaseNodeStyle, IntrusiveDicBaseNodeTagB> BaseB;

    ItemBaseNodeStyle()
        : m_Number(0)
    {
    }

    void Set(int number)
    {
        m_Number = number;
    }

    int Get() const
    {
        return m_Number;
    }

private:
    int m_Number;
};

typedef nn::util::IntrusiveDic<
    ItemBaseNodeStyle, nn::util::IntrusiveDicBaseNodeTraits<
        ItemBaseNodeStyle>> IntrusiveDicBaseNodeStyleA;
typedef nn::util::IntrusiveDic<
    ItemBaseNodeStyle, nn::util::IntrusiveDicBaseNodeTraits<
        ItemBaseNodeStyle, IntrusiveDicBaseNodeTagB>> IntrusiveDicBaseNodeStyleB;

} //anonymous namespace

namespace nn { namespace util {

template class nn::util::IntrusiveDic<
    ItemMemberNodeStyle, nn::util::IntrusiveDicMemberNodeTraits<
        ItemMemberNodeStyle, &ItemMemberNodeStyle::m_NodeA>>;

template class nn::util::IntrusiveDic<
    ItemBaseNodeStyle, nn::util::IntrusiveDicBaseNodeTraits<
        ItemBaseNodeStyle>>;

}} //nn::util

namespace {

// 自身と同じ型を要素の型とする IntrusiveDic をメンバとして持てることのテスト用の辞書アイテムです
class ItemWithChild
    : public nn::util::IntrusiveDicBaseNode<ItemWithChild>
{
public:
    void AddChild(ItemWithChild& child)
    {
        m_Children.insert( "key", child );
    }

private:
    typedef nn::util::IntrusiveDic<ItemWithChild, nn::util::IntrusiveDicBaseNodeTraits<ItemWithChild>> ChildDic;
    ChildDic m_Children;
};

// ノードをメンバとして定義するスタイルのテスト

TEST(IntrusiveDicMemberNodeStyle, MultipleNodes)
{
    TestIntrusiveDicMultipleNodes<IntrusiveDicMemberNodeStyleA, IntrusiveDicMemberNodeStyleB>();
}

TEST(IntrusiveDicMemberNodeStyle, IteratorTo)
{
    TestIntrusiveDicIteratorTo<IntrusiveDicMemberNodeStyleA>();
}

TEST(IntrusiveDicMemberNodeStyle, Insert)
{
    TestDicInsert<IntrusiveDicMemberNodeStyleA>();
}

TEST(IntrusiveDicMemberNodeStyle, Iterator)
{
    TestDicIterator<IntrusiveDicMemberNodeStyleA>();
}

TEST(IntrusiveDicMemberNodeStyle, Find)
{
    TestDicFind<IntrusiveDicMemberNodeStyleA>();
}

TEST(IntrusiveDicMemberNodeStyle, Erase)
{
    TestDicErase<IntrusiveDicMemberNodeStyleA>();
}


// ノードを基底クラスとして定義するスタイルのテスト

TEST(IntrusiveDicBaseNodeStyle, MultipleNodes)
{
    TestIntrusiveDicMultipleNodes<IntrusiveDicBaseNodeStyleA, IntrusiveDicBaseNodeStyleB>();
}

TEST(IntrusiveDicBaseNodeStyle, IteratorTo)
{
    TestIntrusiveDicIteratorTo<IntrusiveDicBaseNodeStyleA>();
}

TEST(IntrusiveDicBaseNodeStyle, Insert)
{
    TestDicInsert<IntrusiveDicBaseNodeStyleA>();
}

TEST(IntrusiveDicBaseNodeStyle, Iterator)
{
    TestDicIterator<IntrusiveDicBaseNodeStyleA>();
}

TEST(IntrusiveDicBaseNodeStyle, Find)
{
    TestDicFind<IntrusiveDicBaseNodeStyleA>();
}

TEST(IntrusiveDicBaseNodeStyle, Erase)
{
    TestDicErase<IntrusiveDicBaseNodeStyleA>();
}

TEST(IntrusiveDicBaseNodeStyle, ChildDic)
{
    ItemWithChild parent;
    ItemWithChild child;

    parent.AddChild(child);

    // コンパイルできればよい
    GTEST_SUCCEED();
}

} //anonymous namespace
