﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/util/util_Exchange.h>

#include <utility>

TEST(testUtil_Exchange, Int)
{
    {
        int a = 10;
        int b = nn::util::Exchange(&a, 20);
        EXPECT_EQ(a, 20);
        EXPECT_EQ(b, 10);
    }
    {
        int a = 10;
        int c = 20;
        int b = nn::util::Exchange(&a, c);
        EXPECT_EQ(a, 20);
        EXPECT_EQ(b, 10);
    }
    {
        int a = 10;
        const int c = 20;
        int b = nn::util::Exchange(&a, c);
        EXPECT_EQ(a, 20);
        EXPECT_EQ(b, 10);
    }
    {
        int a = 10;
        char c = 20;
        int b = nn::util::Exchange(&a, std::move(c));
        EXPECT_EQ(a, 20);
        EXPECT_EQ(b, 10);
    }
    {
        int a = 10;
        char c = 20;
        int b = nn::util::Exchange(&a, c);
        EXPECT_EQ(a, 20);
        EXPECT_EQ(b, 10);
    }
    {
        int a = 10;
        const char c = 20;
        int b = nn::util::Exchange(&a, c);
        EXPECT_EQ(a, 20);
        EXPECT_EQ(b, 10);
    }
}

class B
{
};

class C : public B
{
};

TEST(testUtil_Exchange, Pointer)
{
    B a;
    B b;
    C c;
    {
        B* p = &a;
        B* q = nn::util::Exchange(&p, &b);
        EXPECT_EQ(p, &b);
        EXPECT_EQ(q, &a);
    }
    {
        B* p = &a;
        B* r = &b;
        B* q = nn::util::Exchange(&p, r);
        EXPECT_EQ(p, &b);
        EXPECT_EQ(q, &a);
    }
    {
        B* p = &a;
        B* const r = &b;
        B* q = nn::util::Exchange(&p, r);
        EXPECT_EQ(p, &b);
        EXPECT_EQ(q, &a);
    }
    {
        B* p = &a;
        C* r = &c;
        B* q = nn::util::Exchange(&p, std::move(r));
        EXPECT_EQ(p, &c);
        EXPECT_EQ(q, &a);
    }
    {
        B* p = &a;
        C* r = &c;
        B* q = nn::util::Exchange(&p, r);
        EXPECT_EQ(p, &c);
        EXPECT_EQ(q, &a);
    }
    {
        B* p = &a;
        C* const r = &c;
        B* q = nn::util::Exchange(&p, r);
        EXPECT_EQ(p, &c);
        EXPECT_EQ(q, &a);
    }
}

TEST(testUtil_Exchange, PointerNull)
{
    B a;
    {
        B* p = &a;
        B* q = nn::util::Exchange(&p, nullptr);
        EXPECT_EQ(p, nullptr);
        EXPECT_EQ(q, &a);
    }
    {
        B* p = &a;
        B* q = nn::util::Exchange(&p, 0);
        EXPECT_EQ(p, nullptr);
        EXPECT_EQ(q, &a);
    }
}

class A
{
    bool m_Valid;
    int m_Value;
public:
    A() NN_NOEXCEPT
        : m_Valid(false)
    {
    }
    explicit A(int x) NN_NOEXCEPT
        : m_Valid(true)
        , m_Value(x)
    {
    }
    A(const A& other) NN_NOEXCEPT
        : m_Valid(other.m_Valid)
        , m_Value(other.m_Value)
    {
    }
    A(A&& other) NN_NOEXCEPT
        : m_Valid(other.m_Valid)
        , m_Value(other.m_Value)
    {
        other.m_Valid = false;
    }
    A& operator=(const A& rhs) NN_NOEXCEPT
    {
        auto tmp(rhs);
        tmp.swap(*this);
        return *this;
    }
    A& operator=(A&& rhs) NN_NOEXCEPT
    {
        auto tmp(std::move(rhs));
        tmp.swap(*this);
        return *this;
    }
    void swap(A& other)
    {
        using std::swap;
        swap(m_Valid, other.m_Valid);
        swap(m_Value, other.m_Value);
    }
    bool IsValid() const NN_NOEXCEPT
    {
        return m_Valid;
    }
    int GetValue() const NN_NOEXCEPT
    {
        return m_Value;
    }
};

TEST(testUtil_Exchange, Alias)
{
    {
        A a(10);
        A b = nn::util::Exchange(&a, A(20));
        EXPECT_TRUE(a.IsValid());
        EXPECT_EQ(a.GetValue(), 20);
        EXPECT_TRUE(b.IsValid());
        EXPECT_EQ(b.GetValue(), 10);
    }
    {
        A a(10);
        A b = nn::util::Exchange(&a, a);
        EXPECT_TRUE(a.IsValid());
        EXPECT_EQ(a.GetValue(), 10);
        EXPECT_TRUE(b.IsValid());
        EXPECT_EQ(b.GetValue(), 10);
    }
}
