﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/util/util_IntrusiveDic.h>

template <typename TDic>
void TestDicInsert()
{
    typedef typename TDic::value_type ValueType;
    typedef typename TDic::iterator IteratorType;
    typedef typename TDic::traits_type TraitsType;

    TDic dic;
    const int nodeNumber = 10;
    const char* keys[] = { "0", "1", "2", "3", "4", "5", "6", "7", "8" , "9" };
    ValueType items[nodeNumber];

    EXPECT_TRUE( dic.empty() );
    EXPECT_EQ( 0, dic.size() );

    // insert をテスト
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        std::pair< IteratorType, bool > result = dic.insert( keys[i], items[i]);

        EXPECT_TRUE( result.second );
        EXPECT_EQ( i, result.first->Get() );
        EXPECT_EQ( keys[i], TraitsType::GetNode( *result.first ).GetKey() );
        result = dic.insert( keys[i], items[i]);
        EXPECT_FALSE( result.second );
        EXPECT_EQ( i, result.first->Get() );
    }

    EXPECT_FALSE( dic.empty() );
    EXPECT_EQ( nodeNumber, dic.size() );

    // clear をテスト
    dic.clear();
    EXPECT_TRUE( dic.empty() );
    EXPECT_EQ( 0, dic.size() );
}


template <typename TDic>
void TestDicIterator()
{
    typedef typename TDic::value_type ValueType;
    typedef typename TDic::iterator IteratorType;
    typedef typename TDic::reverse_iterator ReverseIteratorType;
    typedef typename TDic::traits_type TraitsType;

    TDic dic;
    const int nodeNumber = 10;
    const char* keys[] = { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9" };
    ValueType items[nodeNumber];
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        dic.insert( keys[i], items[i]);
    }

    // イテレータの操作
    {
        IteratorType iter = dic.begin();
        IteratorType end = dic.end();
        int n = 0;
        // インクリメントのチェック
        while ( iter != end )
        {
            EXPECT_EQ( TraitsType::GetNode( *iter ).GetKey(), keys[ iter->Get() ] );
            n++;
            iter++;
        }
        EXPECT_EQ(n, nodeNumber);
        n--;
        iter--;

        // デクリメントのチェック
        while (iter != end )
        {
            EXPECT_EQ( TraitsType::GetNode( *iter ).GetKey(), keys[ iter->Get() ] );
            n--;
            iter--;
        }
        EXPECT_EQ(n, -1);
    }

    // リバースイテレータの操作
    {
        ReverseIteratorType reverseIter = dic.rbegin();
        ReverseIteratorType rend = dic.rend();
        int n = nodeNumber - 1;
        // インクリメントのチェック
        while (reverseIter != rend )
        {
            n--;
            reverseIter++;
        }
        EXPECT_EQ(n, -1);
        n++;
        reverseIter--;

        // デクリメントのチェック
        while ( reverseIter != rend )
        {
            n++;
            reverseIter--;
        }
        EXPECT_EQ(n, nodeNumber);
    }
}

template <typename TDic>
void TestDicFind()
{
    typedef typename TDic::value_type ValueType;
    typedef typename TDic::iterator IteratorType;
    typedef typename TDic::const_iterator ConstIteratorType;
    typedef typename TDic::traits_type TraitsType;

    TDic dic;
    const TDic& cdic = dic;
    const int nodeNumber = 10;
    const char* keys[] = { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9" };
    ValueType items[nodeNumber];
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        dic.insert( keys[i], items[i] );
    }

    EXPECT_EQ( dic.end(), dic.find( "" ) );

    for ( IteratorType iter = dic.begin(), end = dic.end(); iter != end; ++iter )
    {
        IteratorType found = dic.find( TraitsType::GetNode( *iter ).GetKey() );
        ConstIteratorType cfound = cdic.find( TraitsType::GetNode( *iter ).GetKey() );

        EXPECT_EQ( found, iter );
        EXPECT_EQ( cfound, iter );
    }
}

template <typename TDic>
void TestDicErase()
{
    typedef typename TDic::value_type ValueType;
    typedef typename TDic::iterator IteratorType;
    typedef typename TDic::traits_type TraitsType;

    TDic dic;
    const int nodeNumber = 10;
    const char* keys[] = { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9" };
    ValueType items[nodeNumber];
    for (int i = 0; i < nodeNumber; i++)
    {
        items[i].Set(i);
        dic.insert( keys[i], items[i] );
    }

    // キーによる erase をチェック
    int empty = dic.erase( "" );
    EXPECT_EQ( 0, empty );

    for (int i = 0; i < nodeNumber; i++)
    {
        int result = dic.erase( keys[i] );
        EXPECT_EQ( 1, result );
        EXPECT_EQ( dic.end(), dic.find( keys[i] ) );
        EXPECT_EQ( nodeNumber - i - 1, dic.size() );
    }

    // 削除した要素を再度追加
    for (int i = 0; i < nodeNumber; i++)
    {
        std::pair< IteratorType, bool > result = dic.insert( keys[i], items[i]);

        // 削除した要素を再度追加したときに正常にアクセスできるかチェック
        EXPECT_TRUE( result.second );
        EXPECT_EQ( i, result.first->Get() );
        EXPECT_EQ( keys[i], TraitsType::GetNode( *result.first ).GetKey() );
        result = dic.insert( keys[i], items[i]);
        EXPECT_FALSE( result.second );
        EXPECT_EQ( i, result.first->Get() );
    }

    // iterator による erase をチェック
    int n = nodeNumber;
    for ( IteratorType iter = dic.begin(), next = dic.begin(), end = dic.end(); iter != end; iter = next, --n )
    {
        ++next;
        IteratorType erased = dic.erase(iter);
        EXPECT_EQ( end, dic.find( TraitsType::GetNode( *iter ).GetKey() ) );
        EXPECT_EQ( next, erased );
        EXPECT_EQ( n - 1, dic.size() );
    }
}
