﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/util/util_Color.h>

#include "testUtil_Color.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.000001f;
}

//
// カラー
//

// 整数カラー
class Color4u8ClassTest : public ::testing::Test
{
};

TEST_F(Color4u8ClassTest, Constant)
{
    uint8_t elementMax = nn::util::Color4u8::ElementMax;
    uint8_t elementMin = nn::util::Color4u8::ElementMin;
    EXPECT_EQ(0xff, elementMax);
    EXPECT_EQ(0x00, elementMin);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Black(),   0x00, 0x00, 0x00, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Gray(),    0x80, 0x80, 0x80, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::White(),   0xff, 0xff, 0xff, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Red(),     0xff, 0x00, 0x00, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Green(),   0x00, 0xff, 0x00, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Blue(),    0x00, 0x00, 0xff, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Yellow(),  0xff, 0xff, 0x00, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Magenta(), 0xff, 0x00, 0xff, 0xff);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(nn::util::Color4u8::Cyan(),    0x00, 0xff, 0xff, 0xff);
}

TEST_F(Color4u8ClassTest, Constractor)
{
    // uint8_t 引数付コンストラクタ
    {
        nn::util::Color4u8 color(53, 41, 244);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 53, 41, 244, 0xff);
    }

    // Vector4fType 引数付コンストラクタ(オーバーフロー)
    {
        nn::util::Vector4fType vectorTypeValue = NN_UTIL_VECTOR_4F_INITIALIZER(47.f, -2.f, -4.f, 20.f);
        nn::util::Color4u8 color(vectorTypeValue);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 0xff, 0x00, 0x00, 0xff);
    }

    // Vector4fType 引数付コンストラクタ(オーバーフロー)
    {
        nn::util::Vector4fType vectorTypeValue = NN_UTIL_VECTOR_4F_INITIALIZER(-3.f, 2.4f, 2.f, -12.f);
        nn::util::Color4u8 color(vectorTypeValue);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 0x00, 0xff, 0xff, 0x00);
    }

    // Vector4fType 引数付コンストラクタ
    {
        nn::util::Vector4fType vectorTypeValue = NN_UTIL_VECTOR_4F_INITIALIZER(64.9f / 255.0f, 14.1f / 255.0f, 83.8f / 255.0f, 124.f / 255.0f);
        nn::util::Color4u8 color(vectorTypeValue);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 64, 14, 83, 124);
    }

    // Vector4fType 引数付コンストラクタ
    {
        nn::util::Color4f color4fValue(64.9f / 255.0f, 14.1f / 255.0f, 83.8f / 255.0f, 124.f / 255.0f);
        nn::util::Color4u8 color(color4fValue);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 64, 14, 83, 124);
    }

    // Unorm8x4 引数付コンストラクタ
    {
        nn::util::Unorm8x4 UnormValue = {{0x41, 0x8f, 0x39, 0x0a}};
        nn::util::Color4u8 color(UnormValue);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 0x41, 0x8f, 0x39, 0x0a);
    }

    // Unorm8x4 引数付コンストラクタ
    {
        nn::util::Color4u8 color4u8Value(0x41, 0x8f, 0x39, 0x0a);
        nn::util::Color4u8 color(color4u8Value);

        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 0x41, 0x8f, 0x39, 0x0a);
    }
}

TEST_F(Color4u8ClassTest, GetRgba)
{
    nn::util::Color4u8 color(53, 81, 244, 52);

    EXPECT_EQ(53, color.GetR());
    EXPECT_EQ(81, color.GetG());
    EXPECT_EQ(244, color.GetB());
    EXPECT_EQ(52, color.GetA());
}

TEST_F(Color4u8ClassTest, Get)
{
    nn::util::Color4u8 color(76, 211, 61, 2);

    uint8_t r, g, b, a;
    color.Get(&r, &g, &b, &a);

    EXPECT_EQ(76, r);
    EXPECT_EQ(211, g);
    EXPECT_EQ(61, b);
    EXPECT_EQ(2, a);
}

TEST_F(Color4u8ClassTest, SetRgba)
{
    nn::util::Color4u8 color;
    color.Set(0, 0, 0, 0);

    color.SetR(76);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 76, 0, 0, 0);

    color.SetG(211);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 76, 211, 0, 0);

    color.SetB(61);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 76, 211, 61, 0);

    color.SetA(2);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 76, 211, 61, 2);
}

TEST_F(Color4u8ClassTest, Set)
{
    {
        nn::util::Color4u8 color;

        color.Set(76, 211, 61, 2);
        NNT_UTIL_COLOR4U8_EXPECT_EQ(color, 76, 211, 61, 2);
    }
}

TEST_F(Color4u8ClassTest, Substitution)
{
    nn::util::Color4u8 color1(76, 211, 61, 2);
    nn::util::Color4u8 color2(color1);

    color1.Black();

    color1 = color2;

    NNT_UTIL_COLOR4U8_EXPECT_EQ(color1, 76, 211, 61, 2);
}

TEST_F(Color4u8ClassTest, Equal)
{
    nn::util::Color4u8 color1(76, 211, 61, 2);
    nn::util::Color4u8 color2(76, 211, 61, 2);
    nn::util::Color4u8 color3(76, 211, 61, 255);

    EXPECT_EQ(true, color1 == color2);
    EXPECT_EQ(false, color1 == color3);
    EXPECT_EQ(false, color1 != color2);
    EXPECT_EQ(true, color1 != color3);
}

TEST_F(Color4u8ClassTest, Lerp)
{
    nn::util::Color4u8 color1(80, 200, 60, 0);
    nn::util::Color4u8 color2(50, 80, 240, 50);

    nn::util::Color4u8 color3;

    color3 = nn::util::Color4u8::Lerp(color1, color2, 0.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color3, 80, 200, 60, 0);

    color3 = nn::util::Color4u8::Lerp(color1, color2, 1.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color3, 50, 80, 240, 50);

    color3 = nn::util::Color4u8::Lerp(color1, color2, 0.5f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color3, 65, 140, 150, 25);

    color3.SetLerp(color1, color2, 0.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color3, 80, 200, 60, 0);

    color3.SetLerp(color1, color2, 1.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color3, 50, 80, 240, 50);

    color3.SetLerp(color1, color2, 0.5f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color3, 65, 140, 150, 25);
}

TEST_F(Color4u8ClassTest, CorrectGamma)
{
    nn::util::Color4u8 color1(80, 200, 60, 200);
    nn::util::Color4u8 color2;

    color2 = color1.CorrectGamma(1.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color2, 80, 200, 60, 200);

    color2 = color1.CorrectGamma(2.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color2, 25, 156, 14, 200);

    color2 = color1.CorrectGamma(-2.f);
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color2, 255, 255, 255, 200);
}

TEST_F(Color4u8ClassTest, ToSrgb)
{
    nn::util::Color4u8 color1(80, 200, 60, 200);
    nn::util::Color4u8 color2;

    color2 = color1.ToSrgb();
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color2, 150, 228, 132, 200);
}

TEST_F(Color4u8ClassTest, ToLinear)
{
    nn::util::Color4u8 color1(80, 200, 60, 200);
    nn::util::Color4u8 color2;

    color2 = color1.ToLinear();
    NNT_UTIL_COLOR4U8_EXPECT_EQ(color2, 19, 149, 10, 200);
}

TEST_F(Color4u8ClassTest, Transparent)
{
    nn::util::Color4u8 color(0, 0, 0, 0);
    EXPECT_EQ(true, color.IsTransparent());

    color.SetA(100);
    EXPECT_EQ(false, color.IsTransparent());
}

// 浮動小数カラー
class Color4fClassTest : public ::testing::Test
{
};

TEST_F(Color4fClassTest, Constant)
{
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Black(),   0.0f, 0.0f, 0.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Gray(),    0.5f, 0.5f, 0.5f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::White(),   1.0f, 1.0f, 1.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Red(),     1.0f, 0.0f, 0.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Green(),   0.0f, 1.0f, 0.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Blue(),    0.0f, 0.0f, 1.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Yellow(),  1.0f, 1.0f, 0.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Magenta(), 1.0f, 0.0f, 1.0f, 1.0f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(nn::util::Color4f::Cyan(),    0.0f, 1.0f, 1.0f, 1.0f);
}

TEST_F(Color4fClassTest, Constractor)
{
    // float 引数付コンストラクタ
    {
        nn::util::Color4f color(-3.f, 1.f, -2.f);

        NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 1.f, -2.f, 1.f);
    }

    // Unorm8x4 引数付コンストラクタ
    {
        nn::util::Unorm8x4 unormValue = {{60, 160, 2, 210}};
        nn::util::Color4f color(unormValue);

        NNT_UTIL_COLOR4F_EXPECT_NEARLY_EQ(color, 0.23529411764f, 0.62745098039f, 0.00784313725f, 0.82352941176f, FloatError);
    }

    // Unorm8x4 引数付コンストラクタ
    {
        nn::util::Color4u8 color4u8Value(60, 160, 2, 210);
        nn::util::Color4f color(color4u8Value);

        NNT_UTIL_COLOR4F_EXPECT_NEARLY_EQ(color, 0.23529411764f, 0.62745098039f, 0.00784313725f, 0.82352941176f, FloatError);
    }

    // Vector4fType 引数付コンストラクタ
    {
        nn::util::Vector4fType vectorTypeValue = NN_UTIL_VECTOR_4F_INITIALIZER(-3.f, 1.f, -2.f, 4.f);
        nn::util::Color4f color(vectorTypeValue);

        NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 1.f, -2.f, 4.f);
    }

    // Vector4fType 引数付コンストラクタ
    {
        nn::util::Color4f color4fValue(-3.f, 1.f, -2.f, 4.f);
        nn::util::Color4f color(color4fValue);

        NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 1.f, -2.f, 4.f);
    }
}

TEST_F(Color4fClassTest, GetRgba)
{
    nn::util::Color4f color(-3.f, 1.f, -2.f, 4.f);

    EXPECT_EQ(-3.f, color.GetR());
    EXPECT_EQ(1.f, color.GetG());
    EXPECT_EQ(-2.f, color.GetB());
    EXPECT_EQ(4.f, color.GetA());
}

TEST_F(Color4fClassTest, Get)
{
    nn::util::Color4f color(-3.f, 1.f, -2.f, 4.f);

    float r, g, b, a;
    color.Get(&r, &g, &b, &a);

    EXPECT_EQ(-3.f, r);
    EXPECT_EQ(1.f, g);
    EXPECT_EQ(-2.f, b);
    EXPECT_EQ(4.f, a);
}

TEST_F(Color4fClassTest, SetRgba)
{
    nn::util::Color4f color;
    color.Set(0.0f, 0.0f, 0.0f, 0.0f);

    color.SetR(-3.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 0.0f, 0.0f, 0.0f);

    color.SetG(1.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 1.f, 0.0f, 0.0f);

    color.SetB(-2.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 1.f, -2.f, 0.0f);

    color.SetA(4.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(Color4fClassTest, Set)
{
    {
        nn::util::Color4f color;

        color.Set(0.3f, 1.f, 0.2f, 0.4f);
        NNT_UTIL_COLOR4F_EXPECT_EQ(color, 0.3f, 1.f, 0.2f, 0.4f);
    }
}

TEST_F(Color4fClassTest, Substitution)
{
    nn::util::Color4f color1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Color4f color2(color1);

    color1.Black();

    color1 = color2;

    NNT_UTIL_COLOR4F_EXPECT_EQ(color1, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(Color4fClassTest, Equal)
{
    nn::util::Color4f color1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Color4f color2(-3.f, 1.f, -2.f, 4.f);
    nn::util::Color4f color3(-3.f, 1.f, -2.f, 40.f);

    EXPECT_EQ(true, color1 == color2);
    EXPECT_EQ(false, color1 == color3);
    EXPECT_EQ(false, color1 != color2);
    EXPECT_EQ(true, color1 != color3);
}

TEST_F(Color4fClassTest, Lerp)
{
    nn::util::Color4f color1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Color4f color2(-6.f, 2.f, 4.f, -8.f);

    nn::util::Color4f color3;

    color3 = nn::util::Color4f::Lerp(color1, color2, 0.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color3, -3.f, 1.f, -2.f, 4.f);

    color3 = nn::util::Color4f::Lerp(color1, color2, 1.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color3, -6.f, 2.f, 4.f, -8.f);

    color3 = nn::util::Color4f::Lerp(color1, color2, 0.5f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color3, -4.5f, 1.5f, 1.f, -2.f);

    color3.SetLerp(color1, color2, 0.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color3, -3.f, 1.f, -2.f, 4.f);

    color3.SetLerp(color1, color2, 1.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color3, -6.f, 2.f, 4.f, -8.f);

    color3.SetLerp(color1, color2, 0.5f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color3, -4.5f, 1.5f, 1.f, -2.f);
}

TEST_F(Color4fClassTest, CorrectGamma)
{
    nn::util::Color4f color1(0.3f, 1.f, 0.2f, 0.4f);
    nn::util::Color4f color2;

    color2 = color1.CorrectGamma(1.f);
    NNT_UTIL_COLOR4F_EXPECT_EQ(color2, 0.3f, 1.f, 0.2f, 0.4f);

    color2 = color1.CorrectGamma(3.f);
    NNT_UTIL_COLOR4F_EXPECT_NEARLY_EQ(color2, 0.027f, 1.f, 0.008f, 0.4f, FloatError);
}

TEST_F(Color4fClassTest, ToSrgb)
{
    nn::util::Color4f color1(0.3f, 1.f, 0.2f, 0.4f);
    nn::util::Color4f color2;

    color2 = color1.ToSrgb();
    NNT_UTIL_COLOR4F_EXPECT_NEARLY_EQ(color2, 0.57853260908f, 1.f, 0.48115650505f, 0.4f, FloatError);
}

TEST_F(Color4fClassTest, ToLinear)
{
    nn::util::Color4f color1(0.3f, 1.f, 0.2f, 0.4f);
    nn::util::Color4f color2;

    color2 = color1.ToLinear();
    NNT_UTIL_COLOR4F_EXPECT_NEARLY_EQ(color2, 0.07074027770f, 1.f, 0.02899118654f, 0.4f, FloatError);
}

TEST_F(Color4fClassTest, Transparent)
{
    nn::util::Color4f color(0.f, 0.f, 0.f, 0.f);
    EXPECT_EQ(true, color.IsTransparent());

    color.SetA(3.f);
    EXPECT_EQ(false, color.IsTransparent());

    color.SetA(-4.f);
    EXPECT_EQ(true, color.IsTransparent());

    color.Set(3.f, -4.f, -4.f, 0.f);
    EXPECT_EQ(true, color.IsTransparent());
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
