﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/util/util_BitFlagSet.h>

using namespace nn::util;

template <typename T>
void TestAllBitOn(const T& set)
{
    EXPECT_TRUE(set.IsAllOn());
    EXPECT_FALSE(set.IsAllOff());
    EXPECT_TRUE(set.IsAnyOn());
    EXPECT_EQ(set.GetCount(), set.CountPopulation());

    int countOn = 0;
    for (int i = 0; i < set.GetCount(); i++)
    {
        if (set[i])
        {
            EXPECT_TRUE(set.Test(i));
            countOn++;
        }
    }

    EXPECT_EQ(set.GetCount(), countOn);
}

template <typename T>
void TestSomeBitOn(const T& set, int expectOn)
{
    EXPECT_FALSE(set.IsAllOn());
    EXPECT_FALSE(set.IsAllOff());
    EXPECT_TRUE(set.IsAnyOn());
    EXPECT_EQ(expectOn, set.CountPopulation());

    int countOn = 0;
    for (int i = 0; i < set.GetCount(); i++)
    {
        if (set[i])
        {
            EXPECT_TRUE(set.Test(i));
            countOn++;
        }
    }

    EXPECT_EQ(expectOn, countOn);
}

template <typename T>
void TestAllBitOff(const T& set)
{
    EXPECT_FALSE(set.IsAllOn());
    EXPECT_TRUE(set.IsAllOff());
    EXPECT_FALSE(set.IsAnyOn());
    EXPECT_EQ(0, set.CountPopulation());

    for (int i = 0; i < set.GetCount(); i++)
    {
        EXPECT_FALSE(set[i]);
        EXPECT_FALSE(set.Test(i));
    }
}

template<int N>
void TestFunc()
{
    SCOPED_TRACE(testing::Message() << "N = " << N);

    typedef typename BitFlagSet<N>::template Flag<N / 2> Median;
    typedef typename BitFlagSet<N>::template Flag<N - 1> Last;

    auto b1 = MakeBitFlagSet<N>();
    auto b2 = MakeBitFlagSet<N>();
    EXPECT_EQ(N, b1.GetCount());
    TestAllBitOff(b1);
    EXPECT_TRUE(b1 == b2);
    EXPECT_FALSE(b1 != b2);

    b1.Set(0);
    EXPECT_FALSE(b1 == b2);
    EXPECT_TRUE(b1 != b2);
    TestSomeBitOn(b1, 1);
    EXPECT_TRUE(b1[0]);

    b1.Reset(0);
    EXPECT_TRUE(b1 == b2);
    EXPECT_FALSE(b1 != b2);
    TestAllBitOff(b1);

    int median = Median::Index;
    b1.Set(median);
    TestSomeBitOn(b1, 1);
    b1.Set(median, false);
    EXPECT_TRUE(b1 == b2);
    EXPECT_FALSE(b1 != b2);
    TestAllBitOff(b1);
    EXPECT_FALSE(b1[median]);

    b1.template Set<Median>();

    auto b3 = b1 & b2;
    TestAllBitOff(b3);
    EXPECT_FALSE(b3[median]);

    b3 = b1 | b2;
    TestSomeBitOn(b3, 1);
    EXPECT_TRUE(b3[median]);

    b3 = b1 ^ b2;
    TestSomeBitOn(b3, 1);
    EXPECT_TRUE(b3[median]);

    auto b4 = MakeBitFlagSet<N>();
    b4 &= b1;
    TestAllBitOff(b4);
    EXPECT_FALSE(b4[median]);

    auto b5 = MakeBitFlagSet<N>();
    b5 |= b1;
    TestSomeBitOn(b5, 1);
    EXPECT_TRUE(b5[median]);

    auto b6 = MakeBitFlagSet<N>();
    b6 ^= b1;
    TestSomeBitOn(b6, 1);
    EXPECT_TRUE(b6[median]);

    b6 ^= b1;
    TestAllBitOff(b6);
    EXPECT_FALSE(b6[median]);

    b2.Set(median);
    EXPECT_TRUE(b1 == b2);
    EXPECT_FALSE(b1 != b2);

    b3 = b1 ^ b2;
    TestAllBitOff(b3);

    b1.Reset();
    TestAllBitOff(b1);

    b1.Flip();
    TestAllBitOn(b1);

    b2.Set();
    TestAllBitOn(b2);
    EXPECT_TRUE(b1 == b2);

    b1.Flip(N - 1);
    TestSomeBitOn(b1, N - 1);
    EXPECT_TRUE(b1[0]);
    EXPECT_FALSE(b1.template Test<Last>());
    EXPECT_TRUE(b1 != b2);
}

template<int N>
void TestMaskFunc()
{
    SCOPED_TRACE(testing::Message() << "N = " << N);

    typedef typename BitFlagSet<N>::template Flag<0    > Head;
    typedef typename BitFlagSet<N>::template Flag<N / 2> Body;
    typedef typename BitFlagSet<N>::template Flag<N - 1> Tail;

    EXPECT_EQ(1, Head::Mask.CountPopulation());
    EXPECT_EQ(1, Body::Mask.CountPopulation());
    EXPECT_EQ(1, Tail::Mask.CountPopulation());

    EXPECT_TRUE(Head::Mask.template Test<Head>());
    EXPECT_TRUE(Body::Mask.template Test<Body>());
    EXPECT_TRUE(Tail::Mask.template Test<Tail>());
}

TEST(BitFlagSet, Basic)
{
    TestFunc<2>();
    TestFunc<4>();
    TestFunc<31>();
    TestFunc<32>();
    TestFunc<33>();
    TestFunc<48>();
    TestFunc<63>();
    TestFunc<64>();
    TestFunc<65>();
    TestFunc<80>();
    TestFunc<123>();
    TestFunc<124>();
    TestFunc<125>();
}

TEST(BitFlagSet, Mask)
{
    TestMaskFunc<1>();
    TestMaskFunc<2>();
    TestMaskFunc<31>();
    TestMaskFunc<32>();
    TestMaskFunc<33>();
    TestMaskFunc<63>();
    TestMaskFunc<64>();
}
