﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nn/util/util_BinaryFormat.h>

namespace {

struct SampleFile
{
    nn::util::BinaryFileHeader header;

    // セクション 0
    nn::util::BinPtr ptr0;
    nn::util::BinPtr ptr1;
    // セクション 1
    nn::util::BinPtr ptr2;
    nn::util::BinPtr ptr3;
};

TEST( RelTableTest, RelocationTable )
{
    typedef nn::util::RelocationTable::Section Section;
    typedef nn::util::RelocationTable::Entry Entry;

    int sectionCount = 2;
    int entryCount = 4;
    size_t fileSize = sizeof( SampleFile ) + nn::util::RelocationTable::CalculateSize( sectionCount , entryCount );
    size_t outOfFileSize = sizeof( nn::util::BinPtr ) * 2; // セクション1（ ptr2, ptr3 ）をファイル外領域に配置する領域のサイズ
    SampleFile* pFile = static_cast< SampleFile* >( malloc( fileSize + outOfFileSize ) );
    void* pOutOfFile = nn::util::BytePtr( pFile, fileSize ).Get();
    std::memset( pFile, 0, fileSize + outOfFileSize );

    nn::util::RelocationTable* pRelt = nn::util::BytePtr( pFile, sizeof( SampleFile ) ).Get< nn::util::RelocationTable >();

    // 全体の設定
    {
        pRelt->SetSignature();
        pRelt->_sectionCount = sectionCount;
        pRelt->_position = sizeof( SampleFile );
    }

    // BinPtr の設定
    // ptr0 → ptr1 → ptr2 → ptr3 → ptr0 と指すように設定します。
    {
        pFile->ptr0.SetOffset( pFile, &pFile->ptr1 );
        pFile->ptr1.SetOffset( pFile, &pFile->ptr2 );
        pFile->ptr2.SetOffset( pFile, &pFile->ptr3 );
        pFile->ptr3.SetOffset( pFile, &pFile->ptr0 );
    }

    // Section の設定
    {
        uint32_t sec0Size = static_cast< uint32_t >( nn::util::BytePtr( pFile ).Distance( &pFile->ptr2 ) );

        Section& sec0 = pRelt->_sections[0];
        sec0._padding = 0;
        sec0._entryCount = 2;
        sec0._position = 0;
        sec0._entryIndex = 0;
        sec0._size = sec0Size;

        Section& sec1 = pRelt->_sections[1];
        sec1._padding = 0;
        sec1._entryCount = 2;
        sec1._position = sec0Size;
        sec1._entryIndex = 2;
        sec1._size = static_cast< uint32_t >( 2 * sizeof( nn::util::BinPtr ) );
    }

    // Entry の設定
    {
        ptrdiff_t entry_size = static_cast< ptrdiff_t >( sizeof( Entry ) );

        // Section 0 を指すポインタの Entry

        // ptr0 の Entry
        Entry* pEntry0 = nn::util::BytePtr( &pRelt->_sections[2] ).Get< Entry >();
        pEntry0->_position = static_cast< uint32_t >( nn::util::BytePtr(pFile).Distance( &pFile->ptr0 ) );
        pEntry0->_offsetCount = 1;
        pEntry0->_structCount = 1;
        pEntry0->_paddingCount = 0;

        // ptr3 の Entry
        Entry* pEntry3 = nn::util::BytePtr( pEntry0, entry_size ).Get< Entry >();
        pEntry3->_position = static_cast< uint32_t >( nn::util::BytePtr(pFile).Distance( &pFile->ptr3 ) );
        pEntry3->_offsetCount = 1;
        pEntry3->_structCount = 1;
        pEntry3->_paddingCount = 0;

        // Section 1 を指すポインタの Entry

        // ptr1 の Entry
        Entry* pEntry1 = nn::util::BytePtr( pEntry3, entry_size ).Get< Entry >();
        pEntry1->_position = static_cast< uint32_t >( nn::util::BytePtr(pFile).Distance( &pFile->ptr1 ) );
        pEntry1->_offsetCount = 1;
        pEntry1->_structCount = 1;
        pEntry1->_paddingCount = 0;

        // ptr2 の Entry
        Entry* pEntry2 = nn::util::BytePtr( pEntry1, entry_size ).Get< Entry >();
        pEntry2->_position = static_cast< uint32_t >( nn::util::BytePtr(pFile).Distance( &pFile->ptr2 ) );
        pEntry2->_offsetCount = 1;
        pEntry2->_structCount = 1;
        pEntry2->_paddingCount = 0;
    }

    // Relocate, Unrelocate の検証

    nn::util::BinPtr::difference_type ptr0offset = pFile->ptr0.GetOffset();
    nn::util::BinPtr::difference_type ptr1offset = pFile->ptr1.GetOffset();
    nn::util::BinPtr::difference_type ptr2offset = pFile->ptr2.GetOffset();
    nn::util::BinPtr::difference_type ptr3offset = pFile->ptr3.GetOffset();

    EXPECT_EQ( pRelt->_signature.GetPacked(), nn::util::RelocationTable::PackedSignature );

    // Relocate でポインタに変える
    {
        pRelt->Relocate();

        EXPECT_EQ( &pFile->ptr1, pFile->ptr0.Get() );
        EXPECT_EQ( &pFile->ptr2, pFile->ptr1.Get() );
        EXPECT_EQ( &pFile->ptr3, pFile->ptr2.Get() );
        EXPECT_EQ( &pFile->ptr0, pFile->ptr3.Get() );
    }

    // Unrelocate でオフセットに戻す
    {
        pRelt->Unrelocate();

        EXPECT_EQ( ptr0offset, pFile->ptr0.GetOffset() );
        EXPECT_EQ( ptr1offset, pFile->ptr1.GetOffset() );
        EXPECT_EQ( ptr2offset, pFile->ptr2.GetOffset() );
        EXPECT_EQ( ptr3offset, pFile->ptr3.GetOffset() );
    }

    // ファイル外領域への配置チェック
    {
        // セクション1( ptr2, ptr3 )を配置する先の、ファイル範囲外のポインタを設定する
        pRelt->_sections[1].SetPtr( pOutOfFile );

        // Relocate でポインタに変える
        pRelt->Relocate();

        // ptr0、ptr1 はそのまま
        EXPECT_EQ( &pFile->ptr1, pFile->ptr0.Get() );
        // ptr2 がファイル外領域に飛んでいるので、ptr1 はファイル外領域の先頭を指すはず
        EXPECT_EQ( pOutOfFile, pFile->ptr1.Get() );
        // ptr2, ptr3 はそれぞれファイル外領域に飛んでいる
        EXPECT_EQ( nn::util::BytePtr( pOutOfFile, sizeof( nn::util::BinPtr ) ).Get(), pFile->ptr2.Get() );
        // ptr3 は元の ptr0 を指すはず
        EXPECT_EQ( &pFile->ptr0, pFile->ptr3.Get() );

        // セクション1の範囲をコピーしてファイル外領域から利用する
        std::memcpy( pOutOfFile, &pFile->ptr2 , sizeof( nn::util::BinPtr) * 2);

        EXPECT_EQ( nn::util::BytePtr( pOutOfFile, sizeof( nn::util::BinPtr ) ).Get(), static_cast< nn::util::BinPtr* >( pOutOfFile )->Get() );
        EXPECT_EQ( &pFile->ptr0, nn::util::BytePtr( pOutOfFile, sizeof( nn::util::BinPtr ) ).Get< nn::util::BinPtr >()->Get() );
    }

    // ファイル外領域から戻すチェック
    {
        // Unrelocate はファイル内のみ変更するので、ファイル外領域からコピーしてくる必要はない
        pRelt->Unrelocate();

        // オフセットが元に戻っていることを確認する
        EXPECT_EQ( ptr0offset, pFile->ptr0.GetOffset() );
        EXPECT_EQ( ptr1offset, pFile->ptr1.GetOffset() );
        EXPECT_EQ( ptr2offset, pFile->ptr2.GetOffset() );
        EXPECT_EQ( ptr3offset, pFile->ptr3.GetOffset() );
    }

    free( pFile );
}

} // anonymous namespace
