﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UsbDiag_Theme.h"
#include "UsbDiag_MenuView.h"
#include "UsbDiag_UsbModel.h"

namespace nnt {
namespace usb {

MenuView::MenuView(const glv::Rect& rect) NN_NOEXCEPT
    : glv::View(rect)
    , m_Selector(glv::Rect(MenuWidth, MenuEntryHeight))
    , m_Frame(glv::Rect(MenuWidth, rect.height()))
{
    enable(glv::Property::HitTest     |
           glv::Property::Controllable);

    disable(glv::Property::DrawBorder |
            glv::Property::Animate    |
            glv::Property::Visible    );

    *this << (m_Frame << m_Selector);

    cloneStyle().colors().back.set(0.0f, 0.6f);

    m_Frame.pos(glv::Place::TR, 0, 0);
    m_Selector.pos(glv::Place::TL, 0, 20);

    m_Frame.disable(glv::Property::DrawBorder     |
                    glv::Property::FocusHighlight |
                    glv::Property::FocusToTop     );
    m_Frame.cloneStyle().colors().back.set(MenuInactiveBackColor);

    g_Model.attach(Show, Event_ShowMenu, this);
    g_Model.attach(Hide, Event_HideMenu, this);

    m_Selector.attach(
        [](const glv::Notification& n) {
            auto *scene = const_cast<Scene*>(n.data<Scene>());
            g_Model.ToggleMenu(false);
            scene->Replace();
        },
        glv::Update::Selection,
        this
    );
}

MenuView::~MenuView() NN_NOEXCEPT
{
    g_Model.detach(Hide, Event_HideMenu, this);
    g_Model.detach(Show, Event_ShowMenu, this);
}

void
MenuView::Select(int index) NN_NOEXCEPT
{
    m_Selector.Select(index);
}

bool
MenuView::onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT
{
    bool bubble = true;

    switch (e)
    {
    case glv::Event::DebugPadDown:
        {
            const auto& pe = g.getDebugPadEvent();
            if (pe.IsButtonDown<glv::DebugPadEventType::Button::Cancel>())
            {
                g_Model.ToggleMenu(false);
                bubble = false;
            }
        }
        break;

    case glv::Event::BasicPadDown:
        {
            const auto& pe = g.getBasicPadEvent();
            if (pe.IsButtonDown<glv::BasicPadEventType::Button::Cancel>())
            {
                g_Model.ToggleMenu(false);
                bubble = false;
            }
        }
        break;

    case glv::Event::MouseDown:
        g_Model.ToggleMenu(false);
        bubble = false;
        break;

    default:
        break;
    }

    return bubble;
}

void
MenuView::onAnimate(double dsec)
{
    glv::space_t delta = m_Speed * dsec;

    bringToFront();

    if (m_Speed > 0) // show
    {
        if (m_Frame.left() + delta >= 0) // done
        {
            disable(glv::Property::Animate);
            m_Frame.pos(glv::Place::TL, 0, 0);
        }
        else
        {
            m_Frame.posAdd(delta, 0);
        }
    }
    else // hide
    {
        if (m_Frame.right() + delta <= 0) // done
        {
            disable(glv::Property::Animate | glv::Property::Visible);
            m_Frame.pos(glv::Place::TR, 0, 0);
        }
        else
        {
            m_Frame.posAdd(delta, 0);
        }
    }
}

void
MenuView::AttachScene(Scene& scene) NN_NOEXCEPT
{
    m_Selector.Add(scene);
}

void
MenuView::Show(const glv::Notification& n)
{
    MenuView *self = n.receiver<MenuView>();

    //self->colors().back.set(0.0f, 0.6f);
    self->enable(glv::Property::DrawBack);

    self->enable(glv::Property::Animate |
                 glv::Property::Visible |
                 glv::Property::HitTest );

    self->m_Speed = AnimationSpeed;

    // auto& g = static_cast< glv::GLV& >( self->root() );
    // g.setFocus(self->m_Frame.focusableView());
    self->m_Selector.UpdateFocus();

    Scene::GetCurrent()->disable(glv::Property::Controllable);
}

void
MenuView::Hide(const glv::Notification& n)
{
    MenuView *self = n.receiver<MenuView>();

    //self->colors().back.set(0.0f, 0.0f);
    self->disable(glv::Property::DrawBack);

    self->enable(glv::Property::Animate |
                 glv::Property::Visible );

    self->disable(glv::Property::HitTest);

    self->m_Speed = -AnimationSpeed;

    auto& g = static_cast< glv::GLV& >( self->root() );

    Scene::GetCurrent()->enable(glv::Property::Controllable);
    auto *view = Scene::GetCurrent()->focusableView();
    g.setFocus(view ? view : Scene::GetCurrent());
}

} // ~usb
} // ~nnt
