﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Fx3Methods/Fx3.h"

namespace nnt {
namespace usb {
namespace hs {

const Fx3DeviceMode deviceModes[] =
{
    {FX3_DEVICE_FULL_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_INTR_3},
    {FX3_DEVICE_HIGH_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_INTR_3},
    {FX3_DEVICE_SUPER_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_INTR_3},
};

const int intrTestValues[] =
    {
        1,
        2,
        3,
        511,
        512,
        513,
        4095,
        4096,
        4097,
        20 * 1024 - 1,
        20 * 1024,
        20 * 1024 + 1,
        32 * 1024 - 1,
        32 * 1024,
        32 * 1024 + 1,
        64 * 1024 - 1,
        64 * 1024,
        64 * 1024 + 1,
        128 * 1024 - 1,
        128 * 1024,
};


static int intrAltEnableSS[FX3_INTR_ALT_SETTINGS_SS + 1] =
{
    0,  // Alt0  Not Use
    0,  // Alt1  MPS:8      Interval:1
    0,  // Alt2  MPS:8      Interval:2
    1,  // Alt3  MPS:8      Interval:4
    0,  // Alt4  MPS:8      Interval:8
    0,  // Alt5  MPS:8      Interval:16
    0,  // Alt6  MPS:16     Interval:1
    0,  // Alt7  MPS:16     Interval:2
    1,  // Alt8  MPS:16     Interval:4
    0,  // Alt9  MPS:16     Interval:8
    0,  // Alt10 MPS:16     Interval:16
    0,  // Alt11 MPS:32     Interval:1
    0,  // Alt12 MPS:32     Interval:2
    1,  // Alt13 MPS:32     Interval:4
    0,  // Alt14 MPS:32     Interval:8
    0,  // Alt15 MPS:32     Interval:16
    0,  // Alt16 MPS:64     Interval:1
    0,  // Alt17 MPS:64     Interval:2
    1,  // Alt18 MPS:64     Interval:4
    0,  // Alt19 MPS:64     Interval:8
    0,  // Alt20 MPS:64     Interval:16
    0,  // Alt21 MPS:128    Interval:1
    0,  // Alt22 MPS:128    Interval:2
    1,  // Alt23 MPS:128    Interval:4
    0,  // Alt24 MPS:128    Interval:8
    0,  // Alt25 MPS:128    Interval:16
    0,  // Alt26 MPS:256    Interval:1
    0,  // Alt27 MPS:256    Interval:2
    1,  // Alt28 MPS:256    Interval:4
    0,  // Alt29 MPS:256    Interval:8
    0,  // Alt30 MPS:256    Interval:16
    0,  // Alt31 MPS:512    Interval:1
    0,  // Alt32 MPS:512    Interval:2
    1,  // Alt33 MPS:512    Interval:4
    0,  // Alt34 MPS:512    Interval:8
    0,  // Alt35 MPS:512    Interval:16
    0,  // Alt36 MPS:1024   Interval:1
    0,  // Alt37 MPS:1024   Interval:2
    1,  // Alt38 MPS:1024   Interval:4
    0,  // Alt39 MPS:1024   Interval:8
    0,  // Alt40 MPS:1024   Interval:16
};


static int intrAltEnableHS[FX3_INTR_ALT_SETTINGS_HS + 1] =
{
    0,  // Alt0  Not Use
    0,  // Alt1  MPS:8      Interval:1
    0,  // Alt2  MPS:8      Interval:2
    1,  // Alt3  MPS:8      Interval:4
    0,  // Alt4  MPS:8      Interval:8
    0,  // Alt5  MPS:8      Interval:16
    0,  // Alt6  MPS:16     Interval:1
    0,  // Alt7  MPS:16     Interval:2
    1,  // Alt8  MPS:16     Interval:4
    0,  // Alt9  MPS:16     Interval:8
    0,  // Alt10 MPS:16     Interval:16
    0,  // Alt11 MPS:32     Interval:1
    0,  // Alt12 MPS:32     Interval:2
    1,  // Alt13 MPS:32     Interval:4
    0,  // Alt14 MPS:32     Interval:8
    0,  // Alt15 MPS:32     Interval:16
    0,  // Alt16 MPS:64     Interval:1
    0,  // Alt17 MPS:64     Interval:2
    1,  // Alt18 MPS:64     Interval:4
    0,  // Alt19 MPS:64     Interval:8
    0,  // Alt20 MPS:64     Interval:16
    0,  // Alt21 MPS:128    Interval:1
    0,  // Alt22 MPS:128    Interval:2
    1,  // Alt23 MPS:128    Interval:4
    0,  // Alt24 MPS:128    Interval:8
    0,  // Alt25 MPS:128    Interval:16
    0,  // Alt26 MPS:256    Interval:1
    0,  // Alt27 MPS:256    Interval:2
    1,  // Alt28 MPS:256    Interval:4
    0,  // Alt29 MPS:256    Interval:8
    0,  // Alt30 MPS:256    Interval:16
    0,  // Alt31 MPS:512    Interval:1
    0,  // Alt32 MPS:512    Interval:2
    1,  // Alt33 MPS:512    Interval:4
    0,  // Alt34 MPS:512    Interval:8
    0,  // Alt35 MPS:512    Interval:16
    0,  // Alt36 MPS:1024   Interval:1
    0,  // Alt37 MPS:1024   Interval:2
    1,  // Alt38 MPS:1024   Interval:4
    0,  // Alt39 MPS:1024   Interval:8
    0,  // Alt40 MPS:1024   Interval:16
};


static int intrAltEnableFS[FX3_INTR_ALT_SETTINGS_FS + 1] =
{
    0,  // Alt0  Not Use
    1,  // Alt1  MPS:8      Interval:1
    0,  // Alt2  MPS:8      Interval:2
    0,  // Alt3  MPS:8      Interval:4
    0,  // Alt4  MPS:8      Interval:8
    0,  // Alt5  MPS:8      Interval:255
    1,  // Alt6  MPS:16     Interval:1
    0,  // Alt7  MPS:16     Interval:2
    0,  // Alt8  MPS:16     Interval:4
    0,  // Alt9  MPS:16     Interval:8
    0,  // Alt10 MPS:16     Interval:255
    1,  // Alt11 MPS:32     Interval:1
    0,  // Alt12 MPS:32     Interval:2
    0,  // Alt13 MPS:32     Interval:4
    0,  // Alt14 MPS:32     Interval:8
    0,  // Alt15 MPS:32     Interval:255
    1,  // Alt16 MPS:64     Interval:1
    0,  // Alt17 MPS:64     Interval:2
    0,  // Alt18 MPS:64     Interval:4
    0,  // Alt19 MPS:64     Interval:8
    0,  // Alt20 MPS:64     Interval:255
};

static int GetIntrAltEnable(uint8_t speed, int index)
{
    int enable = 0;

    switch (speed)
    {
        case FX3_DEVICE_FULL_SPEED:     enable = intrAltEnableFS[index];    break;
        case FX3_DEVICE_HIGH_SPEED:     enable = intrAltEnableHS[index];    break;
        case FX3_DEVICE_SUPER_SPEED:    enable = intrAltEnableSS[index];    break;
        default:                                                            break;
    }

    return enable;
}

enum
{
    TRANSFER_BUFFER_SIZE        = 128 * 1024,
    INTR_TEST_VALUES_COUNT      = 20,
};

static NN_ALIGNAS(4096) uint8_t g_Buffer[TRANSFER_BUFFER_SIZE];

static void PerformIntrTransferTest(uint8_t altSetting)
{
    Fx3TestDataRarameters fx3TestDataParameters;

    FX3_LOG("TRB: head -> tail\n");
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_3_IN,   64 * 1024);
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_3_OUT,  64 * 1024);

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_INTR_3, 0);
    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_INTR_3, altSetting);

    // Fx3TransferData will change the  test params based on Fx3SetMaxTransfer
    // print new params before calling Fx3TransferData
    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress   = FX3_ENDPOINT_INTR_3_IN;
    fx3TestDataParameters.entries           = INTR_TEST_VALUES_COUNT;
    fx3TestDataParameters.dataSeed          = 0xffffeeee;

    for (int i = 0; i < INTR_TEST_VALUES_COUNT; i++)
    {
        fx3TestDataParameters.params[i].requestBytes   = intrTestValues[i];
        fx3TestDataParameters.params[i].transferBytes  = intrTestValues[i];
    }

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);

    // Fx3TransferData will change the  test params based on Fx3SetMaxTransfer
    // print new params before calling Fx3TransferData
    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress   = FX3_ENDPOINT_INTR_3_OUT;
    fx3TestDataParameters.entries           = INTR_TEST_VALUES_COUNT;
    fx3TestDataParameters.dataSeed          = 0xffffeeee;

    for (int i = 0; i < INTR_TEST_VALUES_COUNT; i++)
    {
        fx3TestDataParameters.params[i].requestBytes   = intrTestValues[i];
        fx3TestDataParameters.params[i].transferBytes  = intrTestValues[i];
    }

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);

    FX3_LOG("TRB: head -> middle -> tail\n");
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_3_IN,   128 * 1024);
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_3_OUT,  128 * 1024);

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_INTR_3, 0);
    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_INTR_3, altSetting);

    // Fx3TransferData will change the  test params based on Fx3SetMaxTransfer
    // print new params before calling Fx3TransferData
    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress   = FX3_ENDPOINT_INTR_3_IN;
    fx3TestDataParameters.entries           = INTR_TEST_VALUES_COUNT;
    fx3TestDataParameters.dataSeed          = 0xffffeeee;

    for (int i = 0; i < INTR_TEST_VALUES_COUNT; i++)
    {
        fx3TestDataParameters.params[i].requestBytes   = intrTestValues[i];
        fx3TestDataParameters.params[i].transferBytes  = intrTestValues[i];
    }

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);

    // Fx3TransferData will change the  test params based on Fx3SetMaxTransfer
    // print new params before calling Fx3TransferData
    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress   = FX3_ENDPOINT_INTR_3_OUT;
    fx3TestDataParameters.entries           = INTR_TEST_VALUES_COUNT;
    fx3TestDataParameters.dataSeed          = 0xffffeeee;

    for (int i = 0; i < INTR_TEST_VALUES_COUNT; i++)
    {
        fx3TestDataParameters.params[i].requestBytes   = intrTestValues[i];
        fx3TestDataParameters.params[i].transferBytes  = intrTestValues[i];
    }

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);
}

class HsFx3IntrTest : public ::testing::TestWithParam<Fx3DeviceMode>{};

INSTANTIATE_TEST_CASE_P(DeviceModes, HsFx3IntrTest, ::testing::ValuesIn(deviceModes));


///////////////////////////////////////////////////////////////////////////////
TEST_P(HsFx3IntrTest, VariousBlockSizes)
{
    ASSERT_TRUE(IsFx3Initialized());

    Fx3DeviceMode           fx3DeviceMode;

    fx3DeviceMode.connectSpeed              = GetParam().connectSpeed;
    fx3DeviceMode.maxPacketSize             = GetParam().maxPacketSize;
    fx3DeviceMode.interfaceBitmap           = GetParam().interfaceBitmap;

    Fx3SetDeviceMode(&fx3DeviceMode, WAIT_SECONDS_FOR_ATTACH);

    uint8_t numAltSetting = 0;
    switch (GetParam().connectSpeed)
    {
        case FX3_DEVICE_FULL_SPEED:     numAltSetting = FX3_INTR_ALT_SETTINGS_FS;   break;
        case FX3_DEVICE_HIGH_SPEED:     numAltSetting = FX3_INTR_ALT_SETTINGS_HS;   break;
        case FX3_DEVICE_SUPER_SPEED:    numAltSetting = FX3_INTR_ALT_SETTINGS_SS;   break;
        default:                                                                    break;
    }

    for (uint8_t setting = 0; setting <= numAltSetting; setting++)
    {
        if (GetIntrAltEnable(GetParam().connectSpeed, setting))
        {
            PerformIntrTransferTest(setting);
        }
    }

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_INTR_3, 0);
}


} // hs
} // usb
} // nnt

