﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Fx3Methods/Fx3.h"

namespace nnt {
namespace usb {
namespace hs {

const Fx3DeviceMode deviceModes[] =
{
    {FX3_DEVICE_FULL_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_1},
    {FX3_DEVICE_HIGH_SPEED, FX3_CTRL_MPS_64, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_1},
    {FX3_DEVICE_SUPER_SPEED, FX3_CTRL_MPS_512, FX3_INTERFACE_CTRL | FX3_INTERFACE_BULK_1},
};

const int bulkTestValues[] =
{
    1,
    2,
    3,
    511,
    512,
    513,
    4095,
    4096,
    4097,
    20 * 1024 - 1,
    20 * 1024,
    20 * 1024 + 1,
    32 * 1024 - 1,
    32 * 1024,
    32 * 1024 + 1,
    64 * 1024 - 1,
    64 * 1024,
    64 * 1024 + 1,
    128 * 1024 - 1,
    128 * 1024,
    128 * 1024 + 1,
    256 * 1024 - 1,
    256 * 1024,
    256 * 1024 + 1,
    512 * 1024 - 1,
    512 * 1024,
    512 * 1024 + 1,
    1024 * 1024 - 1,
    1024 * 1024,
    1024 * 1024 + 1,
    2048 * 1024 - 1,
    2048 * 1024
};

enum
{
    TRANSFER_BUFFER_SIZE        = 2048 * 1024,
    BULK_TEST_VALUES_COUNT      = 32,
};

static NN_ALIGNAS(4096) uint8_t g_Buffer[TRANSFER_BUFFER_SIZE];

static void PerformBulkShortTest(uint8_t altSetting)
{
    Fx3TestDataRarameters fx3TestDataParameters;

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_BULK_1, altSetting);

    std::memset(&fx3TestDataParameters, 0, sizeof(Fx3TestDataRarameters));

    fx3TestDataParameters.endpointAddress   = FX3_ENDPOINT_BULK_1_IN;
    fx3TestDataParameters.entries           = BULK_TEST_VALUES_COUNT;
    fx3TestDataParameters.dataSeed          = 0xffffeeee;

    for (int i = 0; i < BULK_TEST_VALUES_COUNT; i++)
    {
        fx3TestDataParameters.params[i].requestBytes   = TRANSFER_BUFFER_SIZE;
        fx3TestDataParameters.params[i].transferBytes  = bulkTestValues[i];
    }

    Fx3TransferData(&fx3TestDataParameters, g_Buffer);
}

class HsFx3BulkShortTest : public ::testing::TestWithParam<Fx3DeviceMode>{};

INSTANTIATE_TEST_CASE_P(DeviceModes, HsFx3BulkShortTest, ::testing::ValuesIn(deviceModes));

///////////////////////////////////////////////////////////////////////////////
TEST_P(HsFx3BulkShortTest, VariousBlockSizes)
{
    ASSERT_TRUE(IsFx3Initialized());

    Fx3DeviceMode           fx3DeviceMode;

    fx3DeviceMode.connectSpeed              = GetParam().connectSpeed;
    fx3DeviceMode.maxPacketSize             = GetParam().maxPacketSize;
    fx3DeviceMode.interfaceBitmap           = GetParam().interfaceBitmap;

    Fx3SetDeviceMode(&fx3DeviceMode, WAIT_SECONDS_FOR_ATTACH);

    uint8_t numAltSetting = 0;
    switch (GetParam().connectSpeed)
    {
        case FX3_DEVICE_FULL_SPEED:     numAltSetting = FX3_BULK_ALT_SETTINGS_FS;   break;
        case FX3_DEVICE_HIGH_SPEED:     numAltSetting = FX3_BULK_ALT_SETTINGS_HS;   break;
        case FX3_DEVICE_SUPER_SPEED:    numAltSetting = FX3_BULK_ALT_SETTINGS_SS;   break;
        default:                                                                    break;
    }

    for (uint8_t setting = 1; setting <= numAltSetting; setting++)
    {
        PerformBulkShortTest(setting);
    }

    Fx3SetAltSetting(FX3_INTERFACE_NUMBER_BULK_1, 0);
}

} // hs
} // usb
} // nnt

