﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/nntest.h>
#include <nnt/usb/testUsb_Fx3Utility.h>

#include <nn/usb/usb_Host.h>
#include <nn/usb/usb_Result.public.h>

#include "../Common/UsbTestSuiteFx3Firmware.h"

namespace nnt {
namespace usb {
namespace hs {


#define FX3_LOG NN_LOG

///////////////////////////////////////////////////////////////////////////////
// Attach timeout

enum
{
    WAIT_SECONDS_FOR_ATTACH = 30,
};

///////////////////////////////////////////////////////////////////////////////
// FX3 internal ctrl buffer size
enum
{
    FX3_CTRL_SIZE                   = 0x10000,
};


///////////////////////////////////////////////////////////////////////////////
// FX3 max transfer size
enum
{
    FX3_MAX_TRANSFER_SIZE            = 2 * 1024 * 1024,
};


///////////////////////////////////////////////////////////////////////////////
// Vendor request definitions (commands to FW)
enum
{
    FX3_REQUEST_RESET               = 0x00,
    FX3_REQUEST_DEVICE_MODE         = 0x01,
    FX3_REQUEST_TRANSFER_DATA       = 0x02,
    FX3_REQUEST_PRINT_STRING        = 0x03,
    FX3_REQUEST_DEVICE_SPEED        = 0x04,
    FX3_REQUEST_DATA                = 0x05,
    FX3_REQUEST_ERROR               = 0x06,
    FX3_REQUEST_FIRMWARE_VERSION    = 0xff,
};

///////////////////////////////////////////////////////////////////////////////
// Initialize HS client, wait for FX3 VID PID ctrl interface to attach for
// specified number of seconds
enum
{
    FX3_VID                         = 0xffff,
    FX3_PID                         = 0xfefe,
};

void Fx3Initialize(int seconds);

bool IsFx3Initialized();


///////////////////////////////////////////////////////////////////////////////
// Finalize HS client
void Fx3Finalize();


///////////////////////////////////////////////////////////////////////////////
// Reset FX3 firmware and wait for FX3 VID PID ctrl interface to attach for
// specified number of seconds
void Fx3Reset(int seconds);


///////////////////////////////////////////////////////////////////////////////
// Set device mode, each mode bit when asserted causes interface to be
// supported by FX3 firmware. Each interface has pair of in and out endpoints
// as defined here.
enum
{
    FX3_DEVICE_FULL_SPEED           = 0x00,
    FX3_DEVICE_HIGH_SPEED           = 0x01,
    FX3_DEVICE_SUPER_SPEED          = 0x02,

    FX3_CTRL_MPS_8                  = 0x08,
    FX3_CTRL_MPS_16                 = 0x10,
    FX3_CTRL_MPS_32                 = 0x20,
    FX3_CTRL_MPS_64                 = 0x40,
    FX3_CTRL_MPS_512                = 0x09, // USB 3.0

    FX3_INTERFACE_CTRL              = 0x00,
    FX3_INTERFACE_BULK_1            = 0x01,
    FX3_INTERFACE_BULK_2            = 0x02,
    FX3_INTERFACE_INTR_3            = 0x04,
    FX3_INTERFACE_INTR_4            = 0x08,
    FX3_INTERFACE_ISOC_5            = 0x10,
    FX3_INTERFACE_ISOC_6            = 0x20,
};



///////////////////////////////////////////////////////////////////////////////
// Fixed interface numbers and endpoint addressed to use at runtime
enum
{
    FX3_INTERFACE_NUMBER_CTRL       = 0x00,
    FX3_INTERFACE_NUMBER_BULK_1     = 0x01,
    FX3_INTERFACE_NUMBER_BULK_2     = 0x02,
    FX3_INTERFACE_NUMBER_INTR_3     = 0x03,
    FX3_INTERFACE_NUMBER_INTR_4     = 0x04,
    FX3_INTERFACE_NUMBER_ISOC_5     = 0x05,
    FX3_INTERFACE_NUMBER_ISOC_6     = 0x06,
    FX3_INTERFACE_MAX               = (FX3_INTERFACE_NUMBER_ISOC_6 + 1),

    FX3_ENDPOINT_CTRL_0_IN          = 0x80,
    FX3_ENDPOINT_CTRL_0_OUT         = 0x00,
    FX3_ENDPOINT_BULK_1_IN          = 0x81,
    FX3_ENDPOINT_BULK_1_OUT         = 0x01,
    FX3_ENDPOINT_BULK_2_IN          = 0x82,
    FX3_ENDPOINT_BULK_2_OUT         = 0x02,
    FX3_ENDPOINT_INTR_3_IN          = 0x83,
    FX3_ENDPOINT_INTR_3_OUT         = 0x03,
    FX3_ENDPOINT_INTR_4_IN          = 0x84,
    FX3_ENDPOINT_INTR_4_OUT         = 0x04,
    FX3_ENDPOINT_ISOC_5_IN          = 0x85,
    FX3_ENDPOINT_ISOC_5_OUT         = 0x05,
    FX3_ENDPOINT_ISOC_6_IN          = 0x86,
    FX3_ENDPOINT_ISOC_6_OUT         = 0x06,
    FX3_ENDPOINT_MAX                = 12,
};


// number of alt settings on FX3 per interface type and speed
enum
{
    FX3_BULK_ALT_SETTINGS_FS        = 4,
    FX3_BULK_ALT_SETTINGS_HS        = 1,
    FX3_BULK_ALT_SETTINGS_SS        = 1,
    FX3_INTR_ALT_SETTINGS_FS        = 20,
    FX3_INTR_ALT_SETTINGS_HS        = 40,
    FX3_INTR_ALT_SETTINGS_SS        = 40,
};

struct Fx3DeviceMode
{
    uint8_t connectSpeed;
    uint8_t maxPacketSize;
    uint8_t interfaceBitmap;
};


void Fx3SetDeviceMode(Fx3DeviceMode *pFx3DeviceMode, int seconds);

///////////////////////////////////////////////////////////////////////////////
// Set max transfer for TRB ring tests
void Fx3SetMaxTransfer(uint8_t endpointAddress, uint32_t maxTransferSize);


///////////////////////////////////////////////////////////////////////////////
// Set alternate setting on interface
void Fx3SetAltSetting(uint8_t interfaceNumber, uint8_t altSetting);


///////////////////////////////////////////////////////////////////////////////
// Transfer data per endpoint address.

enum
{
    FX3_TRANSFER_MAX_ENTRIES    = 63,
};

struct transfer_params
{

    uint32_t        requestBytes;
    uint32_t        transferBytes;

};


struct Fx3TestDataRarameters
{
    uint8_t         endpointAddress;
    uint8_t         entries;
    uint16_t        pad;
    uint32_t        dataSeed;

    transfer_params params[FX3_TRANSFER_MAX_ENTRIES];
};

void Fx3TransferData(Fx3TestDataRarameters *pFx3TestDataParams, uint8_t *pData);


///////////////////////////////////////////////////////////////////////////////
// Loopback data
struct Fx3LoopbackParameters
{
    uint8_t         endpointAddressTx;
    uint8_t        *pTxData;
    uint8_t        *pTxThreadStack;
    uint32_t        txThreadStackSize;
    uint32_t        txBytesTotal;
    uint32_t        txLargestBuffer;

    uint8_t         endpointAddressRx;
    uint8_t        *pRxData;
    uint8_t        *pRxThreadStack;
    uint32_t        rxThreadStackSize;
    uint32_t        rxBytesTotal;

    uint8_t         speed;
    uint8_t         altSetting;

    Fx3TestDataRarameters fx3TestDataParams;
};

void Fx3LoopbackData(Fx3LoopbackParameters *pFx3LoopbackParameters);


///////////////////////////////////////////////////////////////////////////////
// Set clear stall
void Fx3ClrStall(uint8_t endpointAddress);


///////////////////////////////////////////////////////////////////////////////
// Check firmware version
enum
{
    FX3_EXPECT_FIRMWARE_VERSION     = 0x00010006,
};

void Fx3CheckFirmwareVersion();


///////////////////////////////////////////////////////////////////////////////
// Print string on FX3 console
void Fx3PrintString(uint8_t *pStringData, uint32_t size);

} // hs
} // usb
} // nnt
