﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Windows.h>
#include <cstdio>
#include <iostream>

#include "RelayControl.h"

namespace nnt { namespace usb {


namespace {

// These bit masks are a function of custom wiring on our board
const uint8_t SerialBridgeResetMask = 0x80;
const uint8_t VbusMask              = 0x40;
const uint8_t Fx3CoreResetMask      = 0x20;

// Device status
enum
{
    FT_OK,
    FT_INVALID_HANDLE,
    FT_DEVICE_NOT_FOUND,
    FT_DEVICE_NOT_OPENED,
    FT_IO_ERROR,
    FT_INSUFFICIENT_RESOURCES,
    FT_INVALID_PARAMETER,
    FT_INVALID_BAUD_RATE,
    FT_DEVICE_NOT_OPENED_FOR_ERASE,
    FT_DEVICE_NOT_OPENED_FOR_WRITE,
    FT_FAILED_TO_WRITE_DEVICE,
    FT_EEPROM_READ_FAILED,
    FT_EEPROM_WRITE_FAILED,
    FT_EEPROM_ERASE_FAILED,
    FT_EEPROM_NOT_PRESENT,
    FT_EEPROM_NOT_PROGRAMMED,
    FT_INVALID_ARGS,
    FT_NOT_SUPPORTED,
    FT_OTHER_ERROR,
    FT_DEVICE_LIST_NOT_READY,
};

typedef PVOID FT_HANDLE;
typedef ULONG FT_STATUS;

} // namespace {


// ftd2xx Library API
extern "C"
{
FT_STATUS WINAPI FT_Open(int deviceNumber, FT_HANDLE *pHandle);
FT_STATUS WINAPI FT_SetBaudRate(FT_HANDLE ftHandle, ULONG BaudRate);
FT_STATUS WINAPI FT_SetTimeouts(FT_HANDLE ftHandle, ULONG ReadTimeout, ULONG WriteTimeout);
FT_STATUS WINAPI FT_SetBitMode(FT_HANDLE ftHandle, UCHAR ucMask, UCHAR ucEnable);
FT_STATUS WINAPI FT_Write(FT_HANDLE ftHandle, LPVOID lpBuffer, DWORD dwBytesToWrite, LPDWORD lpBytesWritten);
FT_STATUS WINAPI FT_Close(FT_HANDLE ftHandle);
} // extern "C"


#define NNT_FX3_BREAK_UPON_ERROR(attemptedMethodCall)                            \
    if (!((ftStatus=(attemptedMethodCall)) == FT_OK))                            \
    {                                                                            \
        printf("FX3 RelayControl: %s unsuccessful code %d in func-%s @ line-%d.\n",  \
               #attemptedMethodCall, ftStatus,__FUNCTION__,__LINE__);            \
        break;                                                                   \
    }


int DoRelayControl(RelayControlCommand command)
{
    FT_STATUS ftStatus = FT_OK;
    FT_HANDLE ftHandle = 0;


    // Do the command sequence
    do
    {
        DWORD bytesWritten = 0;
        uint8_t writeBuffer = 0;

        NNT_FX3_BREAK_UPON_ERROR(FT_Open(0, &ftHandle));
        NNT_FX3_BREAK_UPON_ERROR(FT_SetBaudRate(ftHandle, 115200));
        NNT_FX3_BREAK_UPON_ERROR(FT_SetTimeouts(ftHandle, 5000, 1000));
        NNT_FX3_BREAK_UPON_ERROR(FT_SetBitMode(ftHandle, 0xff, 4));

        // Do startup sequence, assume no prior state.
        if(command == RelayControlCommand_BootUp)
        {
            printf("FX3 RelayControl: Begin with power off...\n");
            writeBuffer = 0;
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
            Sleep(5000);

            printf("FX3 RelayControl: Cycling resets...\n");
            writeBuffer = Fx3CoreResetMask | SerialBridgeResetMask;
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
            Sleep(500);
            writeBuffer = 0;
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
            Sleep(500);

            printf("FX3 RelayControl: Powering up FX3, resets asserted...\n");
            writeBuffer = Fx3CoreResetMask | SerialBridgeResetMask | VbusMask;
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
            Sleep(500);

            printf("FX3 RelayControl: Releasing CY7C65215 from reset...\n");
            writeBuffer &= ~SerialBridgeResetMask;
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
            Sleep(500);

            printf("FX3 RelayControl: Releasing FX3 core from reset...\n");
            writeBuffer &= ~Fx3CoreResetMask;
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
            Sleep(500);
        }
        // Do shutdown sequence.
        else if(command == RelayControlCommand_ShutDown)
        {
            writeBuffer = 0;
            printf("FX3 RelayControl: Powering Off...\n");
            NNT_FX3_BREAK_UPON_ERROR(FT_Write(ftHandle, &writeBuffer, sizeof(writeBuffer), &bytesWritten));
        }

    }while(NN_STATIC_CONDITION(false));

    if(ftHandle != 0)
    {
        FT_Close(ftHandle);
    }

    return (ftStatus == FT_OK) ? 0 : -1;
}

}} // namespace nnt { namespace usb {
