﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Windows.h>
#include <winusb.h>
#include <memory>
#pragma warning(disable: 4668)
#include <setupapi.h>

// ドライバの GUID
const GUID TARGET_GUID[8] = {
                             { 0x61A954B0, 0xB7F1, 0x4333,{ 0xAF, 0xBD, 0x6A, 0x41, 0x76, 0x9C, 0xE2, 0x66 }},
                             { 0xF30B76BE, 0xE93A, 0x40F2,{ 0xBE, 0x77, 0x28, 0xC8, 0x15, 0xE3, 0xAE, 0x3E }},
                             { 0xB9E90F40, 0x2C5A, 0x41E5,{ 0x98, 0x86, 0x92, 0x20, 0xF4, 0x4D, 0x9A, 0x81 }},
                             { 0xB5700803, 0x2A52, 0x479A,{ 0xB3, 0x8E, 0xEA, 0x5F, 0xAF, 0xE5, 0x34, 0xA0 }},
                             { 0x1B7AFAA9, 0x4685, 0x4AFD,{ 0xB6, 0xB4, 0x95, 0x4B, 0xD2, 0x83, 0xD4, 0xE3 }},
                             { 0x060C5A4D, 0x4544, 0x4909,{ 0xB0, 0xEE, 0x60, 0x46, 0x89, 0x43, 0x36, 0x82 }},
                             { 0xBA79FC20, 0xEE1A, 0x4921,{ 0xA9, 0xAA, 0xE1, 0x3E, 0x6E, 0x8C, 0x2A, 0xAB }},
                             { 0x0BA26FAA, 0x5F70, 0x40C4,{ 0x91, 0xF5, 0x64, 0xF5, 0x3D, 0x26, 0xC5, 0x01 }}
                            };

struct UsbDeviceInfo
{
    HANDLE deviceHandle;
    WINUSB_INTERFACE_HANDLE winUsbHandle;
    USHORT interfaceIndex;
    unsigned char pipeBulkIn;
    unsigned char pipeBulkOut;
};

bool SearchDevice(UsbDeviceInfo* pOutDeviceInfo, UCHAR interfaceNumber) NN_NOEXCEPT
{
    HANDLE deviceHandle = NULL;
    WINUSB_INTERFACE_HANDLE winUsbHandle;
    HDEVINFO deviceInfoHandle = SetupDiGetClassDevs(&TARGET_GUID[interfaceNumber], NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

    if (!deviceInfoHandle)
    {
        NN_LOG("[Host PC] device information was not found (%d)\n", GetLastError());
        return false;
    }

    SP_DEVICE_INTERFACE_DATA interfaceData = { 0 };
    interfaceData.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);
    DWORD dwIndex = 0;

    while (NN_STATIC_CONDITION(true))
    {
        if (SetupDiEnumDeviceInterfaces(deviceInfoHandle, NULL, &TARGET_GUID[interfaceNumber], dwIndex, &interfaceData))
            break;
        else
        {
            if (GetLastError() == ERROR_NO_MORE_ITEMS)
                return false;
        }

        dwIndex++;
    }

    ULONG interfaceDetailDataLength = 256;
    if (!SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, &interfaceData, NULL, 0, &interfaceDetailDataLength, NULL))
    {
        if (GetLastError() != ERROR_INSUFFICIENT_BUFFER)
        {
            NN_LOG("[Host PC] ERROR_INSUFFICIENT_BUFFER\n");
            return false;
        }
    }

    ::std::unique_ptr<char[]> buffer(new char[interfaceDetailDataLength]);
    auto pIfDetailData = reinterpret_cast<PSP_DEVICE_INTERFACE_DETAIL_DATA>(buffer.get());
    pIfDetailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

    if (!SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, &interfaceData, pIfDetailData,interfaceDetailDataLength, &interfaceDetailDataLength, NULL))
    {
        NN_LOG("[Host PC] SetupDiGetDeviceInterfaceDetail\n");
        return false;
    }

    deviceHandle = CreateFile(pIfDetailData->DevicePath,GENERIC_WRITE | GENERIC_READ,FILE_SHARE_WRITE | FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL);
    if (deviceHandle == INVALID_HANDLE_VALUE)
    {
        NN_LOG("Error %x \n", GetLastError());
        return false;
    }

    if (!WinUsb_Initialize(deviceHandle, &winUsbHandle))
    {
        CloseHandle(deviceHandle);
        NN_LOG("[Host PC] WinUsb_Initialize\n");
        return false;
    }

    uint8_t pipeBulkIn = 0, pipeBulkOut = 0;
    USB_INTERFACE_DESCRIPTOR interfaceDescriptor;
    WINUSB_INTERFACE_HANDLE interfaceHandle = winUsbHandle;
    WinUsb_QueryInterfaceSettings(winUsbHandle, interfaceNumber, &interfaceDescriptor);

    for (UCHAR i=0; ;i++)
    {
        WinUsb_QueryInterfaceSettings(interfaceHandle, 0, &interfaceDescriptor);
        if (interfaceDescriptor.bInterfaceNumber == interfaceNumber)
        {
            winUsbHandle = interfaceHandle;
            break;
        }
        if (!WinUsb_GetAssociatedInterface(winUsbHandle, i, &interfaceHandle))
        {
            NN_LOG("[Host PC] Interface number %d not found\n", interfaceNumber);
            break;
        }
    }

    for (int i = 0; i < interfaceDescriptor.bNumEndpoints; i++)
    {
        WINUSB_PIPE_INFORMATION pipeInfo;
        while (!WinUsb_QueryPipe(winUsbHandle, 0, static_cast<uint8_t>(i), &pipeInfo))
            Sleep(10);

        if (pipeInfo.PipeType == UsbdPipeTypeBulk && USB_ENDPOINT_DIRECTION_IN(pipeInfo.PipeId))
        {
            pipeBulkIn = pipeInfo.PipeId;
        }
        else if (pipeInfo.PipeType == UsbdPipeTypeBulk && USB_ENDPOINT_DIRECTION_OUT(pipeInfo.PipeId))
        {
            pipeBulkOut = pipeInfo.PipeId;
        }

        if (!WinUsb_ResetPipe(
                                interfaceHandle,
                                pipeInfo.PipeId
                                ))
        {
            NN_LOG("WinUsb_ResetPipe failed (%d)\n", GetLastError());
        }
    }

    if (!pipeBulkIn)
    {
        NN_LOG("[Host PC] BULK IN pipe was not found\n");
        return false;
    }

    if (!pipeBulkOut)
    {
        NN_LOG("[Host PC] BULK OUT pipe was not found\n");
        return false;
    }

    pOutDeviceInfo->interfaceIndex = interfaceDescriptor.bInterfaceNumber;
    pOutDeviceInfo->deviceHandle = deviceHandle;
    pOutDeviceInfo->winUsbHandle = winUsbHandle;
    pOutDeviceInfo->pipeBulkIn = pipeBulkIn;
    pOutDeviceInfo->pipeBulkOut = pipeBulkOut;
    return true;
} // NOLINT(impl/function_size)

bool SendData(int dataSize, UsbDeviceInfo* pDeviceInfo) NN_NOEXCEPT
{
    std::vector<unsigned char> outBuffer(dataSize);

    NN_LOG("Sending %d bytes\n", dataSize);

    ULONG bytesTransferred = 0;

    if (!WinUsb_WritePipe(
                            pDeviceInfo->winUsbHandle,
                            pDeviceInfo->pipeBulkOut,
                            &outBuffer[0],
                            dataSize,
                            &bytesTransferred,
                            NULL
                            ))
    {
        NN_LOG("WinUsb_WritePipe failed (%d)\n", GetLastError());
        return false;
    }

    NN_LOG("bytesTransferred %d\n", bytesTransferred);

    return true;
}


UsbDeviceInfo info;

bool SetupHost()
{
    NN_LOG("Searching for device 0\n");
    while (!SearchDevice(&info, 0))
    {
        //FAIL() << "[Host PC] failed to find device.\n";
        Sleep(500);
        NN_LOG(".");
    }
    NN_LOG("\n");
    NN_LOG("Device Found\n");

    return true;
}

bool TeardownHost()
{
    WinUsb_Free(info.winUsbHandle);
    CloseHandle(info.deviceHandle);

    // Give time for device side to detach or else we might get a handle that
    // becomes invalid when we do next test. It would be really nice if we
    // can pull for handle to become invalid here :0
    Sleep(500);

    return true;
}

const int TRB_DATA_SIZE     = 64 * 1024;
const int BULK_TRBS_IN_RING = 64;


TEST(DsShortTransfer, 1TRB)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(64, &info));                       // Short on 1st TRB
    EXPECT_TRUE(TeardownHost());
}

TEST(DsShortTransfer, 2TRB)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(64, &info));                       // Short on 1st TRB
    EXPECT_TRUE(SendData(TRB_DATA_SIZE + 64, &info));       // Short on 2nd TRB
    EXPECT_TRUE(TeardownHost());
}

TEST(DsShortTransfer, 3TRB)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(64, &info));                       // Short on 1st TRB
    EXPECT_TRUE(SendData(TRB_DATA_SIZE + 64, &info));       // Short on 2nd TRB
    EXPECT_TRUE(SendData((TRB_DATA_SIZE * 2) + 64, &info)); // Short on 3rd TRB
    EXPECT_TRUE(TeardownHost());
}

TEST(DsShortTransfer, 2_LINKTRB)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(((BULK_TRBS_IN_RING - 2) * TRB_DATA_SIZE) + 64, &info));   // Short on 2 TRBs before link TRB
    EXPECT_TRUE(TeardownHost());
}

TEST(DsShortTransfer, 1_LINKTRB)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE) + 64, &info));   // Short on 2 TRBs before link TRB
    EXPECT_TRUE(TeardownHost());
}

TEST(DsShortTransfer, LINKTRB_1)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE), &info));        // Get to 1 TRB before link TRB
    EXPECT_TRUE(SendData(TRB_DATA_SIZE + 64, &info));                               // Short on 1st TRB after link TRB
    EXPECT_TRUE(TeardownHost());
}

TEST(DsShortTransfer, LINKTRB_2)
{
    EXPECT_TRUE(SetupHost());
    EXPECT_TRUE(SendData(((BULK_TRBS_IN_RING - 1) * TRB_DATA_SIZE), &info));        // Get to 1 TRB before link TRB
    EXPECT_TRUE(SendData((TRB_DATA_SIZE * 2) + 64, &info));                         // Short on 1st TRB after link TRB
    EXPECT_TRUE(TeardownHost());
}
